#ifndef FLUC_INCLUDE
#include "fluctuate.h"
#endif

#ifndef FLUC_MODELLIKE_INCLUDE
#include "fluc_modellike.h"
#endif

#ifdef DMALLOC_FUNC_CHECK
#include "/usr/local/include/dmalloc.h"
#endif

#define USETRUE false /* include Mary's true-theta/true-g tests */

/***************************************************************************
 *  ALPHA                                                                  *
 *  version 1.30. (c) Copyright 1986, 1991, 1992 by the University of      *
 *  Washington and Joseph Felsenstein.  Written by Joseph Felsenstein,     *
 *  Mary K. Kuhner and Jon A. Yamato, with some additional grunt work by   *
 *  Sean T. Lamont.  Permission is granted to copy and use this program    *
 *   provided no fee is charged for it and provided that this copyright    *
 *  notice is not removed.                                                 *
 *                                                                         *
 ***************************************************************************/

/************************************************************
   This file contains all the functions used in calculating
   P(D|theta), the model_likelihood.
 ***********************************************************/

extern FILE *thetafile, *simlog, *outfile;
extern long locus, numseq, totchains, numloci, numtrees, holding;
extern double watttheta, *mu_ratio, *theta_ratio;
extern tree *curtree;
extern option_struct *op;
extern boolean **sametree;

treerec ***sum;
double **growi, **theti, **lntheti, *fixed, *locuslike, **savethetai,
   xinterval;
long numfix;

/*************************************************************
 * zerocheck returns true if value is zero, and false otherwise */
boolean zerocheck(double value)
{
   if (value == 0.0) return(true);
   else return(false);
} /* zerocheck */

/******************************************************************
 * whatsign returns a 0 if value is zero, -1 if value is negative *
 * and 1 if value is positive.                                    */
long whatsign(double value)
{
   if (zerocheck(value)) return(0);
   if (value < 0.0) return(-1);
   else return(1);
} /* whatsign */

/******************************************************************
 *code for confidence intervals of truth:  set USETRUE to 0 if not*
 *desired.  Will bomb if truthfile does not exist.                */

#define CRITVALUE 3.00  /* 95% confidence limit for 2 df likelihood */

void truthmod(double th, double g, long chain, long lowcus)
/* pass locus=-1 for multilocus estimate, in which case chain is
ignored */
{
  FILE *truthfile;
  double true_theta, true_g, maxlike, truelike;

  truthfile = fopen("truthfile","r");

  fscanf(truthfile,"%lf\n",&true_theta);
  fscanf(truthfile,"%lf\n",&true_g);

  if(lowcus!=-1) { /* single locus */
    maxlike = model_llike(th,g,chain,lowcus);
    truelike = model_llike(true_theta,true_g,chain,lowcus);
    fprintf(outfile,"likelihood at truth is %e\n",truelike);
    if (maxlike - CRITVALUE <= truelike)
      fprintf(outfile,"single locus included\n");
    else fprintf(outfile,"single locus excluded\n");
  } else { /* multiple loci */
    maxlike = fluc_locus_llike(th,g);
    truelike = fluc_locus_llike(true_theta,true_g);
    fprintf(outfile,"multilocus likelihood at truth is %e\n",truelike);
    if (maxlike - CRITVALUE <= truelike)
      fprintf(outfile,"multiple locus INCLUDED\n");
    else fprintf(outfile,"multiple locus EXCLUDED\n");
  }
  fclose(truthfile);
} /* truthmod */


/**********************************************************************
 * init_modellike calculates the "fixed" values to be used in         *
 * computing theta estimates, and in general handles locally_global   *
 * variable initialization ONLY FOR NON-DYNAMIC ALLOCATED VARIABLES!!!*/
void init_modellike()
{
  double b, b1, bnum, bdenom;

  b = MINTHETA;
  b1 = exp(log(10.0) * (long)floor(log(b) / log(10.0) + 0.5));
  bdenom = (long)floor(1 / b1 + 0.5);
  bnum = (long)floor(b / b1 + 0.5);
  fixed = (double *)calloc(1,sizeof(double));
  numfix = 0;
  do {
    numfix++;
    fixed = (double *)realloc(fixed,numfix*sizeof(double));
    fixed[numfix - 1] = bnum/bdenom;
    if (fabs(bnum - 2.0) < epsilon)
      bnum = (long)floor(2.5 * bnum + 0.5);
    else
      bnum *= 2.0;
    if (fabs(bnum - 10.0) < epsilon) {
      bnum = 1.0;
      bdenom /= 10.0;
    }
  } while (bnum / bdenom <= MAXTHETA * (1.0 + epsilon));

}  /* init_modellike */

/**********************************************************************
 * model_alloc allocates space for variables that will be used over   *
 * more than one locus in calculating the model_likelihood.           *
 * ONLY CALL ONCE.                                                    */
void model_alloc()
{
  long i, j, k;

  init_modellike(); 


  sum = (treerec ***)calloc(numloci,sizeof(treerec **));
  sum[0] = (treerec **)calloc(numloci*(1+op->numchains[1]),sizeof(treerec *));
  for (i = 1; i < numloci; i++)
    sum[i] = sum[0] + i*(1+op->numchains[1]);
  sum[0][0] = 
    (treerec *)calloc(numloci*(1+op->numchains[1])*numtrees,sizeof(treerec));
  for (i = 0; i < numloci; i++)
     for(j = 0; j < (1+op->numchains[1]); j++)
        sum[i][j] = sum[0][0] + i*(1+op->numchains[1])*numtrees + j*numtrees;
  /* the rest of the sum array is allocated with each locus,
  as its size varies with numtips; this calloc is a placeholder */
  for (i = 0; i < numloci; i++)
     for (j = 0; j < 1+op->numchains[1]; j++)
        for (k = 0; k < numtrees; k++) {
          if(op->growthused) sum[i][j][k].kk = NULL;
          sum[i][j][k].kend = NULL;
        }

  locuslike = (double *)calloc(numloci+numfix,sizeof(double)); 

  growi = (double **)calloc(numloci,sizeof(double *));
  theti = (double **)calloc(numloci,sizeof(double *));
  lntheti = (double **)calloc(numloci,sizeof(double *));
  savethetai = (double **)calloc(numloci,sizeof(double *));
  lntheti[0] = (double *)
     calloc(numloci*(totchains+numfix+1),sizeof(double));
  growi[0] = (double *)
     calloc(numloci*(totchains+numfix+1),sizeof(double));
  theti[0] = (double *)
     calloc(numloci*(totchains+numfix+1),sizeof(double));
  savethetai[0] = (double *)calloc(numloci*totchains,sizeof(double));
  for (i = 1; i < numloci; i++) {
     growi[i] = growi[0] + i*(totchains+numfix+1);
     theti[i] = theti[0] + i*(totchains+numfix+1);
     lntheti[i] = lntheti[0] + i*(totchains+numfix+1);
     savethetai[i] = savethetai[0] + i*totchains;
  }
}  /* model_alloc */

/***********************************************************
 * scoretree saves the values necessary for evaluating the *
 * model_likelihood for a single tree in the 'sum' array   */
void scoretree(long chain)
{
  tlist *t;
  long i, refchain, chaintype;
  double temp;
   
  refchain = REF_CHAIN(chain);
  chaintype = TYPE_CHAIN(chain);

  if (!op->growthused)
     sum[locus][refchain][op->numout[chaintype] - 1].kend[0] = 0.0;

  i = 0;
  temp = 0.0;
  t = curtree->tymelist;
  while (t != NULL) {
    if (!op->growthused) {
       sum[locus][refchain][op->numout[chaintype] - 1].kend[0] +=
          (double)(t->numbranch * (t->numbranch - 1)) 
          * (t->age - t->eventnode->tyme);
    } else {
       if (t->numbranch-1 != 0) {
          sum[locus][refchain][op->numout[chaintype] - 1].kk[i] =
          t->numbranch * (t->numbranch - 1);
          sum[locus][refchain][op->numout[chaintype] - 1].kend[i] = t->age;
          if (t->age > xinterval) xinterval = t->age;
       }
    }
    temp += (double)(t->numbranch * (t->numbranch - 1))
           * (t->age - t->eventnode->tyme);
    i++;
    t = t->succ;
  }
  if (temp == 0.0) fprintf(ERRFILE,"WARNING:  Tree has become length zero\n");
}  /* scoretree */

/********************************************************************
 * fluc_interval_llike returns the Ln(likelihood) of an interval    *
 * within  a tree; for a theta = "theta", and a growth = "growth".  *
 * starttyme = start of the interval                                *
 * endtyme   = end of the interval                                  *
 * kk        = function of number of lineages contained in interval *
 *                                                                  *
 * An approximation is used for growth values close to zero due to  *
 * machine precision.                                               */
double fluc_interval_llike(double theta, double linv_theta, double growth, 
   double starttyme, double endtyme, long kk)
{
double answ;

if (fabs(growth) < LHOP_TOLERANCE)
   answ = linv_theta +
      log(1+(growth*endtyme +
      (growth*3.0*kk/(2.0*theta))*(starttyme*starttyme-endtyme*endtyme)))
      + (kk/theta) * (starttyme - endtyme);
else
   answ = linv_theta + growth*endtyme +
      (kk/(theta*growth))*(exp(growth*starttyme)-exp(growth*endtyme));


return(answ);
} /* fluc_interval_llike */

/*****************************************************************
 * fluc_lliketree returns the Ln(likelihood) of a tree generated *
 * under a theta & growth ("otheta" & "ogrowth") for a given     *
 * theta & growth ("theta" & "growth").  To speed it up, it is   *
 * also supplied with ln(1/theta) and ln(1/otheta).              */
double fluc_lliketree(long tr, long chain, double theta, double otheta,
  double linv_theta, double linv_otheta, double growth, double ogrowth, 
  long lowcus)
{
   long i, kk, refchain;
   double tempnum, tempdenom, starttyme, endtyme, result;

   refchain = REF_CHAIN(chain);

   result = 0.0;
   i = 0;
   do {
      endtyme = sum[lowcus][refchain][tr].kend[i];
      if(!i) starttyme = 0;
      else starttyme = sum[lowcus][refchain][tr].kend[i-1];
      kk = sum[lowcus][refchain][tr].kk[i];
      tempnum = fluc_interval_llike(theta,linv_theta,growth,starttyme,
        endtyme,kk);
      tempdenom = fluc_interval_llike(otheta,linv_otheta,ogrowth,starttyme,
        endtyme,kk);
      result += tempnum - tempdenom;
      i++;
   } while (kk != 2);
   return(result);
} /* fluc_lliketree */


/**********************************************************************
 * fluc_locus_llike() returns the Ln(like) of a theta and growth over *
 * a set of loci, all loci present in the input data.  It uses only   *
 * the last chain of each locus.                                      */
double fluc_locus_llike(double theta, double growth)
{
long lowcus;
double temp;

temp = 0.0;
for (lowcus = 0; lowcus < numloci; lowcus++)
   temp += model_llike(theta_ratio[lowcus]*theta, growth,totchains-1,lowcus);

return(temp);
} /* fluc_locus_llike */

/********************************************************************
 * model_llike returns the Ln(likelihood) of a chain, "chain", at a *
 * given theta, "thgiven".                                          * 
 * EQN:  k = # of active lineages within an interval                *
 *       inter_lngth = length of an interval                        *
 *       thgiven = theta of interest                                *
 *       thchain = theta under which the chain was run              *
 *       numinter = # of intervals in tree                          *
 *       numtree = # of trees in chain                              *
 *                                                                  *
 *  finterval = sum_over_intervals[(k * (k - 1)) * inter_lngth]     *
 *                                                                  *
 *                         exp(-finterval/thgiven)                  *
 *  ftree = sum_over_trees -----------------------                  *
 *                         exp(-finterval/thchain)                  *
 *                                                                  *
 *   likelihood = ((thchain/thgiven) ** numinter) * (ftree/numtree) *
 *                                                                  */
double model_llike(double thgiven, double grgiven, long chain, long lowcus)
{
  double th, th0, g0, smallest, biggest, answ, *likel, linv_theta,
    linv_otheta;
  long trii, refchain, chaintype;

  refchain = REF_CHAIN(chain);
  chaintype = TYPE_CHAIN(chain);

  /* the following code allows use of this function when g=0, giving
  the Coalesce likelihood, even when running as Fluctuate; this is
  useful in constructing the final table.  It is not used when
  maximizing likelihoods as the derivatives are not set up
  appropriately; in that case, if g=0 it is set to epsilon. */

  likel = (double *)calloc(op->numout[chaintype],sizeof(double));

  th0 = theti[lowcus][chain];
  if (op->growthused) {
     g0 = growi[lowcus][chain];
     th = 0.0; /* won't be used */
     if (!grgiven) grgiven = EPSILON;
  } else { /* bail out if theta=theta0 only if growth not used */
    g0 = 0.0; /* won't be used */
    th = 1 / thgiven - 1 / th0;
    if (th == 0) return(0.0);
  }

  /* speed up manuver to avoid taking logs in inner loop */
  linv_theta = log(1/thgiven);
  linv_otheta = log(1/th0);

  smallest = POSMAX;
  biggest = NEGMAX;

  for (trii = 0; trii < op->numout[chaintype]; trii++) {
     if (sametree[lowcus][trii]) likel[trii] = likel[trii-1];
     else {
        if (!op->growthused)
           likel[trii] = sum[lowcus][refchain][trii].kend[0] * th;
        else
           likel[trii] = fluc_lliketree(trii,chain,thgiven,th0,
              linv_theta,linv_otheta,grgiven,g0,lowcus);
     }
     if (!op->growthused) {
        if (likel[trii] < smallest) smallest = likel[trii];
     } else {
        if (likel[trii] > biggest) biggest = likel[trii];
     }
  }

  answ = 0.0;
  for (trii = 0; trii < op->numout[chaintype]; trii++) {
     if (!op->growthused) {
        if (smallest - likel[trii] > EXPMIN)
           answ += exp(smallest - likel[trii]);
     } else {
        if (likel[trii] - biggest > EXPMIN)
           answ += exp(likel[trii] - biggest);
     }
  }

  answ /= op->numout[chaintype];

  if (!op->growthused)
     answ = (numseq - 1) * (lntheti[lowcus][chain] - log(thgiven)) 
         - smallest + log(answ); 
  else answ = log(answ) + biggest;

  free(likel);
  return(answ);
}  /* model_llike */

/*********************************************************************
 * model_likelihood is a driver to call the other model_likelihood   *
 * evaluators.                                                       *
 * Pass "lowcus" < 0 for multiple locus evaluation over chains       *
 *    "firstlong" to "lastlong" (only lastlong used in growth).      *
 * In evaluation of a single chain, use "firstlong" for the chain #  */
double model_likelihood(double theta, double growth, long firstlong,
  long lastlong, long lowcus, double **lthetai)
{

if (!op->growthused) growth = 0.0;
if (lowcus < 0) return(fluc_locus_llike(theta,growth));
else return model_llike(theta,growth,firstlong,lowcus);

} /* model_likelihood */

double fexp1(long kk, double thgiven, double grgiven, double expend,
  double expstart)
{

return(kk/(thgiven*grgiven)*(expstart - expend));

} /* fexp1 */

double fexp2(long kk, double thgiven, double grgiven, double endtyme,
  double starttyme, double expstart, double expend)
{

return(kk/(thgiven*grgiven)*
       (starttyme*expstart - endtyme*expend));

} /* fexp2 */

double fluc_theta_lderiv(long chain, double thgiven, double grgiven,
  double *fx, double *dfx, long *fxplus, long *dfxplus, long lowcus,
  derivlist *d)
{
long i, kk, trii, numintervals, chaintype, refchain;
double temp1, temp2, endtyme, starttyme, th0, gr0, fx1,
   triilike, fmax, smax, result, lthgiven, lth0;

chaintype = TYPE_CHAIN(chain);
refchain = REF_CHAIN(chain);

th0 = theti[lowcus][chain];
gr0 = growi[lowcus][chain];
fmax = NEGMAX;
smax = NEGMAX;
lthgiven = log(thgiven);
lth0 = log(th0);
temp1 = temp2 = triilike = 0.0;

for(trii = 0; trii < op->numout[chaintype]; trii++) {
   if (!sametree[lowcus][trii]) {
   /* we assume the first tree is not sametree */
      i = 0;
      numintervals = numseq - 1; /* WARNING, wrong for recombination */
      temp1 = 0.0;
      temp2 = 0.0;
      triilike = fluc_lliketree(trii,chain,thgiven,th0,-lthgiven,-lth0,
         grgiven,gr0,lowcus);
      do {
         endtyme = sum[lowcus][refchain][trii].kend[i];
         if(!i) starttyme = 0;
         else starttyme = sum[lowcus][refchain][trii].kend[i-1];
         kk = sum[lowcus][refchain][trii].kk[i];
         if(endtyme - starttyme == 0.0) fx1 = 0.0;
         else fx1 = fexp1(kk,thgiven,grgiven,exp(grgiven*endtyme),
                     exp(grgiven*starttyme));
   
         temp1 += 1 + fx1;
         temp2 += 2*fx1 + 1;
         i++;
      } while (kk != 2);
   } /* else it's the sametree case, use last tree's values */

/* sign check reversed since temp1 should be -temp1 in the 1st deriv */
   d->fplus[trii] = 1;
   if (temp1 > 0.0) d->fplus[trii] = -1;
   d->firstd[trii] = log(fabs(temp1)) - lthgiven + triilike;
   if (d->firstd[trii] > fmax) fmax = d->firstd[trii];

   d->splus[trii] = 1;
   if (temp1*temp1 + temp2 < 0.0) d->splus[trii] = -1;
   d->secondd[trii] = triilike - 2*lthgiven + log(fabs(temp1*temp1 + temp2)); 
   if (d->secondd[trii] > smax) smax = d->secondd[trii];
}

*fx = 0.0;
*dfx = 0.0;

for(trii = 0; trii < op->numout[chaintype]; trii++) {
   if((d->firstd[trii] - fmax) > EXPMIN)
      *fx += d->fplus[trii] * exp(d->firstd[trii] - fmax);
   if((d->secondd[trii] - smax) > EXPMIN)
      *dfx += d->splus[trii] * exp(d->secondd[trii] - smax);
}
*fx /= op->numout[chaintype];
*dfx /= op->numout[chaintype];

*fxplus = whatsign(*fx);
*dfxplus = whatsign(*dfx);

*fx = log(fabs(*fx)) + fmax;
*dfx = log(fabs(*dfx)) + smax;

result = *fx - *dfx;

return(result);

} /* fluc_theta_lderiv */

double fluc_growth_lderiv(long chain, double thgiven, double grgiven,
  double *fx, double *dfx, long *fxplus, long *dfxplus, long lowcus,
  derivlist *d)
{
long i, kk, trii, numintervals, chaintype, refchain;
double temp1, temp2, temp3, endtyme, starttyme, th0, gr0, fx1, fx2,
   dkk, triilike, fmax, smax, result, expstart, expend, lth, lth0;

chaintype = TYPE_CHAIN(chain);
refchain = REF_CHAIN(chain);

th0 = theti[lowcus][chain];
gr0 = growi[lowcus][chain];
fmax = NEGMAX;
smax = NEGMAX;
lth = log(thgiven);
lth0 = log(th0);
temp1 = temp2 = temp3 = 0.0;
triilike = 0.0;

for(trii = 0; trii < op->numout[chaintype]; trii++) {
   if (!sametree[lowcus][trii]) {
/* we assume the first tree is not a sametree */
      i = 0;
      numintervals = numseq - 1; /* WARNING, wrong for recombination */
      temp1 = 0.0;
      temp2 = 0.0;
      temp3 = 0.0;
      triilike = fluc_lliketree(trii,chain,thgiven,th0,-lth,-lth0,
        grgiven,gr0,lowcus);
      do {
         endtyme = sum[lowcus][refchain][trii].kend[i];
         if(!i) starttyme = 0;
         else starttyme = sum[lowcus][refchain][trii].kend[i-1];
         kk = sum[lowcus][refchain][trii].kk[i];
         if(endtyme - starttyme == 0.0) {
            temp1 += endtyme;
            temp2 += 0.0;
            temp3 += 0.0;
         } else {
            expstart = exp(grgiven*starttyme);
            expend = exp(grgiven*endtyme);
    
            fx1 = fexp1(kk,thgiven,grgiven,expend,expstart);
            fx2 = fexp2(kk,thgiven,grgiven,endtyme,starttyme,expstart,expend);
            dkk = -fx1/grgiven + fx2;

            temp1 += endtyme + dkk;
            temp2 += endtyme*fx2 - endtyme*dkk + fx2*dkk - dkk*dkk;
            temp3 += (1.0/grgiven) * (-fx1*endtyme - fx1*dkk - 2*dkk) +
                     (kk/(thgiven*grgiven))*
                        (starttyme*starttyme*expstart - endtyme*endtyme*expend);
            }

         i++;
      } while (kk != 2);
   } /* else it's a sametree, use last tree's values */

   d->fplus[trii] = 1;
   if (temp1 < 0.0) d->fplus[trii] = -1;
   if (temp1 == 0.0) {
      d->fplus[trii] = 0;
      d->firstd[trii] = 0.0;
   } else {
      d->firstd[trii] = log(fabs(temp1)) + triilike;
      if (d->firstd[trii] > fmax) fmax = d->firstd[trii];
   }

   d->splus[trii] = 1;
   if (temp1*temp1 + temp2 + temp3 < 0.0) d->splus[trii] = -1;
   if (temp1*temp1 + temp2 + temp3 == 0.0) {
      d->splus[trii] = 0;
      d->secondd[trii] = 0.0;
   } else {
      d->secondd[trii] = triilike + log(fabs(temp1*temp1 + temp2 + temp3)); 
      if (d->secondd[trii] > smax) smax = d->secondd[trii];
   }
}

*fx = 0.0;
*dfx = 0.0;

for(trii = 0; trii < op->numout[chaintype]; trii++) {
   if((d->firstd[trii] - fmax) > EXPMIN)
      *fx += d->fplus[trii] * exp(d->firstd[trii] - fmax);
   if((d->secondd[trii] - smax) > EXPMIN)
      *dfx += d->splus[trii] * exp(d->secondd[trii] - smax);
}
*fx /= op->numout[chaintype];
*dfx /= op->numout[chaintype];

*fxplus = whatsign(*fx);
*dfxplus = whatsign(*dfx);

*fx = log(fabs(*fx)) + fmax;
*dfx = log(fabs(*dfx)) + smax;

result = *fx - *dfx;

return(result);

} /* fluc_growth_lderiv */

double fluc_locus_theta_lderiv(double theta, double growth, double *fx,
   double *dfx, long *fxplus, long *dfxplus, derivlist *d)
{
long lowcus, lfxplus, ldfxplus, lastchain;
double rtheta, fntheta, dfntheta, ddfntheta, temp, temp1, temp2, temp3, 
  temp4, rgrowth;

lastchain = totchains - 1;

temp1 = 0.0;
temp2 = 0.0;
temp3 = 0.0;
temp4 = 0.0;

for (lowcus = 0; lowcus < numloci; lowcus++) {
   rtheta = theta;
   rgrowth = growth;
   fntheta = model_llike(rtheta,rgrowth,lastchain,lowcus);
   temp = fluc_theta_lderiv(lastchain,rtheta,rgrowth,&dfntheta,
      &ddfntheta,&lfxplus,&ldfxplus,lowcus,d);

   temp1 += fntheta;
   if (lfxplus) {
      temp2 += lfxplus * exp(dfntheta - fntheta);
      temp4 -=  exp(2 * (dfntheta - fntheta));
   }
   if (ldfxplus) temp3 +=  ldfxplus * exp(ddfntheta - fntheta);
}

temp = (temp2*temp2+temp3+temp4);

*fxplus = whatsign(temp2);
*dfxplus = whatsign(temp);

if (!*fxplus) *fx = 0.0;
else *fx = temp1 + log(fabs(temp2));
if (!*dfxplus) *dfx = 0.0;
else *dfx = temp1 + log(fabs(temp));

return(*fx - *dfx);

} /* fluc_locus_theta_lderiv */

double fluc_locus_growth_lderiv(double theta, double growth, double *fx,
   double *dfx, long *fxplus, long *dfxplus, derivlist *d)
{
long lowcus, lfxplus, ldfxplus, lastchain;
double rtheta, fngrowth, dfngrowth, ddfngrowth, temp, temp1, temp2, 
   temp3, temp4, rgrowth;

lastchain = totchains - 1;

temp1 = 0.0;
temp2 = 0.0;
temp3 = 0.0;
temp4 = 0.0;

for (lowcus = 0; lowcus < numloci; lowcus++) {
   rtheta = theta;
   rgrowth = growth;
   fngrowth = model_llike(rtheta,rgrowth,lastchain,lowcus);
   temp = fluc_growth_lderiv(lastchain,rtheta,rgrowth,&dfngrowth,
      &ddfngrowth,&lfxplus,&ldfxplus,lowcus,d);

   temp1 += fngrowth;
   if (lfxplus) {
      temp2 += lfxplus * exp(dfngrowth - fngrowth);
      temp4 -= exp(2 * (dfngrowth - fngrowth));
   }
   if (ldfxplus) temp3 += ldfxplus * exp(ddfngrowth - fngrowth);
}

temp = (temp2*temp2+temp3+temp4);

*fxplus = whatsign(temp2);
*dfxplus = whatsign(temp);

if (!*fxplus) *fx = 0.0;
else *fx = temp1 + log(fabs(temp2));
if (!*dfxplus) *dfx = 0.0;
else *dfx = temp1 + log(fabs(temp));

return(*fx - *dfx);

} /* fluc_locus_growth_lderiv */

double fluc_partial_lderiv(long chain, double theta, double growth,
   double *df, long *dfplus, long lowcus)
{
double th0, gr0, endtyme, starttyme, dkk, expstart, expend, fx1, fx2,
   triilike, result, temp1, temp2, temp3, temp4, *triivalue, max, lth,
   lth0;
long i, kk, numintervals, trii, *triisign, chaintype, refchain;

chaintype = TYPE_CHAIN(chain);
refchain = REF_CHAIN(chain);

triisign = (long *)calloc(op->numout[chaintype],sizeof(long));
triivalue = (double *)calloc(op->numout[chaintype],sizeof(double));

numintervals = numseq - 1; /* WARNING--wrong for recombination */
th0 = theti[lowcus][chain];
gr0 = growi[lowcus][chain];

lth = log(theta);
lth0 = log(th0);

max = NEGMAX;
temp1 = temp2 = temp3 = temp4 = 0.0;
triilike = 0.0;

for(trii = 0; trii < op->numout[chaintype]; trii++) {
   if (!sametree[lowcus][trii]) {
   /* assumes first tree not a sametree */
      result = 0.0;
      triilike = fluc_lliketree(trii,chain,theta,th0,-lth,-lth0,
        growth,gr0,lowcus);
      i = 0;
      temp1 = 0.0;
      temp2 = 0.0;
      temp3 = 0.0;
      temp4 = 0.0;
      do {
         endtyme = sum[lowcus][refchain][trii].kend[i];
         if(!i) starttyme = 0;
         else starttyme = sum[lowcus][refchain][trii].kend[i-1];
         kk = sum[lowcus][refchain][trii].kk[i];
         expstart = exp(growth*starttyme);
         expend = exp(growth*endtyme);
   
         fx1 = fexp1(kk,theta,growth,expend,expstart);
         fx2 = fexp2(kk,theta,growth,endtyme,starttyme,expstart,expend);
         dkk = -fx1/growth + fx2;
   
         temp1 -= (1.0+fx1)/theta;
         temp2 += endtyme + dkk;
         temp3 -= (endtyme + endtyme*fx1 + fx1*dkk + 2*dkk)/theta;
         temp4 -= (endtyme+dkk) * (1.0+fx1)/theta;
   
         i++;
      } while(kk != 2);
   } /* else it's a sametree, use last tree's values */

   result = temp1 * temp2 + temp3 - temp4;
   triisign[trii] = 1;
   if (result == 0.0) triisign[trii] = 0;
   if (result < 0.0) triisign[trii] = -1;
   triivalue[trii] = triilike + log(fabs(result));
   if (triivalue[trii] > max) max = triivalue[trii];
}

*df = 0.0;
for(trii = 0; trii < op->numout[chaintype]; trii++)
   if (triivalue[trii] - max > EXPMIN) 
      *df = triisign[trii] * exp(triivalue[trii] - max);
*df /= op->numout[chaintype];

free(triisign);
free(triivalue);

*dfplus = whatsign(*df);

*df = log(fabs(*df)) + max;

return(*df);

} /* fluc_partial_lderiv */


double fluc_locus_partial_lderiv(double theta, double growth, double *df,
   long *dfplus, derivlist *d)
{
long lowcus, growthplus, thetaplus, pplus, lastchain, ldummy;
double rtheta, fn, fngrowth, fntheta, fnp, dummy, temp1, temp2, temp3,
   temp4, temp5, rgrowth;

dummy = 0.0; /* just in case */
ldummy = 0;

lastchain = totchains - 1;

temp1 = 0.0;
temp2 = 0.0;
temp3 = 0.0;
temp4 = 0.0;
temp5 = 0.0;

for (lowcus = 0; lowcus < numloci; lowcus++) {
   rtheta = theta;
   rgrowth = growth;
   fn = model_llike(rtheta,rgrowth,lastchain,lowcus);
   dummy = fluc_growth_lderiv(lastchain,rtheta,rgrowth,&fngrowth,
      &dummy,&growthplus,&ldummy,lowcus,d);
   dummy = fluc_theta_lderiv(lastchain,rtheta,rgrowth,&fntheta,
      &dummy,&thetaplus,&ldummy,lowcus,d);
   dummy = fluc_partial_lderiv(lastchain,rtheta,rgrowth,&fnp,&pplus,
      lowcus);

   temp1 += fn;
   temp2 += growthplus * exp(fngrowth - fn);
   temp3 += thetaplus * exp(fntheta - fn);
   temp4 += pplus * exp(fnp - fn);
   temp5 -= growthplus * thetaplus *
            exp(fngrowth+fntheta - 2.0*fn);
}

temp2 = temp2*temp3+temp4+temp5;

*dfplus = whatsign(temp2);
if (!*dfplus) *df = 0.0;
else *df = temp1 + log(fabs(temp2));

return(*df);
} /* fluc_locus_partial_lderiv */

void init_matrix(double *fx, double **dfx, double g, double theta,
   long *fxplus, long **dfxplus, long chain, derivlist *d, 
   boolean multilocus)
/* row '0' = growth derivatives
   row '1' = theta derivatives */
{
double dummy;

dummy = 0.0; /* just in case */

if (multilocus) {
   if (holding != 2)
      dummy = fluc_locus_growth_lderiv(theta,g,&fx[0],&dfx[0][0],
         &fxplus[0],&dfxplus[0][0],d);
   if (holding != 1)
      dummy = fluc_locus_theta_lderiv(theta,g,&fx[1],&dfx[1][1],
         &fxplus[1],&dfxplus[1][1],d);
   if (!holding)
      dummy = fluc_locus_partial_lderiv(theta,g,&dfx[0][1],
         &dfxplus[0][1],d);
} else {
   if (holding != 2)
      dummy = fluc_growth_lderiv(chain,theta,g,&fx[0],&dfx[0][0],
         &fxplus[0],&dfxplus[0][0],locus,d);
   if (holding != 1)
      dummy = fluc_theta_lderiv(chain,theta,g,&fx[1],&dfx[1][1],
         &fxplus[1],&dfxplus[1][1],locus,d);
   if (!holding)
      dummy = fluc_partial_lderiv(chain,theta,g,&dfx[0][1],&dfxplus[0][1],
         locus);
}

dfx[1][0] = dfx[0][1];
dfxplus[1][0] = dfxplus[0][1];

} /* init_matrix */

boolean check_curvature(double *fx, double **dfx, long *fxplus,
   long **dfxplus)
/* return true if curvature is within maximum's attraction zone */
/*  EQN:
result = (dfx[1][1]*fx[0]*fx[0] - dfx[0][1]*fx[0]*fx[1] +
        dfx[0][0]*fx[1]*fx[1] - dfx[1][0]*fx[0]*fx[1]) /
       (dfx[0][0]*dfx[1][1] - dfx[0][1]*dfx[1][0]);
*/
{
double denom, max, temp1, temp2, temp3, temp4, result;
long denomsign;

if (holding) return(true);

temp1 = dfx[0][0] + dfx[1][1];

if (dfxplus[0][1] && dfxplus[1][0]) {
   temp2 = dfx[0][1] + dfx[1][0];
   if (temp1 > temp2) max = temp1;
   else max = temp2;

   denom = 0.0;
   if (temp1 - max > EXPMIN)
      denom += (double)(dfxplus[0][0] * dfxplus[1][1]) * exp(temp1 - max);
   else temp1 = 0.0;
   if (temp2 - max > EXPMIN)
      denom -= (double)(dfxplus[0][1] * dfxplus[1][0]) * exp(temp2 - max);

   denomsign = whatsign(denom);
   if(zerocheck(denom)) {
      fprintf(ERRFILE,"WARNING--infinite curvature encountered!");
      return(false);
   }
   denom = log(fabs(denom)) + max;
} else {
   denom = temp1;
   denomsign = dfxplus[0][0] * dfxplus[1][1];
}

max = NEGMAX;

temp1 = dfx[1][1] + 2.0 * fx[0] - denom;
if (temp1 > max) max = temp1;
temp2 = dfx[0][0] + 2.0 * fx[1] - denom;
if (temp2 > max) max = temp2;
if (dfxplus[1][0]) {
   temp3 = dfx[1][0] + fx[0] + fx[1] - denom;
   if (temp3 > max) max = temp3;
} else temp3 = 0.0;

if (dfxplus[0][1]) {
   temp4 = dfx[0][1] + fx[0] + fx[1] - denom;
   if (temp4 > max) max = temp4;
} else temp4 = 0.0;

result = 0.0;

if (temp1 - max > EXPMIN)
/* the sign of the 1st derivative is irrelevant since it is squared */
   result += dfxplus[1][1] * denomsign * exp(temp1 - max);
if (temp2 - max > EXPMIN)
/* the sign of the 1st derivative is irrelevant since it is squared */
   result += dfxplus[0][0] * denomsign * exp(temp2 - max);
if (dfxplus[1][0])
   if (temp3 - max > EXPMIN)
      result -= dfxplus[1][0] * fxplus[0] * fxplus[1] * denomsign *
             exp(temp3 - max);
if (dfxplus[0][1])
   if (temp4 - max > EXPMIN)
      result -= dfxplus[0][1] * fxplus[0] * fxplus[1] * denomsign *
             exp(temp4 - max);

if (result < 0) return(true);
else return(false);

} /* check_curvature */

boolean calc_change_denom(double **dfx, long **dfxplus, double *denom,
   long *denomsign)
{
double temp1, temp2, max;

if (holding) return(true);

temp1 = dfx[0][0] + dfx[1][1];

if (dfxplus[0][1] && dfxplus[1][0]) {
   temp2 = dfx[0][1] + dfx[1][0];
   if (temp1 > temp2) max = temp1;
   else max = temp2;
   *denom = 0.0;
   if (temp1 - max > EXPMIN)
      *denom += dfxplus[0][0] * dfxplus[1][1] * exp(temp1 - max);
   if (temp2 - max > EXPMIN)
      *denom -= dfxplus[0][1] * dfxplus[1][0] * exp(temp2 - max);
   *denomsign = whatsign(*denom);
   *denom = log(fabs(*denom)) + max;
} else {
   *denomsign = dfxplus[0][0] * dfxplus[1][1];
   *denom = temp1;
}

/* check that dfx is invertable */
if(!*denom) return(false);
else return(true);

} /* calc_change_denom */

void calc_change_contrib(double *fx, double **dfx, long *fxplus,
   long **dfxplus, double denom, long denomsign, 
   double *answ)
{
double temp1, temp2;

/* In the single dimensional search, use normal newton-raphson,
   except use the absolute value of the 2nd derivative, and add
   rather than subtract the net change. */
if (holding == 1) {
   answ[0] = -1.0 * fxplus[0] * exp(fx[0] - dfx[0][0]);
   return;
}

if (holding == 2) {
   answ[1] = -1.0 * fxplus[1] * exp(fx[1] - dfx[1][1]);
   return;
}

temp1 = dfxplus[1][1] * fxplus[0] * denomsign *
        exp(dfx[1][1] + fx[0] - denom);
if (dfxplus[0][1]) {
   temp2 = dfxplus[0][1] * fxplus[1] * denomsign *
           exp(dfx[0][1] + fx[1] - denom);
   answ[0] = temp1 - temp2;
} else answ[0] = temp1;

temp1 = dfxplus[0][0] * fxplus[1] * denomsign *
        exp(dfx[0][0] + fx[1] - denom);
if (dfxplus[1][0]) {
   temp2 = dfxplus[1][0] * fxplus[0] * denomsign *
           exp(dfx[1][0] + fx[0] - denom);
   answ[1] = temp1 - temp2;
} else answ[1] = temp1;

} /* calc_change_contrib */

boolean calc_change(double *fx, double **dfx, long *fxplus, 
   long **dfxplus, double *answ)
{
double denom;
long denomsign;

if (!calc_change_denom(dfx,dfxplus,&denom,&denomsign)) return(false);

calc_change_contrib(fx,dfx,fxplus,dfxplus,denom,denomsign,answ);

return(true);

} /* calc_change */

boolean check_MAX(double *g)
{

if ((*g)*xinterval > EXPMAX) {
   (*g) = EXPMAX/xinterval - 1.0;
   return(false);
}
else return(true);

} /* check_MAX */

#define STEPPRINT 6

void matrixpoint(long chain, double *thetaresult, double *gresult, 
   derivlist *d, boolean multilocus)
{
  int numloop; /* type "int" because of library function demands! */
  long i, j, samecount, maxcount, lowcus, garbage, *fxplus, 
     **dfxplus, *pplus;
  double theta, newtheta, g, newg, oldlike, newlike, thetachange,
    gchange, **dummy, *pchange, *fmat, **dfmat, ltolerance;
  boolean firstreport;

  garbage = 0; /* just in case */
  dummy = NULL;

  fxplus = (long *)calloc(NUMPARAMETERS,sizeof(long));
  dfxplus = (long **)calloc(NUMPARAMETERS,sizeof(long *));
  pplus = (long *)calloc(NUMPARAMETERS,sizeof(long));
  dfxplus[0] = (long *)calloc(NUMPARAMETERS*NUMPARAMETERS,sizeof(long));
  fmat = (double *)calloc(NUMPARAMETERS,sizeof(double));
  pchange = (double *)calloc(NUMPARAMETERS,sizeof(double));
  dfmat = (double **)calloc(NUMPARAMETERS,sizeof(double *));
  dfmat[0] = (double *)calloc(NUMPARAMETERS*NUMPARAMETERS,sizeof(double));
  for(i = 1; i < NUMPARAMETERS; i++) {
     dfxplus[i] = dfxplus[0] + i * NUMPARAMETERS;
     dfmat[i] = dfmat[0] + i * NUMPARAMETERS;
  } 
  /* point estimate of growth & theta simultaneously */
  theta = *thetaresult;
  g = *gresult;
  if (g == 0.0) g = epsilon;
  if (multilocus) lowcus = -1;
  else lowcus = locus;
  oldlike = model_likelihood(theta,g,chain,garbage,lowcus,dummy);
  i = 0;
  samecount = 0;
  maxcount = 0;
  ltolerance = log(epsilon);
  firstreport = true;

  /* solve by modified Newton Raphson */
  while (1) {

     if (!check_MAX(&g)) {
        if (maxcount < REPEAT_TOLERANCE) {
           fprintf(ERRFILE,"\nmodefied g down!\n");
           maxcount++;
           continue;
        } else {
           fprintf(ERRFILE,"\nWARNING--growth got too large!\n");
           fprintf(outfile,"\nWARNING--growth got too large!\n");
           break;
        }
     }

     init_matrix(fmat,dfmat,g,theta,fxplus,dfxplus,chain,d,
        multilocus);

     /* if within epsilon of the maximum, we're done! */
     if ((fmat[0] < ltolerance && fmat[1] < ltolerance) ||
         (fmat[0] < ltolerance && holding == 2) ||
         (fmat[1] < ltolerance && holding == 1))
        break;

     while ((!dfxplus[0][0] && holding != 2) || 
            (!dfxplus[1][1] && holding != 1)) {
        if (!dfxplus[0][0]) g += epsilon;
        if (!dfxplus[1][1]) theta += epsilon;
        init_matrix(fmat,dfmat,g,theta,fxplus,dfxplus,chain,d,
           multilocus);
     }

     while(!calc_change(fmat,dfmat,fxplus,dfxplus,pchange)) {
        fprintf(ERRFILE,"\nOH NO, no inversion possible!!!!!\n");
        g += epsilon;
        theta += epsilon;
        init_matrix(fmat,dfmat,g,theta,fxplus,dfxplus,chain,d,
           multilocus);
        continue;
     }

     if (check_curvature(fmat,dfmat,fxplus,dfxplus)) {
        gchange = pchange[0];
        thetachange = pchange[1];
     } else {
     /* check to see what direction the newton-raphson said to go,
        don't use generalized sign-checkers here! */
        pplus[0] = -1 * fxplus[0];
        pplus[1] = -1 * fxplus[1];
        if (fabs(g) > LHOP_TOLERANCE) gchange = pplus[0] * fabs(g)/2.0;
        else gchange = pplus[0] * 3.0 * fabs(g)/2.0;
        thetachange = pplus[1] * theta/2.0;
     }

     newg = g - gchange;
     newtheta = theta - thetachange;

     numloop = 0;
     while (newtheta <= 0) {
        numloop++;
        newtheta = theta - ldexp(thetachange,-numloop);
     }
     if (newg == 0.0) newg = epsilon;

     newlike = model_likelihood(newtheta,newg,chain,garbage,lowcus,dummy);
     if(newlike < oldlike) {
    /* in case we overshoot the maximum, don't jump so far...*/
        numloop = 0;
        while(1) {
           numloop++;
           newg = g - ldexp(gchange,-numloop);
           newtheta = theta - ldexp(thetachange,-numloop);
           if(newtheta <= 0) continue;
           newlike = model_likelihood(newtheta,newg,chain,garbage,lowcus,dummy);
           if(newlike >= oldlike) break;
           if((fabs(newg - g) < epsilon) &&
              (fabs(newtheta - theta) < epsilon)) break;
        }
     }
     if (newlike == oldlike) { /* yes, really == */
        samecount++;
        if (samecount == REPEAT_TOLERANCE) break;
     } else samecount = 0;

     oldlike = newlike;
     if (newg > GLID || newtheta > THETALID) {
        g = newg;
        theta = newtheta;
        break;
     }
     if((fabs(newg - g) < epsilon) &&
        (fabs(newtheta - theta) < epsilon))
     {
        g = newg;
        theta = newtheta;
        break;
     }
     g = newg;
     theta = newtheta;
     i++;
     if (!firstreport) for(j = 0; j < STEPPRINT; j++) printf("\b");
     else printf("maximizing ");
     printf("%*ld",(int)STEPPRINT,i);
     fflush(stdout);
     firstreport = false;
  }

/* correct these results downward if they are too large */
  if (g < GLID && theta < THETALID) {
    *gresult = g;
    *thetaresult = theta;
  } else {
    fprintf(ERRFILE,"Warning:  setting theta and/or g lower!\n");
    if (g >= GLID) *gresult = GLID;
    if (theta >= THETALID) *thetaresult = THETALID;
  }
  printf("\n");

  free(pchange);
  free(fmat);
  free(fxplus);
  free(dfmat[0]);
  free(dfmat);
  free(dfxplus[0]);
  free(dfxplus);
  free(pplus);

} /* matrixpoint */

void fluc_locus_estimate()
{
  double th,g,oldth,oldg,like_at_max;
  long i, chaintype, garbage;

  /* the following structure and callocs allocate space for
  temporary variables used in derivatives:  a time-saving
  manuver to avoid calls to malloc in frequently-called routines */

  derivlist *d;

  garbage = 0; /* just in case */
  chaintype = TYPE_CHAIN(totchains-1);

  d = (derivlist *)calloc(1,sizeof(derivlist));
  d->firstd = (double *)calloc(op->numout[chaintype],sizeof(double));
  d->secondd = (double *)calloc(op->numout[chaintype],sizeof(double));
  d->fplus = (long *)calloc(op->numout[chaintype],sizeof(long));
  d->splus = (long *)calloc(op->numout[chaintype],sizeof(long));

  th = watttheta;
  oldth = th;
  g = growi[numloci-1][totchains-1];
  oldg = g;
  for(i=0;i<10;i++) {
    matrixpoint(totchains-1,&th,&g,d,true);
    if((fabs(oldth-th) < epsilon) && (fabs(oldg-g) < epsilon)) break;
    oldth = th;
    oldg = g;
    }

  if (USETRUE) truthmod(th,g,totchains-1,-1L);
  like_at_max = fluc_locus_llike(th,g);
  fprintf(outfile,"-----------------------------------------------\n");
  fprintf(outfile,"\n\nOver %ld loci, best estimate of parameters:\n",numloci);
  fprintf(outfile,"   Theta = %12.6f Growth = %12.6f\n",th,g);
  fprintf(outfile,"Log(Likelihood) of the estimate %12.6f",like_at_max);

  fluc_liketable(th,g,garbage,-1L,op->interact);
  print_covariance(th,g,totchains-1,-1L);
  fluc_printtable(th,g);
  print_locusplot(-1L);

  free(d->firstd);
  free(d->secondd);
  free(d->fplus);
  free(d->splus);
  free(d);
} /* fluc_locus_estimate */

void fluc_estimate(long chain, boolean locusend)
{
  double th,g,oldth,oldg,like_at_max;
  long i, chaintype;

  /* the following structure and callocs allocate space for
  temporary variables used in derivatives:  a time-saving
  manuver to avoid calls to malloc in frequently-called routines */

  derivlist *d;

  chaintype = TYPE_CHAIN(chain);

  d = (derivlist *)calloc(1,sizeof(derivlist));
  d->firstd = (double *)calloc(op->numout[chaintype],sizeof(double));
  d->secondd = (double *)calloc(op->numout[chaintype],sizeof(double));
  d->fplus = (long *)calloc(op->numout[chaintype],sizeof(long));
  d->splus = (long *)calloc(op->numout[chaintype],sizeof(long));

  th = theti[locus][chain];
  oldth = th;
  g = growi[locus][chain];
  oldg = g;

  for (i=0;i<100;i++) {
     matrixpoint(chain,&th,&g,d,false);
     if(((fabs(oldth-th) < epsilon) && (fabs(oldg-g) < epsilon)) ||
        (fabs(model_llike(oldth,oldg,chain,locus)-
        model_llike(th,g,chain,locus)) < epsilon))
        break;
     oldth = th;
     oldg = g;
  }
   
  if (op->progress) printf(" Theta=%10.5f, growth=%10.5f\n",th,g);

  if (th > 0) {
    /* now set the values for the next chain to use */
    if (holding != 1) {
       theti[locus][chain+1] = th;
       lntheti[locus][chain+1] = log(th);
    } else {
       theti[locus][chain+1] = theti[locus][chain];
       lntheti[locus][chain+1] = lntheti[locus][chain];
    }
   if (holding != 2) growi[locus][chain+1] = g;
   else growi[locus][chain+1] = growi[locus][chain];

    if (locusend) {
      if (USETRUE) truthmod(th,g,chain,locus);
      fluc_liketable(th,g,chain,locus,op->interact);
      like_at_max = model_llike(th,g,chain,locus);
      fprintf(outfile,"\nPoint estimate of parameters for locus %ld\n",
         locus+1);
      fprintf(outfile,"Theta %12.6f Growth %12.6f\n",th,g);
      fprintf(outfile,"Log(Likelihood) at maximum %12.6f\n\n",
         like_at_max);
      if (op->growthused) print_covariance(th,g,chain,locus);
      fluc_printtable(th,g);
      if (op->growthused) print_locusplot(locus);
    }
    free(d->firstd);
    free(d->secondd);
    free(d->fplus);
    free(d->splus);
    free(d);
    return;
  }
  /* if we reach this line the point estimation has FAILED */
  if (menu) {
    printf("WARNING, point estimate of theta failed!\n");
    printf("using previous iteration theta estimate\n");
  } else {
    fprintf(simlog, "WARNING, point estimate of theta failed!\n");
    fprintf(simlog, "using previous iteration theta estimate\n");
  }
  theti[locus][chain+1] = theti[locus][chain];
  lntheti[locus][chain+1] = lntheti[locus][chain];
  growi[locus][chain+1] = growi[locus][chain];
  if (locusend) {
     fluc_liketable(theti[locus][chain],growi[locus][chain],
     chain,locus,op->interact);
     if (op->growthused) print_covariance(th,g,chain,locus);
     fluc_printtable(th,g);
     if (op->growthused) print_locusplot(locus);
  }
  free(d->firstd);
  free(d->secondd);
  free(d->fplus);
  free(d->splus);
  free(d);
}  /* fluc_estimate */

#define POP_SCALAR 4       /* theta is 4Nu */
#define NUMBER_RATES 6
#define RATE1 0.0000000001 /* 10e-10 */
#define RATE2 0.00000001   /* 10e-8  */
#define RATE3 0.0000001    /* 10e-7  */
#define RATE4 0.000001     /* 10e-6  */
#define RATE5 0.00001      /* 10e-5  */
#define RATE6 0.001        /* 10e-3  */

/***************************************************************
 * init_mutations sets the mutation rates that will be used in *
 * printing out a more user-readable answer.                   */
void init_mutations(double **mutation_rates)
{

*mutation_rates = (double *)calloc(NUMBER_RATES,sizeof(double));

(*mutation_rates)[0] = RATE1;
(*mutation_rates)[1] = RATE2;
(*mutation_rates)[2] = RATE3;
(*mutation_rates)[3] = RATE4;
(*mutation_rates)[4] = RATE5;
(*mutation_rates)[5] = RATE6;

} /* init_mutations */

#define NUMGENS1 1000  /* number of generations of first displacement */
#define NUMGENS2 100   /* number of generations of second displacement */

/*************************************************************
 * fluc_printout prints out the growth value scaled against  *
 * various mutation rates; and relate the growth rate to the *
 * population size parameter theta.                          */
void fluc_printout(double theta, double growth, double *rates)
{
long i;
double nutheta, nugrowth, change1, change2;

fprintf(outfile,"\n\nTranslation of parameters into population terms\n");
fprintf(outfile,"For a Growth parameter of %f\n",growth);
fprintf(outfile,"And a Theta parameter of %f\n\n",theta);
fprintf(outfile,"                      Population Sizes(generations)\n");
fprintf(outfile,"MutationRate   -1000       -100       Present");
fprintf(outfile,"      +100        +1000\n");

/* old version, sometimes overflows 
for(i = 0; i < NUMBER_RATES; i++) {
   nutheta = theta/(POP_SCALAR*rates[i]);
   nugrowth = growth*rates[i];
   change1 = exp(nugrowth*NUMGENS1);
   change2 = exp(nugrowth*NUMGENS2);
   fprintf(outfile,"%5.4e  %5.4e  %5.4e  %5.4e  %5.4e  %5.4e\n",rates[i],
      nutheta/change1,nutheta/change2,nutheta,nutheta*change2,
      nutheta*change1);
   }
*/
for (i = 0; i < NUMBER_RATES; i++) {
   nutheta = log(theta/(POP_SCALAR*rates[i]));
   nugrowth = growth*rates[i];
   change1 = nugrowth * NUMGENS1;
   change2 = nugrowth * NUMGENS2;
   fprintf(outfile,"%5.4e  ",rates[i]);
   if (nutheta - change1 > EXPMIN) 
        fprintf(outfile,"%5.4e  ",exp(nutheta-change1));
   else fprintf(outfile,"**********  ");
   if (nutheta - change2 > EXPMIN) 
        fprintf(outfile,"%5.4e  ",exp(nutheta-change2));
   else fprintf(outfile,"**********  ");
   fprintf(outfile,"%5.4e  ",exp(nutheta));
   if (nutheta + change2 < EXPMAX)
        fprintf(outfile,"%5.4e  ",exp(nutheta+change2));
   else fprintf(outfile,"**********  ");
   if (nutheta + change1 < EXPMAX)
        fprintf(outfile,"%5.4e  ",exp(nutheta+change1));
   else fprintf(outfile,"**********  ");
   fprintf(outfile,"\n");
}
fprintf(outfile,"\n\n");

} /* fluc_printout */

/**************************************************************
 * fluc_printtable is the driver for the mutation rate scaled *
 * table printer.                                             */
void fluc_printtable(double theta, double growth)
{
double *mutes;

init_mutations(&mutes);
fluc_printout(theta,growth,mutes);

free(mutes);

} /* fluc_printtable */

#define NUM_GROWTH 6
#define STEPINCREM 2
#define FARG   10.0
#define MIDG    5.0
#define NEARG   1.0

/********************************************************************
 * fluc_liketable() prints the likelihood curve in growth and theta *
 * for the chain "chain".                                           */
void fluc_liketable(double th, double gr, long chain, long lowcus,
  boolean to_screen)
{
   long i, j, garbage;
   double ltheta, *theta, rrs[NUM_GROWTH], *llike, **dummy, stepsize;
   double lowtheta, lowg, hightheta, highg, fract;
   FILE *location;
   boolean done, logtheta;
   char inchar;

   garbage = 0; /* just in case */
   dummy = NULL;

   llike = (double *)calloc(numtrees,sizeof(double));
   theta = (double *)calloc(numfix+1,sizeof(double));

   for (i = 0; i < numfix; i++) theta[i] = fixed[i];

   theta[numfix] = th;

   stepsize = gr/STEPINCREM;
   if (gr > 0) {
     rrs[0] = gr * (-1.0);
     rrs[1] = gr * (-0.5);
     rrs[2] = 0.0;
     rrs[3] = gr * 0.5;
     rrs[4] = gr;
     rrs[5] = gr * 2.0;
   } else {
     rrs[0] = gr * 2.0;
     rrs[1] = gr;
     rrs[2] = gr * 0.5;
     rrs[3] = 0.0;
     rrs[4] = gr * (-0.5);
     rrs[5] = gr * (-1.0);
   }

   done = false;

   do {

   if (to_screen) location=stdout;
   else {
     location=outfile;
     if (op->progress) fprintf(stdout,"Table printed to file.\n");
   }

   /* now do the actual likelihood calculations */

   fprintf(location,
     "\n  ============================================================================\n");
   fprintf(location,"             ln(Likelihood) for various values of Theta and growth\n");
   fprintf(location,
     "\n  ============================================================================\n");
   fprintf(location,"                                   growth\n");
   fprintf(location,"\n   Theta  ||  ");
   for(i = 0; i < NUM_GROWTH; i++) {
      fprintf(location," %9.4f",rrs[i]);
      if (rrs[i]==gr) fprintf(location,"*");
      else fprintf(location," ");
   }
   fprintf(location,
     "\n  ============================================================================\n");
   for(i = 0; i < numfix; i++) {
      fprintf(location,"%10.4f ||",theta[i]);
      for(j = 0; j < NUM_GROWTH; j++) {
         ltheta = 
            model_likelihood(theta[i],rrs[j],chain,garbage,lowcus,dummy);
         if (ltheta <= -1000.0) fprintf(location,"     ----  ");
         else fprintf(location," %10.4f",ltheta);
      }
      fprintf(location,"\n");
      if (theta[numfix]>theta[i] && theta[numfix]<=theta[i+1]) {
        fprintf(location,"%10.4f*|| ",theta[numfix]);
        for(j = 0; j < NUM_GROWTH; j++) {
           ltheta = 
              model_likelihood(theta[numfix],rrs[j],chain,garbage,lowcus,dummy);
           if (ltheta <= -1000.0) fprintf(location," %10.2f",ltheta);
           else fprintf(location,"%10.4f",ltheta);
           if (rrs[j]==gr)fprintf(location,"*");
           else fprintf(location," ");
        }
        fprintf(location,"\n");
      }
   }
   fprintf(location,
     "  ============================================================================\n");

   //fprintf(location,"Dashed lines indicate lnL values less than -1000;\n");
   fprintf(location,"Asterisks indicate the MLEs of Theta and growth\n");

   if (to_screen) {
     fprintf(location,"\nAccept this table? [y/n]\n");
     scanf("%c%*[^\n]",&inchar);
     getchar();
     if(inchar=='y' || inchar=='Y') to_screen=false;
     else {
/* change Theta values in table */
       fprintf(location,"\nChange Theta values? [y/n]\n");
       scanf("%c%*[^\n]",&inchar);
       getchar();
       if(inchar=='y' || inchar=='Y') {
         fprintf(location,"\nUse logarithmic scale? [y/n]\n");
         scanf("%c%*[^\n]",&inchar);
         getchar();
         if (inchar=='y' || inchar=='Y') logtheta = true;
         else logtheta = false;
         do {
           fprintf(location,"Lower bound on Theta?\n");
           scanf("%lf%*[^\n]",&lowtheta);
           getchar();
           if (lowtheta <= 0.0) fprintf(location,"Illegal value\n");
         } while (lowtheta <= 0.0);
         if (logtheta) lowtheta = log(lowtheta);
         do {
           fprintf(location,"Upper bound on Theta?\n");
           scanf("%lf%*[^\n]",&hightheta);
           getchar();
           if (hightheta <= lowtheta) fprintf(location,"Illegal value\n");
         } while (hightheta <= lowtheta);
         if (logtheta) hightheta = log(hightheta);
         fract = (hightheta-lowtheta)/(double)(numfix-1);
         for (i = 0; i < numfix; i++) {
           theta[i] = lowtheta + ((double)i * fract);
           if (logtheta) theta[i] = exp(theta[i]);
         }
       }
/* change g values in table */
       fprintf(location,"\nChange growth values? [y/n]\n");
       scanf("%c%*[^\n]",&inchar);
       getchar();
       if(inchar=='y' || inchar=='Y') {
         fprintf(location,"Lower bound on growth?\n");
         scanf("%lf%*[^\n]",&lowg);
         getchar();
         do {
           fprintf(location,"Upper bound on growth?\n");
           scanf("%lf%*[^\n]",&highg);
           getchar();
           if (highg <= lowg) fprintf(location,"Illegal value\n");
         } while (highg <= lowg);
         fract = (highg - lowg)/3.0;
         for (i = 0; i < 4; i++) rrs[i] = lowg + ((double)i * fract);
         for (i = 0; i < 4; i++) {
           if (rrs[i] > 0.0) {
             for (j = 4; j > i; j--) {
               rrs[j] = rrs[j-1];
             }
             rrs[i] = 0.0;
             break;
           }
         }
         for (i = 0; i < 5; i++) {
           if (rrs[i] > gr) {
             for (j = 5; j > i; j--) {
               rrs[j] = rrs[j-1];
             }
             rrs[i] = gr;
             break;
           }
         }
       }
     }
   } else done=true;

   } while (!done);

   free(llike);
   free(theta);

} /* fluc_liketable */

#define BOGUSFLAG -9999999.0

/*******************************************************************
 * Calculate the covariance matrix and approximate SDs.  The       *
 * covariance matrix is the -inverse of a matrix whose terms are   *
 * 1/L d_i d_j (L) - 1/(L^2) d_i (L) d_j (L).  Thanks to Hidetoshi *
 * Shimodaira for providing the equations, though he is not to be  *
 * blamed for the implementation.                                  *
 *******************************************************************/
void print_covariance(double theta, double g, long chain, long lowcus)
{

double covar[NUMPARAMETERS*NUMPARAMETERS];

/* HORRIBLE recreation of variables needed to call init_matrix
   (we need to get rid of this ASAP but Jon doesn't want to
   resort to globals to do it) */

long *fxplus, **dfxplus;
double *fx, **dfx;
boolean multilocus;
derivlist *d;
double denom, temp1, temp2, max, aterm, bterm, cterm, dterm, maxlike;
double **dummy;
long denomsign, i, chaintype, garbage;

garbage = 0; /* just in case */
dummy = NULL;

if (holding) return;

chaintype = TYPE_CHAIN(chain);

fxplus = (long *)calloc(NUMPARAMETERS,sizeof(long));
dfxplus = (long **)calloc(NUMPARAMETERS,sizeof(long *));
dfxplus[0] = (long *)calloc(NUMPARAMETERS*NUMPARAMETERS, sizeof(long));
fx = (double *)calloc(NUMPARAMETERS,sizeof(double));
dfx = (double **)calloc(NUMPARAMETERS,sizeof(double *));
dfx[0] = (double *)calloc(NUMPARAMETERS*NUMPARAMETERS,sizeof(double));
for(i = 1; i < NUMPARAMETERS; i++) {
   dfxplus[i] = dfxplus[0] + i * NUMPARAMETERS;
   dfx[i] = dfx[0] + i * NUMPARAMETERS;
}
d = (derivlist *)calloc(1,sizeof(derivlist));
d->firstd = (double *)calloc(op->numout[chaintype],sizeof(double));
d->secondd = (double *)calloc(op->numout[chaintype],sizeof(double));
d->fplus = (long *)calloc(op->numout[chaintype],sizeof(long));
d->splus = (long *)calloc(op->numout[chaintype],sizeof(long));

/* end of HORRIBLE code */

if(lowcus==-1)multilocus=true;
else multilocus=false;

maxlike = model_likelihood(theta,g,chain,garbage,lowcus,dummy);
init_matrix(fx,dfx,g,theta,fxplus,dfxplus,chain,d,multilocus);

/* all these puppies would have a minus in front, if only they weren't
logs; this explains many of the odd signs in what follows */
aterm = 1 - maxlike + dfx[0][0];
bterm = 1 - maxlike + dfx[0][1];
cterm = 1 - maxlike + dfx[1][0];
dterm = 1 - maxlike + dfx[1][1];

temp1 = aterm + dterm;

if (!bterm && !cterm) {
   temp2 = bterm + cterm;
   if (temp1 > temp2) max = temp1;
   else max = temp2;
   denom = 0.0;
   if (temp1 - max > EXPMIN)
      denom += dfxplus[0][0] * dfxplus[1][1] * exp(temp1 - max);
   if (temp2 - max > EXPMIN)
      denom -= dfxplus[0][1] * dfxplus[1][0] * exp(temp2 - max);
   denomsign = whatsign(denom);
   denom = log(fabs(denom)) + max;
} else {
   denomsign = dfxplus[0][0] * dfxplus[1][1];
   denom = temp1;
}

/* check that dfx is invertable */
if(!denom) {
  for(i=0;i<4;i++) covar[i] = BOGUSFLAG;
} else {

  covar[0]=dterm-denom;
  if((covar[0]<EXPMAX) && (covar[0]>EXPMIN))
    covar[0]=-denomsign*dfxplus[1][1]*exp(covar[0]);
  else covar[0] = BOGUSFLAG;

  covar[1]=bterm-denom;
  if((covar[1]<EXPMAX) && (covar[1]>EXPMIN))
    covar[1]=denomsign*dfxplus[0][1]*exp(covar[1]);
  else covar[1] = BOGUSFLAG;

  covar[2]=cterm-denom;
  if((covar[2]<EXPMAX) && (covar[2]>EXPMIN))
    covar[2]=denomsign*dfxplus[1][0]*exp(covar[2]);
  else covar[2] = BOGUSFLAG;

  covar[3]=aterm-denom;
  if((covar[3]<EXPMAX) && (covar[3]>EXPMIN))
    covar[3]=-denomsign*dfxplus[0][0]*exp(covar[3]);
  else covar[3] = BOGUSFLAG;
}

fprintf(outfile,"************************************************\n");
fprintf(outfile,"Covariance matrix (variance of g in upper left)\n\n");
if(covar[0] != BOGUSFLAG)
   fprintf(outfile,"%e    ",covar[0]);
else fprintf(outfile,"******   ");
if(covar[1] != BOGUSFLAG)
   fprintf(outfile,"%e\n",covar[1]);
else fprintf(outfile,"******\n");
if(covar[2] != BOGUSFLAG)
   fprintf(outfile,"%e    ",covar[2]);
else fprintf(outfile,"******   ");
if(covar[3] != BOGUSFLAG)
   fprintf(outfile,"%e\n\n",covar[3]);
else fprintf(outfile,"******\n\n");

fprintf(outfile,"Approximate SD of g:  %f\n",sqrt(covar[0]));
fprintf(outfile,"Approximate SD of theta:  %f\n",sqrt(covar[3]));
fprintf(outfile,"************************************************\n");

free(fxplus);
free(dfxplus[0]);
free(dfxplus);
free(fx);
free(dfx[0]);
free(dfx);
free(d->firstd);
free(d->secondd);
free(d->fplus);
free(d->splus);
free(d);

} /* print_covariance */

void modelfree()
/* free pointers at end of program--helpful in detecting leaks */
{
  long i, j, k;

  for (i=0; i<numloci; i++) {
    for (j=0; j<op->numchains[1]+1;j++) {
      for (k=0; k<numtrees;k++) {
        free(sum[i][j][k].kk);
        free(sum[i][j][k].kend);
      }
    }
  }
  free(sum[0][0]);
  free(sum[0]);
  free(sum);

  free(locuslike);
  free(theti[0]);
  free(theti);
  free(growi[0]);
  free(growi);
  free(lntheti[0]);
  free(lntheti);
  free(savethetai[0]);
  free(savethetai);
  free(fixed);

} /* modelfree */

