/*****************************************************************************
 *  ENTROPY - emerging network to reduce orwellian potency yield
 *
 *  Copyright (C) 2003 Juergen Buchmueller <pullmoll@stop1984.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software Foundation,
 *  Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 *	$Id: crypt6.c,v 1.3 2005/07/12 23:12:29 pullmoll Exp $
 *****************************************************************************/
#include "crypt.h"
#if	CRYPT6_TEST
#define	xmalloc(size) malloc(size)
#define	xcalloc(num,size) calloc(num,size)
#define	xfree(ptr) if (NULL != ptr) { free(ptr); ptr = NULL; }
#else
#include "memalloc.h"
#endif
#include "logger.h"

/*
 * Lorenz' non linear equations:
 *
 * dx/dt = sigma * y - sigma * x
 * dy/dt = rho * x - y - x * z
 * dz/dt = x * y - gamma * z
 *
 * ---
 * Parameters according to
 * Ian Stewart, "Spielt Gott Roulette", (Insel TB, Frankfurt 1993)
 * sigma = 10.0
 * rho = 28.0
 * gamma = 8.0/3.0
 */

#define	FXBITS	32
#define	FXMULT	((int64_t)1<<FXBITS)
#define	FXMASK	(uint64_t)(FXMULT-1)

/* sigma = 10.0 */
#define	SIGMA	((int64_t)10 << FXBITS)

/* rho = 28.0 */
#define	RHO		((int64_t)28 << FXBITS)

/* gamma = 8.0/3.0 = 2.66666666... (32 bits fraction is 0x55555555) */
#define	GAMMA	(((int64_t)2 << FXBITS) | 0x55555555)

/* step width = 1/8192 */
#define	STEPSHIFT	13

#define	CRYPT6_MAGIC	0x4C4F525A
#define	KEYSIZE			4096

typedef struct crypt6_s {
	int magic;
	int64_t xold;			/* Lorenz attractor current values */
	int64_t yold;
	int64_t zold;
	size_t offs;			/* current offset into the key */
	size_t size;			/* allocated size of the key */
	uint8_t key[KEYSIZE];	/* the key */
}	crypt6_t;

int crypt6_crypto(void)
{
	FUN("crypt6_crypto");

	/* no initialization required */

	return 0;
}

static int64_t fxmul(int64_t a, int64_t b)
{
	uint32_t al, ah, bl, bh;
	uint64_t t1, t2, t3;
	int64_t result;
	int sign = 0;

	if (a < 0) {
		sign ^= 1;
		a = -a;
	}
	al = (uint32_t)a;
	ah = (uint32_t)(a >> 32);

	if (b < 0) {
		sign ^= 1;
		b = -b;
	}
	bl = (uint32_t)b;
	bh = (uint32_t)(b >> 32);

	t1 = al * bl;
	t1 = t1 >> 32;
	t2 = (uint64_t)al * (uint64_t)bh;
	t3 = (uint64_t)ah * (uint64_t)bl;
	result = (uint32_t)(t1 + t2 + t3);
	t1 = (t2 >> 32) + (t3 >> 32);
	t2 = (uint64_t)ah * (uint64_t)bh;
	result |= (int64_t)((t1 + t2) << 32);
	
	if (0 != sign)
		result = -result;
	return result;
}

#if	CRYPT6_TEST_PRINT
static char fxsign(int64_t x)
{
	return (x < 0) ? '-' : '+';
}

static unsigned fxint(int64_t x)
{
	return (unsigned)((x < 0) ? (-x >> FXBITS) : (x >> FXBITS));
}

static unsigned fxfract(uint64_t mult, int64_t x)
{
	if (x < 0)
		x = -x;
	return (unsigned)(mult * (x & FXMASK) / FXMULT);
}
#endif

static uint8_t crypt6_next(void *ptr)
{	
	crypt6_t *c = (crypt6_t *)ptr;
	int64_t x = c->xold;
	int64_t y = c->yold;
	int64_t z = c->zold;
	int64_t dx, dy, dz;
	uint8_t res, px, py, pz;

	dx = fxmul(SIGMA, y - x);
	dy = fxmul(RHO, x) - y - fxmul(x, z);
	dz = fxmul(x, y) - fxmul(GAMMA, z);

	c->xold = x + (dx >> STEPSHIFT);
	c->yold = y + (dy >> STEPSHIFT);
	c->zold = z + (dz >> STEPSHIFT);

	dx >>= 3;
	px = (uint8_t)dx;
	dx >>= 9;

	dy >>= 3;
	py = (uint8_t)dy;
	dy >>= 9;

	dz >>= 3;
	pz = (uint8_t)dz;
	dz >>= 9;

	c->offs += (uint8_t)dx + (uint8_t)dy + (uint8_t)dz;
	c->offs %= c->size;

	res = (c->key[c->offs] ^ px) + py;
	c->key[c->offs] = pz;

#if	CRYPT6_TEST_PRINT
	printf("%c%u.%09u %c%u.%09u %c%u.%09u\n",
		fxsign(x), fxint(x), fxfract(1000000000,x),
		fxsign(y), fxint(y), fxfract(1000000000,y),
		fxsign(z), fxint(z), fxfract(1000000000,z));
#endif
	return res;
}

/***********************************************************************
 *  encrypt_msg()
 *  Encrypt a 'plaintext' of length 'size' into a 'ciphertext'
 ***********************************************************************/
int crypt6_encrypt_msg(void *ptr, void *ciphertext, void *plaintext,
	size_t *outsize, size_t insize)
{
	crypt6_t *c = (crypt6_t *)ptr;
	uint8_t *ct = (uint8_t *)ciphertext;
	size_t i;
	FUN("crypt6_encrypt_msg");

	if (NULL == c || NULL == ciphertext || NULL == plaintext) {
		LOGS(L_CRYPTO,L_ERROR,("one or more arguments are NULL"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == plaintext) {
		LOGS(L_CRYPTO,L_ERROR,("plaintext is NULL"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == ciphertext) {
		LOGS(L_CRYPTO,L_ERROR,("ciphertext is NULL"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == outsize) {
		LOGS(L_CRYPTO,L_ERROR,("outsize is NULL"));
		errno = EINVAL;
		return -1;
	}
	if (insize > *outsize) {
		LOGS(L_CRYPTO,L_ERROR,("outsize is too small (have %x, want %x)",
			(unsigned)*outsize, (unsigned)insize));
		errno = ENOMEM;
		return -1;
	}
	if (CRYPT6_MAGIC != c->magic) {
		LOGS(L_CRYPTO,L_ERROR,("invalid magic %x (expected %x)",
			c->magic, CRYPT6_MAGIC));
		errno = EINVAL;
		return -1;
	}

	if (ciphertext != plaintext) {
		/* copy the plaintext to the ciphertext buffer */
		memcpy(ciphertext, plaintext, insize);
	}

	for (i = 0; i < insize; i++)
		*ct++ ^= crypt6_next(c);

	*outsize = insize;
	return 0;
}

/***********************************************************************
 *  decrypt_msg()
 *  Decrypt a 'ciphertext' of the length 'size' into a 'plaintext'
 ***********************************************************************/
int crypt6_decrypt_msg(void *ptr, void *plaintext, void *ciphertext,
	size_t *outsize, size_t insize)
{
	crypt6_t *c = (crypt6_t *)ptr;
	uint8_t *pt = (uint8_t *)plaintext;
	size_t i;
	FUN("crypt6_decrypt_msg");

	if (NULL == ptr) {
		LOGS(L_CRYPTO,L_ERROR,("ptr is NULL"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == plaintext) {
		LOGS(L_CRYPTO,L_ERROR,("plaintext is NULL"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == ciphertext) {
		LOGS(L_CRYPTO,L_ERROR,("ciphertext is NULL"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == outsize) {
		LOGS(L_CRYPTO,L_ERROR,("outsize is NULL"));
		errno = EINVAL;
		return -1;
	}
	if (insize > *outsize) {
		LOGS(L_CRYPTO,L_ERROR,("outsize is too small (have %x, want %x)",
			(unsigned)*outsize, (unsigned)insize));
		errno = ENOMEM;
		return -1;
	}
	if (CRYPT6_MAGIC != c->magic) {
		LOGS(L_CRYPTO,L_ERROR,("invalid magic %x (expected %x)",
			c->magic, CRYPT6_MAGIC));
		errno = EINVAL;
		return -1;
	}

	if (plaintext != ciphertext) {
		/* copy the ciphertext to the plaintext buffer */
		memcpy(plaintext, ciphertext, insize);
	}

	for (i = 0; i < insize; i++)
		*pt++ ^= crypt6_next(c);

	*outsize = insize;
	return 0;
}

int crypt6_init(void **pptr, int init, void *data, size_t size)
{
	crypt6_t *c;
	FUN("crypt6_init");

	if (NULL == pptr) {
		LOGS(L_CRYPTO,L_ERROR,("pptr is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (0 != init) {
		LOGS(L_CRYPTO,L_ERROR,("init is not 0\n"));
		errno = EINVAL;
		return -1;
	}
	if (size < 12) {
		LOGS(L_CRYPTO,L_ERROR,("need at least 12 bytes session key\n"));
		errno = EINVAL;
		return -1;
	}

	c = (crypt6_t *)xcalloc(sizeof(crypt6_t), 1);
	*pptr = c;

	if (size > KEYSIZE) {
		size = KEYSIZE;
	}

	memcpy(c->key, data, size);
	c->xold = c->key[0];
	c->xold |= (uint32_t)c->key[1] << 8;
	c->xold |= (uint32_t)c->key[2] << 16;
	c->xold |= (uint32_t)c->key[3] << 24;
	c->xold &= FXMASK;

	c->yold = c->key[4];
	c->yold |= (uint32_t)c->key[5] << 8;
	c->yold |= (uint32_t)c->key[6] << 16;
	c->yold |= (uint32_t)c->key[7] << 24;
	c->yold &= FXMASK;

	c->zold = c->key[8];
	c->zold |= (uint32_t)c->key[9] << 8;
	c->zold |= (uint32_t)c->key[10] << 16;
	c->zold |= (uint32_t)c->key[11] << 24;
	c->zold &= FXMASK;

	c->size = size;
	c->magic = CRYPT6_MAGIC;

	/* fill up the key if it was less than KEYSIZE */
	while (size < KEYSIZE) {
		c->key[size++] = crypt6_next(c);
		c->size = size;
	}

	return 0;
}

int crypt6_exit(void *ptr)
{
	crypt6_t *c = ptr;
	FUN("crypt6_exit");

	if (NULL == ptr) {
		LOGS(L_CRYPTO,L_ERROR,("ptr is NULL"));
		errno = EINVAL;
		return -1;
	}
	if (CRYPT6_MAGIC != c->magic) {
		LOGS(L_CRYPTO,L_ERROR,("invalid magic %x (expected %x)",
			c->magic, CRYPT6_MAGIC));
		errno = EINVAL;
		return -1;
	}

	xfree(c);

	return 0;
}

#if	CRYPT6_TEST
configuration_t *g_conf = NULL;
#define	SIZE	1024

int pm_vsnprintf(char *dst, size_t size, const char *fmt, va_list ap)
{
	return vsnprintf(dst, size, fmt, ap);
}

int pm_snprintf(char *dst, size_t size, const char *fmt, ...)
{
	va_list ap;
	int len;

	va_start(ap, fmt);
	len = vsnprintf(dst, size, fmt, ap);
	va_end(ap);
	return len;
}

int main(int argc, char **argv)
{
	char rndname[MAXPATHLEN] = "/dev/urandom";
	char outname[MAXPATHLEN] = "crypt6.bin";
	FILE *rnd, *out;
	void *ctx;
	int mode = 0;
	size_t outsize = SIZE;
	size_t insize = SIZE;
	size_t count = 1;
	size_t n;
	size_t initial_size;
	unsigned char *initial;
	unsigned char *plaintext;
	unsigned char *ciphertext;
	int i;
	int rc;

	for (i = 1; i < argc; i++) {
		if (0 == strcmp(argv[i], "--diehard")) {
			insize = outsize = 0x1000;
			count = 0x1000;
		} else if (0 == strncmp(argv[i], "-r", 2) && strlen(argv[i]) > 2) {
			strcpy(rndname, &argv[i][2]);
		} else if (0 == strcmp(argv[i], "-r")) {
			strcpy(rndname, argv[++i]);
		} else if (0 == strncmp(argv[i], "-o", 2) && strlen(argv[i]) > 2) {
			strcpy(rndname, &argv[i][2]);
		} else if (0 == strcmp(argv[i], "-o")) {
			strcpy(rndname, argv[++i]);
		} else if (0 == strncmp(argv[i], "-m", 2) && strlen(argv[i]) > 2) {
			mode = strtol(&argv[i][2], NULL, 0);
		} else if (0 == strcmp(argv[i], "-m")) {
			mode = strtoul(argv[++i], NULL, 0);
		} else if (0 == strncmp(argv[i], "-s", 2) && strlen(argv[i]) > 2) {
			insize = outsize = strtoul(&argv[i][2], NULL, 0);
		} else if (0 == strcmp(argv[i], "-s")) {
			insize = outsize = strtoul(argv[++i], NULL, 0);
		} else if (0 == strncmp(argv[i], "-c", 2) && strlen(argv[i]) > 2) {
			count = strtoul(&argv[i][2], NULL, 0);
		} else if (0 == strcmp(argv[i], "-c")) {
			count = strtoul(argv[++i], NULL, 0);
		}
	}

	initial_size = 256;

#ifndef	CRYPT6_TEST_PRINT
	printf("open random source: %s\n", rndname);
#endif
	rnd = fopen(rndname, "rb");
	if (NULL == rnd) {
		perror(rndname);
		exit(1);
	}

	initial = xcalloc(initial_size, sizeof(u_int8_t));
	if (initial_size != fread(initial, 1, initial_size, rnd)) {
		perror("PRNG data");
		exit(2);
	}
	fclose(rnd);

#ifndef	CRYPT6_TEST_PRINT
	printf("create output file: %s\n", outname);
#endif
	out = fopen(outname, "wb");
	if (NULL == out) {
		perror(outname);
		exit(3);
	}

	plaintext = xcalloc(insize, sizeof(u_int8_t));
	ciphertext = xcalloc(outsize, sizeof(u_int8_t));

	if (0 != (rc = crypt6_init(&ctx, mode, initial, initial_size))) {
		fprintf(stderr, "crypt6_init failed (%d)", rc);
		exit(4);
	}

	for (n = 0; n < count; n++) {
		outsize = insize;
		if (0 != crypt6_encrypt_msg(ctx, ciphertext, plaintext,
			&outsize, insize)) {
			fprintf(stderr, "crypt6_encrypt_msg failed (%d)", rc);
			exit(5);
		}
		if (outsize != fwrite(ciphertext, 1, outsize, out)) {
			perror("fwrite");
			exit(6);
		}
	}

	fclose(out);

	return 0;
}

#endif
