 /*****************************************************************************
 *  ENTROPY - emerging network to reduce orwellian potency yield
 *
 *  Copyright (C) 2003 Juergen Buchmueller <pullmoll@stop1984.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software Foundation,
 *  Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 *	$Id: crypt5.c,v 1.3 2005/07/12 23:12:29 pullmoll Exp $
 *****************************************************************************/

/* Implementation einer stromverschlsselung fr ENTROPY-Projekt.
 * Durchsatz auf einem 800MHz-K7: ca. 10MB/s (cc -O -O2 -O3 ...).
 *											 Mr.Ikg. 05.May.2003.
 */

#include	"osd.h"
#include	"crypt5.h"
#if	CRYPT5_TEST
#define	xmalloc(size) malloc(size)
#define xcalloc(num,size) calloc(num,size)
#define xfree(ptr) if (NULL != ptr) { free(ptr); ptr = NULL; }
#else
#include	"memalloc.h"
#endif
#include	"logger.h"

#define	EBUFFSIZE	1024

static const uint8_t entropy_crypt6[EBUFFSIZE] =
	"\027\327\365\107\241\035\070\165\233\072\003\104\377\312\040\144"
	"\276\061\350\050\350\233\051\155\053\007\007\124\043\066\072\221"
	"\056\365\065\372\027\067\126\357\214\311\070\016\045\076\234\074"
	"\210\014\012\026\070\010\205\031\246\250\161\002\065\227\104\242"
	"\353\000\247\156\271\045\370\111\050\117\174\226\232\352\244\061"
	"\227\272\327\143\245\106\163\131\270\354\133\047\253\262\211\145"
	"\275\160\135\312\004\122\247\161\312\245\120\161\070\125\336\142"
	"\374\076\042\275\214\374\070\364\377\026\053\240\267\047\131\176"
	"\051\370\347\330\136\030\075\037\375\007\357\100\036\045\214\021"
	"\047\267\214\366\205\217\303\274\365\136\207\375\357\300\166\044"
	"\263\253\156\044\130\250\166\325\304\225\170\014\337\365\143\322"
	"\262\065\120\104\340\022\173\157\137\027\031\265\224\223\301\262"
	"\265\242\223\106\272\303\342\155\037\330\226\150\044\132\337\301"
	"\054\030\354\236\077\161\130\071\300\322\261\204\014\075\347\003"
	"\266\152\065\046\324\312\115\337\103\210\275\163\060\363\246\306"
	"\356\052\161\216\206\225\116\374\301\205\253\366\327\254\054\174"
	"\263\150\157\253\030\252\376\352\120\257\356\340\362\202\307\322"
	"\125\146\175\327\171\006\260\313\224\322\163\353\231\070\256\373"
	"\050\223\245\262\264\324\254\014\365\245\074\036\243\021\100\331"
	"\127\377\253\102\013\330\230\126\132\157\265\231\172\017\301\246"
	"\132\020\057\163\253\075\015\050\262\054\151\131\262\010\033\140"
	"\165\354\252\314\076\167\120\372\202\305\073\355\011\000\161\355"
	"\057\130\077\061\177\251\112\003\323\317\313\150\303\025\251\342"
	"\132\076\336\235\341\321\346\102\035\231\030\272\020\324\022\011"
	"\353\074\025\262\355\315\153\373\233\126\366\306\046\071\360\365"
	"\175\104\151\332\305\324\222\326\200\353\350\022\074\210\232\276"
	"\276\220\153\343\273\343\230\273\225\076\344\241\327\172\117\164"
	"\012\135\354\324\200\373\301\115\011\155\232\366\267\110\045\166"
	"\317\306\343\360\310\241\003\140\167\330\053\331\323\253\313\134"
	"\135\265\264\311\074\247\044\005\067\200\220\037\205\270\142\017"
	"\376\114\336\012\060\154\254\134\107\342\302\037\315\074\253\306"
	"\333\013\113\211\304\147\103\206\342\130\273\217\177\244\007\263"
	"\106\010\236\316\337\152\351\140\035\045\253\045\006\054\172\223"
	"\272\363\332\237\041\130\147\321\300\166\324\225\076\141\121\073"
	"\135\324\353\172\257\230\322\042\330\133\325\036\042\164\364\151"
	"\157\306\132\134\315\105\071\172\053\001\163\250\104\006\130\216"
	"\120\113\141\316\050\253\322\127\124\027\047\175\020\000\013\025"
	"\034\210\271\133\237\300\204\040\201\240\364\360\235\320\124\274"
	"\250\351\074\250\304\000\320\312\107\262\111\073\202\371\167\330"
	"\044\374\241\207\140\002\234\017\072\230\166\007\245\102\320\123"
	"\013\337\042\327\155\266\071\237\041\351\364\324\011\061\147\057"
	"\267\271\036\303\363\040\315\302\225\145\342\356\220\072\161\122"
	"\266\032\206\237\302\221\250\114\364\063\242\131\037\061\324\355"
	"\047\331\155\061\207\322\360\260\310\130\143\110\327\344\122\155"
	"\345\177\302\045\374\177\147\201\147\361\353\212\357\377\304\047"
	"\266\242\133\323\232\357\012\257\172\332\033\331\303\103\223\316"
	"\346\353\311\102\170\035\362\207\351\151\270\332\274\052\372\363"
	"\112\016\053\371\350\313\200\002\103\146\000\047\262\132\355\052"
	"\262\143\371\116\141\130\257\174\051\313\065\004\362\044\151\355"
	"\072\077\106\245\171\026\376\363\257\270\257\117\347\300\264\204"
	"\261\364\150\100\352\055\207\306\250\265\001\175\034\035\010\214"
	"\173\154\330\034\247\051\321\157\231\011\141\076\165\312\012\205"
	"\330\066\004\306\004\061\136\151\252\117\205\007\017\261\350\116"
	"\022\023\264\010\362\055\003\354\134\377\333\240\302\105\104\220"
	"\351\237\070\155\250\334\171\332\346\163\002\327\075\034\302\354"
	"\025\140\125\102\072\356\033\107\123\026\215\343\114\007\002\320"
	"\263\255\073\106\173\040\076\230\110\312\073\354\224\060\251\322"
	"\173\323\074\066\205\073\002\202\034\305\373\103\224\232\325\350"
	"\127\275\067\245\024\245\110\253\313\003\343\271\244\355\312\176"
	"\105\037\175\074\112\070\244\014\152\372\245\101\040\311\207\163"
	"\030\137\260\372\227\345\170\225\140\262\253\051\151\117\320\174"
	"\156\151\050\331\206\024\047\026\106\274\354\351\227\206\034\336"
	"\360\147\225\054\303\024\246\355\154\003\137\376\003\117\376\306"
	"\360\360\374\350\063\050\311\070\277\070\013\264\230\326\327\155"
	;

static const uint8_t entropy_crypt5[EBUFFSIZE] =
	"\025\227\255\317\157\361\356\267\334\247\141\107\070\005\343\252"
	"\374\104\157\031\012\151\036\065\260\204\056\176\161\107\261\123"
	"\145\015\030\333\070\115\131\024\362\361\033\023\145\152\162\023"
	"\031\341\324\031\324\316\007\151\315\010\171\162\055\350\100\136"
	"\132\053\270\274\161\154\345\204\170\342\175\350\277\326\273\237"
	"\004\257\026\063\346\334\201\241\004\115\335\126\124\272\347\137"
	"\032\063\125\316\360\166\102\024\116\303\364\021\263\014\035\340"
	"\243\201\231\172\141\066\306\260\011\351\076\211\331\054\266\117"
	"\315\213\014\260\243\371\365\136\323\075\164\331\273\256\155\242"
	"\166\341\054\013\021\214\351\101\143\366\206\323\030\003\357\351"
	"\345\203\107\136\115\244\135\337\042\265\276\244\112\170\205\215"
	"\123\022\231\335\262\266\330\322\164\354\044\074\343\373\364\352"
	"\173\304\255\374\023\002\051\317\341\206\043\114\032\106\260\073"
	"\075\125\370\034\207\135\222\124\317\361\147\046\227\246\123\254"
	"\320\207\260\222\273\332\255\235\345\021\314\327\333\262\147\353"
	"\312\002\153\317\022\354\241\166\101\205\216\046\116\067\234\360"
	"\114\117\026\174\137\053\175\147\362\363\060\365\364\053\134\112"
	"\033\222\266\031\005\114\122\137\310\310\206\036\027\046\067\106"
	"\254\263\344\257\016\360\054\116\134\367\341\204\363\232\145\031"
	"\116\261\177\074\261\034\154\077\050\111\204\072\167\312\114\113"
	"\343\024\152\234\014\005\320\020\021\367\171\271\331\261\003\164"
	"\204\007\313\244\366\127\124\136\176\206\317\166\131\107\101\224"
	"\272\270\305\355\277\151\177\324\340\154\021\117\123\044\275\376"
	"\273\375\367\073\236\046\225\330\063\240\336\021\013\111\307\352"
	"\036\341\354\234\372\200\236\143\254\150\025\132\355\237\373\340"
	"\014\205\055\024\203\365\027\325\212\236\324\034\004\357\112\001"
	"\267\151\122\306\001\116\306\120\301\012\326\237\203\367\157\237"
	"\047\036\143\253\016\267\035\212\334\322\322\225\060\367\041\365"
	"\004\117\327\341\337\154\122\256\373\064\310\203\141\313\102\306"
	"\352\224\143\217\135\271\237\341\024\306\252\324\071\242\121\264"
	"\102\242\002\034\150\101\112\240\350\261\315\016\171\213\037\313"
	"\110\375\162\237\055\362\243\025\226\217\053\146\144\304\073\057"
	"\072\347\341\350\135\036\221\262\070\031\276\333\222\302\265\053"
	"\305\237\062\364\126\077\241\305\205\173\100\334\134\177\203\127"
	"\033\300\350\264\071\225\005\357\120\160\231\342\307\150\224\220"
	"\361\270\056\204\242\355\330\222\117\032\261\144\217\103\207\371"
	"\271\335\004\171\214\352\015\275\151\241\021\375\000\213\042\067"
	"\165\330\145\375\253\164\055\313\073\351\351\313\166\315\275\230"
	"\125\142\017\243\164\210\304\072\206\310\122\040\147\365\100\201"
	"\307\012\112\275\235\076\226\257\343\142\062\161\101\061\122\155"
	"\353\112\154\326\304\300\223\316\206\037\164\112\030\072\222\107"
	"\060\226\142\240\250\261\231\014\036\313\131\334\165\374\222\125"
	"\310\324\015\206\001\341\300\102\226\205\240\366\015\032\004\272"
	"\215\144\061\241\226\236\274\104\347\154\047\374\254\065\122\172"
	"\200\325\145\077\332\077\206\067\016\311\052\305\057\212\100\275"
	"\373\057\127\041\055\170\061\220\125\131\172\201\102\215\317\143"
	"\125\312\101\352\012\364\147\147\362\105\126\322\001\313\143\202"
	"\005\311\245\002\202\335\151\232\053\153\177\356\016\261\324\352"
	"\116\154\325\226\230\035\154\075\000\304\267\205\234\232\336\163"
	"\061\202\260\143\215\014\107\065\277\077\176\135\122\063\351\045"
	"\025\315\216\062\015\276\220\212\034\135\326\172\063\064\357\012"
	"\126\017\226\015\332\141\060\073\124\142\247\357\067\325\275\341"
	"\320\127\216\125\225\353\056\370\361\115\345\371\162\214\352\242"
	"\123\115\000\170\166\047\341\135\224\043\341\304\354\364\050\325"
	"\333\126\220\047\145\334\117\217\271\024\173\277\255\320\322\105"
	"\154\072\255\110\113\200\127\212\034\222\174\204\007\321\241\052"
	"\356\026\135\055\242\357\026\036\033\120\165\272\332\025\264\217"
	"\051\132\277\332\221\323\303\127\356\360\367\333\242\213\034\142"
	"\043\215\070\373\331\024\224\205\310\046\272\023\327\204\234\237"
	"\362\212\101\275\033\374\122\120\110\345\031\234\304\351\223\005"
	"\157\373\024\171\067\364\366\067\035\353\177\250\240\167\002\144"
	"\054\306\044\230\116\356\113\100\102\144\051\356\050\326\116\026"
	"\217\355\161\223\212\251\010\231\317\342\000\213\250\011\017\015"
	"\100\137\247\223\335\241\071\015\125\314\304\116\312\047\211\342"
	;

#define	MAGICNUM	0x03F902
#define	Mod256(uval)	(uint8_t)(uval)
#define	SWAP(a,b,type)	{type tmp; tmp=a; a=b; b=tmp;}

#define	VICRSTEPS	4
#define	R1	277u
#define	R2	257u
#define	R3	239u
#define	R4	251u
#if	((R1 + R2 + R3 + R4) != EBUFFSIZE)
	#error	Summe der Primzahlen (R1 + R2 + R3 + R4) muss EBUFFSIZE ergeben.
#endif
#define	PRIMACYCLE	((uint32_t)(R1 * R2 * R3 * R4 - 1))

#define	P1	251u
#define	P2	263u
#define	P3	241u
#define	P4	269u
#if	((P1 + P2 + P3 + P4) != EBUFFSIZE)
	#error	The primes sum (P1 + P2 + P3 + P4) must be EBUFFSIZE.
#endif

#define	CYCLESIZE	((uint32_t)(P1 * P2 * P3 * P4 - 1))
/* Die Zykluslngen der ur-PRNGs liegen knapp unter der grenze 2^32	*/

#define	SBOXSIZE	256

typedef	struct	crypt5_s	{
	uint32_t	cyclenum;			/* Zyklen durchgelaufen	*/
	uint32_t	position;			/* 0 .. (CYCLESIZE - 1)	*/
	uint32_t	primapos;
	uint32_t	chngepos;
	uint8_t	pebuf[EBUFFSIZE];	/* primary entropy buffer */
	uint8_t	sebuf[EBUFFSIZE];	/* secondary entropy buffer */
	uint8_t	sbox[SBOXSIZE];		/* s-box permutation */
	uint32_t	last_i, last_j;		/* indices */
	uint32_t	initmagic;
}	crypt5_t;

#define	STARTPOS	41984

#if		CRYPT5_TEST
static  void printsbox(char * titel, uint8_t * sbox)
{
	uint32_t  i, s1 = 0, s2 = 0;

	printf( "%s\n", titel );
	for ( i = 0; i < 256; i++ ) {
		if ( !(i & 15 ) )
			printf( "\n%4u", sbox[i] );
		else printf( "%4u", sbox[i] );
		s1 += i;
		s2 += sbox[i];
	}
	printf( "\n\t%d:%d\n", s1, s2 );
}
#endif

/***********************************************************************
 * Vitus' urgenerator, der fr intialisierung eingestzt wird.
 ***********************************************************************/
static uint8_t vi_urrandom(crypt5_t * ctrlc)
{
	uint8_t *entrbuf, nextdig;
	uint32_t currpos;
	FUN("vi_urrandom");

	currpos = ctrlc->primapos++;
	if (currpos >= PRIMACYCLE)
		ctrlc->primapos = 0;
	entrbuf = ctrlc->pebuf;
	nextdig  = entrbuf[currpos % R1];
	nextdig ^= entrbuf[currpos % R2 + R1];
	nextdig ^= entrbuf[currpos % R3 + R2 + R1];
	nextdig ^= entrbuf[currpos % R4 + R3 + R2 + R1];
	LOGS(L_CRYPTO,L_DEBUGX,("%x: %02x\n",
		currpos, nextdig));
	return nextdig;
}

/***********************************************************************
 * Generierung einer zuflligen permutation aka s-box in der ctrlc.
 * STARTPOS - position der untersequenz, die dafr benutzt werden soll.
 * Error:	(-1) - entropie-puffer ist total untauglich fr die methode.
 ***********************************************************************/
static int gen_perm(crypt5_t * ctrlc)
{
	uint32_t i, n, ppos, chsumm = 0;
	int rc = 0;
	FUN("gen_perm");

	n = 1;
	memset(ctrlc->sbox, 0, sizeof(ctrlc->sbox));
	for (i = 0; n < SBOXSIZE && i < 10000; i++) {
		ppos = vi_urrandom(ctrlc);
		chsumm += ppos;
		if (0 == ctrlc->sbox[ppos])
			ctrlc->sbox[ppos] = n++;
	}
	ctrlc->last_i = 0;
	ctrlc->last_j = 0;
#if	CRYPT5_TEST
	printsbox( "\nDEBUG-INITZUSTAND:", ctrlc->sbox );
#endif
	/* Grobe Taugligkeitskontrolle, nur bei DEBUG?	*/
	chsumm /= i;
	if (n != SBOXSIZE) {
		LOGS(L_CRYPTO,L_ERROR,("Bad entropy-buffer\n"));
		errno = EINVAL;
		rc = -1;
	} else if (chsumm < (127-32) || chsumm > (127+32) || i > 2500) {
		LOGS(L_CRYPTO,L_DEBUG,("Entropy-buffer has faultily statstical quality: %d, %d\n",
			chsumm, i));
	}
	return rc;
}

/***********************************************************************
 * Modifizierung einer s-box. Im gegensatz zu Johnson-Trotter durchlauf
 * verhlt sich folgendes hochgradig nichtlinear. Zu details siehe bitte:
 * http://www.cs.columbia.edu/~dcook/candexam/Y_23_rc4_cryptana.pdf
 ***********************************************************************/
static void changesbox(crypt5_t *ctrlc)
{
	uint32_t i, j;

	i = Mod256(ctrlc->last_i + 1);
	j = Mod256(ctrlc->last_j + ctrlc->sbox[i]);
	SWAP(ctrlc->sbox[i], ctrlc->sbox[j], uint8_t);
	ctrlc->last_i = i;
	ctrlc->last_j = j;
}

/***********************************************************************
 * Krypto-PRNG - liefert nchstes zufallsbyte aus dem pseudo-OTP.
 * Jeder 256-block aus der generierten random-sequenz wird individuell
 * permutiert. Somit werden unanbhngige Ereignisse vierter Art in
 * zuflliger und definitiv unbekannter reihenfolge ausgegeben.
 * Der sekundre entropy-puffer verndert sich auch, wodurch die gasamte
 * statistische qualitt des stromes auch oberhalb CYCLESIZE*N, N > 100
 * konstant bleibt.
 ***********************************************************************/
static uint8_t vicr_random(crypt5_t *ctrlc)
{
	uint32_t currpos, hghpart, lowpart;
	uint8_t *entrbuf, nextdig;

	currpos = ctrlc->position++;
	if (currpos == CYCLESIZE) {	/* um berlauf zu vermeiden	*/
		ctrlc->position = 0;
		ctrlc->cyclenum++;
#if		CRYPT5_TEST
		printsbox("\nDEBUG-CYCLESIZE:", ctrlc->sbox);
#endif
	}
	entrbuf = ctrlc->sebuf;
	lowpart = ctrlc->sbox[Mod256(currpos)];
	/* So mchte es optimizer gerne haben! */
	currpos = (currpos & ~0xff) | lowpart;
	if (0 == Mod256(currpos)) {
		ctrlc->chngepos %= EBUFFSIZE;
		hghpart = ctrlc->chngepos++;
		hghpart = (hghpart & ~0xff) | lowpart;

		entrbuf[hghpart%EBUFFSIZE] = vi_urrandom(ctrlc);	/* nicht ^=	*/
		/* Nach ca. einem TB luft zyklus wieder an, allerdings mit anderem
		 * pufferinhalt und anderer ctrlc->chngepos.
		 */
		changesbox(ctrlc);
	}
	nextdig  = entrbuf[currpos % P1];
	nextdig ^= entrbuf[currpos % P2 + P1];
	nextdig ^= entrbuf[currpos % P3 + P2 + P1];
	nextdig ^= entrbuf[currpos % P4 + P3 + P2 + P1];
	return nextdig;
}

/***********************************************************************
 * Initialisierung der stromverschlsselung.
 ***********************************************************************/
static int vicr_init(crypt5_t *ctrlc, uint8_t *cry_key, size_t keysize,
	const uint8_t *entropy_vector)
{
	uint32_t i, ki, klen;
	int rc;

	klen = (keysize > EBUFFSIZE) ? EBUFFSIZE : keysize;
	/* intialize entropy buffer */
	for (i = 0, ki = 0; i < EBUFFSIZE; i++) {
		ctrlc->pebuf[i] = cry_key[ki];
		if (++ki >= klen)
			ki = 0;
	}
	for (i = 0; i < EBUFFSIZE; i++) {
		ctrlc->pebuf[i] ^= entropy_vector[i];
	}
	/* Dieser rckgriff auf konstanten entropy_vector macht den schlssel
	 * aus kryptographischer sicht weder 'besser' noch 'schlechter', weil
	 * es eben ein konstanter vektor ist.
	 * Durch konstante verschiebung im schlsselraum wird jedoch erreicht,
	 * dass stochastische qualitt des puffers, falls sie im schlssel
	 * nicht vorhanden war, auf notwendiges niveau angehoben wird.
	 * Auerdem werden dadurch wiederholungen im entropy-puffer vermieden,
	 * falls der schlssel krzer als EBUFFSIZE ist. Mit letzterem ist im
	 * regelfall zu rechnen. Die Prozedur ist auch fr ASCII-Key geeignet.
	 */

	rc = -1;
	ctrlc->primapos = STARTPOS;
	if (0 == gen_perm(ctrlc)) {
		ctrlc->initmagic = MAGICNUM;
		rc = 0;
	}
	else ctrlc->initmagic = 0;
	for (i = 0; i < EBUFFSIZE; i++) {
		ctrlc->sebuf[i] = vi_urrandom( ctrlc );
	}
	ctrlc->cyclenum = 0;
	ctrlc->position = 0;
	ctrlc->chngepos = 0;
	return rc;
}

/*****************************************************************************
 *								PUBLIC-Part
 *****************************************************************************/

int crypt5_crypto(void)
{
	FUN("crypt5_crypto");

	/* This method needs no module initialization */

	return 0;
}

/*****************************************************************************
 *	crypt5_init()
 *	out-init: pptr		- pointer to pointer to crypt5_t struct
 *	in: data			- crypt key
 *	in: size			- crypt key size
 *****************************************************************************/
int crypt5_init(void **pptr, int init, void *data, size_t size)
{
	crypt5_t *c;

	FUN("crypt5_init");

	if (NULL == pptr) {
		LOGS(L_CRYPTO,L_ERROR,("pptr is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (init < 0 || init > 1) {
		LOGS(L_CRYPTO,L_ERROR,("init is not 0 or 1\n"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == data) {
		LOGS(L_CRYPTO,L_ERROR,("need crypt key\n"));
		errno = EINVAL;
		return -1;
	}
	if (size < 8) {
		LOGS(L_CRYPTO,L_ERROR,("need min. 8 bytes key\n"));
		errno = EINVAL;
		return -1;
	}
	c = (crypt5_t *)xcalloc(sizeof(crypt5_t),1);
	*pptr = c;
	memset(c, 0, sizeof(crypt5_t));
	
	return vicr_init(c, data, size,
			(0 == init) ? entropy_crypt5 : entropy_crypt6);
}

/*****************************************************************************
 *	crypt5_exit()
 *	in: pptr		- pointer to pointer to crypt5_t struct
 *****************************************************************************/
int crypt5_exit(void *ptr)
{
	crypt5_t *c = ptr;
	FUN("crypt5_exit");

	if (NULL == ptr) {
		LOGS(L_CRYPTO,L_ERROR,("ptr is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (MAGICNUM != c->initmagic) {
		LOGS(L_CRYPTO,L_ERROR,("magic is %x (!= %x)\n", c->initmagic, MAGICNUM));
		errno = EINVAL;
		return -1;
	}
	LOGS(L_CRYPTO,L_DEBUG,("ikg2: Total size = %u * %u + %u;\n",
		c->cyclenum, CYCLESIZE, c->position));
	memset(c, 0, sizeof(*c));
	xfree(ptr);

	return 0;
}

/*
 *	crypt5_encrypt_msg()
 *	ptr				in:out - pointer to crypt5_t struct
 *	ciphertext		out - ciphertext
 *	plaintext		in - plaintext
 *	outsize			out - blocksize
 *	insize 			in - blocksize
 */
int	crypt5_encrypt_msg(void *ptr, void *ciphertext, void *plaintext,
	size_t *outsize, size_t insize)
{
	crypt5_t *c = (crypt5_t *)ptr;
	uint8_t *ct = (uint8_t *)ciphertext;
	uint32_t i;
	FUN("crypt5_encrypt");

	if (NULL == ptr) {
		LOGS(L_CRYPTO,L_ERROR,("ptr is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == ciphertext) {
		LOGS(L_CRYPTO,L_ERROR,("ciphertext is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == plaintext) {
		LOGS(L_CRYPTO,L_ERROR,("plaintext is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == outsize) {
		LOGS(L_CRYPTO,L_ERROR,("outsize is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (insize > *outsize) {
		LOGS(L_CRYPTO,L_ERROR,("outsize is too small (have %x, want %x)\n",
			(unsigned)*outsize, (unsigned)insize));
		errno = ENOMEM;
		return -1;
	}
	if (MAGICNUM != c->initmagic) {
		LOGS(L_CRYPTO,L_ERROR,("magic is %x (!= %x)\n",
			c->initmagic, MAGICNUM));
		errno = EINVAL;
		return -1;
	}
	if (ciphertext != plaintext) {
	     memcpy(ciphertext, plaintext, insize);
	}

	for (i = 0; i < insize; i++)
		*ct++ ^= vicr_random(c);

	*outsize = insize;
	return 0;
}

/*
 *	crypt5_decrypt_msg()
 *	ptr				in:out - pointer to crypt5_t struct
 *	plaintext		out - plaintext
 *	ciphertext		in - ciphertext
 *	outsize			out - blocksize
 *	insize 			in - blocksize
 */
int	crypt5_decrypt_msg(void	*ptr, void *plaintext, void *ciphertext,
	size_t *outsize, size_t insize)
{
	crypt5_t *c = (crypt5_t *)ptr;
	uint8_t *pt = (uint8_t *)plaintext;
	uint32_t i;
	FUN("crypt5_decrypt");

	if (NULL == ptr) {
		LOGS(L_CRYPTO,L_ERROR,("ptr is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == ciphertext) {
		LOGS(L_CRYPTO,L_ERROR,("ciphertext is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == plaintext) {
		LOGS(L_CRYPTO,L_ERROR,("plaintext is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == outsize) {
		LOGS(L_CRYPTO,L_ERROR,("outsize is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (insize > *outsize) {
		LOGS(L_CRYPTO,L_ERROR,("outsize is too small (have %x, want %x)\n",
			(unsigned)*outsize, (unsigned)insize));
		errno = ENOMEM;
		return -1;
	}
	if (MAGICNUM != c->initmagic) {
		LOGS(L_CRYPTO,L_ERROR,("magic is %x (!= %x)\n",
			c->initmagic, MAGICNUM));
		errno = EINVAL;
		return -1;
	}

	if (ciphertext != plaintext) {
	     memcpy(plaintext, ciphertext, insize);
	}
	for (i = 0; i < insize; i++)
		*pt++ ^= vicr_random(c);

	*outsize = insize;
	return 0;
}

#if	CRYPT5_TEST
configuration_t *g_conf = NULL;
#define	SIZE	1024

int pm_vsnprintf(char *dst, size_t size, const char *fmt, va_list ap)
{
	return vsnprintf(dst, size, fmt, ap);
}

int pm_snprintf(char *dst, size_t size, const char *fmt, ...)
{
	va_list ap;
	int len;

	va_start(ap, fmt);
	len = vsnprintf(dst, size, fmt, ap);
	va_end(ap);
	return len;
}

int main(int argc, char **argv)
{
	char rndname[MAXPATHLEN] = "/dev/urandom";
	char outname[MAXPATHLEN] = "crypt5_0.bin";
	FILE *rnd, *out;
	void *ctx;
	int mode = 0;
	size_t outsize = SIZE;
	size_t insize = SIZE;
	size_t count = 1;
	size_t n;
	size_t initial_size;
	unsigned char *initial;
	unsigned char *plaintext;
	unsigned char *ciphertext;
	int i;
	int rc;

	for (i = 1; i < argc; i++) {
		if (0 == strcmp(argv[i], "--diehard")) {
			insize = outsize = 0x1000;
			count = 0x1000;
		} else if (0 == strncmp(argv[i], "-r", 2) && strlen(argv[i]) > 2) {
			strcpy(rndname, &argv[i][2]);
		} else if (0 == strcmp(argv[i], "-r")) {
			strcpy(rndname, argv[++i]);
		} else if (0 == strncmp(argv[i], "-o", 2) && strlen(argv[i]) > 2) {
			strcpy(rndname, &argv[i][2]);
		} else if (0 == strcmp(argv[i], "-o")) {
			strcpy(rndname, argv[++i]);
		} else if (0 == strncmp(argv[i], "-m", 2) && strlen(argv[i]) > 2) {
			mode = strtol(&argv[i][2], NULL, 0);
			snprintf(outname, MAXPATHLEN, "crypt5_%d.bin", mode);
		} else if (0 == strcmp(argv[i], "-m")) {
			mode = strtoul(argv[++i], NULL, 0);
			snprintf(outname, MAXPATHLEN, "crypt5_%d.bin", mode);
		} else if (0 == strncmp(argv[i], "-s", 2) && strlen(argv[i]) > 2) {
			insize = outsize = strtoul(&argv[i][2], NULL, 0);
		} else if (0 == strcmp(argv[i], "-s")) {
			insize = outsize = strtoul(argv[++i], NULL, 0);
		} else if (0 == strncmp(argv[i], "-c", 2) && strlen(argv[i]) > 2) {
			count = strtoul(&argv[i][2], NULL, 0);
		} else if (0 == strcmp(argv[i], "-c")) {
			count = strtoul(argv[++i], NULL, 0);
		}
	}

	initial_size = 256;

	printf("open random source: %s\n", rndname);
	rnd = fopen(rndname, "rb");
	if (NULL == rnd) {
		perror(rndname);
		exit(1);
	}

	initial = xcalloc(initial_size, sizeof(uint8_t));
	if (initial_size != fread(initial, 1, initial_size, rnd)) {
		perror("PRNG data");
		exit(2);
	}
	fclose(rnd);

	printf("create output file: %s\n", outname);
	out = fopen(outname, "wb");
	if (NULL == out) {
		perror(outname);
		exit(3);
	}

	plaintext = xcalloc(insize, sizeof(uint8_t));
	ciphertext = xcalloc(outsize, sizeof(uint8_t));

	if (0 != (rc = crypt5_init(&ctx, mode, initial, initial_size))) {
		fprintf(stderr, "crytp5_init failed (%d)", rc);
		exit(4);
	}

	for (n = 0; n < count; n++) {
		outsize = insize;
		if (0 != crypt5_encrypt_msg(ctx, ciphertext, plaintext,
			&outsize, insize)) {
			fprintf(stderr, "crytp5_encrypt_msg failed (%d)", rc);
			exit(5);
		}
		if (outsize != fwrite(ciphertext, 1, outsize, out)) {
			perror("fwrite");
			exit(6);
		}
	}

	fclose(out);

	return 0;
}

#endif
