#ifndef __MAIL_H__
#define __MAIL_H__ 1
/* 
   elmo - ELectronic Mail Operator

   Copyright (C) 2002, 2003, 2004 rzyjontko

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; version 2.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

*/
/****************************************************************************
 *    INTERFACE REQUIRED HEADERS
 ****************************************************************************/

#include <time.h>

#include "ecurses.h"
#include "address.h"
#include "raddress.h"
#include "rstring.h"
#include "mime.h"

/****************************************************************************
 *    INTERFACE DEFINITIONS
 ****************************************************************************/

#define FLAG_READ       0x0001
#define FLAG_OLD        0x0002
#define FLAG_ANSWERED   0x0004
#define FLAG_TRASHED    0x0008
#define FLAG_DRAFT      0x0010
#define FLAG_FLAGGED    0x0020
#define FLAG_PASSED     0x0040
#define FLAG_LEGITIMATE 0x0080
#define FLAG_NOSPAM     0x0100
#define FLAG_SPAM       0x0200
#define FLAG_DUPLICATE  0x0400
#define FLAG_TO_ME      0x0800
#define FLAG_FETCHED    0x1000
#define FLAG_BROKEN     0x2000

/****************************************************************************
 *    INTERFACE ENUMERATIONS / SIMPLE TYPEDEFS
 ****************************************************************************/

typedef enum {
        BOX_INVALID,
        BOX_MBOX,
        BOX_MAILDIR,
        BOX_POP3,
        BOX_SENDER
} box_type_t;


typedef enum {
        ORDER_NO,
        ORDER_DATE,
        ORDER_DATE_REV,
        ORDER_FROM,
        ORDER_FROM_REV,
        ORDER_SUBJECT,
        ORDER_SUBJECT_REV,
        ORDER_THREAD,
        ORDER_SMTP,
        ORDER_EXTERNAL
} box_order_t;

typedef unsigned short flag_t;

/****************************************************************************
 *    INTERFACE CLASS PROTOTYPES / EXTERNAL CLASS REFERENCES
 ****************************************************************************/
/****************************************************************************
 *    INTERFACE STRUCTURES / UTILITY CLASSES
 ****************************************************************************/

struct mail_array;

/**
 * This simple structure describes a single mail and its connections with
 * other mails grouped in mail_array, and in a tree which is built when
 * grouping messages in threads.
 *
 * parent      -- if this mail is not a reply -1, mail_array index otherwise
 * from        -- From: field as an address
 * reply_to    -- Reply-to: field as an address
 * subject     -- Subject: field
 * date_str    -- Date: field
 * to          -- To: field split into an array of addresses
 * cc          -- Cc: field split into an array of addresses
 * bcc         -- Bcc: field split into an array of addresses
 * recv_for    -- Received: [...] for <user@host>; -- email address as
 *                                                    a string
 * msg_id      -- Message-ID: field
 * in_reply_to -- References: field split into an array of message ids or
 *                one message id taken from In-reply-to: field.
 * tree        -- thread tree without colour
 * tree_len    -- tree length
 * flags       -- bit flags (explained above)
 * date        -- Date: field as 'seconds since the Epoch'
 * child_count -- replies count (this does not include replies to reply)
 * mua         -- Mail User Agent
 * mime        -- mime type
 * headers     -- custom headers
 * smtp        -- smtp account
 *
 * place       -- this *HAS* to be unique for every message (as a pointer)
 *
 */
struct mail_st {

        int            parent;  /* parent index */
        address_t     *from;
        address_t     *reply_to;
        char          *subject;
        char          *date_str;
        raddress_t    *to;
        raddress_t    *cc;
        raddress_t    *bcc;
        address_t     *recv_for;
        char          *msg_id;
        rstring_t     *in_reply_to;
        chtype        *tree;
        short          tree_len;
        flag_t         flags;
        char           child_count;
        char           is_last_child;
        time_t         date;
        char          *mua;
        mime_info_t   *mime;
        rstring_t     *headers;
        char          *smtp;

        box_type_t     type;
  
        union {
                void     *generic;
                unsigned  offset_header;
                char     *file_name;
                int       size;
        } place;

};

typedef struct mail_st mail_t;


/**
 * I decided to hold messages in an array.  It is very convenient because
 * you can easily get any message at any time and sorting is trivial.
 *
 * This structure holds informations about a box.  The mail module is
 * responsible for keeping them consistent.
 *
 * path              -- path to the mailbox
 * array             -- the array itself
 * size              -- how many mails may be held in the array
 * count             -- how many mails is held in the array
 * not_replies_count -- how many messages have no parent
 * old_count         -- how many messages is old
 * read_count        -- how many messages is read
 * type              -- box type
 * order             -- how messages are ordered
 * have_parents      -- is a boolean saying if parents are evaluated
 */
typedef struct mail_array {
        char       *path;
        mail_t     *array;
        int         size;
        int         count;
        int         not_replies_count;
        int         old_count;
        int         read_count;
        box_type_t  type;
        box_order_t order;
        int         have_parents;
} mail_array_t;

/****************************************************************************
 *    INTERFACE DATA DECLARATIONS
 ****************************************************************************/
/****************************************************************************
 *    INTERFACE FUNCTION PROTOTYPES
 ****************************************************************************/

#ifdef __cplusplus
extern "C" {
#endif

extern void mail_destroy (mail_t *mail, box_type_t type);
extern void mail_clear (mail_t *mail);

extern mail_array_t *mail_array_create_size (int size, box_type_t type,
                                             const char *path);
extern mail_array_t *mail_array_create (box_type_t type, const char *path);
extern mail_array_t *mail_array_read_from_file (FILE *fp, box_type_t type,
                          const char *path, void (*)(memchunk_t *, mail_t *));
extern void          mail_array_destroy (mail_array_t *marray);
extern void          mail_array_substitute (mail_array_t *marray,
                                            mail_array_t *with);
extern void          mail_array_insert (mail_array_t *marray, mail_t *mail);
extern mail_t       *mail_array_get (mail_array_t *marray, int index);
extern void          mail_array_find_parents (mail_array_t *marray);
extern int           mail_array_not_replies (mail_array_t *marray);
extern int           mail_array_old (mail_array_t *marray);
extern int           mail_array_read (mail_array_t *marray);
extern int           mail_array_first_unread_index (mail_array_t *marray);
extern int           mail_array_dump (mail_array_t *marray, FILE *fp,
                                      void (*fun)(memchunk_t *, mail_t *));
  
extern void mail_array_sort_external (mail_array_t *marray,
                                      int (*cmp)(const void *, const void *));
extern void mail_array_sort_date (mail_array_t *marray);
extern void mail_array_sort_from (mail_array_t *marray);
extern void mail_array_sort_subject (mail_array_t *marray);
extern void mail_array_sort_smtp (mail_array_t *marray);
extern void mail_array_sort_threads (mail_array_t *marray);

extern int mail_array_find_place (mail_array_t *marray, void *ptr);
extern int mail_array_find (mail_array_t *marray, int (*fun)(mail_t *),
                            int start);
extern int mail_array_find_back (mail_array_t *marray, int (*fun)(mail_t *),
                                 int start);
extern int mail_array_find_subject (mail_array_t *marray, const char *sub,
                                    int start);
extern int mail_array_find_from (mail_array_t *marray, const char *from,
                                 int start);
extern int mail_array_find_unread (mail_array_t *marray, int start);
extern int mail_array_find_unread_back (mail_array_t *marray, int start);

#ifdef __cplusplus
}
#endif

/****************************************************************************
 *    INTERFACE OBJECT CLASS DEFINITIONS
 ****************************************************************************/
/****************************************************************************
 *    INTERFACE TRAILING HEADERS
 ****************************************************************************/
/****************************************************************************
 *
 *    END HEADER mail.h
 *
 ****************************************************************************/
#endif

