/* 
   elmo - ELectronic Mail Operator

   Copyright (C) 2002, 2003, 2004 rzyjontko
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; version 2.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

*/

#define _GNU_SOURCE 1

/****************************************************************************
 *    IMPLEMENTATION HEADERS
 ****************************************************************************/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#ifdef HAVE_GETOPT_H
# include <getopt.h>
#endif

#ifdef HAVE_LOCALE_H
# include <locale.h>
#endif

#ifdef HAVE_HOME_ETC_H
# include <home_etc.h>
# define HOMEDIR_RC _HEdir
#else
# define HOMEDIR_RC getenv("HOME")
#endif

#include "ecurses.h"
#include "confhold.h"
#include "confread.h"
#include "status.h"
#include "folder.h"
#include "cmd.h"
#include "hash.h"
#include "address.h"
#include "mailreader.h"
#include "mail.h"
#include "debug.h"
#include "error.h"
#include "elmo.h"
#include "pop.h"
#include "networking.h"
#include "wrapbox.h"
#include "abook.h"
#include "read.h"
#include "exec.h"
#include "gettext.h"
#include "mybox.h"
#include "bayes.h"
#include "token.h"
#include "xmalloc.h"
#include "file.h"
#include "color.h"
#include "interface.h"
#include "box_selection.h"
#include "clock.h"
#include "label.h"
#include "select.h"
#include "topbar.h"
#include "fetch.h"
#include "frames.h"
#include "attach.h"
#include "help.h"
#include "mime.h"
#include "sender.h"
#include "procmail.h"
#include "smtp.h"
#include "pgp.h"
#include "mlex.h"
#include "property.h"
#include "compose.h"

/****************************************************************************
 *    IMPLEMENTATION PRIVATE DEFINITIONS / ENUMERATIONS / SIMPLE TYPEDEFS
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION PRIVATE CLASS PROTOTYPES / EXTERNAL CLASS REFERENCES
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION PRIVATE STRUCTURES / UTILITY CLASSES
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION REQUIRED EXTERNAL REFERENCES (AVOID)
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION PRIVATE DATA
 ****************************************************************************/

#ifdef HAVE_GETOPT_LONG

static struct option const long_options[] =
{
        {"help", no_argument, 0, 'h'},
        {"version", no_argument, 0, 'V'},
        {NULL, required_argument, 0, 'F'},
        {NULL, 0, NULL, 0}
};

# define GETOPT(a,b,c,d,e) getopt_long (a, b, c, d, e)
#else
# define GETOPT(a,b,c,d,e) getopt (a, b, c)
#endif

/**
 * This pointer is used to pass argv[i] from decode_switches to
 * configuration.
 */
static char *config_file = NULL;

static char *imm_email = NULL;

/****************************************************************************
 *    INTERFACE DATA
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION PRIVATE FUNCTION PROTOTYPES
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION PRIVATE FUNCTIONS
 ****************************************************************************/

static void
usage (int status)
{
        printf (_("%s - ELectronic Mail Operator\n"), PACKAGE);
        printf (_("Usage: %s [OPTION]... \n"), PACKAGE);
        printf (_("Options:\n"));
        printf (_("  -F <file>        use alternate rc file\n"));
        printf (_("  -h, --help       display this help and exit\n"));
        printf (_("  -V, --version    output version information and exit\n"));
        exit (status);
}




static int
decode_switches (int argc, char **argv)
{
        int c;
        int ala;

        while ((c = GETOPT (argc, argv, 
                            "F:"  /* alternative config file */
                            "h"	  /* help */
                            "V",  /* version */
                            long_options, &ala)) != EOF){
    
                switch (c){

                        case 'F':
                                config_file = optarg;
                                break;

                        case 'V':
                                printf ("%s %s\n", PACKAGE, VERSION);
                                exit (EXIT_SUCCESS);

                        case 'h':
                                usage (EXIT_SUCCESS);

                        default:
                                usage (EXIT_FAILURE);
                }
        }
         
        return optind;
}         




static void
variable_registration (void)
{
        confhold_register ("addressbook", 0);
        confhold_register ("indent_string", 0);
        confhold_register ("line_format", 0);
        confhold_register ("sent_format", 0);
        confhold_register ("folder_format", 0);
        confhold_register ("abook_format", 0);
        confhold_register ("sigfile", 0);
        confhold_register ("sig_dashes", 0);
        confhold_register ("relative_names", 0);
        confhold_register ("editor", 0);
        confhold_register ("mailbox", 10, "name", "root", "start", "sent",
                           "drafts", "trash", "protect", "spam", "inbox",
                           "outbox");
        confhold_register ("smtp_acc", 8, "name", "server", "port", "my_name",
                           "email", "username", "password", "ssl");
        confhold_register ("pop_acc", 7, "name", "server", "port", "username",
                           "password", "use_apop", "ssl");
        confhold_register ("pop_info_dir", 0);
        confhold_register ("bayes_file", 0);
        confhold_register ("charset", 0);
        confhold_register ("count_mbox", 0);
        confhold_register ("sound_app", 0);
        confhold_register ("procmail_log", 0);

/* This file contains a list of confhold_register calls, that register
   all variables used in the interface.  It is generated by the interface.pl
   script from interface.desc. */
#include "elmo.inc"
        
        confhold_register ("win_label", 4, "focus_fg", "focus_bg",
                           "nofocus_fg", "nofocus_bg");
        confhold_register ("win_bar", 4, "text_fg", "text_bg",
                           "hilight_fg", "hilight_bg");
}



static void
configuration (void)
{
        char *file;

        file = (config_file) ? xstrdup (config_file)
                : file_with_dir (HOMEDIR_RC, ".elmorc");

        if (confread_read_file (file, 1) == -1){
                confread_read_file (SYSCONFDIR "/elmorc", 1);
        }
        xfree (file);
}


static void
initialization (void)
{
        srand (time (NULL));

        property_init ();
        compose_init ();
        net_init ();
        mlex_init ();
        confhold_init ();
        token_init ();
        pop_init ();
        mime_init ();
        mime_post_setup ();
        exec_init ();

        /* nees: exec */
        interface_setup ();

        /* needs: interface */
        cmd_init ();

        /* needs: cmd */
        error_init ();
        exec_post_setup ();

        /* needs: error */
        variable_registration ();
        configuration ();

        /* needs: configuration */
        smtp_init ();
        wrapbox_init ();
        pgp_init ();
        frames_init ();
        label_init ();
        select_init ();
        topbar_init ();
        clock_init ();
        debug_start ();
        status_start ();
        address_init ();
        procmail_init ();

        sender_init ();
        help_init ();
        attach_init ();
        fetch_init ();
        abook_init ();
        mailreader_init ();
        folder_init ();
        box_selection_init ();

        interface_post_setup ();
}



static void
immediate_compose (void)
{
        sender_open_new_to (imm_email);
        imm_email = NULL;
}


static void
setup_immediate_compose (char *str)
{
        exec_t *exec;

        imm_email  = str;
        exec       = exec_lookup_fun (cmd_after_startup);
        exec->hook = hook_add (exec->hook, immediate_compose);
}

/****************************************************************************
 *    INTERFACE FUNCTIONS
 ****************************************************************************/


int
main (int argc, char **argv)
{
        int i;

#if ENABLE_NLS
        setlocale (LC_ALL, "");
        bindtextdomain(PACKAGE, LOCALEDIR);
        textdomain(PACKAGE);
#endif

        i = decode_switches (argc, argv);

        initialization ();

        if (i < argc)
                setup_immediate_compose (argv[i]);

        cmd_read_loop ();

        return 0;
}



void
elmo_finish (int signum)
{
        property_free_resources ();
        compose_free_resources ();
        net_free_resources ();
        mlex_free_resources ();
        pgp_free_resources ();
        procmail_free_resources ();
        sender_free_resources ();
        help_free_resources ();
        attach_free_resources ();
        frames_free_resources ();
        fetch_free_resources ();
        pop_free_resources ();
        smtp_free_resources ();
        topbar_free_resources ();
        clock_free_resources ();
        interface_free_resources ();
        read_free_resources ();
        box_selection_free_resources ();
        mailreader_free_resources ();
        folder_free_resources ();
        mybox_free_resources (); /* must be after folder_free_resources */
        bayes_free_resources ();
        exec_free_resources ();
        wrapbox_free_resources ();
        abook_save ();
        abook_free_resources ();
        address_free_resources (); /* must be after abook_free_resources */
        mime_free_resources ();
        error_free_resources ();
        status_free_resources ();
        confhold_free_resources ();
        color_free_resources ();
        debug_end ();
        endwin ();
}


/****************************************************************************
 *    INTERFACE CLASS BODIES
 ****************************************************************************/
/****************************************************************************
 *
 *    END MODULE elmo.c
 *
 ****************************************************************************/
