dnl GNET2_CHECKS
dnl
dnl Does all the configure checks needed to compile the
dnl gnet2 that we ship with the GUI source code.

AC_DEFUN([GNET2_CHECKS], [

	dnl --- gnet2: check for IPv6 support
	AC_MSG_CHECKING([for IPv6 availability])
	AC_TRY_LINK([ #include <sys/types.h>
		#include <sys/socket.h>
		#include <netinet/in.h>], [struct sockaddr_in6 s; struct in6_addr t=in6addr_any; int i=AF_INET6; s; t.s6_addr[0] = 0;],
  	[
    	AC_DEFINE(HAVE_IPV6, 1, [Define if IPv6 is potentially available])
    	AC_MSG_RESULT(yes)
  	],
  	[
	    AC_MSG_RESULT(no)
  	]
	)

	dnl --- gnet2: Check for headers ---
	AC_HEADER_STDC
	AC_CHECK_HEADERS([sys/sockio.h sys/param.h ifaddrs.h])

	AC_MSG_CHECKING([for linux/netlink.h])
	AC_TRY_CPP([#include <linux/netlink.h>],
	   [
	     AC_MSG_RESULT(yes)
             AC_DEFINE(HAVE_LINUX_NETLINK_H, 1,
	       [Define if linux/netlink.h header exists])
           ],[
             AC_MSG_RESULT(no)
           ])

	dnl ---- gnet2: Check for types and structs
	AC_CHECK_MEMBER([struct sockaddr.sa_len],
			AC_DEFINE(HAVE_SOCKADDR_SA_LEN, 1,
			  [Define if struct sockaddr has sa_len field]),,
			[#include <sys/types.h>
			 #include <sys/socket.h>])

	AC_MSG_CHECKING([for struct sockaddr_storage])
	AC_TRY_COMPILE([ #include <sys/types.h>
		#include <sys/socket.h>], [struct sockaddr_storage s; s],
  	have_sockaddr_storage=yes, have_sockaddr_storage=no)

	if test "$have_sockaddr_storage" = yes; then
	  AC_DEFINE(HAVE_SOCKADDR_STORAGE,1,[Whether you have struct sockaddr_storage])
  	AC_MSG_RESULT(yes)
	else
  	AC_MSG_RESULT(no)
	fi

	# Most of this code checks for variants of gethostbyname().  For
	# simplicity and curiosity, we check for all variants, though only one
	# working variant is needed.

	# Look for getifaddrs()
	AC_CHECK_FUNC(getifaddrs, AC_DEFINE(HAVE_GETIFADDRS, 1,
  	  [Define if getifaddrs() is available]))


	# The user may be able to tell us if a function is thread-safe.  We
	# know of no good way to test this programaticly.

	# On some systems, getaddrinfo() is thread-safe
	AC_ARG_ENABLE(threadsafe_getaddrinfo,
		      AC_HELP_STRING([--enable-threadsafe-getaddrinfo],
				     [assume getaddrinfo() is threadsafe]))

	# On some systems, gethostbyname() is thread-safe
	AC_ARG_ENABLE(threadsafe_gethostbyname,
		      AC_HELP_STRING([--enable-threadsafe-gethostbyname],
				     [assume gethostbyname() is threadsafe]))

	# Look for getaddrinfo().  If we have that, we don't need
	# gethostbyname().  We assume if the system has getaddrinfo(), they
	# have getnameinfo().

	AC_CHECK_FUNC(getaddrinfo,
	 [ # have getaddrinfo()
  	HAVE_GETADDRINFO=yes
	  AC_DEFINE(HAVE_GETADDRINFO, 1,
  	  [Define if getaddrinfo() is available])
	  # Check for threadsafe getaddrinfo()
  	AC_MSG_CHECKING([for threadsafe getaddrinfo()])
	  if test "x$enable_threadsafe_getaddrinfo" = "xyes" ; then
  	  HAVE_GETADDRINFO_THREADSAFE=yes
    	AC_MSG_RESULT(yes)
	    AC_DEFINE(HAVE_GETADDRINFO_THREADSAFE, 1,
		[Define if getaddrinfo is threadsafe])
	  else
  	  AC_MSG_RESULT(no)
    	# If there is no threadsafe getaddrinfo, use a Glib mutex.
	    if test "x$GTHREAD_CFLAGS" != "x"; then
  	    AC_DEFINE(HAVE_GETADDRINFO_GLIB_MUTEX, 1,
    	            [Define if getaddrinfo requires a mutex for thread-safety])
		HAVE_GETADDRINFO_GLIB_MUTEX=yes
  	  else
		AC_MSG_WARN([

	You do not have a thread-safe getaddrinfo() or GLib threads.
	Calls to getaddrinfo() (called by the GNet address functions) may
	not be thread-safe.  This build of GNet could malfunction if used in
	programs that use threads.  If your system has a thread-safe version
	of getaddrinfo(), pass the option --enable-threadsafe-getaddrinfo
	to configure.  The function's manpage may be able to tell you whether
	it is thread-safe or not.
	])
	    fi
  	fi
	 ],
	 [  # no getaddrinfo()
  	AC_MSG_RESULT(no)
	 ])

	# Check if the user told use gethostbyname() is thread-safe.
	AC_MSG_CHECKING([for threadsafe gethostbyname()])
	if test "x$enable_threadsafe_gethostbyname" = "xyes" ; then
	  AC_MSG_RESULT(yes)
  	AC_DEFINE(HAVE_GETHOSTBYNAME_THREADSAFE, 1,
		[Define if gethostbyname is threadsafe])
  	HAVE_GETHOSTBYNAME_THREADSAFE=yes
	else
  	AC_MSG_RESULT(no)
	fi

	# Look for a gethostbyname_r() we can use.  We assume that if the
	# system has gethostbyname_r(), they also have gethostbyaddr_r().

	# First, look for the glibc varient
	AC_MSG_CHECKING([for glibc gethostbyname_r])
	AC_TRY_LINK([#include <netdb.h>],
		    [struct hostent result_buf;
	  	   char buf[1024];
	    	 struct hostent *result;
	     	int h_erropp;

		     gethostbyname_r("localhost",
				     &result_buf,
				     buf,
				     sizeof(buf),
			  	   &result,
			    	 &h_erropp);],
		     [AC_MSG_RESULT([yes])
		      AC_DEFINE(HAVE_GETHOSTBYNAME_R_GLIBC, 1,
	  	       [Define if glibc variant of gethostbyname() is available])
	    	  HAVE_GETHOSTBYNAME_R=yes],
		     [AC_MSG_RESULT([no])])

	# Whups, maybe we're on Solaris or Irix
	AC_MSG_CHECKING([for Solaris/Irix gethostbyname_r])
	AC_TRY_LINK([#include <netdb.h>],
		    [struct hostent result;
	  	   char buf[1024];
	    	 int h_errnop;

		     gethostbyname_r("localhost",
				     &result,
				     buf,
				     sizeof(buf),
			  	   &h_errnop);],
		    [AC_MSG_RESULT([yes])
		     AC_DEFINE(HAVE_GETHOSTBYNAME_R_SOLARIS, 1,
    	            [Define if Solaris variant of gethostbyname() is available])
	    	 HAVE_GETHOSTBYNAME_R=yes],
		    [AC_MSG_RESULT([no])])

	# Or not.  Perhaps we're on HP-UX?
	AC_MSG_CHECKING([for HP-UX gethostbyname_r])
	AC_TRY_LINK([#include <netdb.h>],
		    [struct hostent result;
	  	   char buf[1024];

	    	 gethostbyname_r("localhost",
			  	   &result,
			    	 buf);],
		    [AC_MSG_RESULT([yes])
		     AC_DEFINE(HAVE_GETHOSTBYNAME_R_HPUX, 1,
    	           [Define if HPUX variant of gethostbyname() is available])
	    	 HAVE_GETHOSTBYNAME_R=yes],
		    [AC_MSG_RESULT([no])])

	# If there is no gethostbyname_r, use Glib threads.
	if test -z "$HAVE_GETHOSTBYNAME_THREADSAFE" -a -z "$HAVE_GETHOSTBYNAME_R" -a "$GTHREAD_CFLAGS"; then
		AC_DEFINE(HAVE_GETHOSTBYNAME_R_GLIB_MUTEX, 1,
  	              [Define if gethostbyname requires a mutex for thread-safety])
		HAVE_GETHOSTBYNAME_R=yes
	fi

	# Duh-oh.  We don't have _any_ thread-safety, so warn the builder
	if test -z "$HAVE_GETADDRINFO_THREADSAFE" -a -z "$HAVE_GETHOSTBYNAME_THREADSAFE" -a -z "$HAVE_GETHOSTBYNAME_R"; then
		AC_MSG_WARN([

		You do not have a thread-safe gethostbyname_r() or GLib threads.
		Calls to gethostbyname() (called by the GNet address functions) may
		not be thread-safe.  This build of GNet could malfunction if used in
		programs that use threads.  If your system has a thread-safe version
		of gethostbyname(), pass the option --enable-threadsafe-gethostbyname
		to configure.  The function's manpage may be able to tell you whether
		it is thread-safe or not.
		])
	fi

	AC_CHECK_SIZEOF(long)
	AC_CHECK_SIZEOF(void*)

	dnl ----gnet2: Check for libraries
	AC_CHECK_LIB([socket], [main])
	AC_CHECK_LIB([nsl], [main])
	AC_CHECK_LIB([resolv], [main])

	# Check for pthreads
	AC_ARG_ENABLE(pthreads,
             AC_HELP_STRING([--disable-pthreads],
                            [ignore pthreads library if present]),
             , # we don't need to do anything here
             enable_pthreads=yes)

	if test "x${enable_pthreads}" = "xyes"; then
  	PTHREAD_LIBS=""

  	AC_MSG_CHECKING([for -pthread])
  	gnet_save_LIBS=$LIBS
  	LIBS="$LIBS -pthread"
  	dash_pthread=""
  	AC_LINK_IFELSE([int main() { return 0; }],
		[dash_pthread="-pthread"
		 AC_MSG_RESULT(yes)],
		[dash_pthread=""
		 AC_MSG_RESULT(no)
  	       AC_MSG_CHECKING([for -pthreads])
    	     LIBS="$gnet_save_LIBS -pthreads"
		 AC_LINK_IFELSE([int main() { return 0; }],
		   [dash_pthread="-pthreads"
	  	 AC_MSG_RESULT(yes)],
		   [dash_pthread=""
		   AC_MSG_RESULT(no)])
		])
  	LIBS="$gnet_save_LIBS $dash_pthread"

	  if test "x$dash_pthread" == "x"; then
  	  AC_CHECK_LIB([pthread],
    	        [pthread_create],
      	      [PTHREAD_LIBS="-lpthread"])
	    AC_CHECK_LIB([c_r],
  	          [pthread_create],
    	        [PTHREAD_LIBS="-lc_r $PTHREAD_LIBS"])
	  fi
  	if test "x$PTHREAD_LIBS" != "x" -o "x$dash_pthread" != "x"; then
    	      AC_DEFINE(HAVE_LIBPTHREAD, 1,
      	            [Define if pthreads library is available])
        	  LIBS="$PTHREAD_LIBS $LIBS"
	  fi
	fi

])

dnl PKG_CHECK_MODULES(GSTUFF, gtk+-2.0 >= 1.3 glib = 1.3.4, action-if, action-not)
dnl defines GSTUFF_LIBS, GSTUFF_CFLAGS, see pkg-config man page
dnl also defines GSTUFF_PKG_ERRORS on error
AC_DEFUN([PKG_CHECK_MODULES], [
  succeeded=no

  if test -z "$PKG_CONFIG"; then
    AC_PATH_PROG(PKG_CONFIG, pkg-config, no)
  fi

  if test "$PKG_CONFIG" = "no" ; then
     echo "*** The pkg-config script could not be found. Make sure it is"
     echo "*** in your path, or set the PKG_CONFIG environment variable"
     echo "*** to the full path to pkg-config."
     echo "*** Or see http://www.freedesktop.org/software/pkgconfig to get pkg-config."
  else
     PKG_CONFIG_MIN_VERSION=0.9.0
     if $PKG_CONFIG --atleast-pkgconfig-version $PKG_CONFIG_MIN_VERSION; then
        AC_MSG_CHECKING(for $2)

        if $PKG_CONFIG --exists "$2" ; then
            AC_MSG_RESULT(yes)
            succeeded=yes

            AC_MSG_CHECKING($1_CFLAGS)
            $1_CFLAGS=`$PKG_CONFIG --cflags "$2"`
            AC_MSG_RESULT($$1_CFLAGS)

            AC_MSG_CHECKING($1_LIBS)
            $1_LIBS=`$PKG_CONFIG --libs "$2"`
            AC_MSG_RESULT($$1_LIBS)
        else
            $1_CFLAGS=""
            $1_LIBS=""
            ## If we have a custom action on failure, don't print errors, but 
            ## do set a variable so people can do so.
            $1_PKG_ERRORS=`$PKG_CONFIG --errors-to-stdout --print-errors "$2"`
            ifelse([$4], ,echo $$1_PKG_ERRORS,)
        fi

        AC_SUBST($1_CFLAGS)
        AC_SUBST($1_LIBS)
     else
        echo "*** Your version of pkg-config is too old. You need version $PKG_CONFIG_MIN_VERSION or newer."
        echo "*** See http://www.freedesktop.org/software/pkgconfig"
     fi
  fi

  if test $succeeded = yes; then
     ifelse([$3], , :, [$3])
  else
     ifelse([$4], , AC_MSG_ERROR([Library requirements ($2) not met; consider adjusting the PKG_CONFIG_PATH environment variable if your libraries are in a nonstandard prefix so pkg-config can find them.]), [$4])
  fi
])


