/***********************************************************
        Copyright 1991,1994 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
******************************************************************/


#ifndef SABER
#ifndef LINT
static char rcs_id[] = "$Id: TargetDB_Check.c,v 1.10 1994/12/16 20:38:02 ww0r Exp $";
#endif /* LINT */
#endif /* SABER */

/*
 * Author: Sohan C. Ramakrishna Pillai
 */

#include "depotlib.h"

#include "util.h"
#include "DepotErrorCodes.h"
#include "File.h"
#include "TargetDB.h"



static Boolean TargetDB_CheckCleanDir();


TARGETSOURCE *
HardLink_SourceToUse(sourcelistp)
     TARGETSOURCELIST *sourcelistp;
{
  register TARGETSOURCE *sp;

  sp = TARGETSOURCELIST_Source(sourcelistp,
			       TARGETSOURCELIST_NSources(sourcelistp) - 1);
  if (sp == NULL) {
    FatalError(E_NULLSOURCE,
	     "Unexpected NULL source when checking source for hardlink.\n");
  } else {
    if (TARGETSOURCE_UpdateSpec(sp) & U_COMMAND) {
      if (TARGETSOURCELIST_NSources(sourcelistp) < 2) {
	/* should never happen! */
	FatalError(E_BADSOURCEUPDTSPEC,
		   "Source to use for hardlink is not being mapped.\n");
      } else {
	sp = TARGETSOURCELIST_Source(sourcelistp,
				TARGETSOURCELIST_NSources(sourcelistp) - 2);
      }
    }
  }

  return (PROGRAM_ErrorNo == E_NULL) ? sp : NULL;
}



/*
 * $$$NOTE$$$
 *      Make sure no noop or delete node has sources with commands.
 */
Boolean
TargetDB_CheckNoop(targetpath, nodep, flags)
     char *targetpath;
     TARGETDBENTRY *nodep;
     unsigned flags;
{
  Boolean CheckStatus;
  FILESTAT targetstatus;

  CheckStatus = FALSE;

  if (flags & TDB_CREATE) {
    CheckStatus = TRUE;
  } else if ((TARGETDB_OldUpdateSpec(nodep) != U_NULL)
	     && !(flags & (TDB_BUILD | TDB_REPAIR))
	     && (flags & TDB_TRUSTTDB)) {
    CheckStatus = TRUE;
    if (TARGETDB_OldUpdateSpec(nodep) & U_IMMACULATE) {		
      /* preempted  earlier  by  link  for a  virgin parent */
      CheckStatus = TRUE;
    } else if ((TARGETDB_OldUpdateSpec(nodep) & (U_NOOP | U_DELETE))
	       != (TARGETDB_UpdateSpec(nodep) & (U_NOOP | U_DELETE))) {
      CheckStatus = FALSE;
    }
  } else {
    File_GetStatus(targetpath, &targetstatus, FALSE /* followlinks */ );
    if (PROGRAM_ErrorNo == E_NULL) {
      if (FILESTAT_Type(&targetstatus) == F_NUL) {
	CheckStatus = TRUE;
      } else {
	CheckStatus = FALSE;
	TARGETDB_UpdateSpec(nodep) |= U_TARGET;
	if (FILESTAT_Type(&targetstatus) & F_DIR) {
	  TARGETDB_UpdateSpec(nodep) |= U_MKDIR;
	}
      }
    }
  }

  if ((PROGRAM_ErrorNo == E_NULL) && !CheckStatus) {
    Filtered_Message(flags & TDB_SHOWACTIONS,
		     "NOOP %s\n", targetpath);
    TARGETDB_Status(nodep) |= S_OUTOFDATE;
  }
  return (PROGRAM_ErrorNo == E_NULL) ? CheckStatus : FALSE;
}



Boolean
TargetDB_CheckDelete(targetpath, nodep, flags)
     char *targetpath;
     TARGETDBENTRY *nodep;
     unsigned flags;
{
  Boolean CheckStatus;
  FILESTAT targetstatus;

  CheckStatus = FALSE;

  if (flags & TDB_CREATE) {
    CheckStatus = TRUE;
  } else if ((TARGETDB_OldUpdateSpec(nodep) != U_NULL)
	     && !(flags & (TDB_BUILD | TDB_REPAIR))
	     && (flags & TDB_TRUSTTDB)) {
    CheckStatus = TRUE;
    if (TARGETDB_OldUpdateSpec(nodep) & U_IMMACULATE) {
/* preempted earlier by  link for a  virgin parent */
      CheckStatus = TRUE;
    } else if (!(TARGETDB_OldUpdateSpec(nodep) & U_DELETE)) {
      CheckStatus = FALSE;
    }
  } else {
    File_GetStatus(targetpath, &targetstatus, FALSE /* followlinks */ );
    if (PROGRAM_ErrorNo == E_NULL) {
      if (FILESTAT_Type(&targetstatus) == F_NUL) {
	CheckStatus = TRUE;
      } else {
	CheckStatus = FALSE;
      }
    }
  }

  if ((PROGRAM_ErrorNo == E_NULL) && !CheckStatus) {
    Filtered_Message(flags & TDB_SHOWACTIONS,
		     "REMOVE %s\n", targetpath);
    TARGETDB_Status(nodep) |= S_OUTOFDATE;
    TARGETDB_UpdateSpec(nodep) &= ~U_MKDIR;
  }
  return (PROGRAM_ErrorNo == E_NULL) ? CheckStatus : FALSE;
}


Boolean
TargetDB_CheckMaintainDir(targetpath, nodep, flags)
     char *targetpath;
     TARGETDBENTRY *nodep;
     unsigned flags;
{
  int i;
  register TARGETSOURCE *sp;

  Boolean CheckStatus;

  FILESTAT targetstatus;
  Boolean TargetExists;

  CheckStatus = FALSE;
  TargetExists = FALSE;

  if (flags & TDB_CREATE) {
    CheckStatus = TRUE;
    TargetExists = FALSE;
  } else if ((TARGETDB_OldUpdateSpec(nodep) != U_NULL)
	     && !(flags & (TDB_BUILD | TDB_REPAIR))
	     && (flags & TDB_TRUSTTDB)
	     && !(TARGETDB_OldUpdateSpec(nodep)
		  & (U_NOOP | U_DELETE | U_IMMACULATE))) {
    TargetExists = TRUE;
    CheckStatus = FALSE;
    if ((TARGETDB_OldUpdateSpec(nodep) & U_MKDIR)
	&& !(TARGETDB_OldUpdateSpec(nodep) & U_VIRGINLINK)) {
      CheckStatus = TRUE;
      TARGETDB_Status(nodep) |= S_TARGETISDIR;
      if ((TARGETDB_Status(nodep) & (S_MODIFIED | S_MODIFIEDCHILDREN))
	  && !TargetDB_CheckCleanDir(targetpath, nodep, flags)) {
	CheckStatus = FALSE;
      } else if ((TARGETDB_UpdateSpec(nodep) & (U_OWNER | U_GROUP | U_MODE))
		 && (TARGETDB_FileStatus(nodep) == NULL)) {
	FatalError(E_BADTARGETFILESTATUS,
		   "Unexpected NULL target file status info while checking target database node.\n");
      } else if ((TARGETDB_UpdateSpec(nodep) & (U_OWNER | U_GROUP | U_MODE))
		 && (TARGETDB_OldFileStatus(nodep) == NULL)) {
	CheckStatus = FALSE;
      } else if ((TARGETDB_UpdateSpec(nodep) & U_OWNER)
		 && (FILESTAT_Uid(TARGETDB_FileStatus(nodep))
		     != FILESTAT_Uid(TARGETDB_OldFileStatus(nodep)))) {
	CheckStatus = FALSE;
      } else if ((TARGETDB_UpdateSpec(nodep) & U_GROUP)
		 && (FILESTAT_Gid(TARGETDB_FileStatus(nodep))
		     != FILESTAT_Gid(TARGETDB_OldFileStatus(nodep)))) {
	CheckStatus = FALSE;
      } else if ((TARGETDB_UpdateSpec(nodep) & U_MODE)
		 && (FILESTAT_Mode(TARGETDB_FileStatus(nodep))
		     != FILESTAT_Mode(TARGETDB_OldFileStatus(nodep)))) {
	CheckStatus = FALSE;
      }
    }
  } else {
    File_GetStatus(targetpath, &targetstatus, FALSE /* followlinks */ );
    if ((PROGRAM_ErrorNo == E_NULL) && (FILESTAT_Type(&targetstatus) != F_NUL)) {
      TargetExists = TRUE;
    }
    if ((PROGRAM_ErrorNo == E_NULL) && (FILESTAT_Type(&targetstatus) & F_DIR)) {
      TARGETDB_Status(nodep) |= S_TARGETISDIR;
      CheckStatus = TRUE;
      if (!TargetDB_CheckCleanDir(targetpath, nodep, flags)) {
	CheckStatus = FALSE;
      } else if ((TARGETDB_UpdateSpec(nodep) & (U_OWNER | U_GROUP | U_MODE))
		 && (TARGETDB_FileStatus(nodep) == NULL)) {
	FatalError(E_BADTARGETFILESTATUS,
		   "Unexpected NULL target file status info while checking target database node.\n");
      } else if ((TARGETDB_UpdateSpec(nodep) & U_OWNER)
		 && (FILESTAT_Uid(TARGETDB_FileStatus(nodep))
		     != FILESTAT_Uid(&targetstatus))) {
	CheckStatus = FALSE;
      } else if ((TARGETDB_UpdateSpec(nodep) & U_GROUP)
		 && (FILESTAT_Gid(TARGETDB_FileStatus(nodep))
		     != FILESTAT_Gid(&targetstatus))) {
	CheckStatus = FALSE;
      } else if ((TARGETDB_UpdateSpec(nodep) & U_MODE)
		 && ((FILESTAT_Mode(TARGETDB_FileStatus(nodep))
		      & FILESTAT_Mode(&targetstatus))
		     != FILESTAT_Mode(&targetstatus))) {
	CheckStatus = FALSE;
      }
    }
  }

  if ((PROGRAM_ErrorNo == E_NULL) && (TARGETDB_SourceList(nodep) != NULL)) {
    for (i = 0;
	 (PROGRAM_ErrorNo == E_NULL)
	 && (i < TARGETSOURCELIST_NSources(TARGETDB_SourceList(nodep)));
	 i++) {
      sp = TARGETSOURCELIST_Source(TARGETDB_SourceList(nodep), i);
      while (sp != NULL) {
	if (TARGETSOURCE_UpdateSpec(sp) & U_MKDIR)
	  TARGETSOURCE_UpdateSpec(sp) |= U_TARGET;
	else
	  TARGETSOURCE_UpdateSpec(sp) &= ~U_TARGET;
	sp = TARGETSOURCE_SecondarySource(sp);
      }
    }
  }
  if (PROGRAM_ErrorNo == E_NULL) {
    if (TargetExists && !(TARGETDB_Status(nodep) & S_TARGETISDIR)) {
      Filtered_Message(flags & TDB_SHOWACTIONS,
		       "REMOVE %s\n", targetpath);
    }
    if (!(TARGETDB_Status(nodep) & S_TARGETISDIR)) {
      Filtered_Message(flags & TDB_SHOWACTIONS, "MKDIR %s\n", targetpath);
    }
    if ((PROGRAM_ErrorNo == E_NULL) && !CheckStatus) {
      TARGETDB_Status(nodep) |= S_OUTOFDATE;
    }
    TARGETDB_UpdateSpec(nodep) |= U_MKDIR;
  }
  return (PROGRAM_ErrorNo == E_NULL) ? CheckStatus : FALSE;
}


Boolean
TargetDB_CheckMkdir(targetpath, nodep, flags, sourcep)
     char *targetpath;
     TARGETDBENTRY *nodep;
     unsigned flags;
     TARGETSOURCE *sourcep;
{
  register unsigned i;
  register TARGETSOURCE *sp;

  Boolean CheckStatus;

  FILESTAT targetstatus;
  Boolean TargetExists;

  CheckStatus = FALSE;
  TargetExists = FALSE;

  if (flags & TDB_CREATE) {
    CheckStatus = FALSE;
    TargetExists = FALSE;
  } else if ((TARGETDB_OldUpdateSpec(nodep) != U_NULL)
	     && !(flags & (TDB_BUILD | TDB_REPAIR))
	     && (flags & TDB_TRUSTTDB)
	     && !(TARGETDB_OldUpdateSpec(nodep)
		  & (U_NOOP | U_DELETE | U_IMMACULATE))
	     && !(TARGETDB_OldUpdateSpec(nodep) & U_VIRGINLINK)
	     && (TARGETSOURCE_OldUpdateSpec(sourcep) & U_TARGET)) {
    /* $$$REFINE$$$ * Refine this - check if some mkdir node is a target as * 
     * well */
    TargetExists = TRUE;
    CheckStatus = TRUE;
    TARGETDB_Status(nodep) |= S_TARGETISDIR;
    if ((TARGETDB_Status(nodep) & (S_MODIFIED | S_MODIFIEDCHILDREN))
	&& !TargetDB_CheckCleanDir(targetpath, nodep, flags)) {
      CheckStatus = FALSE;
    } else if ((TARGETDB_UpdateSpec(nodep) & (U_OWNER | U_GROUP | U_MODE))
	       && (TARGETDB_FileStatus(nodep) == NULL)) {
      FatalError(E_BADTARGETFILESTATUS,
		 "Unexpected NULL target file status info while checking target database node.\n");
    } else if ((TARGETDB_UpdateSpec(nodep) & (U_OWNER | U_GROUP | U_MODE))
	       && (TARGETDB_OldFileStatus(nodep) == NULL)) {
      CheckStatus = FALSE;
    } else if ((TARGETDB_UpdateSpec(nodep) & U_OWNER)
	       && (FILESTAT_Uid(TARGETDB_FileStatus(nodep))
		   != FILESTAT_Uid(TARGETDB_OldFileStatus(nodep)))) {
      CheckStatus = FALSE;
    } else if ((TARGETDB_UpdateSpec(nodep) & U_GROUP)
	       && (FILESTAT_Gid(TARGETDB_FileStatus(nodep))
		   != FILESTAT_Gid(TARGETDB_OldFileStatus(nodep)))) {
      CheckStatus = FALSE;
    } else if ((TARGETDB_UpdateSpec(nodep) & U_MODE)
	       && (FILESTAT_Mode(TARGETDB_FileStatus(nodep))
		   != FILESTAT_Mode(TARGETDB_OldFileStatus(nodep)))) {
      CheckStatus = FALSE;
    }
  } else {
    File_GetStatus(targetpath, &targetstatus, FALSE /* followlinks */ );
    if ((PROGRAM_ErrorNo == E_NULL) && (FILESTAT_Type(&targetstatus) != F_NUL)) {
      TargetExists = TRUE;
    }
    if ((PROGRAM_ErrorNo == E_NULL) && (FILESTAT_Type(&targetstatus) & F_DIR)) {
      TARGETDB_Status(nodep) |= S_TARGETISDIR;
      CheckStatus = TRUE;
      if (!TargetDB_CheckCleanDir(targetpath, nodep, flags)) {
	CheckStatus = FALSE;
      } else if ((TARGETDB_UpdateSpec(nodep) & (U_OWNER | U_GROUP | U_MODE))
		 && (TARGETDB_FileStatus(nodep) == NULL)) {
	FatalError(E_BADTARGETFILESTATUS,
		   "Unexpected NULL target file status info while checking target database node.\n");
      } else if ((TARGETDB_UpdateSpec(nodep) & U_OWNER)
		 && (FILESTAT_Uid(TARGETDB_FileStatus(nodep))
		     != FILESTAT_Uid(&targetstatus))) {
	CheckStatus = FALSE;
      } else if ((TARGETDB_UpdateSpec(nodep) & U_GROUP)
		 && (FILESTAT_Gid(TARGETDB_FileStatus(nodep))
		     != FILESTAT_Gid(&targetstatus))) {
	CheckStatus = FALSE;
      } else if ((TARGETDB_UpdateSpec(nodep) & U_MODE)
		 && ((FILESTAT_Mode(TARGETDB_FileStatus(nodep))
		      & FILESTAT_Mode(&targetstatus))
		     != FILESTAT_Mode(&targetstatus))) {
	CheckStatus = FALSE;
      }
    }
  }

  if (PROGRAM_ErrorNo == E_NULL) {
    for (i = 0;
	 (PROGRAM_ErrorNo == E_NULL)
	 && (i < TARGETSOURCELIST_NSources(TARGETDB_SourceList(nodep)));
	 i++) {
      sp = TARGETSOURCELIST_Source(TARGETDB_SourceList(nodep), i);
      while (sp != NULL) {
	if (TARGETSOURCE_UpdateSpec(sp) & U_MKDIR)
	  TARGETSOURCE_UpdateSpec(sp) |= U_TARGET;
	else
	  TARGETSOURCE_UpdateSpec(sp) &= ~U_TARGET;
	sp = TARGETSOURCE_SecondarySource(sp);
      }
    }
  }
  if (PROGRAM_ErrorNo == E_NULL) {
    if (TargetExists && !(TARGETDB_Status(nodep) & S_TARGETISDIR)) {
      Filtered_Message(flags & TDB_SHOWACTIONS,
		       "REMOVE %s\n", targetpath);
    }
    if (!(TARGETDB_Status(nodep) & S_TARGETISDIR)) {
      Filtered_Message(flags & TDB_SHOWACTIONS, "MKDIR %s\n", targetpath);
    }
    if ((PROGRAM_ErrorNo == E_NULL) && !CheckStatus) {
      TARGETDB_Status(nodep) |= S_OUTOFDATE;
    }
    TARGETDB_UpdateSpec(nodep) |= U_MKDIR;
  }
  return (PROGRAM_ErrorNo == E_NULL) ? CheckStatus : FALSE;
}



static Boolean
TargetDB_CheckCleanDir(targetpath, nodep, flags)
     char *targetpath;
     TARGETDBENTRY *nodep;
     unsigned flags;
{
  register char **cp;

  Boolean CheckStatus;

  STRINGARRAY *filelistarray;
  char **filelist;
  int childindex;

  CheckStatus = TRUE;		/* unless a superfluous child is found */
  File_ListDirectory(targetpath, &filelistarray);
  if (PROGRAM_ErrorNo == E_NULL) {
    if (filelistarray == NULL) {
      filelist = NULL;
    } else {
      filelist = STRINGARRAY_Values(filelistarray);
    }
  }
  if ((PROGRAM_ErrorNo == E_NULL) && (filelist != NULL)) {
    for (cp = filelist;
	 (PROGRAM_ErrorNo == E_NULL) && (*cp != NULL);
	 cp++) {
      childindex = TargetDB_LocateIndexToChildNode(nodep,
						   *cp,
						   TDB_LAX);
      if ((PROGRAM_ErrorNo == E_NULL) && (childindex < 0)) {	
       /* superfluous  child */
	Filtered_Message(flags & TDB_SHOWACTIONS,
			 "REMOVE %s/%s\n", targetpath, *cp);
	CheckStatus = FALSE;
      }
    }
  }
  if (filelistarray != NULL)
    StringArray_Free(filelistarray);

  return (PROGRAM_ErrorNo == E_NULL) ? CheckStatus : FALSE;
}



Boolean
TargetDB_CheckHardLink(targetpath, nodep, flags, parentp, pathprefix, collectionpath, sourcep, depth)
     char *targetpath;
     TARGETDBENTRY *nodep;
     unsigned flags;
     TARGETDBENTRY *parentp;
     char *pathprefix;
     char *collectionpath;
     TARGETSOURCE *sourcep;
     unsigned depth;
{
  register TARGETSOURCE *secondaryp;

  Boolean CheckStatus;

  char hardlinkpath[MAXPATHLEN + 1];
  TARGETDBENTRY *hardlinksourcenodep;
  TARGETSOURCE *hardlinksourcep;
  FILESTAT targetstatus, sourcestatus;
  Boolean TargetExists, FoundHardLinkSourceNode;
  unsigned linkmethod;

  CheckStatus = FALSE;
  TargetExists = FALSE;

  if (TARGETDB_UpdateSpec(nodep) & U_MAPCOPY)
    linkmethod = U_MAPCOPY;
  else if (TARGETDB_UpdateSpec(nodep) & U_MAPLINK)
    linkmethod = U_MAPLINK;
  else if (TARGETSOURCE_UpdateSpec(sourcep) & U_MAPCOPY)
    linkmethod = U_MAPCOPY;
  else if (TARGETSOURCE_UpdateSpec(sourcep) & U_MAPLINK)
    linkmethod = U_MAPLINK;

  if (linkmethod == U_MAPLINK) {
    sourcep = HardLink_SourceToUse(TARGETDB_SourceList(nodep));
    if (PROGRAM_ErrorNo == E_NULL) {
      CheckStatus = TargetDB_CheckLinkFile(targetpath,
					   nodep,
					   flags,
					   collectionpath,
					   sourcep,
					   depth);
    } 
  } else {			/* linkmethod == U_MAPCOPY */
    sourcep = HardLink_SourceToUse(TARGETDB_SourceList(nodep));
    if (PROGRAM_ErrorNo != E_NULL)
      return FALSE; 
    /* 
     * find the first targetdb node corresponding to the
     * secondary sources of sourcep whose source corresponds to this hardlink
     */
    FoundHardLinkSourceNode = FALSE;
    secondaryp = TARGETSOURCE_SecondarySource(sourcep);
    while ((PROGRAM_ErrorNo == E_NULL)
	   && !FoundHardLinkSourceNode && (secondaryp != NULL)
	   && (String_Comparator(TARGETDB_Name(nodep),
				 TARGETSOURCE_Path(secondaryp)) != 0)) {
      hardlinksourcenodep =
	TargetDB_LocateChildNode(parentp,
				 TARGETSOURCE_Path(secondaryp),
				 TDB_LOCATE | TDB_LAX);
      if ((PROGRAM_ErrorNo == E_NULL)
	  && (hardlinksourcenodep != NULL)
	  && (TARGETDB_UpdateSpec(hardlinksourcenodep)
	      == TARGETDB_UpdateSpec(nodep))) {
	hardlinksourcep =
	  HardLink_SourceToUse(TARGETDB_SourceList(hardlinksourcenodep));
	if (PROGRAM_ErrorNo != E_NULL)
	  return FALSE;
	if ((TARGETSOURCE_CollectionId(hardlinksourcep)
	     == TARGETSOURCE_CollectionId(sourcep))
	    && ((TARGETSOURCE_UpdateSpec(hardlinksourcep) & ~U_TARGET)
		== TARGETSOURCE_UpdateSpec(sourcep))
	    && (TargetSource_DuplicateComparator
		(TARGETSOURCE_SecondarySource(hardlinksourcep),
		 TARGETSOURCE_SecondarySource(sourcep)) == TRUE)) {
	  FoundHardLinkSourceNode = TRUE;
	}
      }
      if (!FoundHardLinkSourceNode) {
	secondaryp = TARGETSOURCE_SecondarySource(secondaryp);
      }
    }

    if (PROGRAM_ErrorNo == E_NULL) {
      if (!FoundHardLinkSourceNode) {
	/* we have to map from this node */
	CheckStatus = TargetDB_CheckCopyFile(targetpath,
					     nodep,
					     flags,
					     collectionpath,
					     sourcep);
      } else {
	if (flags & TDB_CREATE) {
	  CheckStatus = FALSE;
	  TargetExists = FALSE;
	} else if (TARGETDB_Status(hardlinksourcenodep) & S_UPTODATE) {
	  if ((TARGETDB_OldUpdateSpec(nodep) != U_NULL)
	      && !(flags & (TDB_BUILD | TDB_REPAIR))
	      && (flags & TDB_TRUSTTDB)
	      && !(TARGETSOURCE_Status(secondaryp) & S_MODIFIED)) {
	    CheckStatus = TRUE;
	    TargetExists = TRUE;
	    if (!(TARGETSOURCE_OldUpdateSpec(secondaryp) & U_TARGET)) {
	      CheckStatus = FALSE;
	    } else if (TARGETDB_OldUpdateSpec(nodep)
		       & (U_DELETE | U_IMMACULATE)) {
	      CheckStatus = FALSE;
	      TargetExists = FALSE;
	    } else if (TARGETDB_OldUpdateSpec(nodep) & U_NOOP) {
	      CheckStatus = FALSE;
	      if (!(TARGETDB_OldUpdateSpec(nodep) & U_TARGET)) {
		TargetExists = FALSE;
	      }
	    } else if (TARGETDB_OldUpdateSpec(nodep)
		       != TARGETDB_UpdateSpec(nodep)) {
	      CheckStatus = FALSE;
	    } else if ((TARGETDB_UpdateSpec(nodep)
			& (U_OWNER | U_GROUP | U_MODE | U_SETUID | U_SETGID))
		       && (TARGETDB_FileStatus(nodep) == NULL)) {
	      FatalError(E_BADTARGETFILESTATUS,
			 "Unexpected NULL target file status info while checking target database node.\n");
	    } else if ((TARGETDB_UpdateSpec(nodep)
			& (U_OWNER | U_GROUP | U_MODE | U_SETUID | U_SETGID))
		       && (TARGETDB_OldFileStatus(nodep) == NULL)) {
	      CheckStatus = FALSE;
	    } else if ((TARGETDB_UpdateSpec(nodep) & (U_OWNER | U_SETUID))
		       && (FILESTAT_Uid(TARGETDB_FileStatus(nodep))
			   != FILESTAT_Uid(TARGETDB_OldFileStatus(nodep)))) {
	      CheckStatus = FALSE;
	    } else if ((TARGETDB_UpdateSpec(nodep) & (U_GROUP | U_SETGID))
		       && (FILESTAT_Gid(TARGETDB_FileStatus(nodep))
			   != FILESTAT_Gid(TARGETDB_OldFileStatus(nodep)))) {
	      CheckStatus = FALSE;
	    } else if ((TARGETDB_UpdateSpec(nodep) & U_MODE)
		       && (FILESTAT_Mode(TARGETDB_FileStatus(nodep))
			 != FILESTAT_Mode(TARGETDB_OldFileStatus(nodep)))) {
	      CheckStatus = FALSE;
	    }
	  } else {
	    /* check by stating etc. */
	    File_GetStatus(targetpath, &targetstatus, FALSE);
	    if ((PROGRAM_ErrorNo == E_NULL)
		&& (FILESTAT_Type(&targetstatus) != F_NUL)) {
	      TargetExists = TRUE;
	    }
	    CheckStatus = FALSE;
	    if ((PROGRAM_ErrorNo == E_NULL)
		&& (FILESTAT_Type(&targetstatus) == F_REG)) {
	      if ((String_Comparator(pathprefix, "/") == 0)
		  || (String_Comparator(pathprefix, ".") == 0)) {
		if (String_Comparator(TARGETSOURCE_Path(secondaryp),
				      "/") == 0)
		  (void) strcpy(hardlinkpath, ".");
		else
		  (void) strcpy(hardlinkpath, TARGETSOURCE_Path(secondaryp));
	      } else
		(void) sprintf(hardlinkpath, "%s/%s",
			       pathprefix,
			       TARGETSOURCE_Path(secondaryp));

	      File_GetStatus(hardlinkpath, &sourcestatus,
			     FALSE /* followlinks */ );
	      if ((PROGRAM_ErrorNo == E_NULL)
		  && (FILESTAT_Type(&sourcestatus)
		      == FILESTAT_Type(&targetstatus))
		  && (FILESTAT_INode(&sourcestatus)
		      == FILESTAT_INode(&targetstatus))) {
		CheckStatus = TRUE;
	      }
	    }
	  }
	} else {		/* !(TARGETDB_Status(hardlinksourcenodep) & * 
				 * S_UPTODATE) */
	  CheckStatus = FALSE;	/* established fact, not initialization */
	  if ((TARGETDB_OldUpdateSpec(nodep) != U_NULL)
	      && !(flags & (TDB_BUILD | TDB_REPAIR))
	      && (flags & TDB_TRUSTTDB)
	      && !(TARGETSOURCE_Status(secondaryp) & S_MODIFIED)) {
	    TargetExists = TRUE;
	    if (TARGETDB_OldUpdateSpec(nodep)
		& (U_DELETE | U_IMMACULATE)) {
	      TargetExists = FALSE;
	    } else if ((TARGETDB_OldUpdateSpec(nodep) & U_NOOP)
		       && !(TARGETDB_OldUpdateSpec(nodep) & U_TARGET)) {
	      TargetExists = FALSE;
	    }
	  } else {
	    File_GetStatus(targetpath, &targetstatus, FALSE);
	    if ((PROGRAM_ErrorNo == E_NULL)
		&& (FILESTAT_Type(&targetstatus) != F_NUL)) {
	      TargetExists = TRUE;
	    }
	  }
	}

	if (PROGRAM_ErrorNo == E_NULL) {
	  TARGETSOURCE_UpdateSpec(secondaryp) |= U_TARGET;
	  TARGETDB_UpdateSpec(nodep) &= ~U_MKDIR;
	  if (CheckStatus) {
	    TARGETDB_Status(nodep) |= S_UPTODATE;
	  } else {		/* !CheckStatus */
	    if (TargetExists) {
	      Filtered_Message(flags & TDB_SHOWACTIONS,
			       "REMOVE %s\n", targetpath);
	    }
	    Filtered_Message(flags & TDB_SHOWACTIONS,
			     "HARD LINK %s %s\n", hardlinkpath, targetpath);
	    TARGETDB_Status(nodep) |= S_OUTOFDATE;
	    TARGETSOURCE_Status(secondaryp) |= S_HARDLINKSOURCE;
	  }
	}
      }
    }
  }

  return (PROGRAM_ErrorNo == E_NULL) ? CheckStatus : FALSE;
}




Boolean
TargetDB_CheckCopyFile(targetpath, nodep, flags, collectionpath, sourcep)
     char *targetpath;
     TARGETDBENTRY *nodep;
     unsigned flags;
     char *collectionpath;
     TARGETSOURCE *sourcep;
{
  Boolean CheckStatus;

  char sourcepath[MAXPATHLEN + 1];
  FILESTAT targetstatus, sourcestatus;
  Boolean TargetExists, TargetToBeDeleted;
  char linksrc[MAXPATHLEN + 1], linktrg[MAXPATHLEN + 1];

  CheckStatus = FALSE;
  TargetExists = FALSE;
  TargetToBeDeleted = FALSE;

  /* compute sourcepath */
  (void) sprintf(sourcepath, "%s/%s",
		 collectionpath, TARGETSOURCE_Path(sourcep));

  if (flags & TDB_CREATE) {
    CheckStatus = FALSE;
    TargetExists = FALSE;
  } else if ((TARGETDB_OldUpdateSpec(nodep) != U_NULL)
	     && !(flags & (TDB_BUILD | TDB_REPAIR))
	     && (flags & TDB_TRUSTTDB)
	     && !(TARGETSOURCE_Status(sourcep) & S_MODIFIED)) {
    CheckStatus = TRUE;
    TargetExists = TRUE;
    if (!(TARGETSOURCE_OldUpdateSpec(sourcep) & U_TARGET)) {
      CheckStatus = FALSE;
    } else if (TARGETDB_OldUpdateSpec(nodep) & (U_DELETE | U_IMMACULATE)) {
      CheckStatus = FALSE;
      TargetExists = FALSE;
    } else if (TARGETDB_OldUpdateSpec(nodep) & U_NOOP) {
      CheckStatus = FALSE;
      if (!(TARGETDB_OldUpdateSpec(nodep) & U_TARGET)) {
	TargetExists = FALSE;
      }
    } else if (TARGETDB_OldUpdateSpec(nodep) != TARGETDB_UpdateSpec(nodep)) {
      CheckStatus = FALSE;
    } else if ((TARGETDB_UpdateSpec(nodep)
		& (U_OWNER | U_GROUP | U_MODE | U_SETUID | U_SETGID))
	       && (TARGETDB_FileStatus(nodep) == NULL)) {
      FatalError(E_BADTARGETFILESTATUS,
		 "Unexpected NULL target file status info while checking target database node.\n");
    } else if ((TARGETDB_UpdateSpec(nodep)
		& (U_OWNER | U_GROUP | U_MODE | U_SETUID | U_SETGID))
	       && (TARGETDB_OldFileStatus(nodep) == NULL)) {
      CheckStatus = FALSE;
    } else if ((TARGETDB_UpdateSpec(nodep) & (U_OWNER | U_SETUID))
	       && (FILESTAT_Uid(TARGETDB_FileStatus(nodep))
		   != FILESTAT_Uid(TARGETDB_OldFileStatus(nodep)))) {
      CheckStatus = FALSE;
    } else if ((TARGETDB_UpdateSpec(nodep) & (U_GROUP | U_SETGID))
	       && (FILESTAT_Gid(TARGETDB_FileStatus(nodep))
		   != FILESTAT_Gid(TARGETDB_OldFileStatus(nodep)))) {
      CheckStatus = FALSE;
    } else if ((TARGETDB_UpdateSpec(nodep) & U_MODE)
	       && (FILESTAT_Mode(TARGETDB_FileStatus(nodep))
		   != FILESTAT_Mode(TARGETDB_OldFileStatus(nodep)))) {
      CheckStatus = FALSE;
    }
  } else {
    File_GetStatus(targetpath, &targetstatus, FALSE /* followlinks */ );
    if ((PROGRAM_ErrorNo == E_NULL)
	&& (FILESTAT_Type(&targetstatus) != F_NUL)) {
      TargetExists = TRUE;
    }
    if ((PROGRAM_ErrorNo == E_NULL)
	&& ((FILESTAT_Type(&targetstatus) == F_REG)
	    || (FILESTAT_Type(&targetstatus) == F_LNK))) {
      /* 
       * $$$OPTIMIZE$$$
       * carry over source info in sourcep with S_STATTHISRUN
       */
      File_GetStatus(sourcepath, &sourcestatus, FALSE /* followlinks */ );
      if ((PROGRAM_ErrorNo == E_NULL) && TargetExists) {
	if (FILESTAT_Type(&sourcestatus) != FILESTAT_Type(&targetstatus)) {
	  TargetToBeDeleted = TRUE;
	}
      }
    }
    if ((PROGRAM_ErrorNo == E_NULL)
	&& ((FILESTAT_Type(&targetstatus) == F_REG)
	    || (FILESTAT_Type(&targetstatus) == F_LNK))
	&& (FILESTAT_Type(&sourcestatus) == FILESTAT_Type(&targetstatus))
      && (FILESTAT_INode(&sourcestatus) != FILESTAT_INode(&targetstatus))) {
      if (FILESTAT_Type(&sourcestatus) == F_LNK) {
	File_ReadSymLink(sourcepath, linksrc, MAXPATHLEN + 1);
	File_ReadSymLink(targetpath, linktrg, MAXPATHLEN + 1);
	if (PROGRAM_ErrorNo == E_NULL) {
	  if (String_Comparator(linksrc, linktrg) == 0) {
	    CheckStatus = TRUE;
	  } else {
	    CheckStatus = FALSE;
	    TargetToBeDeleted = TRUE;
	  }
	}
      } else if ((FILESTAT_Type(&sourcestatus) == F_REG)
		 && (flags & TDB_USEMODTIMES)
		 && (FILESTAT_MTime(&sourcestatus)
		     == FILESTAT_MTime(&targetstatus))) {
	/* 
	 * $$$OPTIMIZE$$$
	 * Refine S_OUTOFDATE and reduce number of update operations
	 */
	CheckStatus = TRUE;
	if ((TARGETDB_UpdateSpec(nodep) & U_OWNER)
	    && (FILESTAT_Uid(TARGETDB_FileStatus(nodep))
		!= FILESTAT_Uid(&targetstatus))) {
	  CheckStatus = FALSE;
	} else if ((TARGETDB_UpdateSpec(nodep) & U_GROUP)
		   && (FILESTAT_Gid(TARGETDB_FileStatus(nodep))
		       != FILESTAT_Gid(&targetstatus))) {
	  CheckStatus = FALSE;
	} else if ((TARGETDB_UpdateSpec(nodep) & U_MODE)
		   && ((FILESTAT_Mode(TARGETDB_FileStatus(nodep))
			& FILESTAT_Mode(&targetstatus))
		       != FILESTAT_Mode(&targetstatus))) {
	  CheckStatus = FALSE;
	} else if ((TARGETDB_UpdateSpec(nodep) & U_SETUID)
		   && ((FILESTAT_Uid(TARGETDB_FileStatus(nodep))
			!= FILESTAT_Uid(&targetstatus))
		   || !(FILESTAT_Mode(&targetstatus) & FSTATMODE_SETUID))) {
	  CheckStatus = FALSE;
	} else if ((TARGETDB_UpdateSpec(nodep) & U_SETGID)
		   && ((FILESTAT_Gid(TARGETDB_FileStatus(nodep))
			!= FILESTAT_Gid(&targetstatus))
		   || !(FILESTAT_Mode(&targetstatus) & FSTATMODE_SETGID))) {
	  CheckStatus = FALSE;
	}
      }
    }
  }

  if (PROGRAM_ErrorNo == E_NULL) {
    TARGETSOURCE_UpdateSpec(sourcep) |= U_TARGET;
    TARGETDB_UpdateSpec(nodep) &= ~U_MKDIR;
    if (CheckStatus) {
      TARGETDB_Status(nodep) |= S_UPTODATE;
    } else {			/* !CheckStatus */
      if (TargetExists && TargetToBeDeleted) {
	Filtered_Message(flags & TDB_SHOWACTIONS,
			 "REMOVE %s\n", targetpath);
      }
      Filtered_Message(flags & TDB_SHOWACTIONS,
		       "COPY %s %s\n", sourcepath, targetpath);
      TARGETDB_Status(nodep) |= S_OUTOFDATE;
    }
  }
  return (PROGRAM_ErrorNo == E_NULL) ? CheckStatus : FALSE;
}



Boolean
TargetDB_CheckLinkFile(targetpath, nodep, flags, collectionpath, sourcep, depth)
     char *targetpath;
     TARGETDBENTRY *nodep;
     unsigned flags;
     char *collectionpath;
     TARGETSOURCE *sourcep;
     unsigned depth;
{
  register int i;
  register char *cp;

  Boolean CheckStatus;

  char dotdotprefix[MAXPATHLEN + 1];
  char sourcepath[MAXPATHLEN + 1];
  FILESTAT targetstatus;
  Boolean TargetExists;
  char linktrg[MAXPATHLEN + 1];

  CheckStatus = FALSE;
  TargetExists = FALSE;

  /* compute sourcepath with .. prefixes, if necessary */
  if (collectionpath[0] == '/') {
    (void) sprintf(sourcepath, "%s/%s",
		   collectionpath, TARGETSOURCE_Path(sourcep));
  } else {
    i = 1;
    cp = dotdotprefix;
    while (i < depth) {
      *cp++ = '.';
      *cp++ = '.';
      *cp = '/';
      i++, cp++;
    }
    *cp = '\0';
    (void) sprintf(sourcepath, "%s%s/%s",
		   dotdotprefix, collectionpath, TARGETSOURCE_Path(sourcep));
  }

  if (flags & TDB_CREATE) {
    CheckStatus = FALSE;
    TargetExists = FALSE;
  } else if ((TARGETDB_OldUpdateSpec(nodep) != U_NULL)
	     && !(flags & (TDB_BUILD | TDB_REPAIR))
	     && (flags & TDB_TRUSTTDB)) {
    CheckStatus = TRUE;
    TargetExists = TRUE;
    if (TARGETSOURCE_UpdateSpec(sourcep) != TARGETSOURCE_OldUpdateSpec(sourcep)) {
      CheckStatus = FALSE;
    } else if (!(TARGETSOURCE_OldUpdateSpec(sourcep) & U_TARGET)) {
      CheckStatus = FALSE;
    } else if (TARGETDB_OldUpdateSpec(nodep) & (U_DELETE | U_IMMACULATE)) {
      CheckStatus = FALSE;
      TargetExists = FALSE;
    } else if (TARGETDB_OldUpdateSpec(nodep) & U_NOOP) {
      CheckStatus = FALSE;
      if (!(TARGETDB_OldUpdateSpec(nodep) & U_TARGET)) {
	TargetExists = FALSE;
      }
    } else if ((TARGETDB_OldUpdateSpec(nodep)
       & ~(U_OWNER | U_GROUP | U_MODE | U_SETUID | U_SETGID | U_VIRGINLINK))
	       != (TARGETDB_UpdateSpec(nodep)
    & ~(U_OWNER | U_GROUP | U_MODE | U_SETUID | U_SETGID | U_VIRGINLINK))) {
      CheckStatus = FALSE;
    }
  } else {
    File_GetStatus(targetpath, &targetstatus, FALSE /* followlinks */ );
    if ((PROGRAM_ErrorNo == E_NULL)
	&& (FILESTAT_Type(&targetstatus) != F_NUL)) {
      TargetExists = TRUE;
    }
    if ((PROGRAM_ErrorNo == E_NULL)
	&& (FILESTAT_Type(&targetstatus) == F_LNK)) {
      File_ReadSymLink(targetpath, linktrg, MAXPATHLEN + 1);
      if ((PROGRAM_ErrorNo == E_NULL)
	  && (String_Comparator(sourcepath, linktrg) == 0))
	CheckStatus = TRUE;
    }
  }

  if (PROGRAM_ErrorNo == E_NULL) {
    TARGETSOURCE_UpdateSpec(sourcep) |= U_TARGET;
    TARGETDB_UpdateSpec(nodep) &= ~U_MKDIR;
    if (TARGETSOURCE_UpdateSpec(sourcep) & U_MKDIR) {
      TARGETDB_UpdateSpec(nodep) |= U_VIRGINLINK;
    }
    if (CheckStatus) {
      TARGETDB_Status(nodep) |= S_UPTODATE;
    } else {			/* !CheckStatus */
      if (TargetExists) {
	Filtered_Message(flags & TDB_SHOWACTIONS, "REMOVE %s\n", targetpath);
      }
      Filtered_Message(flags & TDB_SHOWACTIONS,
		       "SYMLINK %s %s\n", sourcepath, targetpath);
      TARGETDB_Status(nodep) |= S_OUTOFDATE;
    }
  }
  return (PROGRAM_ErrorNo == E_NULL) ? CheckStatus : FALSE;
}


/* Where does this routine really belong? -- Sohan */
/*
 * This routine checks whether the whole path exists under the target directory.
 */
Boolean
TargetDB_CheckPathHierarchy(targetpath)
     char *targetpath;
{
  register int i;

  STRINGARRAY *targetpatharray;
  char filenamebuffer[MAXPATHLEN + 1];
  FILESTAT filestatusbuffer;

  Boolean TargetPathExists;

  TargetPathExists = FALSE;

  /* split target path into array of its components */
  targetpatharray = StringToStringArray(targetpath,
					'/' /* delimited by /s */ ,
					-1 /* no quotechar */ );

  /* go down the targetpatharray - stating nodes till no node found */
  if ((PROGRAM_ErrorNo == E_NULL) && !StringArray_Empty(targetpatharray)) {
    TargetPathExists = TRUE;
    filenamebuffer[0] = '\0';
    for (i = 0;
	 (PROGRAM_ErrorNo == E_NULL)
	 && TargetPathExists && (i < STRINGARRAY_Size(targetpatharray));
	 i++) {
      if (i != 0) {
	(void) strcat(filenamebuffer, "/");
      }
      (void) strcat(filenamebuffer, STRINGARRAY_String(targetpatharray, i));

      File_GetStatus(filenamebuffer, &filestatusbuffer, FALSE	/* followlinks 
								 */ );
      if (PROGRAM_ErrorNo == E_NULL) {
	if (STRINGARRAY_String(targetpatharray, i + 1) != NULL) {
	  if (!(FILESTAT_Type(&filestatusbuffer) & F_DIR)) {
	    TargetPathExists = FALSE;
	  }
	} else {		/* leaf node, just check for existence */
	  if (FILESTAT_Type(&filestatusbuffer) == F_NUL) {
	    TargetPathExists = FALSE;
	  }
	}
      }
    }
  }
  if (targetpatharray != NULL)
    StringArray_Free(targetpatharray);

  return (PROGRAM_ErrorNo == E_NULL) ? TargetPathExists : FALSE;
}

/* $Source: /afs/andrew.cmu.edu/system/src/local/depot2/017/src/lib/TargetDB/RCS/TargetDB_Check.c,v $ */
