/***********************************************************
        Copyright 1991,1994 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
******************************************************************/


#ifndef SABER
#ifndef LINT
static char rcs_id[] = "$Id: FileSystemImage_Read.c,v 1.6 1994/12/08 15:49:21 ww0r Exp $";
#endif /* LINT */
#endif /* SABER */

/*
 *
 * Author: Chris Newman
 *         Modified by Sohan C. Ramakrishna Pillai for depot
 *
 */

#include "depotlib.h"

#include "util.h"
#include "DepotErrorCodes.h"
#include "File.h"
#include "FileSystemImage.h"



/* parse a treeinfo directory */
static FILESYSTEMIMAGE *
FileSystemImage_TreeRead(pptr, cnt)
     char **pptr;
     int cnt;
{
  FILESYSTEMIMAGE *fsimagep;

  register char echar;
  char *scan = *pptr;
  char *stringbegin;
  FILESYSTEMIMAGE *node, *lastnode;

  fsimagep = NULL;

  node = NULL;
  lastnode = NULL;
  while ((PROGRAM_ErrorNo == E_NULL) && (cnt--)) {
    Boolean FoundQuote = FALSE;

    node = (FILESYSTEMIMAGE *) emalloc(sizeof(FILESYSTEMIMAGE));
    if (PROGRAM_ErrorNo == E_NULL) {
      /* get the name */
      stringbegin = scan;
      while (*scan != '\0' && *scan != ' ') {
	/* Allow for 'quoted chars'
	 */
	if (*scan == '\\') {
	  FoundQuote = TRUE;
	  ++scan;
	}
	++scan;
      }
      if (*scan == '\0') {
	FatalError(E_BADFILESYSTEMIMAGE,
		   "Unexpected end of file in file system image file.\n");
	return NULL;
      }
      *scan++ = '\0';
      if (FoundQuote) { 
	char *destp, *srcp;
	char *buf;
	
	buf = (char *)emalloc(strlen(stringbegin) +1);
	destp = buf;
	for (srcp = stringbegin; *srcp; srcp++) {
	  /* Loop through name looking for any backquoted chars 
	   */
	  if (*srcp == '\\') 
	    /* Skipping them, and just copying the next char.
	     */
	    srcp++;
	  *destp++ = *srcp;
	}
	*destp = '\0';
	FILESYSTEMIMAGE_Name(node) = buf;
      } else {
	FILESYSTEMIMAGE_Name(node) = String(stringbegin,
					  strlen(stringbegin));
      }

      /* get the type */
      FILESYSTEMIMAGE_Type(node) = *scan++ - '0';
      while (*scan != ' ') {
	FILESYSTEMIMAGE_Type(node) *= 10;
	FILESYSTEMIMAGE_Type(node) += *scan++ - '0';
      }
      scan++;

      /* initialize other node fields */
      FILESYSTEMIMAGE_Mode(node) = 0;
      FILESYSTEMIMAGE_Uid(node) = 0;
      FILESYSTEMIMAGE_Gid(node) = 0;
      FILESYSTEMIMAGE_Size(node) = 0;
      FILESYSTEMIMAGE_MTime(node) = 0;
      FILESYSTEMIMAGE_DTime(node) = 0;
      FILESYSTEMIMAGE_Count(node) = 0;
      FILESYSTEMIMAGE_Link(node) = (char *) NULL;
      FILESYSTEMIMAGE_Child(node) = (treenode *) NULL;
      FILESYSTEMIMAGE_Next(node) = (treenode *) NULL;

      /* read records */
      echar = ' ';
      while (echar != '\n') {
	char *end;

	for (end = scan; (echar = *end) != ' ' && echar != '\n'; ++end)
	  /* EMPTY */
	  ;
	*end = '\0';
	switch (*scan++) {
	case FSINFO_MODE:
	  while (scan < end) {
	    FILESYSTEMIMAGE_Mode(node)
	      = FILESYSTEMIMAGE_Mode(node) * 8 + (*scan++ - '0');
	  }
	  break;
	case FSINFO_SIZE:
	  FILESYSTEMIMAGE_Size(node) = atoi(scan);
	  break;
	case FSINFO_MODTIME:
	  FILESYSTEMIMAGE_MTime(node) = atoi(scan);
	  break;
	case FSINFO_DIRTIME:
	  FILESYSTEMIMAGE_DTime(node) = atoi(scan);
	  break;
	case FSINFO_COUNT:
	  FILESYSTEMIMAGE_Count(node) = atoi(scan);
	  break;
	case FSINFO_LINK:
	  FILESYSTEMIMAGE_Link(node) = String(scan, strlen(scan));
	  break;
	case FSINFO_FSMT:
	  FILESYSTEMIMAGE_FSMountID(node) = String(scan, strlen(scan));
	  break;
	case FSINFO_UID:
	  FILESYSTEMIMAGE_Uid(node) = atoi(scan);
	  break;
	case FSINFO_GID:
	  FILESYSTEMIMAGE_Gid(node) = atoi(scan);
	  break;
	}
	scan = end + 1;
      }
      if ((FILESYSTEMIMAGE_Type(node) & FS_DIR)
	  && FILESYSTEMIMAGE_Count(node) != 0) {
	FILESYSTEMIMAGE_Child(node) =
	  FileSystemImage_TreeRead(&scan,
				   FILESYSTEMIMAGE_Count(node));
      }
      if (lastnode == NULL) {
	fsimagep = node;
      } else {
	FILESYSTEMIMAGE_Next(lastnode) = node;
      }
      lastnode = node;
    }
  }
  *pptr = scan;

  return (PROGRAM_ErrorNo == E_NULL) ? fsimagep : NULL;
}

/* read a treeinfo file
 */
treenode *
FileSystemImage_Read(fname)
     char *fname;
{
  int fd, entries;
  char *fbuf, *scan;
  treenode *root;
  size_t size;
  FILESTAT stbuf;

  /* check size of file and allocate buffer to hold it */
  if (File_GetStatus(fname, &stbuf, FALSE) < 0) {
    FatalError(E_BADFILESYSTEMIMAGE,
	       "Error checking filesystem image: %s\n", fname);
    return NULL;
  }

  if (FILESTAT_Type(&stbuf) == F_NUL) {
    /* no such file or directory... */
    return NULL;
  } 

  if (FILESTAT_Size(&stbuf) < 0) {
    FatalError(E_BADFILESYSTEMIMAGE,
	       "File system image file %s has size < 0 (%d)!\n", 
	       fname, FILESTAT_Size(&stbuf));
  } else 
    fbuf = emalloc((size = FILESTAT_Size(&stbuf)) + 1);

  /* read entire file into the buffer */
  if (PROGRAM_ErrorNo == E_NULL) {
    if ((fd = open(fname, O_RDONLY)) < 0) {
      FatalError(E_BADFILESYSTEMIMAGE,
		 "Could not open file system image file %s\n", fname);
      (void) free(fbuf);
    } else if (read(fd, fbuf, size) < size) {
      FatalError(E_READFAILED,
		 "Error while reading file system image file %s\n", fname);
      (void) free(fbuf);
      (void) close(fd);
    } else {
      (void) close(fd);
      fbuf[size] = '\0';
    }
  }
  /* verify magic number */
  if (PROGRAM_ErrorNo == E_NULL) {
    if (strncmp(FILESYSTEMIMAGE_magic, fbuf,
		(int) strlen(FILESYSTEMIMAGE_magic)) != 0) {
      FatalError(E_BADFILESYSTEMIMAGE,
		 "Bad magic number for file system image file %s\n",
		 fname);
      (void) free(fbuf);
    }
  }
  /* get number of entries, and allocate space for them */
  if (PROGRAM_ErrorNo == E_NULL) {
    for (scan = fbuf; *scan != '\0' && *scan != '\n'; ++scan)
      /* EMPTY */
      ;
    while (isspace(*scan))
      ++scan;
    for (entries = 0; isdigit(*scan); ++scan) {
      entries = entries * 10 + (*scan - '0');
    }

    while (*scan != '\0' && *scan != '\n')
      ++scan;
    if (*scan++ == '\0') {
      FatalError(E_BADFILESYSTEMIMAGE,
		 "Premature end for file system image file %s\n", fname);
      (void) free(fbuf);
    } else {
      root = NULL;
      if (PROGRAM_ErrorNo != E_NULL) {
	(void) free(fbuf);
      }
    }
  }
  /* recursive parse by directory */
  if (PROGRAM_ErrorNo == E_NULL) {
    root = FileSystemImage_TreeRead(&scan, 1);
    if (PROGRAM_ErrorNo != E_NULL) {
      FatalError(E_BADFILESYSTEMIMAGE,
		 "Error while parsing file system image file %s\n", fname);
      (void) free(fbuf);
    }
  }
  if (PROGRAM_ErrorNo == E_NULL) {
    (void) free(fbuf);
  }
  return ((PROGRAM_ErrorNo == E_NULL) ? root : NULL);
}

/* $Source: /afs/andrew.cmu.edu/system/src/local/depot2/017/src/lib/FileSystemImage/RCS/FileSystemImage_Read.c,v $ */

