/***********************************************************
        Copyright 1991,1994 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
******************************************************************/


#ifndef SABER
#ifndef LINT
static char rcs_id[] = "$Id: FileSystemImage_Build.c,v 1.14 1994/11/08 16:41:05 ww0r Exp $";
#endif /* LINT */
#endif /* SABER */

/*
 *
 * Author: Chris Newman
 *         Rewritten by Sohan C. Ramakrishna Pillai for depot
 *
 */

#include "depotlib.h"

#include "util.h"
#include "DepotErrorCodes.h"
#include "File.h"
#include "FileSystemImage.h"


/* stuff to check inodes for hard links etc. */
typedef struct linktable {
  ino_t inode;
  char *nodename;
  struct linktable *next;
} LINKTABLE;

static char *LinkTable_LocateLink();
static void LinkTable_Free();

static FILESYSTEMIMAGE *FileSystemImageNodeFromFileStatus();
static FILESYSTEMIMAGE *FileSystemImage_TreeBuild();



static FILESYSTEMIMAGE *
FileSystemImageNodeFromFileStatus(filestatus, path, nodename,
				  fsid, hardlinkname)
     FILESTAT *filestatus;
     char *path, *nodename, *fsid, *hardlinkname;
{
  FILESYSTEMIMAGE *fsimagenodep;

  char linkbuf[512];  /* random size to hold volume id + some text */
  time_t fsmodtime;

  fsimagenodep = (FILESYSTEMIMAGE *) emalloc(sizeof(FILESYSTEMIMAGE));

  if (PROGRAM_ErrorNo == E_NULL) {
    FILESYSTEMIMAGE_Mode(fsimagenodep) = FILESTAT_Mode(filestatus);
    FILESYSTEMIMAGE_Uid(fsimagenodep) = FILESTAT_Uid(filestatus);
    FILESYSTEMIMAGE_Gid(fsimagenodep) = FILESTAT_Gid(filestatus);
    FILESYSTEMIMAGE_Size(fsimagenodep) = FILESTAT_Size(filestatus);
    FILESYSTEMIMAGE_DTime(fsimagenodep)
      = FILESYSTEMIMAGE_MTime(fsimagenodep) = FILESTAT_MTime(filestatus);
    FILESYSTEMIMAGE_Name(fsimagenodep) = String(nodename, strlen(nodename));
    FILESYSTEMIMAGE_Count(fsimagenodep) = 0;
    FILESYSTEMIMAGE_Link(fsimagenodep) = NULL;
    FILESYSTEMIMAGE_Child(fsimagenodep) = NULL;
    FILESYSTEMIMAGE_Next(fsimagenodep) = NULL;

    switch (FILESTAT_Type(filestatus)) {
    case F_DIR:
      FILESYSTEMIMAGE_Type(fsimagenodep) = FS_DIR; 
      FILESYSTEMIMAGE_FSMountID(fsimagenodep) = NULL;

      (void)memset(linkbuf,0,sizeof(linkbuf));
      if (FileSystem_GetFSInfo(path, linkbuf, &fsmodtime) < 0) {
	FatalError(E_GETFSIDFAILED,
		   "Could not read the file system id for %s\n", path);
      } else if ((linkbuf[0] != '\0')
		 && ((fsid == NULL) || (strcmp(fsid, linkbuf) != 0))) {
	FILESYSTEMIMAGE_Type(fsimagenodep) |= FS_FSMOUNT;
	FILESYSTEMIMAGE_FSMountID(fsimagenodep) = String(linkbuf,
							 strlen(linkbuf));
	FILESYSTEMIMAGE_DTime(fsimagenodep)
	  = FILESYSTEMIMAGE_MTime(fsimagenodep) = fsmodtime;
      }
      break;

    case F_LNK:
      FILESYSTEMIMAGE_Type(fsimagenodep) = FS_SYMLNK;
      if (File_ReadSymLink(path, linkbuf, MAXPATHLEN) < 0) {
	FatalError(E_READLINKFAILED,
		   "Could not read the symbolic link for %s\n", path);
      } else
	FILESYSTEMIMAGE_Link(fsimagenodep) = String(linkbuf,
						    strlen(linkbuf));
      break;

    case F_REG:
      FILESYSTEMIMAGE_Type(fsimagenodep) = FS_REG;
      if ((FILESTAT_NLinks(filestatus) != 1) && (hardlinkname != NULL)) {
	FILESYSTEMIMAGE_Link(fsimagenodep) = String(hardlinkname,
						    strlen(hardlinkname));
	FILESYSTEMIMAGE_Type(fsimagenodep) = FS_HARDLNK;
      }
      break;

    default:
      FILESYSTEMIMAGE_Type(fsimagenodep) = FS_OTHER;
      break;
    }
  }
  return (PROGRAM_ErrorNo == E_NULL) ? fsimagenodep : NULL;
}


static FILESYSTEMIMAGE *
FileSystemImage_TreeBuild(path, nodename, fsid, linktablep)
     char *path, *nodename, *fsid;
     LINKTABLE **linktablep;
{
  register char **childp;

  FILESYSTEMIMAGE *fsimagep;

  FILESTAT filestatbuffer;
  char *hardlinkname;
  char *newfsid;
  char **childlist;
  STRINGSET *childliststringset;
  char childpath[MAXPATHLEN];
  FILESYSTEMIMAGE *newchildimagep, *lastchildimagep;
  LINKTABLE *childlinktable;

  if (path == NULL) {
    FatalError(E_BADPATHNAME,
	       "Attempt to build file system image for NULL path.\n");
  }
  /* get status for path */
  if (PROGRAM_ErrorNo == E_NULL)
    (void) File_GetStatus(path,
			  &filestatbuffer,
			  FALSE /* followlinks at top level only */ );

  /* check for hard links */
  hardlinkname = NULL;
  if ((PROGRAM_ErrorNo == E_NULL) && (FILESTAT_NLinks(&filestatbuffer) != 1)) {
    hardlinkname = LinkTable_LocateLink(linktablep,
					FILESTAT_INode(&filestatbuffer),
					nodename);
  }
  /* create file system image node from status */
  if (PROGRAM_ErrorNo == E_NULL)
    fsimagep = FileSystemImageNodeFromFileStatus(&filestatbuffer,
						 path, nodename,
						 fsid,
						 hardlinkname);

  if ((PROGRAM_ErrorNo == E_NULL) && (FILESYSTEMIMAGE_Type(fsimagep) & FS_DIR)) {
    /* build subtree for children */
    childliststringset = NULL;
    (void) File_ListDirectory(path, &childliststringset);

    if (PROGRAM_ErrorNo == E_NULL) {
      if (childliststringset == NULL) {
	childlist = NULL;
      } else {
	childlist = STRINGSET_Values(childliststringset);
      }
    }
    if ((PROGRAM_ErrorNo == E_NULL)
	&& (FILESYSTEMIMAGE_Type(fsimagep) & FS_FSMOUNT)
	&& (FILESYSTEMIMAGE_FSMountID(fsimagep) != NULL))
      newfsid = FILESYSTEMIMAGE_FSMountID(fsimagep);
    else
      newfsid = fsid;

    if ((PROGRAM_ErrorNo == E_NULL) && (childlist != NULL)) {
      int pathlen;

      (void) strcpy(childpath, path);
      (void) strcat(childpath, "/");
      pathlen = strlen(childpath);	/* place to insert the NULL */
      childlinktable = NULL;
      lastchildimagep = NULL;
      newchildimagep = NULL;
      for (childp = childlist;
	   (PROGRAM_ErrorNo == E_NULL) && (*childp != NULL);
	   childp++) {
	FILESYSTEMIMAGE_Count(fsimagep)++;
	childpath[pathlen] = '\0';
	(void) strcat(childpath, *childp);
	newchildimagep = FileSystemImage_TreeBuild(childpath,
						   *childp,
						   newfsid,
						   &childlinktable);
	if (FILESYSTEMIMAGE_DTime(newchildimagep)
	    > FILESYSTEMIMAGE_DTime(fsimagep)) {
	  FILESYSTEMIMAGE_DTime(fsimagep)
	    = FILESYSTEMIMAGE_DTime(newchildimagep);
	}
	if (FILESYSTEMIMAGE_Child(fsimagep) == NULL) {
	  FILESYSTEMIMAGE_Child(fsimagep) = newchildimagep;
	} else {
	  FILESYSTEMIMAGE_Next(lastchildimagep) = newchildimagep;
	}
	lastchildimagep = newchildimagep;
      }

      /* free the childlinktable */
      if (childlinktable != NULL)
	LinkTable_Free(childlinktable);
    }
    if (childliststringset != NULL)
      StringArray_Free(childliststringset);
  }
  return (PROGRAM_ErrorNo == E_NULL) ? fsimagep : NULL;
}


/*
 * build a tree structure for a given directory
 */
FILESYSTEMIMAGE *
FileSystemImage_Build(path)
     char *path;
{
  register char **childp;

  FILESYSTEMIMAGE *fsimagep;

  FILESTAT filestatbuffer;
  char *newfsid;
  char **childlist;
  STRINGSET *childliststringset;
  char childpath[MAXPATHLEN];
  FILESYSTEMIMAGE *newchildimagep, *lastchildimagep;
  LINKTABLE *childlinktable;


  if (path == NULL) {
    FatalError(E_BADPATHNAME,
	       "Attempt to build file system image for NULL path.\n");
  }
  (void) FileSystem_Initialize();

  /* get status for path */
  if (PROGRAM_ErrorNo == E_NULL)
    (void) File_GetStatus(path,
			  &filestatbuffer,
			  TRUE /* followlinks at top level only */ );

  if (FILESTAT_Type(&filestatbuffer) == F_NUL) {
    FatalError(E_BADPATHNAME,
	       "No such directory exists at %s\n", path);
  }
  if (FILESTAT_Type(&filestatbuffer) != F_DIR) {
    FatalError(E_BADPATHNAME,
	       "Not a directory at %s\n", path);
  }
  /* check for hard links not necessary at top level */
  /* create file system image node from status */
  if (PROGRAM_ErrorNo == E_NULL)
    fsimagep = FileSystemImageNodeFromFileStatus(&filestatbuffer,
						 path,
						 "/"	/* top level nodename */ ,
						 NULL /* fsid */ ,
						 NULL /* hardlinkname */ );

  if ((PROGRAM_ErrorNo == E_NULL) && (FILESYSTEMIMAGE_Type(fsimagep) & FS_DIR)) {
    if ((FILESYSTEMIMAGE_Type(fsimagep) & FS_FSMOUNT)
	&& (FILESYSTEMIMAGE_FSMountID(fsimagep) != NULL)) {
      newfsid = FILESYSTEMIMAGE_FSMountID(fsimagep);
    } else {
      newfsid = NULL;
      FILESYSTEMIMAGE_Type(fsimagep) |= FS_FSMOUNT;
      FILESYSTEMIMAGE_FSMountID(fsimagep) = String("generic",
						   strlen("generic"));
    }

    /* build subtree for children */
    childliststringset = NULL;
    (void) File_ListDirectory(path, &childliststringset);

    if (PROGRAM_ErrorNo == E_NULL) {
      if (childliststringset == NULL) {
	childlist = NULL;
      } else {
	childlist = STRINGSET_Values(childliststringset);
      }
    }

    if ((PROGRAM_ErrorNo == E_NULL) && (childlist != NULL)) {
      int pathlen;

      (void) strcpy(childpath, path);
      (void) strcat(childpath, "/");
      pathlen = strlen(childpath);
      childlinktable = NULL;
      lastchildimagep = NULL;
      newchildimagep = NULL;
      for (childp = childlist;
	   (PROGRAM_ErrorNo == E_NULL) && (*childp != NULL);
	   childp++) {
	FILESYSTEMIMAGE_Count(fsimagep)++;
	childpath[pathlen] = '\0';
	(void) strcat(childpath, *childp);
	newchildimagep = FileSystemImage_TreeBuild(childpath,
						   *childp,
						   newfsid,
						   &childlinktable);
	if (FILESYSTEMIMAGE_DTime(newchildimagep)
	    > FILESYSTEMIMAGE_DTime(fsimagep)) {
	  FILESYSTEMIMAGE_DTime(fsimagep)
	    = FILESYSTEMIMAGE_DTime(newchildimagep);
	}
	if (FILESYSTEMIMAGE_Child(fsimagep) == NULL) {
	  FILESYSTEMIMAGE_Child(fsimagep) = newchildimagep;
	} else {
	  FILESYSTEMIMAGE_Next(lastchildimagep) = newchildimagep;
	}
	lastchildimagep = newchildimagep;
      }

      /* free the childlinktable */
      if (childlinktable != NULL)
	LinkTable_Free(childlinktable);
    }
    if ((PROGRAM_ErrorNo == E_NULL) && (childliststringset != NULL))
      StringArray_Free(childliststringset);
  }
  (void) FileSystem_End();

  return (PROGRAM_ErrorNo == E_NULL) ? fsimagep : NULL;
}



static char *
LinkTable_LocateLink(linktablep, inode, nodename)
     LINKTABLE **linktablep;
     ino_t inode;
     char *nodename;
{
  register LINKTABLE *lp;

  Boolean LocatedLink;
  char *linkname;

  LocatedLink = FALSE;
  lp = *linktablep;
  while (!LocatedLink && (lp != NULL)) {
    if (inode == lp->inode) {
      linkname = lp->nodename;
      LocatedLink = TRUE;
    }
    lp = lp->next;
  }
  if (!LocatedLink) {
    lp = (LINKTABLE *) emalloc(sizeof(LINKTABLE));
    if (PROGRAM_ErrorNo == E_NULL) {
      lp->inode = inode;
      lp->nodename = String(nodename, strlen(nodename));
      lp->next = *linktablep;
      *linktablep = lp;
      linkname = lp->nodename;
      LocatedLink = TRUE;
    }
  }
  return ((PROGRAM_ErrorNo == E_NULL) && LocatedLink) ? linkname : NULL;
}



static void 
LinkTable_Free(linktable)
     LINKTABLE *linktable;
{
  register LINKTABLE *lp1, *lp2;

  lp1 = linktable;
  while (lp1 != NULL) {
    lp2 = lp1->next;
    String_Free(lp1->nodename);
    (void) free((void *) lp1);
    lp1 = lp2;
  }
  return;
}
/* $Source: /afs/andrew/system/src/local/depot2/016/src/lib/FileSystemImage/RCS/FileSystemImage_Build.c,v $ */
