/***********************************************************
        Copyright 1991,1994 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
******************************************************************/


#ifndef SABER
#ifndef LINT
static char rcs_id[] = "$Id: Collection_PathMap.c,v 1.8 1994/09/28 14:36:04 ww0r Exp $";
#endif /* LINT */
#endif /* SABER */

/*
 * Author: Sohan C. Ramakrishna Pillai
 */

#include "depotlib.h"

#include "util.h"
#include "DepotErrorCodes.h"
#include "Hint.h"
#include "File.h"
#include "FileSystemImage.h"
#include "DepotConf.h"
#include "TargetDB.h"
#include "Collection.h"

/* stuff to check inodes for hard links etc. */
typedef struct secondarysource {
  ino_t inode;
  TARGETSOURCE *source;
  struct secondarysource *next;
} SECONDARYSOURCEINFO, *SECONDARYSOURCELIST;

static SECONDARYSOURCEINFO *SecondarySourceList_IncludeSource();
static void SecondarySourceList_Free();

static TARGETDB *Collection_SubTreeMapFromPath();

TARGETDB *
Collection_PathMap(targetdbp,
		   source, target,
		   collectionp,
		   hintdb_old)
     TARGETDB *targetdbp;
     char *source, *target;
     COLLECTION *collectionp;
     HINTDB hintdb_old;
{
  register char **childp;
  register SECONDARYSOURCEINFO *ssp;
  register TARGETSOURCE *sp;

  TARGETDB *newtargetdbp;

  TARGETDB *targetdbnodep;
  TARGETSOURCE *targetsourcep;
  int locatedsourceindex;
  TARGETSOURCE *locatedtargetsourcep;
  char sourcepath[MAXPATHLEN];
  FILESTAT filestatbuffer;
  HINT *oldhint, *newhint;
  Boolean HintComparison;
  STRINGSET *childlistset;
  int targetdbchildslot;
  TARGETDB *targetdbchildp;
  TARGETDB **targetdbchildpp;
  char childsource[MAXPATHLEN];
  SECONDARYSOURCEINFO *secondarysourcelist;


  if (collectionp == NULL) {
    FatalError(E_NULLCOLLECTION,
	       "Attempt to generate mappings for a NULL collection\n");
  }
  if (source == NULL) {
    FatalError(E_NULLSOURCE,
	       "Attempt to map from NULL source for collection: %s\n",
	       COLLECTION_Name(collectionp));
  }
  if (target == NULL) {
    FatalError(E_NULLTARGET,
	       "Attempt to map to NULL target for collection: %s\n",
	       COLLECTION_Name(collectionp));
  }
  /* 
   * Create non-virgin branch in targetdbp to specified target,
   * except that the leaf is not to be marked as nonvirgin. Source info for the
   * leaf comes from the status info of source.
   */
  if (PROGRAM_ErrorNo == E_NULL)
    newtargetdbp = Collection_TargetDBCreateNonVirginPath(targetdbp,
							  source, target,
							  collectionp);
  if (PROGRAM_ErrorNo == E_NULL)
    targetdbnodep = TargetDB_LocateNode(newtargetdbp, target, TDB_LOCATE);

  if (PROGRAM_ErrorNo == E_NULL) {
    /* stat source following links only at the top level of collection */
    if (String_Comparator(source, "/") == 0) {
      (void) strcpy(sourcepath, COLLECTION_Path(collectionp));
      (void) File_GetStatus(sourcepath, &filestatbuffer, TRUE /* followlinks */ );
    } else {
      (void) sprintf(sourcepath,
		     "%s/%s", COLLECTION_Path(collectionp), source);
      (void) File_GetStatus(sourcepath, &filestatbuffer, FALSE /* followlinks */ );
    }
  }
  /* add source to targetnodep -- unless it is the root node */
  if (PROGRAM_ErrorNo == E_NULL) {
    if (String_Comparator(TARGETDB_Name(targetdbnodep), "/") == 0) {
      if (!(FILESTAT_Type(&filestatbuffer) & F_DIR)) {
	FatalError
	  (E_BADTARGETSOURCEPATH,
	   "Attempt to map non-directory file %s to target directory in collection: %s.\n",
	   source, COLLECTION_Name(collectionp));
      }
    } else {
      targetsourcep = Collection_TargetSourceFromFileStatus(&filestatbuffer,
							    source,
							    collectionp);
      if (PROGRAM_ErrorNo == E_NULL)
	TARGETDB_SourceList(targetdbnodep) =
	  Collection_SourceList_AddSource
	  (TARGETDB_SourceList(targetdbnodep), targetsourcep, collectionp);

      if (PROGRAM_ErrorNo == E_NULL)
	TargetSource_Free(targetsourcep);
    }
  }
  /* 
   * if depotconf specifications haven't changed since last run,
   * check if old hints and new hints match
   */
  HintComparison = FALSE;
  if ((PROGRAM_ErrorNo == E_NULL)
      && !(COLLECTION_Status(collectionp) & C_DEPOTCONFCHANGE)
      && (hintdb_old != NULL)
      && (COLLECTION_HintDB(collectionp) != NULL)) {
    oldhint = HintDB_GetHint(hintdb_old, source);
    if ((PROGRAM_ErrorNo == E_NULL) && (oldhint != NULL)) {
      newhint = HintDB_GetHint(COLLECTION_HintDB(collectionp), source);
      if ((PROGRAM_ErrorNo == E_NULL) && (newhint != NULL))
	HintComparison = Hint_Comparator(oldhint, newhint);
    }
  }
  if (PROGRAM_ErrorNo == E_NULL) {
    if (HintComparison == TRUE) {
      /* mark source to targetdbnodep as enduring subtree */
      locatedsourceindex =
	SourceList_FindIndexToSource(TARGETDB_SourceList(targetdbnodep),
				     source,
				     COLLECTION_Id(collectionp),
				     TDB_LOCATE);
      if ((PROGRAM_ErrorNo == E_NULL) && (locatedsourceindex >= 0)) {
	locatedtargetsourcep =
	  TARGETSOURCELIST_Source(TARGETDB_SourceList(targetdbnodep),
				  locatedsourceindex);
	TARGETSOURCE_Status(locatedtargetsourcep) |= S_ENDURINGSUBTREE;
	TARGETSOURCE_UpdateSpec(locatedtargetsourcep)
	  |= COLLECTION_MapMethod(collectionp);
      }
    } else if (FILESTAT_Type(&filestatbuffer) & F_DIR) {	
      /* recursively  map children, if any, of  source */
      childlistset = NULL;
      (void) File_ListDirectory(sourcepath, &childlistset);

      if ((PROGRAM_ErrorNo == E_NULL)
	  && (childlistset != NULL)
	  && (STRINGSET_Size(childlistset) > 0)
	  && (STRINGSET_Values(childlistset) != NULL)) {
	secondarysourcelist = NULL;
	for (childp = STRINGSET_Values(childlistset);
	     (PROGRAM_ErrorNo == E_NULL) && (*childp != NULL);
	     childp++) {
	  /* compute new source for child node */
	  if (strcmp(source, "/") == 0)
	    (void) strcpy(childsource, *childp);
	  else
	    (void) sprintf(childsource, "%s/%s", source, *childp);

	  /* 
	   * locate slot for child node under targetdbnodep,
	   * creating it if necessary
	   */
	  if (PROGRAM_ErrorNo == E_NULL) {
	    targetdbchildslot =
	      TargetDB_LocateIndexToChildNode(targetdbnodep,
					      *childp,
					      TDB_LOCATE | TDB_CREATE);
	    if (PROGRAM_ErrorNo == E_NULL) {
	      targetdbchildpp =
		TARGETDB_Children(targetdbnodep) + targetdbchildslot;
	    }
	  }
	  /* 
	   * recursively map onto child slot of targetdbnodep
	   * from childp, the current child of source
	   */
	  if (PROGRAM_ErrorNo == E_NULL) {
	    *targetdbchildpp =
	      Collection_SubTreeMapFromPath(*targetdbchildpp,
					    *childp,
					    childsource,
					    collectionp,
					    hintdb_old,
					    &secondarysourcelist);
	  }
	}

	/* add secondary sources from from the secondarysourcelist */
	if (secondarysourcelist != NULL) {
	  for (ssp = secondarysourcelist;
	       (PROGRAM_ErrorNo == E_NULL) && (ssp != NULL);
	       ssp = ssp->next) {
	    for (sp = ssp->source;
		 (PROGRAM_ErrorNo == E_NULL) && (sp != NULL);
		 sp = TARGETSOURCE_SecondarySource(sp)) {
	      if (strcmp(source, "/") == 0)
		(void) strcpy(childsource, TARGETSOURCE_Path(sp));
	      else
		(void) sprintf(childsource, "%s/%s",
			       source, TARGETSOURCE_Path(sp));
	      /* locate child corresponding to this source */
	      targetdbchildp =
		TargetDB_LocateChildNode(targetdbnodep,
					 TARGETSOURCE_Path(sp),
					 TDB_LOCATE);
	      /* locate source corresponding to this node in child */
	      if (PROGRAM_ErrorNo == E_NULL) {
		locatedsourceindex =
		  SourceList_FindIndexToSource
		  (TARGETDB_SourceList(targetdbchildp),
		   childsource,
		   COLLECTION_Id(collectionp),
		   TDB_LOCATE);
	      }
	      if ((PROGRAM_ErrorNo == E_NULL)
		  && (locatedsourceindex >= 0)) {
		locatedtargetsourcep =
		  TARGETSOURCELIST_Source
		  (TARGETDB_SourceList(targetdbchildp),
		   locatedsourceindex);
		TARGETSOURCE_SecondarySource(locatedtargetsourcep) =
		  TargetSource(ssp->source);
	      }
	    }
	  }
	  /* free the secondarysourcelist */
	  SecondarySourceList_Free(secondarysourcelist);
	}
	StringArray_Free(childlistset);
      }
    }
  }
  return (PROGRAM_ErrorNo == E_NULL) ? newtargetdbp : NULL;
}


static TARGETDB *
Collection_SubTreeMapFromPath(targetdbp,
			      nodename,
			      source,
			      collectionp,
			      hintdb_old,
			      secondarysourcelistp)
     TARGETDB *targetdbp;
     char *nodename, *source;
     COLLECTION *collectionp;
     HINTDB hintdb_old;
     SECONDARYSOURCEINFO **secondarysourcelistp;
{
  register char **childp;
  register SECONDARYSOURCEINFO *ssp;
  register TARGETSOURCE *sp;

  TARGETDB *newtargetdbp;

  TARGETSOURCE *targetsourcep;
  int locatedsourceindex;
  TARGETSOURCE *locatedtargetsourcep;
  char sourcepath[MAXPATHLEN];
  FILESTAT filestatbuffer;
  HINT *oldhint, *newhint;
  Boolean HintComparison;
  STRINGSET *childlistset;
  int targetdbchildslot;
  TARGETDB *targetdbchildp;
  TARGETDB **targetdbchildpp;
  char childsource[MAXPATHLEN];
  SECONDARYSOURCEINFO *childsecondarysourcelist;

  newtargetdbp = targetdbp;

  /* 
   * stat source without following links - this is a static function
   * which will not be called to map from the top level of collection
   */
  (void) sprintf(sourcepath, "%s/%s", COLLECTION_Path(collectionp), source);
  (void) File_GetStatus(sourcepath, &filestatbuffer, FALSE /* followlinks */ );

  /* add source for newtargetdbp from the status info of source */
  if (PROGRAM_ErrorNo == E_NULL) {
    /* include possible secondary source in secondarysourcelistp */
    if ((PROGRAM_ErrorNo == E_NULL)
	&& (FILESTAT_Type(&filestatbuffer) & F_REG)
	&& (FILESTAT_NLinks(&filestatbuffer) != 1)) {
      *secondarysourcelistp =
	SecondarySourceList_IncludeSource(*secondarysourcelistp,
					  nodename,
					  &filestatbuffer,
					  collectionp);
    }
    /* get source from file status */
    if (PROGRAM_ErrorNo == E_NULL)
      targetsourcep = Collection_TargetSourceFromFileStatus(&filestatbuffer,
							    source,
							    collectionp);
    /* add the source to sourcelist of newtargetdbp */
    if (PROGRAM_ErrorNo == E_NULL)
      TARGETDB_SourceList(newtargetdbp) =
	Collection_SourceList_AddSource(TARGETDB_SourceList(newtargetdbp),
					targetsourcep,
					collectionp);

    if (PROGRAM_ErrorNo == E_NULL)
      TargetSource_Free(targetsourcep);
  }
  /* 
   * if depotconf specifications haven't changed since last run,
   * check if old hints and new hints match
   */
  HintComparison = FALSE;
  if ((PROGRAM_ErrorNo == E_NULL)
      && !(COLLECTION_Status(collectionp) & C_DEPOTCONFCHANGE)
      && (hintdb_old != NULL)
      && (COLLECTION_HintDB(collectionp) != NULL)) {
    oldhint = HintDB_GetHint(hintdb_old, source);
    if ((PROGRAM_ErrorNo == E_NULL) && (oldhint != NULL)) {
      newhint = HintDB_GetHint(COLLECTION_HintDB(collectionp), source);
      if ((PROGRAM_ErrorNo == E_NULL) && (newhint != NULL))
	HintComparison = Hint_Comparator(oldhint, newhint);
    }
  }
  if (PROGRAM_ErrorNo == E_NULL) {
    if (HintComparison == TRUE) {
      /* mark source to newtargetdbp as enduring subtree */
      locatedsourceindex =
	SourceList_FindIndexToSource(TARGETDB_SourceList(newtargetdbp),
				     source,
				     COLLECTION_Id(collectionp),
				     TDB_LOCATE);
      if ((PROGRAM_ErrorNo == E_NULL) && (locatedsourceindex >= 0)) {
	locatedtargetsourcep =
	  TARGETSOURCELIST_Source(TARGETDB_SourceList(newtargetdbp),
				  locatedsourceindex);
	TARGETDB_Status(locatedtargetsourcep) |= S_ENDURINGSUBTREE;
	TARGETDB_UpdateSpec(locatedtargetsourcep)
	  |= COLLECTION_MapMethod(collectionp);
      }
    } else if (FILESTAT_Type(&filestatbuffer) & F_DIR) {	
      /* source represents a directory, recursively map children */
      childlistset = NULL;
      (void) File_ListDirectory(sourcepath, &childlistset);

      if ((PROGRAM_ErrorNo == E_NULL)
	  && (childlistset != NULL)
	  && (STRINGSET_Size(childlistset) > 0)
	  && (STRINGSET_Values(childlistset) != NULL)) {
	childsecondarysourcelist = NULL;
	for (childp = STRINGSET_Values(childlistset);
	     (PROGRAM_ErrorNo == E_NULL) && (*childp != NULL);
	     childp++) {
	  /* compute new source for child node */
	  (void) sprintf(childsource, "%s/%s", source, *childp);

	  /* 
	   * locate slot for child node under newtargetdb,
	   * creating it if necessary
	   */
	  if (PROGRAM_ErrorNo == E_NULL) {
	    targetdbchildslot =
	      TargetDB_LocateIndexToChildNode(newtargetdbp,
					      *childp,
					      TDB_LOCATE | TDB_CREATE);
	    if (PROGRAM_ErrorNo == E_NULL) {
	      targetdbchildpp =
		TARGETDB_Children(newtargetdbp) + targetdbchildslot;
	    }
	  }
	  /* 
	   * recursively map onto child slot of newtargetdbp
	   * from chilp, the current child of source
	   */
	  if (PROGRAM_ErrorNo == E_NULL) {
	    *targetdbchildpp =
	      Collection_SubTreeMapFromPath(*targetdbchildpp,
					    *childp,
					    childsource,
					    collectionp,
					    hintdb_old,
					    &childsecondarysourcelist);
	  }
	}
	StringArray_Free(childlistset);

	/* add secondary sources from from the childsecondarysourcelist */
	if (childsecondarysourcelist != NULL) {
	  for (ssp = childsecondarysourcelist;
	       (PROGRAM_ErrorNo == E_NULL) && (ssp != NULL);
	       ssp = ssp->next) {
	    for (sp = ssp->source;
		 (PROGRAM_ErrorNo == E_NULL) && (sp != NULL);
		 sp = TARGETSOURCE_SecondarySource(sp)) {
	      (void) sprintf(childsource, "%s/%s",
			     source, TARGETSOURCE_Path(sp));
	      /* locate child corresponding to this source */
	      targetdbchildp =
		TargetDB_LocateChildNode(newtargetdbp,
					 TARGETSOURCE_Path(sp),
					 TDB_LOCATE);
	      /* locate source corresponding to this node in child */
	      if (PROGRAM_ErrorNo == E_NULL) {
		locatedsourceindex =
		  SourceList_FindIndexToSource
		  (TARGETDB_SourceList(targetdbchildp),
		   childsource,
		   COLLECTION_Id(collectionp),
		   TDB_LOCATE);
	      }
	      if ((PROGRAM_ErrorNo == E_NULL)
		  && (locatedsourceindex >= 0)) {
		locatedtargetsourcep =
		  TARGETSOURCELIST_Source
		  (TARGETDB_SourceList(targetdbchildp),
		   locatedsourceindex);
		TARGETSOURCE_SecondarySource(locatedtargetsourcep) =
		  TargetSource(ssp->source);
	      }
	    }
	  }
	  /* free the secondarysourcelist */
	  SecondarySourceList_Free(childsecondarysourcelist);
	}
      }
    }
  }
  return (PROGRAM_ErrorNo == E_NULL) ? newtargetdbp : NULL;
}



static SECONDARYSOURCEINFO *
SecondarySourceList_IncludeSource(secondarysourcelist,
				  name,
				  filestatp,
				  collectionp)
     SECONDARYSOURCEINFO *secondarysourcelist;
     char *name;
     FILESTAT *filestatp;
     COLLECTION *collectionp;
{
  register SECONDARYSOURCEINFO *ssp;
  register TARGETSOURCE *sp;

  SECONDARYSOURCEINFO *newsecondarysourcelist;

  TARGETSOURCE *targetsourcep;
  SECONDARYSOURCEINFO *locatedsecondarysourcep;
  Boolean LocatedSource;

  newsecondarysourcelist = secondarysourcelist;
  targetsourcep = TargetSource_Create(name,
				      (U_HARDLINK
				       | COLLECTION_MapMethod(collectionp)),
				      COLLECTION_Id(collectionp),
				      S_NULL,
				      U_NULL /* update_spec_old */ ,
				      NULL);

  LocatedSource = FALSE;
  ssp = newsecondarysourcelist;
  while (!LocatedSource && (ssp != NULL)) {
    if (FILESTAT_INode(filestatp) == (ssp->inode)) {
      locatedsecondarysourcep = ssp;
      LocatedSource = TRUE;
    } else {
      ssp = ssp->next;
    }
  }

  if (PROGRAM_ErrorNo == E_NULL) {
    if (LocatedSource) {
      sp = locatedsecondarysourcep->source;
      while (TARGETSOURCE_SecondarySource(sp) != NULL)
	sp = TARGETSOURCE_SecondarySource(sp);
      TARGETSOURCE_SecondarySource(sp) = targetsourcep;
    } else {			/* !LocatedSource */
      ssp = (SECONDARYSOURCEINFO *) emalloc(sizeof(SECONDARYSOURCEINFO));
      if (PROGRAM_ErrorNo == E_NULL) {
	ssp->inode = FILESTAT_INode(filestatp);
	ssp->source = targetsourcep;
	ssp->next = newsecondarysourcelist;
	newsecondarysourcelist = ssp;
	LocatedSource = TRUE;
      }
    }
  }
  return ((PROGRAM_ErrorNo == E_NULL)
	  && LocatedSource) ? newsecondarysourcelist : NULL;
}


static void 
SecondarySourceList_Free(secondarysourcelist)
     SECONDARYSOURCEINFO *secondarysourcelist;
{
  register SECONDARYSOURCEINFO *ssp1, *ssp2;

  ssp1 = secondarysourcelist;
  while (ssp1 != NULL) {
    ssp2 = ssp1->next;
    if (ssp1->source != NULL)
      TargetSource_Free(ssp1->source);
    (void) free((void *) ssp1);
    ssp1 = ssp2;
  }

  return;
}
/* $Source: /afs/andrew.cmu.edu/system/src/local/depot2/018/src/lib/Collection/RCS/Collection_PathMap.c,v $ */
