import './style.scss';

import { useMutation, useQueryClient } from '@tanstack/react-query';
import { isUndefined } from 'lodash-es';
import { ChangeEvent, useEffect, useState } from 'react';

import { useI18nContext } from '../../../../i18n/i18n-react';
import SvgIconCheckmark from '../../../../shared/components/svg/IconCheckmark';
import { Button } from '../../../../shared/defguard-ui/components/Layout/Button/Button';
import {
  ButtonSize,
  ButtonStyleVariant,
} from '../../../../shared/defguard-ui/components/Layout/Button/types';
import { Card } from '../../../../shared/defguard-ui/components/Layout/Card/Card';
import { CheckBox } from '../../../../shared/defguard-ui/components/Layout/Checkbox/CheckBox';
import { Input } from '../../../../shared/defguard-ui/components/Layout/Input/Input';
import { MessageBox } from '../../../../shared/defguard-ui/components/Layout/MessageBox/MessageBox';
import { MessageBoxType } from '../../../../shared/defguard-ui/components/Layout/MessageBox/types';
import { TextareaAutoResizable } from '../../../../shared/defguard-ui/components/Layout/TextareaAutoResizable/TextareaAutoResizable';
import useApi from '../../../../shared/hooks/useApi';
import { useToaster } from '../../../../shared/hooks/useToaster';
import { QueryKeys } from '../../../../shared/queries';
import { useEnrollmentStore } from '../../hooks/useEnrollmentStore';

export const EnrollmentEmail = () => {
  const {
    settings: { editSettings },
  } = useApi();
  const queryClient = useQueryClient();
  const { LL } = useI18nContext();
  const [duplicateMessage, setDuplicateMessage] = useState(false);
  const settings = useEnrollmentStore((state) => state.settings);
  const [email, setEmail] = useState(settings?.enrollment_welcome_email ?? '');
  const [subject, setSubject] = useState(
    settings?.enrollment_welcome_email_subject ?? '',
  );
  const componentLL = LL.enrollmentPage.settings.welcomeEmail;
  const toaster = useToaster();

  const { isPending: isLoading, mutate } = useMutation({
    mutationFn: editSettings,
    onSuccess: () => {
      void queryClient.invalidateQueries({
        queryKey: [QueryKeys.FETCH_SETTINGS],
      });
      toaster.success(LL.enrollmentPage.messages.edit.success());
    },
    onError: (e) => {
      toaster.error(LL.enrollmentPage.messages.edit.error());
      console.error(e);
    },
  });

  const handleSave = () => {
    if (!isLoading && settings) {
      mutate({
        ...settings,
        enrollment_use_welcome_message_as_email: duplicateMessage,
        enrollment_welcome_email: email,
        enrollment_welcome_email_subject: subject,
      });
    }
  };

  useEffect(() => {
    if (settings) {
      setDuplicateMessage(settings.enrollment_use_welcome_message_as_email);
      setEmail(settings.enrollment_welcome_email);
      setSubject(settings.enrollment_welcome_email_subject);
    }
  }, [settings]);

  return (
    <div id="enrollment-email">
      <header>
        <h3>{componentLL.title()}</h3>
      </header>
      <MessageBox type={MessageBoxType.INFO} message={componentLL.messageBox()} />
      <Card shaded hideMobile>
        <div className="controls">
          <div className="checkbox-wrap">
            <CheckBox
              value={duplicateMessage}
              onChange={() => setDuplicateMessage((state) => !state)}
              disabled={isLoading}
            />
            <span onClick={() => setDuplicateMessage((state) => !state)}>
              {componentLL.controls.duplicateWelcome()}
            </span>
          </div>
          <Button
            text={LL.enrollmentPage.controls.save()}
            styleVariant={ButtonStyleVariant.SAVE}
            size={ButtonSize.SMALL}
            icon={<SvgIconCheckmark />}
            onClick={() => handleSave()}
            loading={isLoading}
            disabled={isUndefined(settings)}
          />
        </div>
        <Input
          label={componentLL.subject.label()}
          value={subject}
          onChange={(e) => setSubject(e.target.value)}
          disabled={isLoading || isUndefined(settings)}
        />
        <div className="text-wrapper">
          <TextareaAutoResizable
            value={email}
            onChange={(ev: ChangeEvent<HTMLTextAreaElement>) => setEmail(ev.target.value)}
            disabled={isLoading || isUndefined(settings)}
          />
        </div>
      </Card>
    </div>
  );
};
