package session

import (
	"io"
	"io/ioutil"
	"os"
	"path/filepath"

	log "github.com/sirupsen/logrus"
	"github.com/sirupsen/logrus/hooks/test"
	. "gopkg.in/check.v1"
)

type SessionLogSuite struct{}

var _ = Suite(&SessionLogSuite{})

func (s *SessionLogSuite) Test_openLogFile_returnsNilIfNoFilenameGiven(c *C) {
	c.Assert(openLogFile(""), IsNil)
}

func (s *SessionLogSuite) Test_openLogFile_failsIfCantOpenFile(c *C) {
	tf, ex := ioutil.TempFile("", "coyim-session-9-")
	c.Assert(ex, IsNil)
	ex = tf.Close()
	c.Assert(ex, IsNil)
	defer func() {
		ex2 := os.Remove(tf.Name())
		c.Assert(ex2, IsNil)
	}()

	ll := log.StandardLogger()
	orgLevel := ll.Level
	defer func() {
		ll.SetLevel(orgLevel)
	}()

	ll.SetLevel(log.DebugLevel)
	hook := test.NewGlobal()

	c.Assert(openLogFile(filepath.Join(tf.Name(), "something elsE")), IsNil)

	c.Assert(len(hook.Entries), Equals, 2)
	c.Assert(hook.LastEntry().Level, Equals, log.WarnLevel)
	c.Assert(hook.LastEntry().Message, Equals, "Failed to open log file.")
	c.Assert(hook.LastEntry().Data["error"], ErrorMatches, ".*(cannot find the (file|path)|not a directory).*")
}

func (s *SessionLogSuite) Test_openLogFile_succeeds(c *C) {
	tf, ex := ioutil.TempFile("", "coyim-session-10-")
	c.Assert(ex, IsNil)
	ex = tf.Close()
	c.Assert(ex, IsNil)
	defer func() {
		ex2 := os.Remove(tf.Name())
		c.Assert(ex2, IsNil)
	}()

	ll := log.StandardLogger()
	orgLevel := ll.Level
	defer func() {
		ll.SetLevel(orgLevel)
	}()

	ll.SetLevel(log.DebugLevel)
	hook := test.NewGlobal()

	res := openLogFile(tf.Name())
	c.Assert(res, Not(IsNil))

	c.Assert(len(hook.Entries), Equals, 1)
	c.Assert(hook.LastEntry().Level, Equals, log.DebugLevel)
	c.Assert(hook.LastEntry().Message, Equals, "Logging XMPP messages to file")
	c.Assert(hook.LastEntry().Data["file"], Equals, tf.Name())

	exx := res.(io.Closer).Close()
	c.Assert(exx, IsNil)
}
