/*  $Id: sdlmain.cpp,v 1.28 2011/10/31 01:37:54 sarrazip Exp $

    cosmosmash - A space rock shooting video game.
    Copyright (C) 2000-2011 Pierre Sarrazin <http://sarrazip.com/>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public
    License along with this program; if not, write to the Free
    Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA  02110-1301, USA.
*/

#include "CosmoSmashEngine.h"
#include "UserController.h"
#include "AutoController.h"
#include "ScriptController.h"

#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <fstream>
#include <memory>

using namespace std;


static const int defaultMillesecondsPerFrame = 50;


#ifdef HAVE_GETOPT_LONG
#include <getopt.h>


static struct option knownOptions[] =
{
    { "help",              no_argument,       NULL, 'h' },
    { "version",           no_argument,       NULL, 'v' },
    { "initial-score",     required_argument, NULL, 'i' },
    { "ms-per-frame",      required_argument, NULL, 'm' },
    { "mirror-hyperspace", no_argument,       NULL, 'y' },
    { "no-sound",          no_argument,       NULL, 'n' },
    { "full-screen",       no_argument,       NULL, 'f' },
    { "auto",              no_argument,       NULL, 'a' },
    { "record",            required_argument, NULL, 'r' },
    { "play",              required_argument, NULL, 'p' },
    { "no-auto-pause",     no_argument,       NULL, 'P' },

    { NULL, 0, NULL, 0 }  // marks the end
};


static void
displayVersionNo()
{
    printf("%s %s\n", PACKAGE, VERSION);
}


static void
displayHelp()
{
    displayVersionNo();

    printf(
"\n"
"Copyright (C) 2001, 2002 Pierre Sarrazin <http://sarrazip.com/>\n"
"This program is free software; you may redistribute it under the terms of\n"
"the GNU General Public License.  This program has absolutely no warranty.\n"
    );

    printf(
"\n"
"Known options:\n"
"--help               Display this help page and exit\n"
"--version            Display this program's version number and exit\n"
"--initial-score=N    Start game at score N (N >= 0) [default=0]\n"
"--ms-per-frame=N     N milliseconds per animation frame [default=%d]\n"
"                     Min. 1, max. 1000.  50 ms means 20 frames per second\n"
"--mirror-hyperspace  Hyperspace pivots around the center of the ground\n"
"                     instead of begin random [defaut=random]\n"
"--no-sound           Disable sound effects [default is to enable them]\n"
"--full-screen        Attempt to use full screen mode [default is window mode]\n"
"--auto               Automatic mode (the computer plays the game itself)\n"
"--record=FILENAME    Record the game into the specified file\n"
"                     (named file is overwritten if it already exists)\n"
"--play=FILENAME      Replay the game in the named file (written by --record)\n"
"--no-auto-pause      Do not pause automatically when window loses focus\n"
"\n"
    ,
    defaultMillesecondsPerFrame);
}
#endif  /* __GNUG__ */


int main(int argc, char *argv[])
{
    /*  Default game parameters:
    */
    long initScore = 0;
    int millisecondsPerFrame = defaultMillesecondsPerFrame;
    bool mirrorHyperspace = false;
    bool useSound = true;
    bool fullScreen = false;
    enum Mode { USER, AUTO, REPLAY } mode = USER;
    string recordFilename;  // game recorded only if this is not empty
    string playFilename;  // only relevant if mode == REPLAY
    bool processActiveEvent = true;


    #ifdef HAVE_GETOPT_LONG

    /*  Interpret the command-line options:
    */
    int c;
    do
    {
        c = getopt_long(argc, argv, "", knownOptions, NULL);

        switch (c)
        {
            case EOF:
                break;  // nothing to do

            case 'i':
                {
                    errno = 0;
                    long n = strtol(optarg, NULL, 10);
                    if (errno == ERANGE || n <= 0 || n >= 1000000)
                    {
                        displayHelp();
                        printf("\n\nInvalid initial score.\n");
                        return EXIT_FAILURE;
                    }

                    initScore = n;
                }
                break;

            case 'm':
                {
                    errno = 0;
                    long n = strtol(optarg, NULL, 10);
                    if (errno == ERANGE || n < 1 || n > 1000)
                    {
                        displayHelp();
                        printf("\n\nInvalid number of ms per frame.\n");
                        return EXIT_FAILURE;
                    }

                    millisecondsPerFrame = int(n);
                }
                break;
            
            case 'y':
                mirrorHyperspace = true;
                break;

            case 'n':
                useSound = false;
                break;

            case 'f':
                fullScreen = true;
                break;

            case 'a':
                mode = AUTO;
                break;

            case 'v':
                displayVersionNo();
                return EXIT_SUCCESS;

            case 'h':
                displayHelp();
                return EXIT_SUCCESS;

            case 'r':
                recordFilename = optarg;
                break;

            case 'p':
                mode = REPLAY;
                playFilename = optarg;
                break;

            case 'P':
                    processActiveEvent = false;
                break;

            default:
                displayHelp();
                return EXIT_FAILURE;
        }
    } while (c != EOF && c != '?');

    #endif  /* __GNUG__ */


    /*  Initialize the random number generator:
    */
    const char *s = getenv("SEED");
    unsigned int seed = (s != NULL ? atol(s) : time(NULL));
    #ifndef NDEBUG
    fprintf(stderr, "seed = %u\n", seed);
    fprintf(stderr, "init-score = %ld\n", initScore);
    fprintf(stderr, "ms-per-frame = %d\n", millisecondsPerFrame);
    #endif
    srand(seed);


    try
    {
        // Create a controller (i.e., a source of commands)
        // according to 'mode':

        Controller *controller = NULL;
        ifstream playFile;  // game script to REPLAY, if applicable

        switch (mode)
        {
            case USER:
                controller = new UserController();
                break;
            case AUTO:
                controller = new AutoController();
                break;
            case REPLAY:
                {
                    // Open the game script file, checks its format version,
                    // use its random number seed.

                    playFile.open(playFilename.c_str(), ios::in);
                    if (!playFile)
                    {
                        int e = errno;
                        fprintf(stderr, "%s: failed to open file %s: %s\n",
                                            PACKAGE, playFilename.c_str(), strerror(e));
                        return EXIT_FAILURE;
                    }

                    int requiredFileFormatVersion = 1;
                    try
                    {
                        ScriptController *sc = new ScriptController(
                                                    playFile,
                                                    requiredFileFormatVersion);
                        srand(sc->getSeed());
                        controller = sc;
                    }
                    catch (int errorCode)
                    {
                        switch (errorCode)
                        {
                            case -1:
                                fprintf(stderr, "%s: failed to read from file %s\n",
                                                    PACKAGE, playFilename.c_str());
                                break;
                            case -2:
                                fprintf(stderr, "%s: invalid data at start of file %s\n",
                                                    PACKAGE, playFilename.c_str());
                                break;
                            default:
                                fprintf(stderr, "%s: invalid file format (requiring %d, got %d)\n",
                                                    PACKAGE, requiredFileFormatVersion, errorCode);
                        }
                        return EXIT_FAILURE;
                    }
                }
                break;
        }

        // Create a file into which the game will be recorded, if requested:

        ofstream gameScriptFile;
        if (!recordFilename.empty())
        {
            gameScriptFile.open(recordFilename.c_str(), ios::out);
            if (!gameScriptFile)
            {
                int e = errno;
                fprintf(stderr, "%s: failed to create file %s: %s\n",
                                    PACKAGE, recordFilename.c_str(), strerror(e));
                return EXIT_FAILURE;
            }
            gameScriptFile << GAME_SCRIPT_FORMAT_VERSION << '\n' << static_cast<int>(seed) << '\n';
        }

        // Create and run the game engine:
        // Ownership of the Controller instance is given to the
        // CosmoSmashEngine instance, which is thus responsible
        // for destroying it.

        CosmoSmashEngine theCosmoSmashEngine(
                                initScore, mirrorHyperspace, useSound,
                                fullScreen, controller,
                                recordFilename.empty() ? NULL : &gameScriptFile,
                                mode == REPLAY ? false : processActiveEvent);

        controller = NULL;  // object now owned by theCosmoSmashEngine

        theCosmoSmashEngine.run(millisecondsPerFrame);
    }
    catch (const string &s)
    {
        fprintf(stderr, "%s: engine init error:\n%s\n", PACKAGE, s.c_str());
        return EXIT_FAILURE;
    }
    catch (int e)
    {
        fprintf(stderr, "%s: init error # %d\n", PACKAGE, e);
        return EXIT_FAILURE;
    }

    return EXIT_SUCCESS;
}
