/*  $Id: c-api.cpp,v 1.1 2012/07/08 00:46:01 sarrazip Exp $

    flatzebra - SDL-based sound renderer
    Copyright (C) 2011 Pierre Sarrazin <http://sarrazip.com/>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public
    License along with this program; if not, write to the Free
    Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA  02110-1301, USA.
*/

#include <roundbeetle/c-api.h>

#include <roundbeetle/Engine.h>
#include <roundbeetle/SoundRenderer.h>
#include <roundbeetle/ADSR.h>

#include <map>
#include <iostream>

using namespace roundbeetle;
using namespace std;


typedef map<int, FrequencyFunction *> RequestMap;

static RequestMap requestMap;


// This callback will be invoked under the renderer mutex.
//
static void
onRequestFinished(int reqHandle, void *)
{
    RequestMap::iterator it = requestMap.find(reqHandle);
    if (it == requestMap.end())
        return;
    assert(it->second != NULL);
    //cout << "onRequestFinished(" << reqHandle << "): " << it->second << endl;
    delete it->second;
    requestMap.erase(it);
}


int
flatzebra_sound_create(int rendererFreq)
{
    int status = Engine::create(rendererFreq, 0, NULL);
    if (status != 0)
        return status;  // failure
    Engine &engine = Engine::instance();
    engine.registerRequestFinishedCallback(onRequestFinished, NULL);
    engine.postBusInit();
    return 0;
}


void
flatzebra_sound_destroy()
{
    Engine::destroy();
}


void
flatzebra_sound_setMasterVolume(float linVol)
{
    SoundRenderer::instance().getTopFrameSourceAdder().setLinearAttenuation(linVol);
}


int
flatzebra_sound_requestSquareWave(float startFreq,
                              float endFreq,
                              float attackLevel,
                              float sustainLevel,
                              float attackTime,
                              float decayTime,
                              float sustainTime,
                              float releaseTime,
                              size_t loopCount)
{
    ADSR adsr(attackLevel, sustainLevel,
              attackTime, decayTime, sustainTime, releaseTime);

    LinearMovingFreq *ff = new LinearMovingFreq(startFreq, endFreq, adsr.getTotalTime());

    Engine &engine = Engine::instance();
    Bus &mainBus = engine.getMainBus();
    int reqHandle = engine.requestSquareWave(*ff, adsr, loopCount, mainBus);

    if (reqHandle == -1)  // if failed
        delete ff;  // onRequestFinished() will not be invoked in this case
    else
        requestMap[reqHandle] = ff;

    return reqHandle;
}


#if 0
int
flatzebra_sound_setSquareWaveFreq(int sqWaveHandle, float newWaveFreq)
{
    Engine &engine = Engine::instance();
    return engine.setSquareWaveFreq(sqWaveHandle, newWaveFreq);
}
#endif


int
flatzebra_sound_requestWhiteNoise(float attackLevel,
                              float sustainLevel,
                              float attackTime,
                              float decayTime,
                              float sustainTime,
                              float releaseTime,
                              size_t loopCount)
{
    Engine &engine = Engine::instance();
    ADSR adsr(attackLevel, sustainLevel,
              attackTime, decayTime, sustainTime, releaseTime);
    Bus &mainBus = engine.getMainBus();
    return engine.requestWhiteNoise(adsr, loopCount, mainBus);
}


int
flatzebra_sound_isRequestAlive(int reqHandle)
{
    Engine &engine = Engine::instance();
    return engine.isRequestAlive(reqHandle);
}


int
flatzebra_sound_pauseMainBus()
{
    Engine &engine = Engine::instance();
    return engine.getMainBus().pause();
}


int
flatzebra_sound_resumeMainBus()
{
    Engine &engine = Engine::instance();
    return engine.getMainBus().resume();
}
