/*  $Id: UserController.h,v 1.14 2011/10/31 01:37:54 sarrazip Exp $

    cosmosmash - A space rock shooting video game.
    Copyright (C) 2007-2011 Pierre Sarrazin <http://sarrazip.com/>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public
    License along with this program; if not, write to the Free
    Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA  02110-1301, USA.
*/

#ifndef _H_UserController
#define _H_UserController

#include "Controller.h"

#include <flatzebra/KeyState.h>
#include <flatzebra/Joystick.h>


class UserController : public Controller
{
public:

    UserController()
      : leftKS(SDLK_LEFT),
        rightKS(SDLK_RIGHT),
        startKS(SDLK_SPACE),
        hyperspaceKS(SDLK_DOWN),
        extendedKS(SDLK_e),
        pauseKS(SDLK_p),
        noshootKS(SDLK_LCTRL, SDLK_RCTRL),
        fullScreenKS(SDLK_F11),
        joystick()
    {
    }

    virtual ~UserController()
    {
    }

    virtual void processKey(SDLKey keysym, bool pressed)
    {
        leftKS.check(keysym, pressed);
        rightKS.check(keysym, pressed);
        hyperspaceKS.check(keysym, pressed);
        startKS.check(keysym, pressed);
        extendedKS.check(keysym, pressed);
        pauseKS.check(keysym, pressed);
        noshootKS.check(keysym, pressed);
        fullScreenKS.check(keysym, pressed);
    }

    /** Indicates if the player has just asked to start a game.
        Returns true if yes, and then 'isExtendedMode' indicates
        if the non-Intellivision extensions must be activated.
        Otherwise, returns false and 'isExtendedMode' is false.
    */
    virtual bool isStartRequested(bool &isExtendedMode)
    {
        bool ext = extendedKS.justPressed();
        bool start = (startKS.justPressed() || ext || joystick.buttonJustPressed(flatzebra::Joystick::START_BTN));
        isExtendedMode = start && ext;
        return start;
    }

    virtual bool isPauseRequested()
    {
        return pauseKS.justPressed() || joystick.buttonJustPressed(flatzebra::Joystick::BACK_BTN);
    }

    virtual bool isResumeRequested()
    {
        return pauseKS.justPressed() || joystick.buttonJustPressed(flatzebra::Joystick::START_BTN);
    }

    virtual bool isShootingActive()
    {
        return !noshootKS.isPressed() && !joystick.getButton(flatzebra::Joystick::A_BTN);
    }

    virtual bool isHyperspaceRequested()
    {
        return hyperspaceKS.justPressed() || joystick.buttonJustPressed(flatzebra::Joystick::B_BTN);
    }

    virtual bool isLeftMoveRequested()
    {
        return leftKS.isPressed() || joystick.getXAxisDisplacement() < 0;
    }

    virtual bool isRightMoveRequested()
    {
        return rightKS.isPressed() || joystick.getXAxisDisplacement() > 0;
    }

    virtual bool isFullScreenToggleRequested()
    {
        return fullScreenKS.justPressed();
    }

    /** Do any processing that needs to be done at the end of
        a tick.  In this case, we remember the state of some
        keys (pressed or not), so that justPressed() will work
        in the next tick.
        We also update the state of the joystick buttons and axes.
    */
    virtual void endOfTick()
    {
        startKS.remember();
        hyperspaceKS.remember();
        extendedKS.remember();
        pauseKS.remember();

        joystick.update();
    }

private:

    flatzebra::KeyState leftKS;
    flatzebra::KeyState rightKS;
    flatzebra::KeyState startKS;
    flatzebra::KeyState hyperspaceKS;
    flatzebra::KeyState extendedKS;
    flatzebra::KeyState pauseKS;
    flatzebra::KeyState noshootKS;
    flatzebra::KeyState fullScreenKS;
    flatzebra::Joystick joystick;

};


#endif  /* _H_UserController */
