#include <stdlib.h>
#include <stdio.h>
#include <assert.h>

#include "chess.h"
#include "tactics.h"
#include "eval.h"

#include "misc.h"

extern int posdat[6][8][8]; /*  Holds the score bonus for pieces on each position on the board */
extern int posdat_pawn[8][8]; /* Just the pawn position data to speed things up (-1 multiply) */
extern int rank_bonus[8];/*  Holds the bonus for pawns being on distant ranks */
extern int rank_bonus_half[8];/*  Holds half the bonus for pawns being on distant ranks */
extern int early_queen_penalty[8]; /*  Penalty for moving queen out early */
extern int piece_value[7]; /*  Values for the pieces (0,1,5,3,3,9,1000) */
extern int file_bonus[8],file_bonus2[8]; /*  Bonus for passed pawns on flank files, f_b*2 */
extern int first_pawn[8][2]; /*  Holds the first pawn on each file, if any */
extern int centre_board[8][8],corner_board[8][8]; /*  Bonus for corner proximity, centre proximity */
extern int halved[1000]; /* Take a guess */
extern int squared[20]; /* Take a guess */

/*  In all these tact_ functions, x,y hold piece co-ords.  sq is current board. */
/*   isdef holds defence status of the square */
int tact_pawn(const board *sq,int side,int x,int y,const compdat *cd,int def,int att) {
  int sc=PAWN_SCORE,b=y,pp=0,ks=0,cpp=0,passed=0,np=0,tpp=0;
  int friendly_pawn=1,pdir=-1,isdef=def-att,a;
  /*  pp=passed pawn score, ks=safe from king or not? 1/0 
   *  pdir is direction of friendly pawn travel */

  assert(y>0 && y<7); /* Must be in a legal position! */

  if (side==BLACK) {b=7-y;friendly_pawn=-1;pdir=1;}
  sc+=posdat_pawn[b][x]; /*  Position score data */

   /*  Pawn advance bonus depending on rank, doubled if opening is over */
  if (isdef>=0) {
    if (cd->gamestage<60) sc+=rank_bonus[b]; /*  Full value */
    else sc+=rank_bonus_half[b]; /*  Halved */
  }
   /* Sort out weak pawn chains */
  if (isdef<1) {
    if (b==6) sc-=UNDEF_BACK_PAWN;
    else {
      if (x>0 && sq->sq[y+pdir][x-1]==friendly_pawn) sc-=UNDEF_BACK_PAWN;
      else if (x<7 && sq->sq[y+pdir][x+1]==friendly_pawn) sc-=UNDEF_BACK_PAWN;
    }
  }

   /*  Penalise doubled (or tripled etc...) pawns on one file */
  if (sq->pawnfile[x][side-1]>1) {
    if (cd->gamestage>60) sc -= DOUBLED_PAWNS;
    else sc -= DOUBLED_PAWNS_LATE;
     /* Further penalise if they are isolated as well */
    if ((x==0 || sq->pawnfile[x-1][side-1]==0) && (x==7 || sq->pawnfile[x+1][side-1]==0)) {
      sc -= DOUBLED_PAWNS_ISO;
    }
  }
   /*  Passed and Backward Pawns */
  if (cd->gamestage<66) { /*  If not in very beginning of opening */
     /*  Reward Passed Pawns */
    if (b<5 || cd->gamestage<40) {
      pp=passed_pawn(sq,x,y,side); /*  Returns 0 or PASSED_PAWN_(1-3) */
      passed=pp;
      if (passed) sc+=rank_bonus[b]; /*  Bigger pawn advance bonus if passed */
      if (pp==PASSED_PAWN_3) tpp=1; /*  Totally passed pawn */
      if (tpp) {
        if (isdef>0) pp += PP_DEF_BON; /*  Bonus if passed pawn is defended */
        else if (isdef<0) pp -= PP_ATTACKED; /*  Penalty if attacked */
	if (cd->gamestage<50) pp += file_bonus[x]; /*  Bonus for passed pawns on the flanks */
        if (cd->gamestage<20) pp += file_bonus[x]; /*  ... especially very near end of game */
      }
      if (cd->gamestage>60) sc += halved[pp]; /*  Still beginning of middlegame - don't reward too much as pp's are weak */
      else sc += pp;
    }
     /*  Penalise backward and/or isolated pawns (unless passed) */
    if (!passed && b>3) sc += backward_pawn(sq,x,y,side);  /*  Returns BACKWARD_PAWN_(1-5) */
  }

  /*  Do other endgame stuff as well. */
  if (cd->gamestage<54) {
     /*  Reward if opp. king is not within pawn's promotion square */
    if (side==WHITE) { /*  White pawn */
      if (sq->kpos[3]<=y) { /*  If ahead of pawn */
        if (abs(sq->kpos[2]-x)>y) ks=1; /*  If outside promotion square then reward */
      }
       /*  If behind pawn then reward automatically */
      else ks=1;
       /*  If pawn is blockaded then penalise. */
      if (sq->sq[y-1][x]==-2 || sq->sq[y-1][x]==-5) sc-=HOSTILE_BLOCKADE;
      else if (sq->sq[y-1][x]<0) sc-=PIECE_BLOCKADE;
       /* Check to see if TPP is defended from behind by a rook/queen */
      if (tpp) {
        for (a=y+1;a<8;a++) if (sq->sq[a][x]!=0) break;       
        if (a<8 && (sq->sq[a][x]==2 || sq->sq[a][x]==5)) sc+=ROOK_DEF_PP;
      }
       /*  Connected passed pawns if no opp. bish/queen - large bonus */
      if (y<3 && tpp) {
	cpp=0;
	if (x>0 && sq->sq[y-1][x-1]==1) cpp=CONNECTED_PP_DEF;
	else if (x<7 && sq->sq[y-1][x+1]==1) cpp=CONNECTED_PP_DEF;
	else if (x<7 && sq->sq[y][x+1]==1) cpp=CONNECTED_PP;
	if (cpp>0) {
	    /* Reward CPPs on last rank */
	  if (y==1 && cpp==CONNECTED_PP) cpp+=CPP_7TH_RANK;
	  if (sq->npieces[1][1]>0) np++; /*  Penalise if opp. has knight */
	  if (sq->npieces[1][2]>0) np++; /*  ... or bishop */
	  if (sq->npieces[1][3]>0) np++; /*  ... or queen */
	  if (abs(sq->kpos[2]-x)<2 && abs(sq->kpos[3]-y)<2) np++; /* Penalise if opp. king is too near */
	  if (np==1) sc+=halved[cpp];
	  else sc+=cpp>>np;
	}
      }
    }  
    else { /*  Ditto for Black */
      if (sq->kpos[1]>=y) {
        if (abs(sq->kpos[0]-x)>b) ks=1;
      } 
      else ks=1;
       /*  Test for blockade */
      if (sq->sq[y+1][x]==2 || sq->sq[y+1][x]==5) sc-=HOSTILE_BLOCKADE;
      else if (sq->sq[y+1][x]>0) sc-=PIECE_BLOCKADE;
       /* Check to see if TPP is defended from behind by a rook/queen */
      if (tpp) {
        for (a=y-1;a>=0;a--) if (sq->sq[a][x]!=0) break;       
        if (a>=0 && (sq->sq[a][x]==-2 || sq->sq[a][x]==-5)) sc+=ROOK_DEF_PP;
      }
       /*  Connected Passed Pawns if no opp. bishop/queen - large bonus */
      if (y>4 && tpp) {
	cpp=0;
	if (x>0 && sq->sq[y+1][x-1]==-1) cpp=CONNECTED_PP_DEF;
	else if (x<7 && sq->sq[y+1][x+1]==-1) cpp=CONNECTED_PP_DEF;
	else if (x<7 && sq->sq[y][x+1]==-1) cpp=CONNECTED_PP;
	if (cpp>0) {
	  if (y==6 && cpp==CONNECTED_PP) cpp+=CPP_7TH_RANK;
	  if (sq->npieces[0][1]) np++; /*  Penalise if opp. has knight */
	  if (sq->npieces[0][2]) np++; /*  ... or bishop */
	  if (sq->npieces[0][3]) np++; /*  ... or queen */
	  if (abs(sq->kpos[0]-x)<2 && abs(sq->kpos[1]-y)<2) np++; /* Penalise if opp. king is too near */
	  if (np==1) sc+=halved[cpp];
	  else sc+=cpp>>np;
	}
      }
    }
    if (ks) { /*  If safe from king */
      sc+=KING_SAFETY; /*  Score bonus for safety from opp. king */
      if (tpp) sc+=KING_SAFETY_PP;  /*  Totally passed pawn & safe from king */
      /* If totally passed, safe from opponent's king and no
       *  opposing pieces on the board then give big bonus */
      if (tpp && sq->npieces[2-side][0]==0 && sq->npieces[2-side][1]==0 && sq->npieces[2-side][2]==0 && sq->npieces[2-side][3]==0) {
	 /* Don't reward connected pawns too much or else they don't want to promote :) */
	if (cpp==0) sc+=UNSTOPPABLE_PP;
	sc+=rank_bonus[b];
      }
    }
     /*  King Tropism in endgame (not in opening or else tends to discourage central pawn advance) */
    if (cd->gamestage<20) {
      if (side==WHITE && (max(abs(x-sq->kpos[0]),abs(y-sq->kpos[1]))) < 3) sc+=KING_TROPISM;
      if (side==BLACK && (max(abs(x-sq->kpos[2]),abs(y-sq->kpos[3]))) < 3) sc+=KING_TROPISM;
    }
    else if (cd->gamestage<40) {
      if (side==WHITE && (max(abs(x-sq->kpos[0]),abs(y-sq->kpos[1]))) < 3) sc+=KING_TROPISM_MILD;
      if (side==BLACK && (max(abs(x-sq->kpos[2]),abs(y-sq->kpos[3]))) < 3) sc+=KING_TROPISM_MILD;
    }
  }
  return sc;
}

int tact_rook(const board *sq,int side,int x,int y) {
  int sc=ROOK_SCORE,b=y,m,sl=0,sr=0,mov=0,a;

  if (side==BLACK) b=7-y;
  sc+=posdat[1][b][x]; /*  Precalculated position score */
  if (sq->pawnfile[x][side-1]==0) { /*  Freefile bonus */
    if (sq->pawnfile[x][2-side]==0) sc+=OPEN_FILE; /*  Fully open file */
    else sc+=HALF_OPEN_FILE; /*  Half open file */
  }
   /*  Penalise if a pawn can drive it away */
  sc-=drive_away(sq,x,y,side);
   
   /*  Bonus for attacking opp. king */
  if (side==WHITE) {
    m=min(abs(x-sq->kpos[2]),abs(y-sq->kpos[3]));
    sc -= m * ROOK_ATTACK;
  }
  if (side==BLACK) {
    m=min(abs(x-sq->kpos[0]),abs(y-sq->kpos[1]));
    sc -= m * ROOK_ATTACK;
  }

   /* Check to see if blocked in to the sides by pieces which aren't
    * defending it.  Kings aren't included */
  if (x>0) sl=sq->sq[y][x-1];
  if (x<7) sr=sq->sq[y][x+1];
  if (side==WHITE) {
    if (sl>0 && sl!=2 && sl!=5) sc-=ROOK_TRAPPED_SIDE;
    if (sr>0 && sr!=2 && sr!=5) sc-=ROOK_TRAPPED_SIDE;
  }
  if (side==BLACK) {
    if (sl<0 && sl!=-2 && sl!=-5) sc-=ROOK_TRAPPED_SIDE;
    if (sr<0 && sr!=-2 && sr!=-5) sc-=ROOK_TRAPPED_SIDE;
  }
   
   /* Check horizontal freedom */
  for (a=x-1;a>=0;a--) if (sq->sq[y][a]==0) mov++; else break;
  for (a=x+1;a<8;a++) if (sq->sq[y][a]==0) mov++; else break;
   sc+=mov*2;
   /* Penalise if totally blocked horizontally */
  if (mov==0) sc-=ROOK_BLOCKED_H;

   /* Return the resultant score */
  return sc;
}

int tact_knight(const board *sq,int side,int x,int y,const compdat *cd) {
  int sc=KNIGHT_SCORE,b=y;

  if (side==BLACK) b=7-y;
  if (cd->gamestage<50) sc+=(cd->gamestage-50); /*  Worth less in open endgame */
  if (cd->gamestage>60) sc+=(cd->gamestage-60); /*  Worth more in cramped opening */
  if (cd->gamestage>76 && b<4) sc+=(b-7); /*  Don't move it too far forward in opening */
  sc+=posdat[2][b][x]; /*  Precalculated position score */
   /*  Penalise if a pawn can drive it away */
  sc-=drive_away(sq,x,y,side);

   /*  Opponent king tropism */
  if (side==WHITE) sc -= (abs(x-sq->kpos[2]) + abs(y-sq->kpos[3]));
  if (side==BLACK) sc -= (abs(x-sq->kpos[0]) + abs(y-sq->kpos[1]));

   /*  Don't block in central pawns */
  if (side==WHITE && (x==3 || x==4) && y==5 && sq->sq[6][x]==1) sc-=BLOCK_PAWNS;
  if (side==BLACK && (x==3 || x==4) && y==2 && sq->sq[1][x]==-1) sc-=BLOCK_PAWNS;

   /* Return the resultant score */
  return sc;
}

int tact_bishop(const board *sq,int side,int x,int y,const compdat *cd) {
  int sc=BISHOP_SCORE,b=y;

  if (side==BLACK) b=7-y;
   /*  Precalculated Position Score */
  sc+=posdat[3][b][x];
   /*  Worth more in open endgame */
  if (cd->gamestage<50) sc+=(50-cd->gamestage);
   /*  Don't block in central pawns */
  if (side==WHITE && (x==3 || x==4) && y==5 && sq->sq[6][x]==1) sc-=BLOCK_PAWNS;
  if (side==BLACK && (x==3 || x==4) && y==2 && sq->sq[1][x]==-1) sc-=BLOCK_PAWNS;

   /* Return the resultant score */
  return sc;
}

 /* Evaluate a queen score */
int tact_queen(const board *sq,int side,int x,int y,const compdat *cd) {
  int sc=QUEEN_SCORE,b=y,pen=0,m,dist;

  if (side==BLACK) b=7-y;
  sc+=posdat[4][b][x]; /*  Precalculated Position Score */
  if (sq->pawnfile[x][side-1]==0) { /*  No friendly pawns on file */
    if (sq->pawnfile[x][2-side]==0) sc+=OPEN_FILE_Q; /*  Fully open file */
    else sc+=HALF_OPEN_FILE_Q; /*  Half open file */
  }
   /*  Substantial bonus for having a queen in the endgame  */
  if (cd->gamestage<35) sc+=((35-cd->gamestage)*QUEEN_ENDGAME_BON);
  if (b<7) { /*  Penalise for bringing queen out at wrong time */
    if (cd->gamestage==78) sc-=early_queen_penalty[b]; /*  Penalise for bringing queen out early */
    if (side==WHITE) { /*  Penalise for bringing Queen out before minor pieces */
      if (sq->sq[7][1]==3) pen++;
      if (sq->sq[7][6]==3) pen++;
      if (sq->sq[7][2]==4) pen++;
      if (sq->sq[7][5]==4) pen++;
    }
    else if (side==BLACK) {
      if (sq->sq[0][1]==-3) pen++;
      if (sq->sq[0][6]==-3) pen++;
      if (sq->sq[0][2]==-4) pen++;
      if (sq->sq[0][5]==-4) pen++;
    }
    if (pen==4) sc-=EARLY_QUEEN_1; /*  ... before both knights and both bishops */
    else if (pen==3) sc-=EARLY_QUEEN_2; /*  ... before any 3 of the above */
    else if (pen==2) sc-=EARLY_QUEEN_3; /*  ... before any 2 of the above */
  }

   /*  Penalise if a pawn can drive it away */
  sc-=halved[drive_away(sq,x,y,side)];
   /*  Don't block in central pawns */
  if (side==WHITE && (x==3 || x==4) && y==5 && sq->sq[6][x]==1) sc-=BLOCK_PAWNS;
  if (side==BLACK && (x==3 || x==4) && y==2 && sq->sq[1][x]==-1) sc-=BLOCK_PAWNS;

   /*  Bonus if no opponent queen */
  if (sq->npieces[2-side][3]==0) sc+=LONE_QUEEN_BONUS;
   
   /*  Bonus for attacking opp. king */
  if (side==WHITE) {
    m=(min(abs(x-sq->kpos[2]),abs(y-sq->kpos[3]))); /* Minimum cartesian displacement */
    if (abs(x-sq->kpos[2]) == abs(y-sq->kpos[3])) m--; /* Bonus for same diagonal */
    sc -= m * QUEEN_ATTACK;
    dist=abs(x-sq->kpos[2]) + abs(y-sq->kpos[3]); /* Bonus for attacking the enemy king */
    sc += (8-dist)*2;
  }
  if (side==BLACK) {
    m=(min(abs(x-sq->kpos[0]),abs(y-sq->kpos[1]))); /* Minimum cartesian displacement */
    if (abs(x-sq->kpos[0]) == abs(y-sq->kpos[1])) m--; /* Bonus for same diagonal */
    sc -= m * QUEEN_ATTACK;
    dist=abs(x-sq->kpos[0]) + abs(y-sq->kpos[1]); /* Bonus for attacking the enemy king */
    sc += (8-dist)*2;
  }
     
   /* Return the resultant score */
  return sc;
}

int tact_king(const board *sq,int side,int x,int y,const compdat *cd) {
  int sc=0,flg,b=y;

  if (side==BLACK) b=7-y;
  sc+=posdat[5][b][x]; /*  Precalculated Position Score */
  if (cd->gamestage>50) sc+=corner_board[b][x]; /* Move to corners in opening */
  else if (cd->gamestage<44) {
     /*  Move towards the centre */
    if (cd->gamestage>30) sc+=centre_board[b][x];
     /*  More encouragement nearer the end */
    else if (cd->gamestage>20) sc+=centre_board[b][x]*2;
    else { /* If cd->gamestage <= 20 */
      sc+=centre_board[b][x]*4;  /* Still more central tropism */
       /* Check to make sure that the King doesn't get trapped in the corners
	* or on the sides if the opponent has rooks or queens left and
	* the side to score is not winning */
      if (cd->pts[side-1]<=cd->pts[2-side]) {
	if (sq->npieces[2-side][0]>0 || sq->npieces[2-side][3]>0) {
          sc+=centre_board[b][x]*4;  /* Still more central tropism */
	}
      }
    }
     /* If opp has <5 points then reward for
      * keeping your king close to his */
    if (cd->pts[2-side]<5) {
      sc+=18-(3*(abs(sq->kpos[0]-sq->kpos[2])+abs(sq->kpos[1]-sq->kpos[3])));
    }
  }
  flg=0;
  if (cd->gamestage<50) {
    if (y==0) {
      /*  Check to see if boxed in on back rank (vulnerable to back rank mates) */
      if (sq->sq[1][x]==0) flg=1;   /*  ... and penalise if undefended */
      else if (x>0 && sq->sq[1][x-1]==0) flg=1;
      else if (x<7 && sq->sq[1][x+1]==0) flg=1;
      if (flg==0) sc-=BACK_RANK_UNSAFE;
    }
    else if (y==7) { /*  Ditto for white */
      if (sq->sq[6][x]==0) flg=1;
      else if (x>0 && sq->sq[6][x-1]==0) flg=1;
      else if (x<7 && sq->sq[6][x+1]==0) flg=1;
      if (flg==0) sc-=BACK_RANK_UNSAFE;
    }
  }
   /* Add on overall king defence score if opp. has bishop(s), rooks(s) or queen(s) */
  if (sq->npieces[2-side][0]>0 || sq->npieces[2-side][3]>0 || sq->npieces[2-side][2]>0) {
    sc += king_defence(sq,x,y,side);
  }
   
  return sc;
}

 /*  Overall score bonuses for various things */
int tact_overall(const board *sq,int side,const compdat *cd) {
  int sc=0,castledw=1,kx,castledb=1;

   /* I previously used a flag to see if the side had castled, but I decided that
    * this was a crap idea simply because two positions could look identical
    * but one could have been got from castling and one could have been got
    * without and they would be scored differently.  Also, it's dumb to penalise
    * players in the endgame if they didn't castle in the opening.  If failing to castle
    * was such a bad move then it would have caused other problems by then...
    * Instead I test on the problems caused by not castling, and check for the
    * general positional problems which ensue */
  if (sq->kpos[1]==7) {
    kx=sq->kpos[0];
    if (cd->gamestage>64 && kx>2 && kx<6) castledw=0;
    else if ((kx==1 || kx==2) && (sq->sq[7][0]==2 || sq->sq[7][1]==2)) castledw=0;
    else if ((kx==5 || kx==6) && (sq->sq[7][6]==2 || sq->sq[7][7]==2)) castledw=0;
  }
  if (sq->kpos[3]==0) {
    kx=sq->kpos[2];
    if (cd->gamestage>64 && kx>2 && kx<6) castledb=0;
    else if ((kx==1 || kx==2) && (sq->sq[0][0]==-2 || sq->sq[0][1]==-2)) castledb=0;
    else if ((kx==5 || kx==6) && (sq->sq[0][6]==-2 || sq->sq[0][7]==-2)) castledb=0;
  }

   /*  Castling bits */
  if (castledw==1) sc+=REWARD_CASTLE; /*  if castled then reward */
  else { /*  If not castled and spoilt chance then penalise */
    if (cd->cp[0]==0 || cd->cp[1]==0) {
      sc-=SPOILT_CASTLE_1; /*  Spoilt at least one side */
      if (cd->cp[0]==0 && cd->cp[1]==0) sc-=SPOILT_CASTLE_2; /*  Both sides */
    }
  }
  if (castledb==1) sc-=REWARD_CASTLE; /*  Ditto for opponent */
  else { /*  If not castled and spoilt chance then penalise */
    if (cd->cp[2]==0 || cd->cp[3]==0) {
      sc+=SPOILT_CASTLE_1; /*  Spoilt at least one side */
      if (cd->cp[2]==0 && cd->cp[3]==0) sc+=SPOILT_CASTLE_2; /*  Both sides */
    }
  }

   /*  Encourage retention of centre-file pawns */
  if (cd->gamestage>45) {
    if (sq->pawnfile[3][0]>0) sc+=CENTRE_PAWN_BONUS;
    else if (sq->pawnfile[4][0]>0) sc+=CENTRE_PAWN_BONUS;
    else sc-=NO_CENTRE_PAWNS;
    if (sq->pawnfile[3][1]>0) sc-=CENTRE_PAWN_BONUS;
    else if (sq->pawnfile[4][1]>0) sc-=CENTRE_PAWN_BONUS;
    else sc+=NO_CENTRE_PAWNS;
  }
   
   /* Return this score if scoring for white, or minus the score if scoring for black */
  if (side==WHITE) return sc;
  return -sc;
}


/* Used for preliminary move ordering. */
void tact_general(move *mv, const board *sq) {
  int fx,fy,tx,ty,cdist,fcdist,side;

  fx=mv->fx;
  fy=mv->fy;
  tx=mv->tx;
  ty=mv->ty;
  if (sq->sq[fy][fx]>0) side=WHITE;
  else side=BLACK;
  
   /*  Moving towards centre */
  if (fx!=tx && abs(sq->sq[fy][fx])!=4) { /*  Changing files & not a bishop... */
    if (tx<4) cdist=3-tx;
    else cdist=tx-4;
    if (fx<4) fcdist=3-fx;
    else fcdist=fx-4;
    (mv->score)+=(fcdist-cdist)*8;
  }

   /* Discourage moves which end up next to enemy pawns */
  if (side==WHITE && ty>1) {
    if (tx>1 && sq->sq[ty-1][tx-1]==-1) mv->score-=4;
    if (tx<7 && sq->sq[ty-1][tx+1]==-1) mv->score-=4;
  }
  else if (side==BLACK && ty<6) {
    if (tx>1 && sq->sq[ty+1][tx-1]==1) mv->score-=4;
    if (tx<7 && sq->sq[ty+1][tx+1]==1) mv->score-=4;
  }
}


 /*   ----------  SPECIALIST ENDGAME FUNCTIONS  ----------   */

/* Score a pawn when there are only pawns and kings on
  * the board.  Optimised for this particular case. */
int tact_pawn_kp(const board *sq,int side,int x,int y,int att) {
  int sc=PAWN_SCORE,b=y,pp=0,ks=0,cpp=0,passed=0,tpp=0,kt=0;
  /*  pp=passed pawn score, ks=safe from king or not? 1/0 */

  assert(y>0 && y<7); /* Must be in a legal position! */

   /* 'b' stores rank advancement */
  if (side==BLACK) b=7-y;
   /*  Pawn advance bonus depending on rank */
  sc+=rank_bonus[b];
   /*  Penalise doubled (or tripled etc...) pawns on one file */
  if (sq->pawnfile[x][side-1]>1) sc-=DOUBLED_PAWNS;
   
   /*  Reward Passed Pawns */
  if (b<5) {
    pp=passed_pawn_kp(sq,x,y,side); /*  Returns 0 or PASSED_PAWN_(1-3) */
    passed=pp;
     /*  Bigger pawn advance bonus if passed or semi-passed */
    if (pp>PASSED_PAWN_1) sc+=rank_bonus[b];
     /* Check if totally passed (no pawns on adjacent files either) */
    if (pp==PASSED_PAWN_3) {
      tpp=1;
       /* Reward if unattacked */
      if (att==0) pp+=pp;
       /* Reward totally passed pawns for advancing (and penalise for not advancing) */
      sc+=rank_bonus[b];
       /* Reward totally passed pawns on the flanks */
      pp+=file_bonus2[x];
    }
    sc+=pp;
    /*  Penalise backward and/or isolated pawns (unless passed) 
     *  Returns BACKWARD_PAWN_(1-5) */
    if (!passed && b>3) sc+=backward_pawn(sq,x,y,side);
  }
   /* Reward if friendly king is ahead of pawn */
  if (tpp==1) {
     /* ...in the three adjoining squares.  Theoretically a very strong position */
    if (side==WHITE && sq->kpos[1]==y-1 && abs(sq->kpos[0]-x)<2) sc+=KING_BUFFER;
    else if (side==BLACK && sq->kpos[3]==y+1 && abs(sq->kpos[2]-x)<2) sc+=KING_BUFFER;
     /* ...in x+/- 1 square plus 1 rank gap. Also strong */
    if (side==WHITE && sq->kpos[1]==y-2 && abs(sq->kpos[0]-x)<2) sc+=KING_BUFFER;
    else if (side==BLACK && sq->kpos[3]==y+2 && abs(sq->kpos[2]-x)<2) sc+=KING_BUFFER;
  }
  /*  Reward if opp. king is not within pawn's promotion square *
   *  Remembering that next move is for opponent! */
  if (side==WHITE) { /*  White pawn */
    if (sq->kpos[3]<=y) { /*  If opp. king ahead of pawn */
      if (abs(sq->kpos[2]-x)>y) ks=1; /*  If outside promotion square then reward */
    }
    else ks=1; /*  If behind pawn then reward automatically */
     /*  If pawn is blockaded by opp. King then penalise. */
    if (sq->kpos[2]==x && sq->kpos[3]<y) {
      sc-=HOSTILE_BLOCKADE;
      if (sq->sq[y-1][x]==-6) sc-=HOSTILE_BLOCKADE;
    }
     /*  Connected passed pawns - large bonus */
    if (y<3 && tpp) {
      cpp=0;
      if (x>0 && sq->sq[y-1][x-1]==1) cpp=CONNECTED_PP_DEF;
      if (x<7 && sq->sq[y-1][x+1]==1) cpp+=CONNECTED_PP_DEF;
      if (x<7 && sq->sq[y][x+1]==1) cpp+=CONNECTED_PP;
      if (abs(sq->kpos[2]-x)<2 && abs(sq->kpos[3]-y)<2) cpp=halved[cpp]; /* Penalise if opp. king is too near */
      if (cpp>0) sc+=cpp;
    }
  }  
  else if (side==BLACK) { /*  Ditto for Black */
    if (sq->kpos[1]>=y) {
      if (abs(sq->kpos[0]-x)>b) ks=1;
    } 
    else ks=1;
     /*  Test for blockade by opp. King */
    if (sq->kpos[0]==x && sq->kpos[1]>y) {
      sc-=HOSTILE_BLOCKADE;
      if (sq->sq[y+1][x]==6) sc-=HOSTILE_BLOCKADE;
    }
     /*  Connected Passed Pawns - large bonus */
    if (y>4 && tpp) {
      cpp=0;
      if (x>0 && sq->sq[y+1][x-1]==-1) cpp=CONNECTED_PP_DEF;
      if (x<7 && sq->sq[y+1][x+1]==-1) cpp+=CONNECTED_PP_DEF;
      if (x<7 && sq->sq[y][x+1]==-1) cpp+=CONNECTED_PP;
      if (abs(sq->kpos[0]-x)<2 && abs(sq->kpos[1]-y)<2) cpp=halved[cpp]; /* Penalise if opp. king is too near */
      if (cpp>0) sc+=cpp;
    }
  }
  if (ks) { /*  If safe from king */
    sc+=KING_SAFETY; /*  Score bonus for safety from opp. king */
    sc+=file_bonus2[x]; /* Bonus for being on the sides */
     /*  If totally passed, safe from opp. king and no opp. pieces on the board then give big bonus */
    if (tpp) {
       /* Don't reward connected pawns too much or else they don't want to promote :) */
      if (cpp==0) sc+=UNSTOPPABLE_PP;
       /*  Try to get them to advance */
      sc+=rank_bonus[b];
    }
  }
   
   /*  King Tropism */
  if (side==WHITE) kt=-(abs(sq->kpos[0]-x)+abs(sq->kpos[1]-y));
  else kt=-(abs(sq->kpos[2]-x)+abs(sq->kpos[3]-y));
   /* Enemy King anti-tropism */
  if (side==WHITE) kt+=(abs(sq->kpos[2]-x)+abs(sq->kpos[3]-y));
  else kt+=(abs(sq->kpos[0]-x)+abs(sq->kpos[1]-y));
   /* Add on tropism score times 2 */
  sc+=kt*2;

   /* Return Accumulated Score */
  return sc;
}

 /* Evaluate king when only kings and pawns left on the board */
int tact_king_kp(const board *sq,int side,int x,int y) {
  int sc=0,b=y;

  if (side==BLACK) b=7-y;

   /* Move towards the centre */
  sc+=centre_board[b][x]*4;
   
   /* Move in front of your own pawns to help protect them */
  if (side==WHITE && y<6) {
    if (sq->sq[y+1][x]==1) sc+=KING_PROT;
    if (x>0 && sq->sq[y+1][x-1]==1) sc+=KING_PROT;
    if (x<7 && sq->sq[y+1][x+1]==1) sc+=KING_PROT;
  }
  else if (side==BLACK && y>1) {
    if (sq->sq[y-1][x]==-1) sc+=KING_PROT;
    if (x>0 && sq->sq[y-1][x-1]==-1) sc+=KING_PROT;
    if (x<7 && sq->sq[y-1][x+1]==-1) sc+=KING_PROT;
  }

  return sc;
}

 /* Evaluate pawn for KPR endgame */
int tact_pawn_kpr(const board *sq,int side,int x,int y,const compdat *cd,int att) {
  int sc=PAWN_SCORE,b=y,pp=0,ks=0,cpp=0,passed=0,tpp=0,kt=0;

   /* No opponent rooks so use simpler analysis */
  if (sq->npieces[2-side][0]==0) return tact_pawn_kp(sq,side,x,y,att);

  assert(y>0 && y<7); /* Must be in a legal position! */
   
  if (side==BLACK) b=7-y;

  sc+=posdat_pawn[b][x]; /*  Position score data */

   /*  Pawn advance bonus depending on rank, doubled if opening is over */
   sc+=rank_bonus[b]; /*  Full value */
   /*  Penalise doubled (or tripled etc...) pawns on one file */
  if (sq->pawnfile[x][side-1]>1) sc-=DOUBLED_PAWNS_KPR;
   /*  Reward Passed Pawns */
  if (b<5) {
    pp=passed_pawn_kp(sq,x,y,side); /*  Returns 0 or PASSED_PAWN_(1-3) */
    passed=pp;
    if (passed) sc+=rank_bonus[b]; /*  Bigger pawn advance bonus if passed */
    if (pp==PASSED_PAWN_3) tpp=1; /*  Totally passed pawn */
    if (tpp) {
      if (att<0) pp+=pp; /*  Bonus doubled if passed pawn is defended */
      else if (att>0) sc-=PP_ATTACKED; /*  Penalty if attacked */
      sc+=rank_bonus[b];
      pp+=file_bonus[x]; /*  Bonus for passed pawns on the flanks */
    }
    sc+=pp;
    /*  Penalise backward and/or isolated pawns (unless passed) */
    if (!passed) sc+=backward_pawn(sq,x,y,side);  /*  Returns BACKWARD_PAWN_(1-5) */
  }

  /*  Reward if opp. king is not within pawn's promotion square */
  if (side==WHITE) { /*  White pawn */
    if (sq->kpos[3]<=y) { /*  If ahead of pawn */
      if (abs(sq->kpos[2]-x)>y) ks=1; /*  If outside promotion square then reward */
    }
     /*  If behind pawn then reward automatically */
    else ks=1;
     /*  If pawn is blockaded then penalise. */
    if (sq->sq[y+1][x]==-6 || sq->sq[y+1][x]==-2) sc-=HOSTILE_BLOCKADE;
     /*  Connected passed pawns - large bonus */
    if (y<3 && tpp) {
      cpp=0;
      if (x>0 && sq->sq[y-1][x-1]==1) cpp=CONNECTED_PP_DEF;
      else if (x<7 && sq->sq[y-1][x+1]==1) cpp=CONNECTED_PP_DEF;
      else if (x<7 && sq->sq[y][x+1]==1) cpp=CONNECTED_PP;
      if (cpp>0) {
         /* Reward CPPs on last rank */
        if (y==1 && cpp==CONNECTED_PP) cpp+=CPP_7TH_RANK;
        if (abs(sq->kpos[2]-x)<2 && abs(sq->kpos[3]-y)<2) cpp=halved[cpp]; /* Penalise if opp. king is too near */
	sc+=cpp;
      }
    }
  }
  else if (side==BLACK) { /*  Ditto for Black */
    if (sq->kpos[1]>=y) {
      if (abs(sq->kpos[0]-x)>b) ks=1;
    } 
    else ks=1;
     /*  Test for blockade */
    if (sq->sq[y+1][x]==6 || sq->sq[y+1][x]==2) sc-=HOSTILE_BLOCKADE;
     /*  Connected Passed Pawns - large bonus */
    if (y>4 && tpp) {
      cpp=0;
      if (x>0 && sq->sq[y+1][x-1]==-1) cpp=CONNECTED_PP_DEF;
      else if (x<7 && sq->sq[y+1][x+1]==-1) cpp=CONNECTED_PP_DEF;
      else if (x<7 && sq->sq[y][x+1]==-1) cpp=CONNECTED_PP;
      if (cpp>0) {
        if (y==6 && cpp==CONNECTED_PP) cpp+=CPP_7TH_RANK;
        if (abs(sq->kpos[0]-x)<2 && abs(sq->kpos[1]-y)<2) cpp=halved[cpp]; /* Penalise if opp. king is too near */
	sc+=cpp;
      }
    }
  }
  if (ks) { /*  If safe from king */
    sc+=KING_SAFETY; /*  Score bonus for safety from opp. king */
     /*  Totally passed pawn & safe from king */
    if (tpp) {
      sc+=KING_SAFETY_PP;
      sc+=rank_bonus[b];
       /* Give a bonus if this one could cause problems for opp. */
      if (cpp==0) sc+=DANGEROUS_PP_KPR;
    }
  }
   /*  King Tropism */
  if (side==WHITE) kt=-(abs(sq->kpos[0]-x)+abs(sq->kpos[1]-y));
  else kt=-(abs(sq->kpos[2]-x)+abs(sq->kpos[3]-y));
   /* Enemy King anti-tropism */
  if (side==WHITE) kt+=(abs(sq->kpos[2]-x)+abs(sq->kpos[3]-y));
  else kt+=(abs(sq->kpos[0]-x)+abs(sq->kpos[1]-y));
   /* Add on tropism score times 2 */
  sc+=kt*2;

  return sc;
}

 /* Evaluate rook for KPR endgame */
int tact_rook_kpr(const board *sq,int side,int x,int y) {
  int sc=ROOK_SCORE,b=y,a,dpp=0;

  if (side==BLACK) b=7-y;

   /* No pawn of yours on this file */
  if (sq->pawnfile[x][side-1]==0) {
      /*  Fully open file */
    if (sq->pawnfile[x][2-side]==0) sc+=OPEN_FILE_KPR;
      /*  Attacking an enemy pawn */
/*    else sc+=ATTACK_PAWN_KPR;*/
  }
   /* You have a pawn on this file */
  else {
     /*  Rook is just defending a pawn. */
    if (sq->pawnfile[x][2-side]==0) { /* No opp. pawns */
       /* Slight bonus for defending a pawn */
      sc+=DEF_PAWN_KPR;
       /* See if this defended pawn is passed.
        * Reward only if the rook is behind this pawn
        * (and therefore not blocking its progress) */
      if (x==0 || sq->pawnfile[x-1][2-side]==0) {
        if (x==7 || sq->pawnfile[x+1][2-side]==0) {
	  dpp=0;
	  if (side==WHITE) {
	    for (a=(y-1);a>0;a--) if (sq->sq[a][x]==1) {dpp=1;break;}
	  }
	  if (side==BLACK) {
	    for (a=(y+1);a<7;a++) if (sq->sq[a][x]==-1) {dpp=1;break;}
	  }
	  if (dpp==1) sc+=DEF_PP_KPR;
	}
      }
    }
    else sc -= DEFENDING_BLOCKED;
  }

  return sc;
}

 /* Evaluate rook for KPB endgame */
int tact_pawn_kpb(const board *sq,int side,int x,int y,const compdat *cd,int att) {
  int sc=PAWN_SCORE,b=y,pp=0,ks=0,cpp=0,passed=0,tpp=0,kt=0;

   /* No opponent bishops so use simpler analysis */
  if (sq->npieces[2-side][2]==0) return tact_pawn_kp(sq,side,x,y,att);

  assert(y>0 && y<7); /* Must be in a legal position! */

  if (side==BLACK) b=7-y;

   /*  Pawn advance bonus depending on rank */
  if (att<1) sc+=rank_bonus[b]; /*  Full value */

   /*  Penalise doubled (or tripled etc...) pawns on one file */
  if (sq->pawnfile[x][side-1]>1) sc-=DOUBLED_PAWNS_KPB;
   /*  Reward Passed Pawns */
  if (b<5) {
    pp=passed_pawn_kp(sq,x,y,side); /*  Returns 0 or PASSED_PAWN_(1-3) */
    passed=pp;
    if (passed) sc+=rank_bonus[b]; /*  Bigger pawn advance bonus if passed */
    if (pp==PASSED_PAWN_3) tpp=1; /*  Totally passed pawn */
    if (tpp) {
      if (att<0) pp+=pp; /*  Bonus doubled if passed pawn is defended */
      else if (att>0) sc-=PP_ATTACKED; /*  Penalty if attacked */
      sc+=rank_bonus[b];
      if (cd->gamestage<50) pp+=file_bonus[x]; /*  Bonus for passed pawns on the flanks */
      if (cd->gamestage<20) pp+=file_bonus[x]; /*  ... especially very near end of game */
    }
    sc+=pp;
    /*  Penalise backward and/or isolated pawns (unless passed) */
    if (!passed) sc+=backward_pawn(sq,x,y,side);  /*  Returns BACKWARD_PAWN_(1-5) */
  }

  /*  Reward if opp. king is not within pawn's promotion square */
  if (side==WHITE) { /*  White pawn */
    if (sq->kpos[3]<=y) { /*  If ahead of pawn */
      if (abs(sq->kpos[2]-x)>y) ks=1; /*  If outside promotion square then reward */
    }
     /*  If behind pawn then reward automatically */
    else ks=1;
     /*  If pawn is blockaded then penalise. */
    if (sq->sq[y-1][x]==-6) sc-=HOSTILE_BLOCKADE;
     /*  Connected passed pawns if no opp. bish/queen - large bonus */
    if (y<3 && tpp) {
      cpp=0;
      if (x>0 && sq->sq[y-1][x-1]==1) cpp=CONN_PP_DEF_KPB;
      else if (x<7 && sq->sq[y-1][x+1]==1) cpp=CONN_PP_DEF_KPB;
      else if (x<7 && sq->sq[y][x+1]==1) cpp=CONN_PP_KPB;
      if (cpp>0) {
          /* Reward CPPs on last rank */
        if (y==1 && cpp==CONN_PP_KPB) cpp+=CPP_7TH_RANK_KPB;
        if (abs(sq->kpos[2]-x)<2 && abs(sq->kpos[3]-y)<2) cpp=halved[cpp]; /* Penalise if opp. king is too near */
        sc+=cpp;
      }
    }
  }  
  else if (side==BLACK) { /*  Ditto for Black */
    if (sq->kpos[1]>=y) {
      if (abs(sq->kpos[0]-x)>b) ks=1;
    } 
    else ks=1;
     /*  Test for blockade */
    if (sq->sq[y-1][x]==6) sc-=HOSTILE_BLOCKADE;
     /*  Connected Passed Pawns if no opp. bishop/queen - large bonus */
    if (y>4 && tpp) {
      cpp=0;
      if (x>0 && sq->sq[y+1][x-1]==-1) cpp=CONN_PP_DEF_KPB;
      else if (x<7 && sq->sq[y+1][x+1]==-1) cpp=CONN_PP_DEF_KPB;
      else if (x<7 && sq->sq[y][x+1]==-1) cpp=CONN_PP_KPB;
      if (cpp>0) {
        if (y==6 && cpp==CONN_PP_KPB) cpp+=CPP_7TH_RANK_KPB;
        if (abs(sq->kpos[0]-x)<2 && abs(sq->kpos[1]-y)<2) cpp=halved[cpp]; /* Penalise if opp. king is too near */
	sc+=cpp;
      }
    }
  }
  if (ks) { /*  If safe from king */
    sc+=KING_SAFETY; /*  Score bonus for safety from opp. king */
    if (tpp) {
      sc+=KING_SAFETY_PP;  /*  Totally passed pawn & safe from king */
      sc+=rank_bonus[b];
       /* Give a bonus if this one could cause problems for opp. */
      if (cpp==0) sc+=DANGEROUS_PP_KPB;
    }
  }

   /*  King Tropism */
  if (side==WHITE) kt=-(abs(sq->kpos[0]-x)+abs(sq->kpos[1]-y));
  else kt=-(abs(sq->kpos[2]-x)+abs(sq->kpos[3]-y));
   /* Enemy King anti-tropism */
  if (side==WHITE) kt+=(abs(sq->kpos[2]-x)+abs(sq->kpos[3]-y));
  else kt+=(abs(sq->kpos[0]-x)+abs(sq->kpos[1]-y));
   /* Add on tropism score times 2 */
  sc+=kt*2;

  return sc;
}

 /* Evaluate Bishop for KPB Endgame.  Not much to do :) */
int tact_bishop_kpb(int x,int y,const compdat *cd) {
  int sc=BISHOP_SCORE;

   /*  Precalculated Position Score */
  sc+=centre_board[y][x]*2;
   /*  Worth more in open endgame */
  sc+=(55-cd->gamestage);
  return sc;
}

 /* Evaluate pawn for a KPN endgame */
int tact_pawn_kpn(const board *sq,int side,int x,int y,const compdat *cd,int att) {
  int sc=PAWN_SCORE,b=y,pp=0,ks=0,cpp=0,passed=0,tpp=0,kt=0;

   /* No opponent knights so use simpler analysis */
  if (sq->npieces[2-side][1]==0) return tact_pawn_kp(sq,side,x,y,att);

  assert(y>0 && y<7); /* Must be in a legal position! */

  if (side==BLACK) b=7-y;

   /*  Pawn advance bonus depending on rank */
  if (att<1) sc+=rank_bonus[b]; /*  Full value */

   /*  Penalise attached doubled (or tripled etc...) pawns on one file */
  if (side==WHITE && sq->sq[y+1][x]==1) sc-=DOUBLED_PAWNS_KPN;
  if (side==BLACK && sq->sq[y-1][x]==-1) sc-=DOUBLED_PAWNS_KPN;
   /*  Reward Passed Pawns */
  if (b<5) {
    pp=passed_pawn_kp(sq,x,y,side); /*  Returns 0 or PASSED_PAWN_(1-3) */
    passed=pp;
    if (passed) sc+=rank_bonus[b]; /*  Bigger pawn advance bonus if passed */
    if (pp==PASSED_PAWN_3) tpp=1; /*  Totally passed pawn */
    if (tpp) {
      if (att<0) pp+=pp; /*  Bonus doubled if passed pawn is defended */
      else if (att>0) sc-=PP_ATTACKED; /*  Penalty if attacked */
      sc+=rank_bonus[b];
      if (cd->gamestage<50) pp+=file_bonus[x]; /*  Bonus for passed pawns on the flanks */
      if (cd->gamestage<20) pp+=file_bonus[x]; /*  ... especially very near end of game */
    }
    sc+=pp;
    /*  Penalise backward and/or isolated pawns (unless passed) */
    if (!passed) sc+=backward_pawn(sq,x,y,side);  /*  Returns BACKWARD_PAWN_(1-5) */
  }

  /*  Reward if opp. king is not within pawn's promotion square */
  if (side==WHITE) { /*  White pawn */
    if (sq->kpos[3]<=y) { /*  If ahead of pawn */
      if (abs(sq->kpos[2]-x)>y) ks=1; /*  If outside promotion square then reward */
    }
     /*  If behind pawn then reward automatically */
    else ks=1;
     /*  If pawn is blockaded then penalise. */
    if (sq->sq[y-1][x]==-6) sc-=HOSTILE_BLOCKADE;
     /*  Connected passed pawns if no opp. bish/queen - large bonus */
    if (y<3 && tpp) {
      cpp=0;
      if (x>0 && sq->sq[y-1][x-1]==1) cpp=CONNECTED_PP_DEF;
      else if (x<7 && sq->sq[y-1][x+1]==1) cpp=CONNECTED_PP_DEF;
      else if (x<7 && sq->sq[y][x+1]==1) cpp=CONNECTED_PP;
      if (cpp>0) {
          /* Reward CPPs on last rank */
        if (y==1 && cpp==CONNECTED_PP) cpp+=CPP_7TH_RANK;
        if (abs(sq->kpos[2]-x)<2 && abs(sq->kpos[3]-y)<2) cpp=halved[cpp]; /* Penalise if opp. king is too near */
        sc+=cpp;
      }
    }
  }  
  else if (side==BLACK) { /*  Ditto for Black */
    if (sq->kpos[1]>=y) {
      if (abs(sq->kpos[0]-x)>b) ks=1;
    } 
    else ks=1;
     /*  Test for blockade */
    if (sq->sq[y-1][x]==6) sc-=HOSTILE_BLOCKADE;
     /*  Connected Passed Pawns if no opp. bishop/queen - large bonus */
    if (y>4 && tpp) {
      cpp=0;
      if (x>0 && sq->sq[y+1][x-1]==-1) cpp=CONNECTED_PP_DEF;
      else if (x<7 && sq->sq[y+1][x+1]==-1) cpp=CONNECTED_PP_DEF;
      else if (x<7 && sq->sq[y][x+1]==-1) cpp=CONNECTED_PP;
      if (cpp>0) {
        if (y==6 && cpp==CONNECTED_PP) cpp+=CPP_7TH_RANK;
        if (abs(sq->kpos[0]-x)<2 && abs(sq->kpos[1]-y)<2) cpp=halved[cpp]; /* Penalise if opp. king is too near */
	sc+=cpp;
      }
    }
  }
  if (ks) { /*  If safe from king */
    sc+=KING_SAFETY; /*  Score bonus for safety from opp. king */
    if (tpp) {
      sc+=KING_SAFETY_PP;  /*  Totally passed pawn & safe from king */
      sc+=rank_bonus[b];
       /* Give a bonus if this one could cause problems for opp. */
      if (cpp==0) sc+=DANGEROUS_PP_KPN;
    }
  }

   /*  King Tropism */
  if (side==WHITE) kt=-(abs(sq->kpos[0]-x)+abs(sq->kpos[1]-y));
  else kt=-(abs(sq->kpos[2]-x)+abs(sq->kpos[3]-y));
   /* Enemy King anti-tropism */
  if (side==WHITE) kt+=(abs(sq->kpos[2]-x)+abs(sq->kpos[3]-y));
  else kt+=(abs(sq->kpos[0]-x)+abs(sq->kpos[1]-y));
   /* Add on tropism score times 2 */
  sc+=kt*2;

  return sc;
}

 /* Evaluate Knight in a KPN endgame */
int tact_knight_kpn(const board *sq,int side,int x,int y,const compdat *cd) {
  int sc=KNIGHT_SCORE,b=y;

  if (side==BLACK) b=7-y;
  sc+=(cd->gamestage-50); /*  Worth less in open endgame */
  sc+=posdat[2][b][x]; /* Stay in the centre of the board */
   /*  Strong opponent king tropism */
  if (side==WHITE) sc += (5 - (abs(x-sq->kpos[2]) + abs(y-sq->kpos[3])) )*4;
  if (side==BLACK) sc += (5 - (abs(x-sq->kpos[0]) + abs(y-sq->kpos[1])) )*4;
  return sc;
}

 /*   ----------  ANALYSIS FUNCTIONS  ----------  */


/* Returns a score depending on how well defended or badly attacked 
  * your king is */
int king_defence(const board *sq,int kx,int ky,int side) {
  int x,y,p,xmin,xmax,ymin,ymax,def=0,nopatt=0,infront;

  xmin = max(kx-2,0);
  xmax = min(kx+2,7);
  ymin = max(ky-2,0);
  ymax = min(ky+2,7);

  for (y = ymin ; y <= ymax ; y++) {   /*--*/

    for (x = xmin ; x <= xmax ; x++) {   /*--*/

      if (y==ky && x==kx) continue;
      p=sq->sq[y][x];
      if (side==BLACK) p=-p;
       /* Test to see if pawn is infront.  Was gonna use side, not king y-pos
	* but then realised that was a stupid idea */
      if ((ky>3 && y<ky) || (ky<4 && y>ky)) infront=1;
      else infront=0;
      switch(p) {
        case (1) : if (infront==1) def++; 
	           if (y==ky && (ky>1 && ky<6)) def++; /* Flanking pawn */
	           break; /* Pawn near king */
        case (2) : def++; break; /* Nearby rook */
        case (3) : def++; break;  /* Nearby knight */
        case (4) : def++; break;  /* Nearby bishop */
        case (5) : def+=2; break;  /* Nearby queen */
        case (-1) : def--; break;  /* Nearby enemy rook */
        case (-2) : def-=2; break;  /* Nearby enemy rook */
        case (-3) : def-=2; break;  /* Nearby enemy knight */
        case (-4) : def-=2; break;  /* Nearby enemy knight */
        case (-5) : def-=4; break;  /* Nearby enemy queen */
        default: break; 
      }
       
    }
     
  }
   
   /* Penalty for being on an empty file. */
  if (sq->pawnfile[kx][side-1]==0) {
    def-=HALF_OPEN_FILE_K; /*  King on half-open file */
    if (sq->pawnfile[kx][2-side]==0) def-=OPEN_FILE_K; /*  ..or fully open file */
  }
   
   /* Pawn fortress */
  if (side==WHITE) {
    if (ky>2 && sq->sq[ky-1][kx]==1) def++;
    if (ky>2 && kx>0 && sq->sq[ky-1][kx-1]==1) def++;
    if (ky>2 && kx<7 && sq->sq[ky-1][kx+1]==1) def++;
  }
  else {
    if (ky<5 && sq->sq[ky+1][kx]==-1) def++;
    if (ky<5 && kx>0 && sq->sq[ky+1][kx-1]==-1) def++;
    if (ky<5 && kx<7 && sq->sq[ky+1][kx+1]==-1) def++;
  }

  nopatt=sq->npieces[2-side][0]+sq->npieces[2-side][3];
  switch (nopatt) {
    case (0) : break;
    case (1) : def *= 2; break;
    default  : def *= 3; break;
  }
   
   /* Return score */
  return def;
}

/* Calculate if a pawn is passed, i.e. unobstructed by enemy pawns. */
int passed_pawn(const board *sq,int x,int y,int side) {
  int score=2;
   
  if (side==WHITE) {
     /*  Check there isn't an opp. pawn in front */
    if (sq->pawnfile[x][1]>0 && first_pawn[x][1]<y) return 0;
     /*  Check to see if LH file contains blocker  */
    if (x>0 && sq->pawnfile[x-1][1]>0 && first_pawn[x-1][1]<y) score--;
     /*  Check to see if RH file contains blocker */
    if (x<7 && sq->pawnfile[x+1][1]>0 && first_pawn[x+1][1]<y) score--;
  }
  if (side==BLACK) { /*  See above */
     /*  Check there isn't an opp. pawn in front */
    if (sq->pawnfile[x][0]>0 && first_pawn[x][0]>y) return 0;
     /*  Check to see if LH file contains blocker  */
    if (x>0 && sq->pawnfile[x-1][0]>0 && first_pawn[x-1][0]>y) score--;
     /*  Check to see if RH file contains blocker */
    if (x<7 && sq->pawnfile[x+1][0]>0 && first_pawn[x+1][0]>y) score--;
  }
  if (score==2) return PASSED_PAWN_3; /*  Open file, no pawn blockers */
  if (score==1) return PASSED_PAWN_2; /*  Open file, one side defended by enemy pawns */
  return PASSED_PAWN_1; /*  Open file, both sides defended by enemy pawns */
}

/*  Calculate if a pawn is backward, */
/*   i.e. left behind the rest of the pawns and undefended by other friendly pawns */
int backward_pawn(const board *sq,int x,int y,int side) {
  int score=2;

  if (side==WHITE) {
    if (x>0 && sq->pawnfile[x-1][0]>0) { /*  If LH file contains friendly pawn */
      if (sq->sq[y+1][x-1]==1) return BACKWARD_PAWN_1; /*  Behind you */
      if (sq->sq[y][x-1]==1) return BACKWARD_PAWN_2; /*  Next to you :) */
      score--;
    }
    if (x<7 && sq->pawnfile[x+1][0]>0) { /*  Same for RH file */
      if (sq->sq[y+1][x+1]==1) return BACKWARD_PAWN_1;
      if (sq->sq[y][x+1]==1) return BACKWARD_PAWN_2;
      score--;
    }
  }
  if (side==BLACK) { /*  Ditto for black */
    if (x>0 && sq->pawnfile[x-1][1]>0) {
      if (sq->sq[y-1][x-1]==-1) return BACKWARD_PAWN_1;
      if (sq->sq[y][x-1]==-1) return BACKWARD_PAWN_2;
      score--;
    }
    if (x<7 && sq->pawnfile[x+1][1]>0) {
      if (sq->sq[y-1][x+1]==-1) return BACKWARD_PAWN_1;
      if (sq->sq[y][x+1]==-1) return BACKWARD_PAWN_2;
      score--;
    }
  }  
  if (score==2) return BACKWARD_PAWN_5; /*  Isolated - no pawns on either side */
  if (score==1) return BACKWARD_PAWN_4; /*  One pawn on one of the two sides */
  return BACKWARD_PAWN_3; /*  Backward, but both sides have pawns somewhere */
}

int drive_away(const board *sq,int x,int y,int side) {
  int sc=0;

  if (side==WHITE) {
    if (y<3) return 0;
    if (x>0 && sq->sq[y-1][x-1]==0) {
      if (sq->sq[y-2][x-1]==-1) sc+=DRIVE_AWAY_SHORT;
      else if (y==4 && sq->sq[1][x-1]==-1 && sq->sq[2][x-1]==0) sc+=DRIVE_AWAY_LONG;
    }
    if (x<7 && sq->sq[y-1][x+1]==0) {
      if (sq->sq[y-2][x+1]==-1) sc+=DRIVE_AWAY_SHORT;
      else if (y==4 && sq->sq[1][x+1]==-1 && sq->sq[2][x+1]==0) sc+=DRIVE_AWAY_LONG;
    }
  }
  else {
    if (y>4) return 0;
    if (x>0 && sq->sq[y+1][x-1]==0) {
      if (sq->sq[y+2][x-1]==1) sc+=DRIVE_AWAY_SHORT;
      else if (y==3 && sq->sq[6][x-1]==1 && sq->sq[5][x-1]==0) sc+=DRIVE_AWAY_LONG;
    }
    if (x<7 && sq->sq[y+1][x+1]==0) {
      if (sq->sq[y+2][x+1]==1) sc+=DRIVE_AWAY_SHORT;
      else if (y==3 && sq->sq[6][x+1]==1 && sq->sq[5][x+1]==0) sc+=DRIVE_AWAY_LONG;
    }
  }
  return sc;
}

/* Calculate if a pawn is passed, i.e. unobstructed by enemy pawns. */
int passed_pawn_kp(const board *sq,int x,int y,int side) {
  int score=2,n;
   
  if (side==WHITE) {
     /*  Check there isn't an opp. pawn in front */
    if (sq->pawnfile[x][1]>0) for (n=y-1;n>0;n--) if (sq->sq[n][x]==-1) return 0;
     /*  Check to see if LH file contains blocker  */
    if (x>0 && sq->pawnfile[x-1][1]>0) for (n=y-1;n>0;n--) if (sq->sq[n][x-1]==-1) score--;
     /*  Check to see if RH file contains blocker */
    if (x<7 && sq->pawnfile[x+1][1]>0) for (n=y-1;n>0;n--) if (sq->sq[n][x+1]==-1) score--;
  }
  if (side==BLACK) { /*  See above */
     /*  Check there isn't an opp. pawn in front */
    if (sq->pawnfile[x][0]>0) for (n=y+1;n<7;n++) if (sq->sq[n][x]==1) return 0;
     /*  Check to see if LH file contains blocker  */
    if (x>0 && sq->pawnfile[x-1][0]>0) for (n=y+1;n<7;n++) if (sq->sq[n][x-1]==1) score--;
     /*  Check to see if RH file contains blocker */
    if (x<7 && sq->pawnfile[x+1][0]>0) for (n=y+1;n<7;n++) if (sq->sq[n][x+1]==1) score--;
  }
  if (score==2) return PASSED_PAWN_3; /*  Open file, no pawn blockers */
  if (score==1) return PASSED_PAWN_2; /*  Open file, one side defended by enemy pawns */
  return PASSED_PAWN_1; /*  Open file, both sides defended by enemy pawns */
}
