#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <assert.h>

#include "chess.h"
#include "comp.h"
#include "checks.h"
#include "moves.h"
#include "misc.h"
#include "tactics.h"
#include "mprocs.h"
#include "eval.h"


/*       --------  SCORE WEIGHTINGS  ---------        */


#define INFINITY       (100000000) /* Initial value for beta & -alpha */
#define CMSCORE        (50000000)  /* Score for a CM */
#define PVBONUS        (10000000)  /* Score bonus for a move in the PV before preliminary move ordering */
#define HASHMOVE       (5000000)   /* Score bonus for a best move derived from the hash table */
#define CAPTURE_BONUS  (10000)     /* Initial score bonus for a capture move */
#define CHECK_BONUS    (1000)      /* Initial score bonus for a checking move */
#define CAPT_LAST      (0)         /* Initial score bonus for capturing last moved piece */
#define MOVE_KING      (0)         /* Ordering penalty for moving the king at any stage */

#define CMBOUNDS       (10000)     /* If move scores within this value of a CM then it _is_ a CM */

#define FUT_MARGIN     (1)         /* Margin for futility pruning at leaf nodes (pawn scores) */

#define KILLER1        (1000)      /* Score bonus for primary killer move */
#define KILLER2        (500)       /* Score bonus for secondary killer move */

#define BEST_CAPT      (1000)      /* Score bonus for best capture move */

#define MOST_NODES     (500)       /* Score bonus for highest node count */
#define SECOND_NODES   (300)       /* Score bonus for secondary node count */


/*       --------  HEURISTIC SWITCHES  ---------        */


 /*  Overall History heuristic ON/OFF */
#define historyon       (1)

 /*  History for each ply ON/OFF */
#define historyplyon    (0)

 /*  Determine how to increment the history counter.  If this is
  *    set to 1 then the current depth is added, otherwise the
  *    score is just incremented by 1.  Occasional speed benefits for
  *    worse tactical performance.  Dubious but sound. */
#define historyincr     (1)

 /* Principal Variation Search Heuristic ON/OFF */
#define pvsearch        (1)

 /*  Search extensions ON/OFF */
#define extensionson    (1)

 /*  Do weak futility pruning at frontier nodes. Very dodgy.
  *    Not theoretically sound either. Might miss stuff in
  *    the qsearch  */
#define futility_prune  (0)

 /*  Do move ordering based on node counts */
#define node_ordering   (0)

 /* Do search extensions based on king safety fluctuations */
#define ksextend        (0)

 /* Use the hashtable (0 = overrides and turns it off, 1 = as set by user) */
#define hashoverride    (1)

/*       --------  DISPLAY OPTIONS  ---------        */


 /*  Debugging info. */
#define debug
 /* Print progress dots to see how far the computer has thought */
#define progress_dots


/*       --------  LOSS AVOIDING TACTICS  --------       */

/*  When the best move so far appears to be losing material,
 *  extend the search time allowed based on the two values below.
 *  For every LOSS_SCALE centipawns that the move appears to
 *  lose then multiply the available search time by LOSS_INCR,
 *  interpolating logarithmically.
 *  (See below in the code to find out how this works in practice)  */

#define LOSS_INCR       (1.33)     /*  Base of search time extension factor  */
#define LOSS_SCALE      (40.0)     /*  Exponential scale of search time extension factor  */



/*       --------  MISCELLANEOUS DEFINES  --------       */


/*  This one rules :)  It returns 1 if a score is a CM for current side,
  *    -1 if it is CM for opposition, and 0 otherwise */
#define iscm(a)       ((((a) + (CMBOUNDS)) > (CMSCORE)) ? (1) : ((((a) - (CMBOUNDS)) < (-CMSCORE)) ? (-1) : (0)))



/*  -- GLOBAL VARIABLES --   (I know, I know...) */

extern int piece_value[7],piece_value_10[7],piece_value_100[7];
extern int depth_searched,initscore,xboardmode;
extern board defsq,defsqop;
extern compdat cdd[MAX_PV];
long int _nanaly,_nhash,_nlookup,_neval;
long int history[64][64],historyply[64][64][MAX_PV],nodecount[64][64][MAX_PV];
long int killerscore[MAX_PV],killerscore2[MAX_PV],second_best,hash_size=0,max_hash=0;
long int capturescore[MAX_PV];
int killer1[MAX_PV][2],killer2[MAX_PV][2],pvlength[MAX_PV],capturesq[MAX_PV];;
int rollover=0,prev_rollover=0,istest,npvold,isdraw[MAX_PV];
int hashtable=1,initlev,abort_flag,sbcm=0;  /* sbcm=1 if second best move at beg=2 is cm<0 */
int kswhite=0,ksblack=0;  /* Initial King safety scores */
double climit,time_spent,climit_base;
move pv[MAX_PV][MAX_PV],movestack[MAX_PV][256],pvold[MAX_PV],bestmove[MAX_PV],junk_move;
hash *hashbank,temphash;
hash_elt *table=NULL;
clock_t start;

/* Comp function which runs the iterative deepening process, sets up
 * all the variables beforehand and frees the necessary memory
 * after the search is complete.  Also handles a few bits of
 * thinking statistics. */
move *comp(board *sq,int mem,int post,int test,int cdepth) {
  int n=0,nmvs=0,depth,i,sameaslast=0;
  long int tnanal=0,tneval=0,nn,tmem;
  move *best=NULL,prevbest,*bestret;
  compdat *cd=&cdd[0];
  double rate=0.0,sc,ctlimit;

  _nanaly=_nlookup=_nhash=_neval=0;
  tmem=(1<<mem)*1024;
  hashtable=1;
  if (hashoverride==0 || mem==0) {tmem=0;hashtable=0;}
  if (!test && cd->tlimit<10) {tmem=0;hashtable=0;}
  if (hashtable) {
    hash_size=(tmem/(sizeof(hash_elt)*3));
    max_hash=(tmem-(hash_size*sizeof(hash_elt)));
    max_hash/=sizeof(hash);
    table=malloc(sizeof(hash_elt)*(size_t)hash_size);
    assert(table!=NULL);
    hashbank=malloc(sizeof(hash)*(size_t)max_hash);
    assert(hashbank!=NULL);
    for (nn=0;nn<hash_size;nn++) (table+nn)->h=NULL;
  }
  
   
   /* Reset all the memory properly */
  memset(sq->pawnfile, 0, sizeof(sq->pawnfile));
  for (n=0;n<8;n++) {
    for (i=0;i<8;i++) {
      if (sq->sq[n][i]==1) sq->pawnfile[i][0]++;
      if (sq->sq[n][i]==-1) sq->pawnfile[i][1]++;
    }
  }
  memset(history, 0, sizeof(history));
  memset(historyply, 0, sizeof(historyply));
  memset(nodecount, 0, sizeof(nodecount));
  memset(killer1, 0, sizeof(killer1));
  memset(killer2, 0, sizeof(killer2));
  memset(isdraw, 0, sizeof(isdraw));
  memset(capturesq, -1, sizeof(capturesq));
  memset(capturescore, -CMSCORE, sizeof(capturescore));
  for (i=0;i<MAX_PV;i++) {
    bestmove[i].fx=pvold[i].fx=-1;
    for (n=0;n<MAX_PV;n++) pv[i][n].fx=-1;
  }

   /* Setup the default junk move */

  junk_move.score=-INFINITY-1;
  junk_move.fx=junk_move.fy=-1;
  junk_move.tx=junk_move.ty=-1;
  junk_move.capt=junk_move.cm=junk_move.castle=0;
  junk_move.ep=junk_move.promote=junk_move.gchk=0;  /*  Set all to null values */
  junk_move.type=0;
   
   
  initialise(2);
  istest=test;
 
  sbcm=1; /* Check if second best move at beg=2 is a CM against */
  second_best=-INFINITY-1; /* And set the second best score initially to a low value */
   
   /*  Get initial move list (for total number of moves) */
  mvs(sq,cd->side,&nmvs,cd->cp[cd->side*2-2],cd->cp[cd->side*2-1],cd->last,1,0);
  if (xboardmode==0) {
    post=1;
    fprintf(stderr,"[%d Move",nmvs);
    if (nmvs>1) fprintf(stderr,"s");
    fprintf(stderr,"]  ");
  }
  memset(defsq.sq, 0, sizeof(defsq.sq));
  memset(defsqop.sq, 0, sizeof(defsqop.sq));

  cd->pts[0]=pts(1,sq);
  cd->pts[1]=pts(2,sq);
   /*  Get initial static score so that we can determine by how much
    *  this move improves it */
  initscore=defsc(cd->side,sq,cd);

  if (ksextend==1) {
    kswhite=king_defence(sq,sq->kpos[0],sq->kpos[1],WHITE);
    ksblack=king_defence(sq,sq->kpos[2],sq->kpos[3],BLACK);
  }
   
  if (xboardmode==0) fprintf(stderr,"\nCurrent Position=%.2f",((double)initscore)/(double)PAWN_SCORE);
  else fprintf(stderr,"Current Position=%.2f\n",((double)initscore)/(double)PAWN_SCORE);
   
  prevbest = junk_move;
   
   /* Set timer going.  I use clock() because I want CPU time not actual time.
    * CPU time is much more important because I occasionally want to run several
    * instances of ColChess simultaneously, perhaps running test suites.  */
  start=clock();
   /* Keep a count of how many times the clock() has overrun the signed long int bounds
    * since this search started.  Mainly only used in unfortunate circumstances when the
    * search started near to an overrun, or when we are doing a very long duration search. */
  rollover=0;
   /* If elapsed CPU time is less than zero (i.e. we have had a modulus error already)
    * then set up to _not_ add on a correction yet - wait until next loop round in this
    * search, should it occur.  Loop round occurs roughly every hour and a bit so this
    * should hopefully be very rare unless a long match is played, or many short matches
    * are played without restarting ColChess. */
  if ((double)start<0.0) prev_rollover=1;
  else prev_rollover=0;

  initlev=cdepth;
  depth=cdepth;
  ctlimit=(double)cd->tlimit;
  climit_base=ctlimit;
    /* Setup time to continue searching another ply if <=1/2 of max. time left.
     * If we're doing a test then we use all available time. */
  if (!test) ctlimit*=0.5;

   /* Setup counter to keep track of consequitive identical moves */
  sameaslast=0;
   
   /*  ---  START ITERATIVE DEEPENING LOOP  --- */
   
  for (n=0;n<depth;n++) {
    
     /* These scores need to be reset each iteration */
    memset(killerscore, -CMSCORE, sizeof(killerscore));
    memset(killerscore2, -CMSCORE, sizeof(killerscore2));
    
    if (xboardmode==0) fprintf(stderr,"\n{L%d}  ",n+2);
     
    /* climit_base is the base time limit for the computer which needs to be
     * reset here as it gets altered if the previous ply was bad.  Actually, 
     * it's probably safe not to reset it, but what the heck :) */
    climit=climit_base;
    cd->followpv=1;
    cd->extend=0;
    abort_flag=0;
    _nanaly=_neval=0;
    i=0;

    compiter(sq,2,n+1);       /*    <---- Actually do the search here   ***    */
     
    best=&bestmove[0];
    if (compare(best,&prevbest)) sameaslast++;
    else sameaslast=0;
    
    if (best->fx>=0) {
      prevbest=*best;
      npvold=pvlength[0];
      for (i=0;i<npvold;i++) pvold[i]=pv[0][i];
    }
     /*  Move aborted on first move at top ply.  Replace move with previous best. */
    else *best=prevbest;
     
#ifdef progress_dots
    if (xboardmode==0) fprintf(stderr,"\b ");
#endif
     
    if (xboardmode==0 && best->fx>=0) printmove(best);
    sc=(double)(best->score-initscore);  /*  Use score improvement not absolute score */
    sc/=(double)PAWN_SCORE; /*  Normalise to pawn score */
     
    if (!best->cm) { /*  If not CM */
      if (xboardmode==0 && isdraw[0]==0) {
	  /*  Theoretically won position */
	if (best->score<-THEORETICAL_WIN) fprintf(stderr," (Lost)");
	else if (best->score>THEORETICAL_WIN) fprintf(stderr," (Won) ");
	  /*  Print normal score */
	else fprintf(stderr," (%.2f)",sc);
      }
      else if (xboardmode==0) {fprintf(stderr," (Draw)");best->score=0;}
      best->score-=initscore; /* Modify score to relative value */
      second_best-=initscore; /* Modify second best score */
      if (xboardmode==0) {
        if (best->score>=0) fprintf(stderr," "); /*  Presentation */
        if (best->score>-1000 && best->score<1000) fprintf(stderr," ");
      }
    }
     /* Otherwise, print out that this move was a CM/Loss */
    else if (xboardmode==0 && best->cm>0) fprintf(stderr," (Mate %ld ply)  ",CMSCORE-best->score);
    else if (xboardmode==0) fprintf(stderr," (Mated %ld ply) ",best->score+CMSCORE);

#ifdef debug
    if (xboardmode==0) fprintf(stderr,"  Nodes=%ld  Eval=%ld",_nanaly,_neval);
#endif

     /* A few debugging things to keep track of node counts. */
    tnanal+=_nanaly;
    tneval+=_neval;

    rate=get_time(start);
    if (xboardmode==0) {
      if (rate<1000.0) fprintf(stderr,"  [%.2fs]",rate/(double)10);
      else if (rate<10000.0) fprintf(stderr,"  [%.1fs]",rate/(double)10);
      else fprintf(stderr,"  [%.0fs]",rate/(double)10);
    }
    else if (post) { /*  Post thinking to xboard :) */
      fprintf(stderr,"%d %ld %d %ld ",n+2,best->score,(int)(rate*100),tnanal);
      for (i=0;i<pvlength[0];i++) {  /*  Print off formatted PV */
        if (pv[0][i].type==-1) fprintf(stderr,"<HT>");
        else printmove2(&(pv[0][i]));
        fprintf(stderr," ");
        if ((i&1)==1) fprintf(stderr," ");
      }
      fprintf(stderr,"\n");
    }

     /*  Immediate Draw/Stalemate */
    if (best->fx==-1) break;
     /*  If only 1 move possible or CM found or only one move which does not lose */
    if ((!test && nmvs==1) || best->cm || sbcm==1) break;
     /*  Check it isn't stuck in a simple straight draw loop */
    if (n>3 && (_nanaly/n) < 10) break;
     /*  Seemingly good move found & has been best move for at least 4
      *  iterations & used at least 30% of max. time */
    if (!test && sameaslast>2 && best->score-second_best>80 && rate>(ctlimit*0.6)) break;
     /* If winning move found & sufficient depth & used 40% of time then play it */
    if (!test && n>4 && (best->score+initscore)>THEORETICAL_WIN && (n+1)>=initlev && rate>(ctlimit*0.8)) break;
     /*  Is it worth searching another ply given the time limit? */
    if (n==depth-1 && rate<ctlimit) depth++;
  }

  if (isdraw[0]==1) {best->score=initscore=0;}
  depth_searched=n;
  /*  Print out info :) */
  if (xboardmode==0) {
    fprintf(stderr,"\n");
    print_time(rate);
    time_spent=rate;
    if (cd->tlimit>0 && rate>(double)cd->tlimit) fprintf(stderr," - D'oh");
#ifdef debug /*  Print off debugging stats */
    if (xboardmode==0) {
      fprintf(stderr,"\nNodes=%ld  Eval=%ld\n",tnanal,tneval);
      _nanaly=tnanal;
      if (hashtable) fprintf(stderr,"Table Hits=%ld  [%ld Entr",_nlookup,_nhash);
      if (hashtable && _nhash!=1) fprintf(stderr,"ies]\n");
      else if (hashtable && _nhash==1) fprintf(stderr,"y]\n");
    }
#endif
  }
   
   /* Print off PV to screen */
  for (i=0;i<=npvold;i++) pv[0][i]=pvold[i];
  pvlength[0]=npvold;
  if (xboardmode==0) print_pv(0);

  if (hashtable) {
    free(hashbank);
    hashbank=NULL;
    free(table);
    table=NULL;
  }
  bestret=malloc(sizeof(move));
  assert(bestret!=NULL);
  *bestret=bestmove[0];

  return bestret;
}


 /*  -----=====#####  THE RECURSIVE SEARCH FUNCTION  #####=====-----   */


/* This function recursively searches a given node in the full width search
 * phase.  'sq' is the board, 'beg' is a number indicating the ply, starting
 * at 2 for the top ply and decreasing (this is a fossil of the old version
 * of ColChessand I never got round to changing it!). 'lev' is the decreasing
 * depth counter indicating when to start the quiescence search (i.e. at zero). */

void compiter(board *sq,int beg,int lev) {
  int nm,a,cpk,cpq,pc=0,inch=0,hvars=0,epx,epy,aa;
  int flg=0,nbmv=-1,drawn_game=0,abcutoff=0,depth=2-beg;
  long int hcheck=0,hcheckp=0,alpha=0,beta=0,nodes=0;
  int hcheck2=0;
  move *mv,*best,*temp=NULL,movetemp;
  compdat *cd=&cdd[2-beg],*cd2=&cdd[3-beg];
  hash *h=NULL;
  double rate=0.0,search_incr=0.0;

   /* Set up PV length */
  pvlength[depth]=depth;
   
   /*  Setup the computer data structure ready to pass on to the next depth */
  setup_compdat(beg);
   
   /* Setup the return data */
  isdraw[depth]=0;
  bestmove[depth]=junk_move;
  best=&bestmove[depth];
   
   /*  Setup castling permissions */
  if (cd->side==WHITE) {cpk=cd->cp[0];cpq=cd->cp[1];}
  else {cpk=cd->cp[2];cpq=cd->cp[3];}
  nm=0;

   /*  Check to see if this position is drawn */
  drawn_game=is_drawn_game(sq,cd->gamestage);
   /*  If not then get list of legal moves. */
  if (!drawn_game) mvs(sq,cd->side,&nm,cpk,cpq,cd->last,cd->inchk,depth);

   /*  Drawn Game - return draw score */
  if (nm==0) {     
#ifdef progress_dots
    if (beg==1 && xboardmode==0) fprintf(stderr," ");  /*  Sort out presentation :) */
#endif   
    isdraw[depth]=1;
    best->score=0;
    return;
  }
   
   
  /*  --  DO SIMPLE ANALYSIS ON ALL MOVES  --  */

   /* If this move in the PV was a hash move before then stop following the PV */
  if (pv[depth][depth].type==-1) cd->followpv=0;
   /* Naively analyse all moves */
  nbmv=analyse_all(beg,depth,nm,cd,sq,0);
  assert(nbmv>=0);

   /*  CM already? */
  if (movestack[depth][nbmv].cm) {
    *best=movestack[depth][nbmv];
    best->score=CMSCORE-1;
#ifdef progress_dots
    if (beg==1 && xboardmode==0) fprintf(stderr," ");  /*  Sort out presentation :) */
#endif
    pv[depth][depth]=*best;
    return;
  }

   /*  Put best move at head of list */
  if (nbmv>0) {
    movetemp=movestack[depth][nbmv];
    movestack[depth][nbmv]=movestack[depth][0];
    movestack[depth][0]=movetemp;
  }
   

   /* Setup the cutoff bounds */
  beta=cd->beta;
  alpha=cd->alpha;
   
  /*  --  Test for Extensions  --  */
  
  if (extensionson && cd->extend<2) {
     /*  If this is the PV terminal node then extend more!! */
    if (cd->followpv && lev<1) cd->extend=3;
     /*  Always do at least depth 1 qui-search if in check at any point */
    else if (cd->quiesce==0 && cd->extend==0 && cd->inchk) cd->extend=1;
     /*  Only one legal reply (max. 2 extensions) */
    else if (nm==1) cd->extend++;
  }

     
   /*  ----==   RECURSIVELY ANALYSE MOVES TO REQUIRED DEPTH   ==---- */

  for (a=0;a<nm;a++) { 
     /*  If first move then use the best move already calculated & sorted */
    if (a==0) mv=&movestack[depth][0];
    else {
       /*  Put 'best of the rest' in next place */
      sort_from(nm,a,depth);
      mv=&movestack[depth][a];
    }
    if (mv->cm) {fprintf(stderr,"ERROR IN COMP : CM=%d\n",mv->cm);exit(0);}
    mv->score=0; /*  Reset score - preliminary ordering score is irrelevant */
    flg=0; /*  Indicates if this score has been taken from the hashtable (1/0) */
     /* Check for illegal moves */
    assert(mv->fx>=0);

#ifdef progress_dots  /*  Display the progress data dots etc... */
    if (xboardmode==0) progress(beg,a,nm);
#endif

     /*  pc stores identifier of piece being moved., or zero if move is a castle */
    if (mv->castle==0) pc=sq->sq[mv->fy][mv->fx];
    else pc=0;
     /*  Perform move 'mv' on board 'sq'. */
    do_move2(sq,mv,cd->side);
     /*  Setup computer data ready for next iteration. */
    setup_cd2(mv,pc,beg);
     /*  If there was a simple check then no need to do a full test */
    if (mv->gchk!=0) { /*  If move gave direct check in preliminary scoring */
      inch=pc; /*  inch stores identifier of checking piece */
      if (mv->promote!=0) inch=mv->promote;  /*  ...or piece a pawn has just promoted to, if that gives check */
    }
     /* Otherwise, make sure there isn't a revealed check. */
    else if (mv->ep==0 && mv->castle==0) inch=revchck(sq,3-cd->side,mv->fx,mv->fy,mv->tx,mv->ty);
     /* Be a bit more careful if the move was a castle or an en-passant capture */
    else inch=chck(sq,3-cd->side);

     /*  Extend if this is a check evasion by a king move */
    if (extensionson && cd->inchk && abs(pc)==6 && cd->extend<2) cd->extend++;
     
    if (inch!=0) { /*  If this move gave check (of any sort) */
      cd2->inchk=1;  /*  Computer will be in check for next time.  Store this as a flag. */
       /*  Test to see if it's actually CM, but no need if the move has been tested before */
      if (mv->gchk==0) {
        /* Set up en-passant square */
	if (abs(pc)==1 && abs(mv->fy-mv->ty)==2) {epx=mv->fx;epy=(mv->ty+mv->fy)/2;}
	else {epx=epy=-1;}
	if (chkmte(cd2->side,sq,inch,epx,epy)==1) {(mv->score)=CMSCORE-1;mv->cm=1;}
      }
      mv->gchk=1;
    }

     /* --- DO NASTY RECURSIVE BIT --- */
     
    if (!mv->cm) {  /*  Move is not a checkmate (not worth searching if it is!) */
      cd2->last=mv;
      if (pvsearch && a>0) cd2->alpha=-alpha-1; /*  Zero window search bounds */
      else cd2->alpha=-beta;
      cd2->beta=-alpha;

         /* ---- PROBE HASHTABLE FIRST ---- */
       
      if (hashtable) { /*  Check to see if this position has been stored in hash table */
         /*  Hash current position */
	make_hash(&temphash,sq,cd2->cp,cd2->side);
        hcheck=temphash.check;hvars=temphash.vars;hcheckp=hcheck%hash_size;hcheck2=temphash.check2;
	 /*  Update information for the draw-by-repetition check */
	if (abs(pc)!=1 && mv->capt==0 && mv->castle==0 && isinposlist(cd->twice[cd2->side-1],hcheck,hcheck2)) {mv->score=0;flg=1;}
	else update_poslist(cd2,cd2->side,hcheck,hcheck2);
          /*  See if current position has been stored before. */
	h=NULL;
        h=find_hash((table+hcheckp),hcheck,hvars,hcheck2);	
        if (h && flg==0) {
	    /*  Current position has been found in the hashtable :)) */
          _nlookup++; /*  Increment hash lookup counts */
          cd2->smove=h->move;cd2->svars=h->mvars;
	  flg=0;
            /*  If this hash value is deeper than current search or it returned exact CM. */
	  if ((int)h->depth>=lev || (h->type==exact && h->cm!=0)) {
              /*  If previously searched exactly before then use stored value instead */
            if (h->type==exact) {mv->score=h->score;mv->cm=h->cm;flg=1;}
              /*  Otherwise if lower_bound>=beta then don't bother searching any more */
            else if (h->type==lower_bound && h->score>=beta) {mv->score=h->score;flg=1;}
              /*  Otherwise if upper_bound<=best_score then don't bother searching any more */
            else if (h->type==upper_bound && h->score<=alpha) {mv->score=h->score;flg=1;}
              /*  Flag that this move's score was recovered from the hash table. */
            if (flg==1 && mv->score>best->score) mv->type=-1;
	  }
        }
      }
      /*  If no cutoff then search properly */
      if (!hashtable || flg==0) {

         /* --- QUIESCENCE SEARCH AT ROOT NODES --- */
	 
	if (lev<1) {
	    /* A bit of experimental code to extend if the king safety has
	     * changed appreciably for either side.  Probably a bit pants.  */
           if (ksextend==1) {
	     if (kswhite - king_defence(sq,sq->kpos[0],sq->kpos[1],WHITE) > 10) cd->extend++;
	     if (ksblack - king_defence(sq,sq->kpos[2],sq->kpos[3],BLACK) > 10) cd->extend++;
	   }
	    /*  If we're not in check and this move neither checks nor promotes
	     *  a pawn and the (theoretically unsound) futiltity condition holds
	     *  then do a narrow quiescence search. Pretty drastic, huh? */
          if (futility_prune && cd->inchk==0 && mv->gchk==0 && mv->promote==0 && cd->capt+piece_value[(mv->capt)]+FUT_MARGIN<0) {
	     mv->score=-(quiesce(sq,0,0,beg-1,lev-1));
	  }
	    /*  Otherwise do a full & proper quiescence search including accumulated
	     *  search extensions  */
          else mv->score=-(quiesce(sq,cd->quiesce+cd->extend,0,beg-1,lev-1));
	  mv->cm=iscm(mv->score); /* See if the returned move score signals a CM */
	  temp=NULL;
        }
	 
	 /* --- MINIMAL WINDOW SEARCH --- */
	 
        else {
	  nodes=_nanaly;
	  compiter(sq,beg-1,lev-1); /*  Recurse tree */
	  temp=&bestmove[depth+1];  /*  Get the move */
	   
          mv->score=-(temp->score);  /*  Update move score */
 
           /*   -- MORE ACCURATE SEARCH RECURSION -- */

	  if (!abort_flag && pvsearch && a>0 && mv->score>alpha && mv->score<beta) {
            cd2->alpha=-beta; /*  Reset alpha and beta */
	    cd2->beta=-alpha;
#ifdef progress_dots
	    if (xboardmode==0 && beg==2) fprintf(stderr,"\b"); /*  Formatting */
#endif
	    compiter(sq,beg-1,lev-1); /*  Search again */

	    temp=&bestmove[depth+1];  /*  Get the move	     */
            mv->score=-(temp->score);  /*  Update move score */
	    if (abort_flag==1) { /* Move aborted but we know it was good */
	      mv->score=alpha+1;
	      *best=*mv;
              pv[depth][depth]=*mv;
              for (aa=depth+1;aa<pvlength[depth+1];aa++) pv[depth][aa]=pv[depth+1][aa];
              pvlength[depth]=pvlength[depth+1];
	    }
  	  }
	   
	   /* Work out node count for this move */
	  nodes=_nanaly-nodes;
           /*  Set up CM flag */
	  mv->cm=-temp->cm;
        }
	 /* Check for CM score which needs decreasing/increasing to track ply */
        if (mv->cm==1) (mv->score)--;
        if (mv->cm==-1) (mv->score)++;

	 /* Check for drawn position */
	if (mv->score>best->score) {
	  if (isdraw[depth+1]==1) drawn_game=1;
	  else drawn_game=0;
	}
	 	 
     	 /* --- UPDATE HASH TABLE WITH NEW DATA --- */

	if (mv->score<=alpha) abcutoff=1;
	else if (mv->score>=beta) abcutoff=-1;
	else abcutoff=0;
	if (!abort_flag && hashtable && lev>0) {
          if (!h && _nhash<max_hash) { /*  Store position if enough space left. */
	    add_hash((table+hcheckp),temp,hcheck,hvars,hcheck2,lev,mv->score,mv->cm,abcutoff);
            _nhash++; /*  Increment counter. */
          }
           /* Update hash entry if better new information is found. */
          else if (h && (lev>h->depth || (lev==h->depth && abcutoff==0 && h->type!=exact))) alter_hash(h,mv->score,mv->cm,temp,abcutoff,lev);
        }

      }
#ifdef progress_dots
      else if (xboardmode==0 && beg==2) fprintf(stderr," ");
#endif
    }
     
    undo_move(sq,mv,cd->side);  /*  Put board back !! */

     /* Store node count for this move */
    if (node_ordering==1) nodecount[(mv->fy*8)+mv->fx][(mv->ty)*8+mv->tx][depth]=nodes;
     
     /*  This move beats the previous best move!! */
    if (!abort_flag && mv->score>best->score) { 
      if (beg==2) {second_best=best->score;if (a>0 && best->cm>=0) sbcm=0;}
      *best=*mv;
      if (flg==1) best->type=-1;
      if (mv->score>alpha) {
        increase_counts(mv,depth,0,lev);
	alpha=mv->score;
	if (alpha<beta) { /* Update PV */
          pv[depth][depth]=*mv;
          for (aa=depth+1;aa<pvlength[depth+1];aa++) pv[depth][aa]=pv[depth+1][aa];
          pvlength[depth]=pvlength[depth+1];
	}
      }

    }
     /* Update second best move score if necessary */
    else if (!abort_flag && beg==2 && mv->score>second_best) {second_best=mv->score;if (a>0 && best->cm>=0) sbcm=0;}

     /*  'Fail-High' cutoff if alpha>=beta - no more searching needed */
    if (alpha>=beta && !abort_flag) {
      free_compdat(beg-1);
      if (drawn_game) isdraw[depth]=1;
       /* Update capture square information if this move scores highly */
      if (sq->sq[mv->ty][mv->tx]!=0 && mv->score > capturescore[depth]) {
	capturesq[depth] = (mv->fy*8) + mv->fx;
      }
      return;
    }
     
     /*  This move is a CM or won move, but didn't cause a beta cutoff.
      *  Make sure that, if you are at the first iterative deepening
      *  level then you search all the moves to make sure that you
      *  don't mess around aimlessly. */
    if ((beg<2 || lev!=1) && mv->cm>0 && !abort_flag) {
      free_compdat(beg-1);
      if (drawn_game) isdraw[depth]=1;
      return;
    }

     /* If game has a time limit and we are near top ply,
      * search is deep enough and search is not near the end */
    if (abort_flag || (cd->tlimit>0 && (beg>=0 || lev>5) && (depth+lev)>initlev)) {
      rate=get_time(start);
       /* If this is the top level and not a test */
      if (beg==2 && istest==0) {
         /* --- LOSS AVOIDING TACTICS ---- */
	 /* Increase the time allowed if current best move loses,
	  * i.e. give more time to think in difficult positions.
	  * This is slow, but is only done at beg=2, i.e. top level.
	  * Note that best->score - initscore is current best move's
	  * positional _improvement_ score in centipawns. */
	if (best->score<initscore) { /* (Best move so far is bad...) */
	  search_incr=(double)(min(initscore-best->score,(LOSS_SCALE*6)));
	  if (best->score>0) search_incr/=2;
	  else if (best->score>-50) search_incr*=2/3;
	  climit=climit_base*pow((double)LOSS_INCR,search_incr/(double)LOSS_SCALE);
	}
	else climit=climit_base;
      }
       /*  climit holds the maximum duration for which computer should think.
        *  Test if ColChess has thought for this time limit... */
      if ((rate>climit) || abort_flag) {
        free_compdat(beg-1);
         /*  Set a flag to show that all levels of computation should be aborted */
	abort_flag=1;
	if (xboardmode==0 && beg==2) fprintf(stderr," (ABORT) ");
        return;
      }
    }
  } /* Loop end */

  /*  --  BEST<BETA AND NO CM -- */

  free_compdat(beg-1);
  if (drawn_game) isdraw[depth]=1;
}


/*  Perform quiescence search on terminal nodes.
 *  Search down to a specified depth with checks,
 *  check evasions, pawn pushes and captures
 *  Search to infinite depth on captures only
 *  This is quite slow, but maintains accuracy 
 *  The 'force' parameter forces a full width search from this node. */
long int quiesce(board *sq,int qdepth,int force,int beg, int lev) {
  int nm,a,pc=0,inch=0,epx,epy,aa;
  int nbmv=-1,depth=2-beg;
  long int alpha=0,beta=0,bsc=0;
  move *mv=NULL,movetemp;
  compdat *cd=&cdd[2-beg],*cd2=&cdd[3-beg];

   /* --  SETUP VARIABLES  -- */

  pvlength[depth]=depth;
  nm=0;
  isdraw[depth]=0;
   
   /*  Check to see if this position is drawn */
  if (is_drawn_game(sq,cd->gamestage)) { isdraw[depth]=1; return 0; }
    /*  If past quiescence depth then get list of capture moves (faster than mvs2) */
  if (!force && qdepth<1) gencapt(sq,cd->side,&nm,cd->last,cd->inchk,depth);
    /*  Else get list of _all_ non-quiescent moves */
    /*  Firstly, if side to move is in check then get all moves */
  else if (force || cd->inchk) mvs(sq,cd->side,&nm,0,0,cd->last,1,depth);
    /*  Otherwise just get captures, direct checks and pawn pushes */
    /*  (Note that discovered checks are ignored here, for speed) */
  else mvs2(sq,cd->side,&nm,cd->last,depth);

    /*  Always give the option of 'no move', i.e. accepting the current position */
    /*  unless side to move is in check & we are still in wide quiescence search */
  if (!force && (qdepth<1 || cd->inchk==0)) {
    bsc=defsc(cd->side,sq,cd);
/*  
    if (bsc+defsc(3-cd->side,sq,cd)!=0) {fprintf(stderr,"Score asymmetry!\n");print_board(sq);}
*/
  }
  else {bsc=-INFINITY;if (nm==1) qdepth++;}

  if (nm==0) {  /*  No legal moves - return static score */
    if (bsc>-INFINITY) return bsc;
    else return 0; /* Stalemate */
  }

   /*  Current position good enough for cutoff.
    *  Known as a 'stand pat cutoff' */
  if (bsc>cd->beta) {
    return bsc;
  }

  /*  Setup computer data structure ready for next depth */
  setup_compdat(beg);

  /*  --  DO SIMPLE ANALYSIS ON ALL MOVES  --  */

   /* Naively analyse all moves */
  nbmv=analyse_all(beg,depth,nm,cd,sq,1);
  assert(nbmv>=0);

   /*  CM already? */
  if (movestack[depth][nbmv].cm) {
    free_compdat(beg-1);
    movestack[depth][nbmv].score=CMSCORE-1;
    if (beta>=CMSCORE) pv[depth][depth]=movestack[depth][nbmv];
    return (CMSCORE-1);
  }
 
   /* Put the best move at the front */
  if (nbmv>0) {
    movetemp=movestack[depth][nbmv];
    movestack[depth][nbmv]=movestack[depth][0];
    movestack[depth][0]=movetemp;
  }

  /*  --  RECURSIVELY ANALYSE MOVES TO REQUIRED DEPTH  -- */

  beta=cd->beta;
  alpha=max(cd->alpha,bsc);

  for (a=0;a<nm;a++) {    /*     MAIN LOOP BEGINS HERE */
     /*  If first move then use the best move already calculated & sorted */
    if (a==0) mv=&movestack[depth][0];
    else {
       /*  Get best of the rest :) */
      sort_from(nm,a,depth);
      mv=&movestack[depth][a];
    }
     
     /*  This move is screwed */
    if (mv->fx==-1) continue;

    pc=sq->sq[mv->fy][mv->fx]; /*  pc stores identifier of piece being moved. */
if (pc==1 && mv->ty==0 && mv->promote==0) {fprintf(stderr,"!W!");printmove(mv);}
else if (pc==-1 && mv->ty==7 && mv->promote==0) {fprintf(stderr,"!B!");printmove(mv);}
if (abs(sq->sq[mv->ty][mv->tx])==6) {fprintf(stderr,"Oh Dear! (P=%d)",pc);printmove(mv);print_board(sq);exit(0);}

    do_move2(sq,mv,cd->side); /*  Do this move */
    setup_cd2(mv,pc,beg);  /*  Setup cd2 for next iteration. */

    if (mv->gchk) {
      inch=pc;
      if (mv->promote) inch=mv->promote;
    }
     /*  Test for revealed check */
    else if (mv->ep==0 && mv->castle==0) inch=revchck(sq,3-cd->side,mv->fx,mv->fy,mv->tx,mv->ty);
     /* Be a bit more careful if the move was a castle or an en-passant capture */
    else inch=chck(sq,3-cd->side); 

    if (inch!=0) {
      cd2->inchk=1;  /*  Computer will be in check for next time. */
      if (mv->gchk==0) {
        /* Set up en-passant square */
        if (abs(pc)==1 && abs(mv->fy-mv->ty)==2) {epx=mv->fx;epy=(mv->ty+mv->fy)/2;}
        else {epx=epy=-1;}
        if (chkmte(cd2->side,sq,inch,epx,epy)==1) {(mv->score)=CMSCORE-1;mv->cm=1;}
      }
    }
    if (!mv->cm) {
      cd2->last=mv;
      cd2->beta=-alpha;
      cd2->alpha=-beta;
      cd2->smove=-1;
      mv->score=-(quiesce(sq,qdepth-1,0,beg-1,lev-1));
      mv->cm=iscm(mv->score); /* See if this move is a CM */
      if (mv->cm==1) (mv->score)--;
      if (mv->cm==-1) (mv->score)++;
    }

    undo_move(sq,mv,cd->side);  /*  Put board back !! */

    if (mv->score>bsc) {  /*  Update best score */
      if (mv->score>alpha) {
	alpha=mv->score;
        increase_counts(mv,depth,1,lev);
        if (alpha<beta) {
	  pv[depth][depth]=*mv;
          for (aa=depth+1;aa<pvlength[depth+1];aa++) pv[depth][aa]=pv[depth+1][aa];
          pvlength[depth]=pvlength[depth+1];
	}
      }
      bsc=mv->score;
    }
     /*  Cutoff if alpha>=beta or this move is a cm - no more searching needed */
    if (alpha>=beta || mv->cm>0) {
      free_compdat(beg-1);
      return bsc;
    }
  }

  /*  --  FINISH OFF --  */

  free_compdat(beg-1);
  return bsc; 
}

/*       --------------  SIMPLE ANALYSIS FUNCTION  --------------  */

/* Perform preliminary move analysis for sorting purposes */
void analyse(board *sq,move *mv,compdat *cd) {
  int p=abs(sq->sq[mv->fy][mv->fx]),epx,epy;

  _nanaly++;
  do_move2(sq,mv,cd->side); /*  Do the move */
   /*  Score highly if capture -
    *  Give bonus of 100 * Captured piece value*/
  mv->score += piece_value_100[abs(mv->capt)];
   /*  Subtract 10 * this piece value if capturing -
    *  This encourage exchanges starting with weakest piece  */
  if (mv->capt) mv->score -= piece_value_10[p];
   /*  Reward capture of last moved piece */
  if (cd->last && (cd->last)->castle==0 && mv->tx==(cd->last)->tx && mv->ty==(cd->last)->ty) {
    (mv->score)+=CAPT_LAST;
  }
   /* Test to see if move gives direct check(mate) */
  if (mv->castle==0 && give_direct_check(sq,mv->tx,mv->ty)==1) {
    mv->gchk=1;
    /* Set up en-passant square */
    if (p==1 && abs(mv->fy-mv->ty)==2) {epx=mv->fx;epy=(mv->ty+mv->fy)/2;}
    else {epx=epy=-1;}
    if (chkmte(3-cd->side,sq,sq->sq[mv->ty][mv->tx],epx,epy)) {mv->cm=1;mv->score=CMSCORE-1;}
  }
  undo_move(sq,mv,cd->side);  /*  Put board back !! */
   /*  Bonus for giving check */
  if (mv->gchk) mv->score+=CHECK_BONUS;
   /*  Bonus for capture of any sort */
  if (mv->capt) mv->score+=CAPTURE_BONUS;
   /*  Bonus for promotion moves = new piece value * 10 */
  mv->score+=(piece_value_10[abs(mv->promote)]);
}



/*      ------------ MISC. COMP FUNCTIONS ------------      */

/* Grrr... untidy and slow... */
void setup_cd2(move *mv,int pc,int beg) {
  int b=0,v=0,depth=2-beg;
  compdat *cd=&cdd[2-beg],*cd2=&cdd[3-beg];
  
  v=piece_value[abs(mv->capt)];
  _neval++;
  cd2->inchk=0;
  cd2->gamestage=cd->gamestage-v;
  cd2->pts[cd->side-1]=cd->pts[cd->side-1];
  cd2->pts[2-cd->side]=cd->pts[2-cd->side]-v;
  cd2->capt+=v;
  cd2->smove=-1;
  cd2->svars=-1;
  cd2->extend=cd->extend;
  if (mv && mv->promote!=0) {
    cd2->gamestage+=piece_value[abs(mv->promote)]-1;
    cd2->pts[cd->side-1]+=piece_value[abs(mv->promote)]-1;
  }
   /* Check to see if following PV */
  if (cd->followpv==1) {
    if (compare(&(pv[0][depth]),mv)==1) cd2->followpv=1;
    else cd2->followpv=0;   
  }
  else cd2->followpv=0;

    /*  Setup castle flags */
  for (b=0;b<4;b++) cd2->cp[b]=cd->cp[b];
  if (mv && cd->side==WHITE) {
    if (mv->castle!=0 || pc==6) {cd2->cp[0]=cd2->cp[1]=0;}
    else if (pc==2) {
      if (mv->fx==0 && mv->fy==7) cd2->cp[1]=0;
      if (mv->fx==7 && mv->fy==7) cd2->cp[0]=0;
    }
  }
  if (mv && cd->side==BLACK) {
    if (mv->castle!=0 || pc==-6) {cd2->cp[2]=cd2->cp[3]=0;}
    else if (pc==-2) {
      if (mv->fx==0 && mv->fy==0) cd2->cp[3]=0;
      if (mv->fx==7 && mv->fy==0) cd2->cp[2]=0;
    }
  }
  /*  Free the draw by repetition stats */
  free_poslist(cd2->once[0]);cd2->once[0]=NULL;
  free_poslist(cd2->once[1]);cd2->once[1]=NULL;
  free_poslist(cd2->twice[0]);cd2->twice[0]=NULL;
  free_poslist(cd2->twice[1]);cd2->twice[1]=NULL;
   /*  Need new copy? */
  if (mv && mv->castle==0 && mv->ep==0 && abs(pc)!=1 && !mv->capt) {
    cd2->once[0]=copy_poslist(cd->once[0]);
    cd2->once[1]=copy_poslist(cd->once[1]);
    cd2->twice[0]=copy_poslist(cd->twice[0]);
    cd2->twice[1]=copy_poslist(cd->twice[1]);
  }
}

/* Still more tidiness.  Get all this stuff out of the main search functions */
void setup_compdat(int beg) {
  compdat *cd2=&cdd[3-beg],*cd=&cdd[2-beg];
   
  cdd[3-beg]=cdd[2-beg];
  cd2->side=3-cd->side;
  cd2->alpha=-cd->beta;
  cd2->beta=-cd->alpha;
  cd2->capt=-cd->capt;
  cd2->once[0]=NULL;
  cd2->once[1]=NULL;
  cd2->twice[0]=NULL;
  cd2->twice[1]=NULL;
}

/* Does exactly what it says on the tin */
void free_compdat(int beg) {
  compdat *cd=&cdd[2-beg];
  free_poslist(cd->once[0]);
  free_poslist(cd->once[1]);
  free_poslist(cd->twice[0]);
  free_poslist(cd->twice[1]);
}

/* Set up a few variables */
void initialise(int beg) {
  compdat *cd=&cdd[2-beg];
   
  cd->alpha=-INFINITY;
  cd->beta=INFINITY;
  cd->capt=cd->tflg=0;
  cd->smove=cd->svars=-1;
}

 /* Display the principal variation */
void print_pv(int ply) {
  int n;
  fprintf(stderr,"\nPV : ");
  for (n=ply;n<pvlength[ply];n++) {
    if ((pv[ply][n]).type==-1) fprintf(stderr,"<HT>");
    else printmove2(&(pv[ply][n]));
    fprintf(stderr," ");
    if ((n&1)==1) fprintf(stderr," ");
  }
  fprintf(stderr,"\n");
}

 /* Computer decides whether or not to resign */
int choose_resign(long int mvsc, board *sq,compdat *cd) {
  int posn;
   
  if (depth_searched<5) return 0;
  cd->pts[0]=pts(1,sq);
  cd->pts[1]=pts(2,sq);
  cd->gamestage=pts(1,sq)+pts(2,sq);
  posn=defsc(cd->side,sq,cd); /*  Calc. static position evaluation */
  if (posn<-250 && mvsc<-200) return 1;
  if (posn<-300 && mvsc<-150) return 1;
  if (posn<-350 && mvsc<-100) return 1;
  if (posn<-400 && mvsc<-50) return 1;
  return 0;
}
  
 /* Do simple analysis on all of the moves in the movelist */
int analyse_all(int beg, int depth, int nm, compdat *cd, board *sq, int qui) {
  int nbmv=-1,a,from,to,nmnd=-1,nsmnd=-1,pvmove=cd->followpv;
  long int mostnodes=-1,secondmostnodes=-1;
  move *mv;

  for (a=0;a<nm;a++) {
    mv=&movestack[depth][a];
    mv->score=mv->gchk=mv->type=mv->cm=mv->capt=0;

     /*  If following the PV and this move is in the PV at this level. */
    if (pvmove==1 && compare(mv,&pvold[depth])==1) { mv->score = PVBONUS; pvmove=0; }

     /*  Found in table? smove is passed in the call to this function to save
      *  another hash lookup :) */
    else if (beg<2 && cd->smove>-1 && compare2(mv,cd->smove,cd->svars)==1) mv->score += HASHMOVE;

    analyse(sq,mv,cd);  /*  Simple analysis function */

     /*  CM already? */
    if (mv->cm) return a;

     /* Get values for from & to squares in 0-63 range */
    from=(mv->fy*8)+mv->fx;
    to=(mv->ty*8)+mv->tx;
     /*  Score bonus from history databases */
    if (qui==0 && historyon) (mv->score)+=history[from][to];
    if (historyplyon) (mv->score)+=historyply[from][to][depth];
     /*  Bonus score based on killermoves stored for each ply. */
    if (from==killer1[depth][0] && to==killer1[depth][1]) mv->score += KILLER1;
    if (from==killer2[depth][0] && to==killer2[depth][1]) mv->score += KILLER2;
     /* Bonus score for capturing same piece as this ply's best capture move */
    if (to==capturesq[depth] && sq->sq[mv->ty][mv->tx]!=0) mv->score += BEST_CAPT;

     /*  General positional statistics */
    if (qui==0 && !mv->ep && !mv->castle) tact_general(mv,sq);
     /*  Keep a track of the best move */
    if (nbmv<0 || mv->score>movestack[depth][nbmv].score) nbmv=a;
     /* Penalise if moving the king */
    if (abs(sq->sq[mv->fy][mv->fx])==6) mv->score -= MOVE_KING;
     /* Check node count */
    if (node_ordering==1) {
      if (nodecount[from][to][depth]>mostnodes) {
	secondmostnodes=mostnodes;
	nsmnd=nmnd;
        mostnodes=nodecount[from][to][depth];
	nmnd=a;
      }
      else if (nodecount[from][to][depth]>secondmostnodes) {
        secondmostnodes=nodecount[from][to][depth];
	nsmnd=a;
      }
    }
  }   
   
   /* Score bonus for moves with largest node counts */
  if (node_ordering) {
    if (nmnd>-1) movestack[depth][nmnd].score+=MOST_NODES;
    if (nsmnd>-1) movestack[depth][nsmnd].score+=SECOND_NODES;
  }
  return nbmv;
}

 /* Sort out increasing the values for the killer move and history
  * heuristic tables */
void increase_counts(move *mv, int depth, int qui, int lev) {
  int from=(mv->fy*8)+mv->fx,to=(mv->ty*8)+mv->tx;

   /*  Update history tables */
  if (qui==0 && historyon) {
    if (historyincr) (history[from][to])+=lev; /* Increase by the depth if flag is set */
    else (history[from][to])++; /* Otherwise just incremement by one */
  }
   /* Update history for this move at this ply by one. */ 
  if (historyplyon) (historyply[from][to][depth])++;
   /*  If this move is a good one then make it killer1
    *  and move the old killer1 to killer2 */	
  if (mv->score>killerscore[depth]) {
    if (from!=killer1[depth][0] || to!=killer1[depth][1]) {
      killer2[depth][0]=killer1[depth][0];
      killer2[depth][1]=killer1[depth][1];
      killerscore2[depth]=killerscore[depth];
      killer1[depth][0]=from;
      killer1[depth][1]=to;
    }
    killerscore[depth]=mv->score;
  }
  else if (mv->score>killerscore2[depth]) {
    if (from!=killer1[depth][0] || to!=killer1[depth][1]) {
      if (from!=killer2[depth][0] || to!=killer2[depth][1]) {
        killer2[depth][0]=from;
        killer2[depth][1]=to;
      }
      killerscore2[depth]=mv->score;
    }
  }
}
