use crate::types::context::BergContext;
use crate::{actions::GeneralArgs, render::option::option_display};
use forgejo_api::structs::{NotificationThread, NotifyGetListQuery};

use clap::Parser;

#[derive(Debug, Parser)]
pub struct ListNotificationArgs {
    /// just list everything
    #[arg(short, long, default_value_t = false)]
    pub all: bool,

    /// filter notifications by status
    #[arg(long, default_values_t = ["Unread", "Pinned"].map(String::from).to_vec())]
    pub status_types: Vec<String>,

    /// filter notifications by subject type
    #[arg(long)]
    pub subject_type: Option<String>,

    /// filter notifications by date
    #[arg(short, long)]
    pub dates: bool,

    /// control how many pages of notifications should be shown
    #[arg(short, long, default_value_t = 1)]
    pub page: usize,

    /// control how many notifications each page should hold
    #[arg(short, long, default_value_t = usize::MAX)]
    pub limit: usize,
}

impl ListNotificationArgs {
    pub async fn run(self, general_args: GeneralArgs) -> anyhow::Result<()> {
        let _ = general_args;
        let ctx = BergContext::new(self, general_args).await?;

        let notification_threads_list = ctx
            .client
            .notify_get_list(NotifyGetListQuery {
                ..Default::default()
            })
            .await?;

        tracing::debug!("{notification_threads_list:?}");

        present_notification_threads(&ctx, notification_threads_list);

        Ok(())
    }
}

fn present_notification_threads(
    ctx: &BergContext<ListNotificationArgs>,
    notification_threads_list: Vec<NotificationThread>,
) {
    let header = if notification_threads_list.is_empty() {
        "Notification Threads (empty)"
    } else {
        "Notification Threads"
    };

    let mut table = ctx.make_table();

    table
        .set_header(vec![header])
        .add_rows(
            notification_threads_list
                .into_iter()
                .map(|notification_thread| {
                    vec![option_display(
                        &notification_thread
                            .subject
                            .as_ref()
                            .and_then(|subject| subject.title.as_ref()),
                    )]
                }),
        );

    println!("{table}", table = table.show());
}
