/*
    Neville's ini reading object
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>              /* toupper, isspace */

#include "llist.h"
#include "inifile.h"

typedef struct ini_node {
    char *entry;
    char *value;
} ini_node_t, * ini_node_ptr;

typedef struct ini_section {
    char *name;
    LLIST nodes;
} ini_section_t, * ini_section_ptr;

typedef struct inifile {
    LLIST sections;
} inifile_t, * inifile_ptr;

/* Local prototypes */
static void lowerit(char *section);
static char isComment(char *line);
static char isSection(char *line, char **name);
static char isEntry(char *line, char **entry, char **value);


static void freenode(void *node)
{
    ini_node_ptr theNode;

    theNode = (ini_node_ptr) node;
    free(theNode->entry);
    free(theNode->value);
}

static void freesection(void *section)
{
    ini_section_ptr     theSection;

    theSection = (ini_section_ptr) section;
    free(theSection->name);
    ll_destroy(theSection->nodes);
}

extern INI ini_create()
{
    inifile_ptr         theObject;

    theObject = malloc(sizeof(inifile_t));
    if (theObject == NULL)
        return(NULL);

    theObject->sections = NULL;

    return(theObject);
}

extern void ini_destroy(INI object)
{
    inifile_ptr         theObject;

    theObject = (inifile_ptr) object;

    ll_destroy(theObject->sections);
    free(theObject);
}

extern char ini_load(INI object, const char *filename)
{
    FILE               *infile;
    char                linebuf[1024];
    inifile_ptr         theObject;
    INISECT             curSect = NULL;
    char               *loc1;
    char               *loc2;

    theObject = (inifile_ptr) object;

    infile = fopen(filename, "rb");
    if (!infile)
        return(INI_FAIL);

    while (fgets(linebuf, 1024, infile) != NULL)
    {
        if (isComment(linebuf))
            continue;
        if (isSection(linebuf, &loc1)) {
            curSect = ini_addSection(theObject, loc1);
            continue;
        }
        if (isEntry(linebuf, &loc1, &loc2))
        {
            if (curSect == NULL)
            {
                /* error */
                fprintf(stderr, "fatal ini_load error: invalid format\n");
                return(INI_FAIL);
            }
            ini_addEntrySect(theObject, curSect, loc1, loc2);
        }
    }

    fclose(infile);
    return(INI_PASS);
}

extern char ini_write(INI object, const char *filename)
{
    FILE               *outfile;
    char                returned;

    outfile = fopen(filename, "wb");
    if (!outfile)
        return(INI_FAIL);

    returned = ini_writeFile(object, outfile);

    fclose(outfile);

    return(returned);
}

extern char ini_writeFile(INI object, FILE *outfile)
{
    ini_node_ptr        theNode;
    ini_section_ptr     theSect;
    inifile_ptr         theObject;

    theObject = (inifile_ptr) object;

    ll_reset(theObject->sections);
    theSect = (ini_section_ptr) ll_next(theObject->sections);
    while (theSect != NULL)
    {
        fprintf(outfile, "[%s]\n", theSect->name);

        ll_reset(theSect->nodes);
        theNode = (ini_node_ptr) ll_next(theSect->nodes);
        while (theNode != NULL)
        {
            fprintf(outfile, "%s='%s'\n", theNode->entry, theNode->value);

            theNode = (ini_node_ptr) ll_next(theSect->nodes);
        }

        fprintf(outfile, "\n");
        theSect = (ini_section_ptr) ll_next(theObject->sections);
    }
    
    fprintf(outfile, "# EOF\n");

    return(INI_PASS);
}

static void lowerit(char *section)
{
    int                 i;

    for (i = 0; section[i] != '\0'; i++)
        section[i] = tolower(section[i]);
}

static char isComment(char *line)
{
    if ((line[0] == '#') ||
        ((line[0] == '/') && (line[1] == '/')))
        return(1);
    return(0);
}

static char isSection(char *line, char **name)
{
    char               *end;

    if (line[0] != '[')
        return(0);
    *name = line + 1;
    end = strchr(*name, ']');
    if (end == NULL)
        return(0);
    *end = '\0';
    
    lowerit(*name);
    return(1);
}

static char isEntry(char *line, char **entry, char **value)
{
    char       *separate;
    char       *theend;
    int         hastick = 0;

    separate = strchr(line, '=');
    if (separate == NULL)
        return(0);

    *entry = line;
    for (theend = separate - 1; isspace(*theend); theend -= 1)
        ;   /* Empty Loop */
    theend += 1;
    *theend = '\0';

    for (theend = separate + 1; isspace(*theend); theend += 1)
        ;   /* Empty Loop */

    if (*theend == '\'')
    {
        hastick = 1;
        *value = theend + 1;
    } else {
        *value = theend;
    }

    if (hastick == 1)
    {
        for (theend = *value + 1;
             (*theend != '\'') && (*theend != '\0');
             theend += 1)
            ;   /* Empty Loop */
    } else {
        for (theend = *value + 1;
             (!isspace(*theend)) && (*theend != '\0');
             theend += 1)
            ;   /* Empty Loop */
    }
    *theend = '\0';

    lowerit(*entry);
    return(1);
}

extern INISECT ini_getSection(INI object,
                              const char *sectname)
{
    inifile_ptr         theObject;
    ini_section_ptr     theSect;

    if ((object == NULL) || (sectname == NULL))
        return(NULL);

    theObject = (inifile_ptr) object;

    ll_reset(theObject->sections);
    theSect = (ini_section_ptr) ll_next(theObject->sections);
    while (theSect != NULL)
    {
        if (strcmp(theSect->name, sectname) == 0)
            return(theSect);
        theSect = (ini_section_ptr) ll_next(theObject->sections);
    }

    return(NULL);
}

extern INISECT ini_addSection(INI object,
                              const char *sectname)
{
    ini_section_ptr     theSect;
    ini_section_t       addSect;
    inifile_ptr         theObject;

    theObject = (inifile_ptr) object;

    if (theObject->sections == NULL)
    {
        theObject->sections = ll_create(sizeof(ini_section_t), &freesection);
        if (theObject->sections == NULL)
        {
            return(NULL);
        }
    }

    addSect.name  = (char*) strdup(sectname);
    addSect.nodes = NULL;
    theSect = (ini_section_ptr) ll_addlast(theObject->sections, &addSect);

    return(theSect);
}

extern char ini_addEntry(INI object,
                         const char *sectname,
                         const char *entryname,
                         const char *valuename)
{
    ini_section_ptr     theSect;
    inifile_ptr         theObject;

    theObject = (inifile_ptr) object;

    theSect = (ini_section_ptr) ini_getSection(object, sectname);
    if (theSect == NULL)
    {
        theSect = ini_addSection(object, sectname);
        if (theSect == NULL)
            return(INI_FAIL);
    }
    return(ini_addEntrySect(object, theSect, entryname, valuename));
}

extern char ini_addEntrySect(INI object,
                             INISECT section,
                             const char *entryname,
                             const char *valuename)
{
    ini_node_ptr        theNode;
    ini_node_t          addNode;
    ini_section_ptr     theSect;
    inifile_ptr         theObject;

    theObject = (inifile_ptr) object;
    theSect   = (ini_section_ptr) section;

    if (theSect->nodes == NULL)
    {
        theSect->nodes = ll_create(sizeof(ini_node_t), &freenode);
        if (theSect->nodes == NULL)
        {
            return(INI_FAIL);
        }
    }

    addNode.entry = (char*) strdup(entryname);
    addNode.value = (char*) strdup(valuename);
    theNode = (ini_node_ptr) ll_addlast(theSect->nodes, &addNode);

    if (theNode == NULL)
        return(INI_FAIL);
    return(INI_PASS);
}

extern char *ini_getValueString(INI object,
                                const char *sectname,
                                const char *entryname)
{
    ini_section_ptr     theSect;
    inifile_ptr         theObject;

    theObject = (inifile_ptr) object;

    theSect = (ini_section_ptr) ini_getSection(object, sectname);
    if (theSect == NULL)
        return(NULL);
    return(ini_getValueStringSect(object, theSect, entryname));
}

extern char *ini_getValueStringSect(INI object,
                                    INISECT section,
                                    const char *entryname)
{
    ini_node_ptr        theNode;
    ini_section_ptr     theSect;
    inifile_ptr         theObject;

    if ((object == NULL) || (section == NULL) || (entryname == NULL))
        return(NULL);

    theObject = (inifile_ptr) object;
    theSect   = (ini_section_ptr) section;

    ll_reset(theSect->nodes);
    theNode = (ini_node_ptr) ll_next(theSect->nodes);
    while (theNode != NULL)
    {
        if (strcmp(theNode->entry, entryname) == 0)
            return(theNode->value);
        theNode = (ini_node_ptr) ll_next(theSect->nodes);
    }

    return(NULL);
}

extern char ini_getValueNumber(INI object,
                               const char *sectname,
                               const char *entryname,
                               long *numloc)
{
    ini_section_ptr     theSect;
    inifile_ptr         theObject;

    theObject = (inifile_ptr) object;

    theSect = (ini_section_ptr) ini_getSection(object, sectname);
    if (theSect == NULL)
        return(INI_FAIL);
    return(ini_getValueNumberSect(object, theSect, entryname, numloc));
}

extern char ini_getValueNumberSect(INI object,
                                   INISECT section,
                                   const char *entryname,
                                   long *numloc)
{
    char               *theValue;
    inifile_ptr         theObject;

    theObject = (inifile_ptr) object;

    theValue = ini_getValueStringSect(object, section, entryname);
    if (theValue == NULL)
        return(INI_FAIL);
    *numloc = atol(theValue);

    return(INI_PASS);
}

extern char ini_getValueBool(INI object,
                             const char *sectname,
                             const char *entryname,
                             const char thedefault)
{
    ini_section_ptr     theSect;
    inifile_ptr         theObject;

    theObject = (inifile_ptr) object;

    theSect = (ini_section_ptr) ini_getSection(object, sectname);
    if (theSect == NULL)
        return(thedefault);
    return(ini_getValueBoolSect(object, theSect, entryname, thedefault));
}

extern char ini_getValueBoolSect(INI object,
                                 INISECT section,
                                 const char *entryname,
                                 const char thedefault)
{
    char               *theValue;
    inifile_ptr         theObject;

    theObject = (inifile_ptr) object;

    theValue = ini_getValueStringSect(object, section, entryname);
    if (theValue == NULL)
        return(thedefault);
    if ((toupper(*theValue) == 'Y') ||
        (toupper(*theValue) == '1') ||
        (toupper(*theValue) == 'T'))
    {
        return(1);
    }

    return(0);
}

/* EOF */
