/*
   NAME tests.c CREATED 14:31 16 Mar, 1994

   perform tests as described in MAIN.TEX MT_ALGO.TEX   
   */


#include <stdio.h>
#include <stdlib.h>

#include "config.h"

#ifdef USE_TESTS
#include "commands.h"
#include "chipmunk.h"
#include "tests.h"


int global_noise;/* global, sorry */


                                  /* generate random word with probability p */
UINT32 randomword (UINT32 p) {
  if ((rand () % 100) < p) 
     return ((rand () & 0xffff) | ((rand () & 0xffff) << 16));
  return 0;
}

#ifdef DEBUG

UINT32 CAsa0 = 0,
       CAsa1 = 0,
       FA0sa0 = 0,
       FA0sa1 = 0,
       FA1sa0 = 0,
       FA1sa1 = 0,
       WAsa0 = 0,
       WAsa1 = 0,
       CAobf = 0,
       CAabf = 0,
       np    = 0,
       dp    = 0,
       CDsa0 = 0,
       CDsa1 = 0,
       FD0sa0 = 0,
       FD0sa1 = 0,
       FD1sa0 = 0,
       FD1sa1 = 0,
       CDobf1= 0,
       CDabf1= 0,
       CDobf2= 0,
       CDabf2= 0,
       FD0obf1= 0,
       FD0abf1= 0,
       FD0obf2= 0,
       FD0abf2= 0,
       FD1obf1= 0,
       FD1abf1= 0,
       FD1obf2= 0,
       FD1abf2= 0;

#define INSTALL_OBF(p, s)  if ( p & (s)) p |=  (s);
#define INSTALL_ABF(p, s)  if (~p & (s)) p &= ~(s); 

                                                    /* simulate addressfault */
UINT32 addressfault (UINT32 a) {
  UINT32 sa1, sa0;

  sa0 = CAsa0 & ~0x3;               /* CA SAF, ignore CA[1,0] */
  sa1 = CAsa1 & ~0x3;
  if (a >= 0xc0000)
    if (R(0x10000) & 0x8000) {
      sa0 |= FA0sa0 & ~0xc0203;     /* FA0 SAF, ignore CA[31,30,9,1,0] */
      sa1 |= FA0sa1 & ~0xc0203;
    }else{
      sa0 |= FA1sa0 & ~0xc0203;     /* FA1 SAF, ignore CA[31,30,9,1,0] */
      sa1 |= FA1sa1 & ~0xc0203;
    }
  else
    if (a >= 0x80000) {
      sa0 |= WAsa0 & ~0xe0003;      /* WA SAF, ignore CA[31,30,17,1,0] */
      sa1 |= WAsa1 & ~0xe0003;
  }
  a = (a & ~sa0) | sa1;
  INSTALL_OBF (a, CAobf & ~0x3);    /* address line BF, ignore CA[1,0] */
  INSTALL_ABF (a, CAabf & ~0x3);

  return (a);
}

                                                   /* simulate datalinefault */
UINT32 datalinefault (UINT32 data, UINT32 address) {
  UINT32 sa1, sa0, shift;

  shift = address & 3;/* align data */
  if (shift)
    data = (data << (shift * 8)) | (data >> (32 - shift * 8));
    
  data ^= randomword (np);           /* not dominant random fault */
  sa0 = CDsa0;                       /* CD SAF */
  sa1 = CDsa1;
  if (address >= 0xc0000) 
    if (R(0x10000) & 0x8000) {
      sa0 |= FD0sa0;                 /* FD0 SAF */
      sa1 |= FD0sa1;
    }else {
      sa0 |= FD1sa0;                 /* FD1 SAF */
      sa1 |= FD1sa1;
    }
  data = (data & ~sa0) | sa1;
  INSTALL_OBF (data, CDobf1);        /* CD BF */
  INSTALL_ABF (data, CDabf1);
  INSTALL_OBF (data, CDobf2);
  INSTALL_ABF (data, CDabf2);
  if (address >= 0xc0000) 
    if (R(0x10000) & 0x8000) {
      INSTALL_OBF (data, FD0obf1);   /* FD0 BF */
      INSTALL_ABF (data, FD0abf1);
      INSTALL_OBF (data, FD0obf2);
      INSTALL_ABF (data, FD0abf2);
    }else {
      INSTALL_OBF (data, FD1obf1);   /* FD1 BF */
      INSTALL_ABF (data, FD1abf1);
      INSTALL_OBF (data, FD1obf2);
      INSTALL_ABF (data, FD1abf2);
    }
  data ^= randomword (dp);           /* dominant random fault */

  if (shift)
    data = (data >> (shift * 8)) | (data << (32 - shift * 8));

  return (data);
}

                                              /* simulate faulty read access */
UINT32 debugread (UINT32 address) {
  UINT32 data, faulty_address, faulty_data;

  faulty_address = addressfault (address);
  data           = A (faulty_address);
  faulty_data    = datalinefault (data, faulty_address);

  if (VERBOSE == 3 )
    printf ("@ %08x \n", address);
  if (VERBOSE >  3 )
    fprintf (stderr, "@ %08x > %08x R %08x > %08x\n",
            address, faulty_address, data, faulty_data);
  return (faulty_data);
}

                                             /* simulate faulty write access */
void debugwrite (UINT32 address, UINT32 data) {
  UINT32 faulty_address;
  
  faulty_address = addressfault (address);
  A (faulty_address) = data;

  if (VERBOSE >= 4)
    fprintf (stderr, "@ %08x > %08x W %08x\n",
            address, faulty_address, data);
}

#endif DEBUG


#define DOTX(x)   (x) ? "x" : "."
#define SETDLF(s, h)                            \
  if (((s = getintvar (#s)) && v) || V) {	\
    printf (" %-7s = %08x  ", #s, s); n++;	\
    if (V) 		 			\
      printf ("%s", h);				\
    else 					\
      DWN(i, 31, 0) {				\
        printf (DOTX(s & (1 << i)));		\
        if (!(i % 8)) printf(" ");		\
      } 					\
    printf("\n");				\
  }
#define SETALF(s, h) 				\
  if (((s = getintvar (#s)) && v) || V) {	\
    printf (" %-7s = %05x     ", #s, s); n++; 	\
    if (V)					\
      printf ("%s", h);				\
    else					\
      DWN(i,19,0) {				\
        printf (DOTX(s & (1 << i))); 		\
        if (!(i % 8)) printf(" ");		\
        if (i == 18)  printf(" ");		\
      }						\
    printf("\n");				\
  }
#define SETF(s, h)   				\
 if (((s = getintvar (#s)) && v) || V) {	\
   printf (" %-7s = %-8d  %s\n",#s,s,(V)?h:"");	\
   n++;}

int setfaults (int argc, char **argv) {
#ifdef DEBUG
  int i, v, V, n = 0;

  V = (argc && strcmp (argv[1], "-V") == 0);/* verbose */
  v = (argc && strcmp (argv[1], "-v") == 0);/* verbose */
  if (v)
    printf ("Setting the following fault (s):\n");
  SETALF (CAsa0,  "CA[31,30,17..2] SA0. Bit 0 and 1 are ignored,");
  SETALF (CAsa1,  "CA[31,30,17..2] SA1. bit 19,18 correspond with CA[31,30]");
  SETALF (FA0sa0, "FA[14..0]       SA0. Frame Unit 0");
  SETALF (FA0sa1, "   ...          SA1  The corresponding CA are specified.");
  SETALF (FA1sa0, "FA[14..0]       SA0. Frame Unit 1");
  SETALF (FA1sa1, "   ...          SA1  bit 19, 18, 9, 1, 0 are ignored.");
  SETALF (WAsa0,  "WA[16..2]       SA0. Other bits are ignored");
  SETALF (WAsa1,  "   ...          SA1   ...");
  SETALF (CAobf,  "Only one or-BF can be set. See CAsa1/0");
  SETALF (CAabf,  "   ...   and-BF       ...");
  SETF   (np,     "Non-dominant noise probabilty");
  SETF   (dp,     "Dominant     noise probabilty");
  SETDLF (CDsa0,  "CD[31..0] SA0");
  SETDLF (CDsa1,  "   ...    SA1");
  SETDLF (FD0sa0, "FD[31..0] SA0,  Frame Unit 0");
  SETDLF (FD0sa1, "   ...    SA1,  Frame Unit 0");
  SETDLF (FD1sa0, "FD[31..0] SA0,  Frame Unit 1");
  SETDLF (FD1sa1, "   ...    SA1,  Frame Unit 1");
  SETDLF (CDobf1, "Two CD or-BFs can be set,  this is the first");
  SETDLF (CDabf1, "Two CD and-BFs can be set, this is the first");
  SETDLF (CDobf2, "second CD or-BF");
  SETDLF (CDabf2, "second CD and-BF");
  SETDLF (FD0obf1,"first  FD or-BF,  Frame Unit 0");
  SETDLF (FD0abf1,"first  FD and-BF, Frame Unit 0");
  SETDLF (FD0obf2,"second FD or-BF,  Frame Unit 0");
  SETDLF (FD0abf2,"second FD and-BF, Frame Unit 0");
  SETDLF (FD1obf1,"first  FD or-BF,  Frame Unit 1");
  SETDLF (FD1abf1,"first  FD and-BF, Frame Unit 1");
  SETDLF (FD1obf2,"second FD or-BF,  Frame Unit 1");
  SETDLF (FD1abf2,"second FD and-BF, Frame Unit 1");
  if (v && !(n))
    printf (" (none)\n");

  return 0;
#else
  if (VERBOSE > 0)
    printf ("Can't set faults: not in DEBUG mode.\n");
  return 1;
#endif DEBUG

}

                                                     /* display version info */
int tver (int argc, char **argv) { 
  printf ("Test parts version of %.5s %s.\n", __TIME__, __DATE__);
# ifdef DEBUG
  printf ("DEBUG mode:\n");
  printf ("try  setfaults -V  for programable faults.\n");  
  printf ("(these work for all tests)\n");
# endif DEBUG
  return 0;
}

                                                /* check for crash on access */
int crashtest (int argc, char **argv) {
  /* usage : crashtest <address> [<message>]
     <address> target address
     <message> message to print other than default:
               "ERR: AT crashed"
     returns 0 
     Prints message if host crashed 
     */
  UINT32 target;
  int    i;

  if (argc < 2) {
    printf ("usage : crashtest address [message]\n");
    printf ("address   target address\n");
    printf ("message   message to print (ERR: AT crashed ...)\n");
    printf ("returns: 0.\n");
    fflush (stderr);
    return (1);
  }
  target = myatoi (argv[1]);
                                   /* leap before you look */
  if (argc > 2) {
    for (i = 2; i < argc; i++)
      printf ("%s ", argv[i]);
    printf ("\r");
  }else{
    printf ("ERR: AT crashed at access !! \r");
  }
  fflush (stdout);
  (void) R (target);
  if (VERBOSE >= 1) {                 /* passed, erase line */
    printf ("OK : Crashtest passed.");
    for (i = strlen ("OK : Crashtest passed."); i < 80; i++) 
      printf (" ");
    printf ("\n");
  }else{
    for (i = 0; i < 80; i++) 
      printf (" ");
    printf ("\r");
    fflush (stdout);
  }
  return (0);
}

                                            /* count bits set to one in word */
int count_ones (UINT32 w) {
  w = (w & 0x55555555) + ((w >> 1) & 0x55555555);
  w = (w & 0x33333333) + ((w >> 2) & 0x33333333);
  w = (w & 0x0f0f0f0f) + ((w >> 4) & 0x0f0f0f0f);
  w = (w & 0x00ff00ff) + ((w >> 8) & 0x00ff00ff);
  w = (w & 0x0000ffff) + ((w >>16) & 0x0000ffff);
  return w;
}
                                                        /* return ones count */
int do_count_ones (int argc, char **argv) { 

  if (argc < 2) {
    printf ("usage: cntones <value>\n");
    printf ("returns: ones count.\n");
    return (0);
  }
  return (count_ones (myatoi (argv[1])));
}


UINT32 robust_read (UINT32 a_ref, int ws, int p) {
/* robust read, updates global variable noise !! */
  UINT32 dr = 0;       /* data read */
  int b, i, dbsum[32]; /* loop cnt, data bit sum */

  if (p > 1)
    for (b = 0; b < ws; b++)
      dbsum[b] = 0;
  for (i = 0; i < p; i++) {
    switch (ws) {
    case 8:
      dr = RB(a_ref);
      break;
    case 16:
      dr = RS(a_ref);
      break;
    default:
      dr = R (a_ref);
      break;
    }
    if (p > 1)
      for (b = 0; b < ws; b++)
        dbsum[b] += 1 & (dr >> b);
  }
  if (p > 1) {
    dr = 0;
    global_noise = 0;
    for (b = 0; b < ws; b++)
      if ((dbsum[b] << 1) > p) {
        dr |= 1 << b;
        global_noise += p - dbsum[b];
      }else
        global_noise += dbsum[b];
    if (global_noise) {
      global_noise = (global_noise * 200 + ws * p / 2) / (ws * p);
      if (!global_noise)
        global_noise = 1;
    }
  }
  else
    global_noise = 0;
  return dr;
}
      
                                                         /* ghost image test */
int ghost_test (int argc, char **argv) { 
  /*  
     usage: readghost [-r]ref [-x]xmask [options] 
     ref      : reference address 
     xmask    : address line toggle mask 
     -a<amask>: addressing address lines
     -d<dmask>: datamask (f...f)
     -e       : expected data
     -w<n>    : word size
     -v       : write a random value
     -t<n>    : threshold 0..31 max. bits diff. (0)
     -p<n>    : number of passes (1)
     returns: fraction of misses times 100 (error percentage)
     */
  UINT32 a_ar,       /* alternative reference address */
         a_t,        /* target address */
         a_ref = 0,  /* ref address */
         ab  = 0,    /* address bit */
         am  = 0,    /* address mask */
         de  = 0,    /* data expected */
         dm  =~0,    /* data mask */
         xb  = 0,    /* x bit */
         xm  = 0;    /* x mask */
  int    de_spec = 0,/* data expected specified flag */
         e   = 0,    /* error */
         gtr = 0,    /* ghosttest threshold */
         n   = 0,    /* locations counter */
         noise = 0,  /* noise percentage */
         p   = 1,    /* passes */
         wperm = 0,  /* write permision */
         ws  = 32,   /* word-size */
         cnt = 0,   /* argument passing vars */
         i, 
         j = 0,
         iinc = 1;
  UINT32 value;
  char   option;

                             /* argument passing */
  for (i = 1; i < argc; i += iinc) {
    if (argv[i][0] == '-') {  /* an option */
      option = argv[i][1];
      if (argv[i][2] == '\0') {
        value = myatoi (argv[i+1]);/* seperate value */
        iinc = 2;
      }else{
        value = myatoi (argv[i] + 2);
        iinc = 1;
      }
      switch (option) {
      case 'r': a_ref= value; j = 1; break;
      case 'x': xm   = value; break;
      case 'a': am   = value; break;
      case 'd': dm   = value; break;
      case 'w': ws   = value; break;
      case 'p': p    = value; break;
      case 'v': wperm= 1;     iinc    = 1; break;
      case 'i': de   = value; de_spec = 1; break;
      case 't': gtr  = value; break;
      default : e    = 1;     iinc = 1;
      }
    }else{
      cnt++;
      switch (cnt + j) {
      case 1  : a_ref= myatoi (argv[i]); break;
      case 2  : xm   = myatoi (argv[i]); break;
      default : e    = 1;
      }
    }
  }
  
  if (!((ws == 32) || (ws == 16) || (ws == 8))) {
    printf ("Invalid wordsize.\n");
    return (1);
  }
  if (wperm && de_spec) {
    printf ("Cannot expect a random value to be %x !\n", de);
    return (1);
  }
  if (e || ((cnt + j) < 1) || (argc < 3)) {    /* print help */
    printf ("usage: readghost [-r]ref [-x]xmask [options]\n");
    printf ("-r<ref>   reference address\n");
    printf ("-x<xmask> don't care address lines\n");
    printf ("-a<amask> addressing address lines (%x)\n", am);
    printf ("-d<dmask> datamask (%08x)\n", dm);
    if (de_spec)
      printf ("-i<val>   initial value (%x)\n", de);
    else
      printf ("-i<val>   initial value (none)\n");
    printf ("-w<n>     wordsize 8, 16 or 32 (%d)\n", ws);
    printf ("-v        write random value (off)\n");
    printf ("-t<n>     threshold 0..31 max. bits diff. (%d)\n", gtr);
    printf ("-p<n>     number of passes (%d)\n", p);
    printf ("returns: percentage misses.\n");
    fflush (stderr);
    return 1;
  }
  if (!xm) {
    printf ("Invalid xmask.\n");
    return (1);
  }
                                             /* initialize */
  if (!dm || !p) {
    if (VERBOSE > 0)
      printf ("Nothing to do.\n");
    return 0;
  }
  e = 0;
  do {
    if (am) {                         /* address line toggle */
      while (!((am >> ab) & 1)) ab++; 
      a_ar = a_ref ^ (1 << ab);
      am &= ~(1 << ab);       
    }else{ 
      a_ar = a_ref;
    }
    if (de_spec || wperm) {
      if (wperm) {
        de = randomword (100);
        switch (ws) {
        case 8:
          WB(a_ar, de);
          break;
        case 16:
          WS(a_ar, de);
          break;
        default:
          W (a_ar, de);
        }
      }
    }
    else
      de = robust_read (a_ar, ws, p);

    while (xm) {                      /* address line toggle */
      while (!((xm >> xb) & 1)) xb++; 
      a_t = a_ar ^ (1 << xb);
      xm &= ~(1 << xb);       
                                            /* actual test */
      n++;
      e     += (count_ones ((de ^ robust_read (a_t, ws, p)) & dm) > gtr);
      noise += global_noise;
    }
  }while (a_ar != a_ref);
  
  e = (e * 100 + n/2) /n;
  if (noise) {
    noise = (noise + n/2)/n;
    if (!noise)
      noise = 1;
    if (VERBOSE > 0)
      printf ("WRN: detected %d%% noise (ave).\n", noise);
  }
  internal_set_int ("NOISE", noise);

  if (VERBOSE >= 1)                              /* report */
    if (e)
      printf ("ERR: ghostimage not found for %d%% .\n", e);
    else
      printf ("OK : ghostimage found.\n");
                                                 /* return */
  return (e);
}

#define PATT   0x55555555
                                                        /* address line test */
int addrtest (int argc, char **argv) { 
  /* 
     usage: addrtest [-r]ref [-a]amask [options] 
            -t<n> threshold 
     ref     : reference address 
     amask   : ata read from reference address */
  UINT32 a_ref = 0,
         a_t,
         am  = 0,
         av,
         dm  =~0,
         dr  = 0,
         dt  = 0;    /* data read from target address */
  int    ab  = 0, 
         aftr = 0,   /* address fault threshold */
         drerr = 0,  /* target fault count */
         dterr = 0,  /* refnce fault count */
         e   = 0,    /* error */
         n   = 0,    /* locations counter */
         noise = 0,  /* noise percentage */
         p   = 1,    /* passes */
         ws  = 32,   /* word-size */
         cnt = 0,    /* fault count & argument passing vars */
         i, j = 0,
         iinc = 1;
  UINT32 value;
  char   option;

                                      /* argument passing */
  for (i = 1; i < argc; i += iinc) {
    if (argv[i][0] == '-') {  /* an option */
      option = argv[i][1];
      if (argv[i][2] == '\0') {
        value = myatoi (argv[i+1]);/* seperate value */
        iinc = 2;
      }else{
        value = myatoi (argv[i]+2);
        iinc = 1;
      }
      switch (option) {
      case 'r': a_ref= value; j = 1; break;
      case 'a': am   = value; break;
      case 'w': ws   = value; break;
      case 'd': dm   = value; break;
      case 't': aftr = value; break;
      case 'p': p    = value; break;
      default : iinc = 1; e = 1;
      }
    }else{
      cnt++;
      switch (cnt + j) {
      case 1  : a_ref= myatoi (argv[i]); break;
      case 2  : am   = myatoi (argv[i]); break;
      default : e    = 1;
      }
    }
  }
  e += ((cnt + j) < 1) || (argc < 3);

  if (!((ws == 32) || (ws == 16) || (ws == 8))) {
    printf ("Invalid wordsize.\n");
    return (1);
  }

  if (ws<32)            /* reduce dmask */
    dm &= ~(~0 << ws);
  if (!aftr)            /* deduce threshold if not specified */
    aftr = (count_ones (dm) + 1) / 2;
  if (((count_ones (dm) < 2) || !am || !p) && !e) { 
    if (VERBOSE > 0)
      printf ("Nothing to do.\n");
    return 0;
  }

  if (count_ones (dm)<aftr) { 
    if (VERBOSE > 0)
      printf ("Ones (dmask) < threshold makes no sense.\n");
    return 0;
  }

  if ((aftr < 1) || (aftr > ws)) {
    printf ("threshold out of range");
    return 1;
  }

  if (e) {                            /* print help */
    printf ("usage: addrtest [-r]ref [-a]amask [options]\n");
    printf ("-r<ref>   reference address\n" );
    printf ("-a<amask> address line toggle mask\n");
    printf ("-d<dmask> datamask (%x)\n", dm);
    printf ("-w<n>     wordsize 8, 16 or 32 (%d)\n", ws);
    printf ("-t<n>     threshold (max. faulty bits)(%d)\n", aftr);
    printf ("-p<n>     number of passes (%d)\n", p);
    printf ("returns: faulty address lines vector.\n"); 
    fflush (stderr);
    return 1;
  }

  av    = am;
  noise = 0;
  while (av) {                  /* generate target address */
    while (!(av & (1 << ab))) ab++; 
    a_t = (a_ref ^ (1 << ab));
    av &= ~(1 << ab);  
                                            /* actual test */
    cnt   = 0;
    drerr = 0;
    dterr = 0;
    for (i = 0; i < p; i++) {
      switch (ws) {
      case 8:
        WB(a_t,  ~PATT);
        WB(a_ref, PATT);
        dt = RB(a_t);
        dr = RB(a_ref);
        break;
      case 16:
        WS(a_t,  ~PATT);
        WS(a_ref, PATT);
        dt = RS(a_t);
        dr = RS(a_ref);
        break;
      default:/*32*/
        W (a_t,  ~PATT);
        W (a_ref, PATT);
        dt = R (a_t);
        dr = R (a_ref);
      }
      dr = ( dr ^  PATT) & dm;
      dt = ( dt ^ ~PATT) & dm;
      cnt   += count_ones ( dt );
      drerr += count_ones ( dr );
   /* dterr += count_ones ( dt ); */
#if 0
      /* previous code is optimized version of following code */
      cnt   += count_ones ((~dr^dt  ) & dm);
      drerr += count_ones (( dr^PATT) & dm);
      dterr += count_ones ((~dt^PATT) & dm);
#endif
    }
    n++;
    dterr = cnt; /* count_ones ( dt );*/
    if ( cnt > (p * aftr)) {
      e |= 1 << ab;    /* store faulty line  */
      noise += MIN (dterr, drerr) + count_ones (dm) * p - MAX (dterr, drerr);
    }
    else
      noise += dterr + drerr;
  }
  
  if (VERBOSE >= 1) {                               /* report */
    if (e) {
      n = count_ones (e);
      printf ("ERR: address line fault: %d error%s.\n", n, n == 1 ? "" : "s");
    }else
      printf ("OK : no address line faults.\n");
  }
  if (noise) {
    noise = (100 * noise + n*p*count_ones (dm)/2)/(n*p*count_ones (dm));
    if (!noise)
      noise = 1;
    if (VERBOSE > 0)
      printf ("WRN: detected %d%% noise (ave).\n", noise);
  }
  internal_set_int ("NOISE", noise);

  if ((VERBOSE >= 2) && e) 
      if (getintvar ("notable")) {
        printf ("ERR: address line fault A[");
        DWN(i,31,0) if (e & ( 1 << i)) {
          printf ("%d", i);
          if (e & ~(~0 << i))
            printf (", ");
        }
        printf ("]\n");
      }
      else{
        printf (  "bitnr ");
        DWN(i,31,0) if (am & (1 << i))
          printf (" %d", i / 10);else printf (" .");
        printf ("\n      ");
        DWN(i,31,0) if (am & (1 << i))
          printf (" %d", i % 10);else printf ("  ");
        printf ("\nAF    ");
        DWN(i,31,0)
          printf (" %s", AN_X(e & (1 << i)));
        printf ("\n");
      }
                                                 /* return */
  return (e);
}

                                                 /* dataline SAF & BF detect */
int datatest (int argc, char **argv) {
  /* 
    Usage: datatest [-r]ref [[-c]cmask] [options]
    -r<ref>   reference address 
    -c<cmask> count mask
    -a<amask> address mask
    -w<n>     wordsize
    -p<n>     number of passes
    -d<dmask> data mask
    -s        use walking one testset
    returns number of faulty locations
    */
  UINT32 a_t,        /* target address */
         a_ref = 0,  /* reference address */
         a_tt,       /* temporary address */
         cm  = 0,    /* count mask */
         am  = 0,    /* address mask */
         dm  =~0,    /* data mask */
         count = 0,  /* count vector */
         tv;         /* testvector */
  int    e   = 0,    /* error */
         eflag = 0,  /* error flag */
         n   = 0,    /* locations counter */
         db  = 0,    /* data bit */
         ab  = 0,    /* address bit */
         ws  = 32,   /* word-size */
         p   = 1,    /* passes */
         noise = 0,  /* noise percentage */
         walk = 0,   /* walking one set flag */
         cnt = 0,    /* fault count & argument passing vars */
         j   = 0,
         i,
         iinc= 1;
  UINT32 value;
  char   option;

                                      /* argument passing */
  for (i = 1; i < argc; i += iinc) {
      if (argv[i][0] == '-') {  /* an option */
      option = argv[i][1];
      if (argv[i][2] == '\0') {
        value = myatoi (argv[i+1]);/* seperate value */
        iinc = 2;
      }else{
        value = myatoi (argv[i]+2);
        iinc = 1;
      }
      switch (option) {
      case 'r': a_ref= value; j = 1; break;
      case 'c': cm   = value; break;
      case 'a': am   = value; break;
      case 'w': ws   = value; break;
      case 'd': dm   = value; break;
      case 'p': p    = value; break;
      case 'm': walk = 1; iinc = 1; break;
      default : e    = 1; iinc = 1;
      }
    }else{
      cnt++;
      switch (cnt + j) {
      case 1  : a_ref= myatoi (argv[i]); break;
      case 2  : cm   = myatoi (argv[i]); break;
      default : e    = 1;
      }
    }
  }

  if (!((ws == 32) || (ws == 16) || (ws == 8))) {
    printf ("Invalid wordsize.\n");
    return (1);
  }

  if (e || ((cnt + j) < 1) || (argc < 2)) {
    printf ("usage: datatest [-r]ref [options]\n");
    printf ("-r<ref>   reference address\n");
    printf ("-c<cmask> count mask (%x)\n", cm);
    printf ("-a<amask> address mask (%x)\n", am);
    printf ("-w<n>     wordsize 8, 16 or 32 (%d)\n", ws);
    printf ("-p<n>     number of passes (%d)\n", p);
    printf ("-d<dmask> data mask (%x)\n", dm);
    printf ("-m        use walking one test set (off)\n");
    printf ("returns: number of faulty locations.\n");
    fflush (stderr);
    return 1;
  }

  if (!dm) { 
    if (VERBOSE > 0)
      printf ("Nothing to do.\n");
    return 0;
  }
  if (ws < 32)    /* reduce dmask */
    dm &= ~(~0 << ws);
  do {
    if (am) {                        /* address line toggle */
      while (!((am >> ab) & 1)) ab++;
      a_tt = a_ref ^ (1 << ab);
      am  &= ~(1 << ab);       
    }
    else
      a_tt = a_ref;
    count = 0;
    do {                                  /* counting */
      count--;
      count &= cm;
      a_t = a_tt ^ count;
                                        /* actual test */
      if (walk) {
        db = ws;
        tv = 1; /* walking one */
      }
      else {
        tv = ~1; /* tv according to modified counting sequence */
        db = ws+1;/* round fix */
      }
      while (db > 0) {
        if (!walk) {
          if (db != ws + 1)
            tv = (tv << db) ^ tv;
        }
        switch (ws) {
        case 8:
          WB(a_t, tv);
          break;
        case 16:
          WS(a_t, tv);
          break;
        default:/*32*/
          W (a_t, tv);
        }
        if (dm & (robust_read (a_t, ws, p) ^ tv))
          eflag = 1;
        noise += global_noise;
        n++;
        if (walk) {
          tv <<= 1;
          db--;
        }else{
          db /= 2;
          tv |= 1;
        }
      }
      if (eflag) {
        eflag = 0;
        e++;
      }
    }while (count);
  }while (a_tt != a_ref);

  if (noise) {    
    noise = (noise + n/2)/n;
    if (!noise)
      noise = 1;
    if (VERBOSE > 0)                                 /* report */
      printf ("WRN: detected %d%% noise.\n", noise);
  }
  internal_set_int ("NOISE", noise ); /* store noise */

  if (VERBOSE > 0) {
    if (e)
      printf ("ERR: dataline fault (s) detected.\n");
    else {
    if (walk) {
       if (dm & (robust_read (a_t, ws, p) ^ (1<<(ws-1))))
         printf ("WRN: data retention fault./n");
     } else
       if (dm & (robust_read (a_t, ws, p) ^ (0x55555555)))
          printf ("WRN: data retention fault./n");
      printf ("OK : no dataline faults.\n");
    }
  }

  return e;                                      /* return */
}

                                                   /* visit */
int visit (int argc, char **argv) {
  /* 
    Usage: visit [-r]ref [options]
           -r<ref>   reference address 
           -a<amask> address mask
           -c<cmask> count mask
           -w<n>     wordsize
           -p<n>     number of passes
           -
           returns number of visits
           sets environment DATA and NOISE
  */
  UINT32 a_t;        /* target address */
  UINT32 a_ref = 0,  /* reference address */
         a_tt,       /* temporary target address */
         am  = 0,    /* address mask */
         cm  = 0,    /* count mask */
         count = 0;  /* count vector */
  int    ab  = 0,    /* address bit */
         e   = 0,    /* error */
         n   = 0,    /* locations counter */
         p   = 1,    /* passes */
         ws  = 32,   /* word-size */
         cnt = 0,    /* argument passing vars */
         j   = 0,
         i,
         iinc = 1;
  UINT32 value;
  char   option;

                                      /* argument passing */
  for (i = 1; i < argc; i += iinc) {
    if (argv[i][0] == '-') {  /* an option */
      option = argv[i][1];
      if (argv[i][2] == '\0') {
        value = myatoi (argv[i+1]);/* seperate value */
        iinc = 2;
      }else{
        value = myatoi (argv[i]+2);
        iinc = 1;
      }
      switch (option) {
      case 'r': a_ref= value; j = 1; break;
      case 'a': am   = value; break;
      case 'c': cm   = value; break;
      case 'w': ws   = value; break;
      case 'p': p    = value; break;
      default : e    = 1; iinc = 1;
      }
    }else{
      cnt++;
      switch (cnt + j) {
      case 1  : a_ref= myatoi (argv[i]); break;
      case 2  : am   = myatoi (argv[i]); break;
      case 3  : cm   = myatoi (argv[i]); break;
      default : e    = 1;
      }
    }
  }
  if (p <= 0) {
    if (VERBOSE)
      printf ("Nothing to do.\n");
    return (0);
  }
  if (!((ws == 32) || (ws == 16) || (ws == 8))) {
    printf ("Invalid wordsize.\n");
    return (1);
  }

  if (e || ((cnt + j) < 1) || (argc < 2)) {
    printf ("usage: visit [-r]ref [options]\n");
    printf ("-r<ref>   reference address\n");
    printf ("-a<amask> address mask (%x)\n", am);
    printf ("-c<cmask> count mask (%x)\n", cm);
    printf ("-w<n>     wordsize 8, 16 or 32 (%d)\n", ws);
    printf ("-p<n>     number of passes (%d)\n", p);
    printf ("returns: number of locations.\n");
    fflush (stderr);
    return 1;
  }

  do{                           /* generate target address */
    if (am) {                         /* address line toggle */
      while (!((am >> ab) & 1)) ab++; 
      a_tt = a_ref ^ (1 << ab);
      am &= ~(1 << ab);       
    }else{
      a_tt = a_ref;
    }
    count = 0;                                 /* counting */
    do {
      count--;
      count &= cm;
      a_t = a_tt ^ count;
                                            /* actual test */
      switch (ws) {
      case 8:
        for (i = 0; i < p; i++)
          (void) AB(a_t);
        break;
      case 16:
        for (i = 0; i < p; i++)
          (void) AS(a_t);
        break;
      default:
        for (i = 0; i < p; i++)
          (void) A (a_t);
      }
      n+=p;
      
    }while (count);
  }while (a_tt != a_ref);

  return (n);
}

                                                   /* read_test */
int read_test (int argc, char **argv) {
  /* 
    Usage: read [-r]ref [options]
           -r<ref>   reference address 
           -a<amask> address mask
           -c<cmask> count mask
           -n<n>     length
           -s<n>     stride
           -w<n>     wordsize
           -p<n>     number of passes
           returns number of visits
           sets environment DATA and NOISE
  */
  UINT32 a_t,        /* target address */
         a_ref = 0,  /* reference address */
         a_tt,       /* temporary target address */
         dr,         /* data read */
         am    = 0,  /* address mask */
         cm    = 0,  /* count mask */
         count = 0;  /* count vector */
  int    ab    = 0,  /* address bit */
         e     = 0,  /* error */
         n_loc = 1,  /* number of locations */
         stride= 0,  /* stride */
         n     = 0,  /* locations counter */
         p     = 1,  /* passes */
         ws    = 32, /* word-size */
         noise = 0,  /* noise */
         b,          /* data bit pointer */
         dbsum[32],  /* data bit sum vector */
         cnt = 0,    /* argument passing vars */
         j   = 0,
         i,
         iinc = 1;
  UINT32 value;
  char   option;

                                      /* argument passing */
  for (i = 1; i < argc; i += iinc) {
    if (argv[i][0] == '-') {  /* an option */
      option = argv[i][1];
      if (argv[i][2] == '\0') {
        value = myatoi (argv[i+1]);/* seperate value */
        iinc = 2;
      }else{
        value = myatoi (argv[i]+2);
        iinc = 1;
      }
      switch (option) {
      case 'r': a_ref = value; j = 1; break;
      case 'a': am    = value; break;
      case 'c': cm    = value; break;
      case 'n': n_loc = value; break;
      case 's': stride= value; break;
      case 'w': ws    = value; break;
      case 'p': p     = value; break;
      default : e     = 1; iinc = 1;
      }
    }else{
      cnt++;
      switch (cnt + j) {
      case 1  : a_ref= myatoi (argv[i]); break;
      case 2  : p    = myatoi (argv[i]); break;
      default : e    = 1;
      }
    }
  }

  if ((n_loc > 1) && (am || cm)) {
    printf ("ERR: specify either -n OR -a and -c.\n");
    return 1;
  }
  if (!((ws == 32) || (ws == 16) || (ws == 8))) {
    printf ("Invalid wordsize.\n");
    return (1);
  }

  if (e || ((cnt + j) < 1) || (argc < 2)) {
    printf ("usage: read [-r]ref [options]\n");
    printf ("-r<ref>   reference address\n");
    printf ("-a<amask> address mask (%x)\n", am);
    printf ("-c<cmask> count mask (%x)\n", cm);
    printf ("-n<n>     length\n");
    printf ("-s<n>     stride (%d)\n", ws);
    printf ("-w<n>     wordsize 8, 16 or 32 (%d)\n", ws);
    printf ("-p<n>     number of passes (%d)\n", p);
    printf ("specify either -n and -s OR -c and -a.\n");
    printf ("returns: data read.\n");
    fflush (stderr);
    return 1;
  }

  if (!(n_loc || cm || am) || (p == 0)) {
    if (VERBOSE > 0)
      printf ("Nothing to do.\n");
    return 0;
  }

  if (!n_loc) {/* cm and am specified */
    n_loc = 1;
    if (cm) {
      if ((cm & ((ws / 8)- 1)) && (VERBOSE >= 2))
        printf ("WRN: ignoring cmask bits that address within word.\n");
      cm &= ~((ws / 8)- 1);
    }
  }
  else {       /* n_loc specified */
    if (!stride)
      stride = ws /8;
    if ((stride % (ws/8)) && (VERBOSE >= 2))
      printf ("WRN: stride not in wordsize units causing misalignment.\n");
    if ((n_loc % stride) && (VERBOSE >= 2))
      printf ("WRN: using length %x instead of %x to fit stride.\n",
               n_loc - n_loc % stride, n_loc);
    n_loc /= stride;
    if (!n_loc) n_loc = 1;
  }
  
  for (b = 0; b < ws; b++)
    dbsum[b] = 0;
  do {                          /* generate target address */
    if (am) {                         /* address line toggle */
      while (!((am >> ab) & 1)) ab++; 
      a_tt = a_ref ^ (1 << ab);
      am &= ~(1 << ab);       
    }
    else {
      a_tt = a_ref;
    }
    count = 0;                                 /* counting */
    do {
      count--;
      count &= cm;
      a_t = a_tt ^ count;
      for (j = 0; j < n_loc; j++, a_t += stride)
        for (i = 0; i < p; i++) {             /* passes */
                                            /* actual test */
          switch (ws) {
          case 8:
            dr = RB(a_t);
            break;
          case 16:
            dr = RS(a_t);
            break;
          default:/*32*/
            dr = R (a_t);
          }
          if (dr)
            for (b = 0; b < ws; b++)
              dbsum[b] += 1 & (dr >> b);
          n++;
        }
    }while (count);
  }while (a_tt != a_ref);
  dr = 0;
  for (b = 0; b < ws; b++)
    if ((dbsum[b] << 1) > n) {
        dr |= 1 << b;
        noise += n - dbsum[b];
      }else
        noise += dbsum[b];
  if (noise) {
    noise = (noise * 200 + ws * n / 2) / (ws * n);
    if (!noise)
      noise = 1;
  }
  if (VERBOSE > 1) {
    printf("Read  0x%08x at %d location%s", dr, n/p, (n/p) == 1 ? "" : "s");
    if (p > 1)
      printf(" %d times", p);
    printf (".\n");
    if (noise)
      printf ("WRN: detected %d%% noise (ave).\n", noise);
  }
  internal_set_int ("NOISE", noise);
  internal_set_int ("N", n);

  return (dr);
}

                                                   /* write_test */
int write_test (int argc, char **argv) {
  /* 
    Usage: write [-r]ref [options]
           -r<ref>   reference address 
           -a<amask> address mask
           -c<cmask> count mask
           -n<n>     length
           -s<n>     stride
           -w<n>     wordsize
           -p<n>     number of passes
           -
           returns number of visits
           sets environment DATA and NOISE
  */
  UINT32 a_t,        /* target address */
         a_ref = 0,  /* reference address */
         a_tt,       /* temporary target address */
         d = 0,      /* data to write */
         am  = 0,    /* address mask */
         cm  = 0,    /* count mask */
         count = 0;  /* count vector */
  int    ab  = 0,    /* address bit */
         e   = 0,    /* error */
         n_loc = 1,  /* number of locations */
         stride= 0,  /* stride */
         n   = 0,    /* locations counter */
         p   = 1,    /* passes */
         ws  = 32,   /* word-size */
         cnt = 0,    /* argument passing vars */
         j   = 0,
         i,
         iinc = 1;
  UINT32 value;
  char   option;

                                      /* argument passing */
  for (i = 1; i < argc; i += iinc) {
    if (argv[i][0] == '-') {  /* an option */
      option = argv[i][1];
      if (argv[i][2] == '\0') {
        value = myatoi (argv[i+1]);/* seperate value */
        iinc = 2;
      }else{
        value = myatoi (argv[i]+2);
        iinc = 1;
      }
      switch (option) {
      case 'r': a_ref= value; j = 1; break;
      case 'd': d    = value; break;
      case 'a': am   = value; break;
      case 'n': n_loc = value; break;
      case 's': stride= value; break;
      case 'c': cm   = value; break;
      case 'w': ws   = value; break;
      case 'p': p    = value; break;
      default : e    = 1; iinc = 1;
      }
    }else{
      cnt++;
      switch (cnt + j) {
      case 1  : a_ref= myatoi (argv[i]); break;
      case 2  : d    = myatoi (argv[i]); break;
      default : e    = 1;
      }
    }
  }
  if ((n_loc > 1) && (am || cm)) {
    printf ("ERR: specify either -n and -s OR -a and -c.\n");
    return 1;
  }
  if (!((ws == 32) || (ws == 16) || (ws == 8))) {
    printf ("Invalid wordsize.\n");
    return (1);
  }

  if (e || ((cnt + j) < 1) || (argc < 2)) {
    printf ("usage: write [-r]ref [-d]data [options]\n");
    printf ("-r<ref>   reference address\n");
    printf ("-d<value> data to write\n");
    printf ("-a<amask> address mask (%x)\n", am);
    printf ("-c<cmask> count mask (%x)\n", cm);
    printf ("-w<n>     wordsize 8, 16 or 32 (%d)\n", ws);
    printf ("-n<n>     length\n");
    printf ("-s<n>     stride (%d)\n", ws);
    printf ("-p<n>     number of passes (%d)\n", p);
    printf ("specify either -n and -s OR -c and -a.\n");
    printf ("returns: number of write accesses.\n");
    fflush (stderr);
    return 1;
  }

  if (!(n_loc || cm || am) || (p == 0)) {
    if (VERBOSE > 0)
      printf ("Nothing to do.\n");
    return 0;
  }

  if (cm || am) {/* cm and am specified */
    n_loc = 1;
    if (cm) {
      if ((cm & ((ws / 8)- 1)) && (VERBOSE >= 2))
        printf ("WRN: ignoring cmask bits that address within word.\n");
      cm &= ~((ws / 8)- 1);
    }
  }
  else {       /* n_loc specified */
    if (!stride)
      stride = ws /8;
    if ((stride % (ws/8)) && (VERBOSE >= 2))
      printf ("WRN: stride not in wordsize units causing misalignment.\n");
    if ((n_loc % stride) && (VERBOSE >= 2))
      printf ("WRN: using length %x instead of %x to fit stride.\n",
               n_loc - n_loc % stride, n_loc);
    n_loc /= stride;
    if (!n_loc) n_loc = 1;
  }
  
  do {                          /* generate target address */
    if (am) {                         /* address line toggle */
      while (!((am >> ab) & 1)) ab++; 
      a_tt = a_ref ^ (1 << ab);
      am &= ~(1 << ab);       
    }
    else {
      a_tt = a_ref;
    }
    count = 0;                                 /* counting */
    do {
      count--;
      count &= cm;
      a_t = a_tt ^ count;
      for (j = 0; j < n_loc; j++, a_t += stride)
        for (i = 0; i < p; i++) {             /* passes */
                                            /* actual test */
          switch (ws) {
          case 8:
            WB(a_t, d);
            break;
          case 16:
            WS(a_t, d);
            break;
          default:/*32*/
            W (a_t, d);
          }
          n++;
        }
    }while (count);
  }while (a_tt != a_ref);

  if (VERBOSE > 1) {
    printf("Wrote 0x%08x at %d location%s", d, n/p, (n/p)== 1 ? "" : "s");
    if (p > 1)
      printf(" %d times", p);
    printf (".\n");
  }
  return (n);
}

                                                     /* dataline test for BF */
int dlsatest (int argc, char **argv) {
  /* 
    Usage: dlsatest [-r]ref [options]
           -r<ref>   reference address
           -a<amask> address mask (0)
           -c<cmask> count mask (0)
           -w<n>     wordsize (32)
           -d<dmask> data mask (ffffffff)
           -t<n>     threshold ()
           -p<n>     number of passes
           returns number of SAFs
  */
  UINT32 a_t,        /* target address */
         a_ref = 0,  /* reference address */
         a_tt,       /* temporary target address */
         am  = 0,    /* address mask */
         cm  = 0,    /* count mask */
         count = 0,  /* count vector */
         dm  =~0,    /* data mask */
         nv  = 0,    /* noise vector */
         dr0,        /* data read 0 */
         dr1,        /* data read 0 */
         saone = 0,  /* word with SA1 faults */
         sazero = 0; /* word with SA0 faults */
  int    ab  = 0,    /* address bit */
         dltr,       /* data line test threshold */
         e   = 0,    /* error */
         n   = 0,    /* locations counter */
         cntf[32],   /* fault count per data line*/
         noise = 0,  /* noise percentage */
         p   = 1,    /* passes */
         ones[32],   /* ones count per bit */
         ws  = 32,   /* word-size */
         cnt = 0,    /* argument passing vars */
         j   = 0,
         i,
         iinc = 1;
  UINT32 value;
  char   option;

  dltr = getintvar ("dltr");
  if (!dltr)
    dltr = 100;

  j = 0;                               /* argument passing */
  e = 0;
  cnt = 0;
  iinc = 1;
  for (i = 1; i < argc; i += iinc) {
    if (argv[i][0] == '-') {  /* an option */
      option = argv[i][1];
      if (argv[i][2] == '\0') {
        value = myatoi (argv[i+1]);/* seperate value */
        iinc  = 2;
      }else{
        value = myatoi (argv[i] + 2);
        iinc  = 1;
      }
      switch (option) {
      case 'r': a_ref= value; j = 1; break;
      case 'a': am   = value; break;
      case 'c': cm   = value; break;
      case 'w': ws   = value; break;
      case 'd': dm   = value; break;
      case 't': dltr = value; break;
      case 'p': p    = value; break;
      default : iinc = 1; e = 1;
      }
    }else{
      cnt++;
      switch (cnt + j) {
      case 1  : a_ref= myatoi (argv[i]); break;
      case 2  : am   = myatoi (argv[i]); break;
      case 3  : cm   = myatoi (argv[i]); break;
      default : e    = 1;
      }
    }
  }
  if (!((ws == 32) || (ws == 16) || (ws == 8))) {
    printf ("Invalid wordsize.\n");
    return (1);
  }
  if ((dltr > 100) || (dltr <= 50)) {
    printf ("Threshold out of range.\n");
    return (1);
  }

  if (e || ((cnt + j) < 1) || (argc < 2)) {
    printf ("usage: dlsatest [-r]ref [options]\n");
    printf ("-r<ref>   reference address\n");
    printf ("-a<amask> address mask (%x)\n", am);
    printf ("-c<cmask> count mask (%x)\n", cm);
    printf ("-w<n>     wordsize 8, 16 or 32 (%d)\n", ws);
    printf ("-d<dmask> data mask (%x)\n", dm);
    printf ("-t<n>     threshold (%d)\n", dltr);
    printf ("-p<n>     number of passes (%d)\n", p);
    printf ("returns: number of SAFs.\n");
    fflush (stderr);
    return 1;
  }
  if ((!dm) || (p <= 0)) {
    internal_set_int ("DLSA0", 0);
    internal_set_int ("DLSA1", 0);
    if (VERBOSE > 0)
      printf ("Nothing to do.\n");
    return (0);
  }
                                             /* initialize */
  if (ws < 32)
    dm &= ~(~0 << ws);
  for (i = 0; i < 32; i++) {
    ones[i] = 0;
    cntf[i] = 0;
  }
  /* test */
  do{                           /* generate target address */
    if (am) {                        /* address line toggle */
      while (!((am >> ab) & 1)) ab++;
      a_tt = a_ref ^ (1 << ab);
      am &= ~(1 << ab);       
    }
    else
      a_tt = a_ref;
    count = 0;                                 /* counting */
    do {
      count--;
      count &= cm;
      a_t = a_tt ^ count;
      for (j = 0; j < p; j++) {                   /* passes */
                                            /* actual test */
        switch (ws) {
        case 8:
          WB(a_t,~0); dr1 = (UINT32)RB(a_t);
          WB(a_t, 0); dr0 = (UINT32)RB(a_t);
          break;
        case 16:
          WS(a_t,~0); dr1 = (UINT32)RS(a_t);
          WS(a_t, 0); dr0 = (UINT32)RS(a_t);
          break;
        default:
          W (a_t,~0); dr1 = (UINT32)R (a_t);
          W (a_t, 0); dr0 = (UINT32)R (a_t);
        }
        n += 2;
        for (i = 0; i < ws; i++) {
          ones[i]+= (1 &(dr0 >> i)) + (1 &( dr1 >> i));/* count 1 bits */
          cntf[i]+= (1 &(dr0 >> i)) + (1 &(~dr1 >> i));/* count wrong bits */
        }
      }
    }while (count);
  }while (a_tt != a_ref);
                                        /* process results */
               /* generate saf words and normalize saf cnt */
  dltr   = dltr * n;
  for (i = 0; i < ws; i++)
    if (1 & (dm >> i)) {          /* within datamask */
      if ( ((n - ones[i])* 100) >= dltr) {
        sazero |= 1 << i;        /* SA0 */
        cntf[i] = ones[i];
      }
      else{
        if ( (ones[i] * 100) >= dltr) {
          saone  |= 1 << i;       /* SA1 */
          cntf[i] = n - ones[i];
        }
        else{                 
          if (cntf[i]) {         /* no SAF but noisy */
            nv |= 1 << i;
            if ((cntf[i] << 1) > n)
              cntf[i]= n - cntf[i];
          }
        }
      }
      noise += cntf[i];
      cntf[i] = (200 * cntf[i] + n/2) / n; /* scale 0..100 */
      if ((cntf[i] == 0) && (nv & (1 << i)))
        cntf[i] = 1;
    }
    else{
      cntf[i] = 0; /* join for speck and beans */
    }
  
  e = count_ones (saone | sazero); 
  noise = (200 * noise + n * count_ones (dm)/2) / (n * count_ones (dm));
  if ((noise == 0) && nv)
    noise = 1;
  /* noise average per bit */
                                          /* store results */  
  internal_set_int ("DLSA0", sazero); /* store sazero */
  internal_set_int ("DLSA1", saone ); /* store saone */
  internal_set_int ("NOISE", noise ); /* store noise */
                                         /* report results */
  if (VERBOSE >= 1) {
    if (e)
      printf ("ERR: dataline SAF: %d error%s.\n", e, e == 1 ? "" : "s");
    else {
      printf ("OK : no dataline SAF.\n");
      if (R(a_t) & dm)
        printf ("WRN: retention fault\n");
    }
    if (noise)
      printf ("WRN: detected %d%% noise (ave).\n", noise);
  }
                                         /* print table */
  if (VERBOSE >= 2) {
    if (getintvar ("notable")) {
      if (sazero) {
        printf ("ERR: dataline SA0 D[");
        DWN(i,ws-1,0) {
          if (sazero & ( 1 << i)) {
            printf ("%d", i);
            if (sazero & ~(~0 << i)) printf (", ");
          }
        }
        printf ("]\n");
      }
      if (saone) {
        printf ("ERR: dataline SA1 D[");
        DWN(i,ws-1,0)
          if (saone & ( 1 << i)) {
            printf ("%d", i);
            if (saone & ~(~0 << i)) printf (", ");
          }
        printf ("]\n");
      }
    }else{
      if (getintvar ("wide")) {
        if ((saone | sazero | nv) & 0xffff0000) {
          printf (  "bitnr ");
          DWN(i,31,16) printf (" %2d ", i);
          printf ("\nSA0   ");
          DWN(i,31,16) printf ("  %s ", AN_X(sazero & (1 << i)));
          printf ("\nSA1   ");
          DWN(i,31,16) printf ("  %s ", AN_X(saone  & (1 << i)));
          printf ("\nnoise ");
          DWN(i,31,16) if (cntf[i])
            printf ("%3d%%", cntf[i]);
          else
            if (dm & (1 << i))
              printf ("    ");
            else
              printf (" -na");
          printf ("\n");
          if ((saone | sazero | nv) & 0xffff)
            printf ("\n");
        }
        if ((saone | sazero | nv) & 0xffff) {
          printf (  "bitnr ");
          DWN(i,15,0) printf (" %2d ", i);
          printf ("\nSA0   ");
          DWN(i,15,0) printf ("  %s ", AN_X(sazero & (1 << i)));
          printf ("\nSA1   ");
          DWN(i,15,0) printf ("  %s ", AN_X(saone  & (1 << i)));
          printf ("\nnoise ");
          DWN(i,15,0) if (cntf[i])
            printf ("%3d%%", cntf[i]);
          else
            if (dm & (1 << i))
              printf ("    ");
            else
              printf (" -na");
          printf ("\n");
        }
      }
      else
        if (e | nv) {
          printf (  "bitnr ");
          DWN(i,31,0) if (dm & (1 << i))
            printf (" %d", i / 10);
          else
            printf (" .");
          printf ("\n      ");
          DWN(i,31,0) if (dm & (1 << i))
            printf (" %d", i % 10);
          else
            printf ("  ");
          printf ("\nSA0   ");
          DWN(i,31,0) printf (" %s", AN_X(sazero & (1 << i)));
          printf ("\nSA1   ");
          DWN(i,31,0) printf (" %s", AN_X(saone  & (1 << i)));
          printf ("\nnoise ");
          DWN(i,31,0) printf (" %s", AN_X(nv     & (1 << i)));
          printf ("\n");
        }
    }
  }
                                                 /* return */
  return (e);
}

                                                     /* dataline test for BF */
int dlbftest (int argc, char **argv) {
  /* Usage: dlsatest [-r]ref [options]
           -r<ref>   reference address
           -a<amask> address mask (0)
           -c<cmask> count mask (0)
           -w<n>     wordsize (32)
           -d<dmask> data mask (ffffffff)
           -t<n>     threshold ()
           -p<n>     number of passes ()
           returns number of BFs
     */
  UINT32 a_t,        /* target address */
         a_ref = 0,  /* reference address */
         a_tt,       /* temporary target address */
         am  = 0,    /* address mask */
         cm  = 0,    /* counting mask */
         count,      /* count vector */
         dlbf = 50,  /* word with all BFs */
         dm  =~0,    /* data mask */
         fv,         /* fault vector */
         r0,         /* data read walking zero */
         r1,         /* data read walking zero */
         tv,         /* test vector */
         cf[32][32], /* cummulative fault */
         bf[32][32], /* bridging fault */
         rf[32][32]; /* recomputed fault */
  int    ab  = 0,    /* address bit */
         bftr = 50,  /* threshold */
         db,         /* data bit */
         e   = 0,    /* error */
         n   = 0,    /* location counter */
         noise = 0,  /* noise percentage */
         p   = 1,    /* passes */
         x,          /* loop cnt, flag */
         ws  = 32,   /* word-size */
         i,          /* argument passing vars*/
         j   = 0,
         cnt = 0,
         iinc = 1;
  UINT32 value;
  char   option;
  
  for (i = 1; i < argc; i += iinc) {
    if (argv[i][0] == '-') {  /* an option */
      option = argv[i][1];
      if (argv[i][2] == '\0') {
        value = myatoi (argv[i+1]);/* seperate value */
        iinc  = 2;
      }
      else{
        value = myatoi (argv[i] + 2);
        iinc  = 1;
      }
      switch (option) {
      case 'r': a_ref= value; j = 1; break;
      case 'a': am   = value; break;
      case 'c': cm   = value; break;
      case 'w': ws   = value; break;
      case 'd': dm   = value; break;
      case 't': bftr = value; break;
      case 'p': p    = value; break;
      default : iinc = 1; e = 1;
      }
    }
    else{
      cnt++;
      switch (cnt + j) {
      case 1  : a_ref= myatoi (argv[i]); break;
      case 2  : am   = myatoi (argv[i]); break;
      case 3  : cm   = myatoi (argv[i]); break;
      default : e    = 1;
      }   
    }
  }
  if ((bftr < 1) || (bftr > 100)) {
    printf ("Threshold out of range.\n");
    return (1);
  }
  if (!((ws == 32) || (ws == 16) || (ws == 8))) {
    printf ("Invalid wordsize.\n");
    return (1);
  }

  if (e || ((cnt + j) < 1) || (argc < 2)) {
    printf ("usage: dlbftest [-r]ref [options]\n");
    printf ("-r<ref>   reference address\n");
    printf ("-a<amask> address mask (%x)\n", am);
    printf ("-c<cmask> count mask (%x)\n", cm);
    printf ("-w<n>     wordsize 8, 16 or 32 (%d)\n", ws);
    printf ("-d<dmask> data mask (%x)\n", dm);
    printf ("-t<n>     threshold (%d)\n", bftr);
    printf ("-p<n>     number of passes (%d)\n", p);
    printf ("returns: number of BFs.\n");
    fflush (stderr);
    return 1;
  }
  if ((count_ones (dm) < 2) || (p <= 0)) {
    internal_set_int ("DLBF", 0);
    if (VERBOSE > 0)
      printf ("Nothing to do.\n");
    return (0);
  }
                                             /* initialize */
  e = 0;
  tv = 0;
  if (ws < 32)
    dm &= ~(~0 << ws); 
  for (i = 0; i < 32; i ++)
    for (j = 0; j < 32; j++)
      cf[i][j] = 0;
                                                   /* test */
  do {                          /* generate target address */
    if (am) {                        /* address line toggle */
      while (!((am >> ab) & 1)) ab++; 
      a_tt = a_ref ^ (1 << ab);
      am  &= ~(1 << ab);
    }else{
      a_tt = a_ref;
    }
    count = 0;                                 /* counting */
    do {
      count--;
      count &= cm;
      a_t    = a_tt ^ count;
      for (i = 0; i < p; i++) {
                                            /* actual test */
        n++;
        for (db = 0; db < ws; db++) {
          tv = (1 << db);           /* generate testvector */
          switch (ws) {
          case 8:
            WB(a_t,~tv); r1 = RB(a_t);  /* walking zero */
            WB(a_t, tv); r0 = RB(a_t);  /* walking one */
            break;
          case 16:
            WS(a_t,~tv); r1 = RS(a_t);  /* walking zero */
            WS(a_t, tv); r0 = RS(a_t);  /* walking one */
            break;
          default:
            W (a_t,~tv); r1 = R (a_t);  /* walking zero */
            W (a_t, tv); r0 = R (a_t);  /* walking one */
          }
          fv  = r0 ^  tv;         /* store faulty bits */
          fv |= r1 ^ ~tv;
          fv &= dm;               /* apply datamask */
          if (fv)
            for (j = 0; j < ws; j++)/* upd. cumm.fault matrix */
              cf[db][j] += (fv >> j) & 1;
        }
      }
    }while (count);
  }while (am);
                                        /* process results */
            /* generate bf (symmetric localisation matrix) */
  for (i = 0; i < ws; i++)
    for (j = i; j < ws; j++) {
      bf[i][j] = ((cf[i][j]*cf[j][i]*100) >= (bftr*SQR (n)));
      bf[j][i] = bf[i][j];
    }

                       /* remove redundant entries in UTbf */
  for (i = 0; i < ws; i++)
    for (j = i+1; j < ws; j++)
      if (bf[i][j])
        for (x = i + 1; x < j; x++)
          if (bf[i][x])
            bf[x][j] = 0;
  
  e = 0;                                      /* count BFs */
  for (i = 0; i < ws; i++)
    for (j = i+1, x = 0; j < ws; j++)
      if (bf[i][j])
        if (!x++)
          e++;
                                             /* report BFs */
  if (e && (VERBOSE >= 2))
    printf ("ERR: dataline BF: %d error%s.\n", e, e == 1 ? "" : "s");
  dlbf = 0;
  for (i = 0; i < ws; i++) {
    x = 0;
    for (j = i+1; j < ws; j++)
      if (bf[i][j]) {
        if (VERBOSE >= 2) 
          if (!x)
            printf ("ERR: bridging fault D[%d, %d", i, j);
          else
            printf (", %d", j);
        x = 1;
        dlbf |= (1 << j) | (1 << i);/* store */
      }
    if (x && (VERBOSE >= 2))
      printf ("]\n");
  }
  /* noise is defined as the fraction */
  /*   for LT if not bf then sum cf (noisy entries) / */
  /*   for LT if not bf then sum n (total possible noise)*/
  x = 0;
  for (i = 0; i < ws; i++)
    for (j = i+1; j < ws; j++)
      if (!bf[j][i]) {
        noise += cf[i][j];
        noise += cf[j][i];
        x += 2;
      }
  for (i = 0; i < ws; i++)
    if (!bf[i][i]) {
      noise += cf[i][i];
      x ++;
    }
  if (noise) {
    noise = (100 * noise + x*n/2) / (x * n);
    if (!noise)
      noise = 1;
  }
                                          /* store results */  
  internal_set_int ("DLBF" , dlbf );
  internal_set_int ("NOISE", noise);

                            /* report the "missing" faults */
                                        /* copy bfUT => cf */
  for (i = 0; i < ws; i++) {
    cf[i][i] = 1;
    for (j = i+1; j < ws; j++) {
      cf[i][j] = bf[i][j];
      cf[j][i] = bf[i][j];
    }
  }
                              /* compute square cf => UTrf */
  for (i = 0; i < ws; i++) {
    rf[i][i] = 1;
    for (j = i+1; j < ws; j++) {
      rf[i][j] = 0;
      for (x = 0; x < ws; x++)
        if (bf[i][x] * bf[x][j])
          rf[i][j] = 1;
    }
  }
  x = 0;                              /* compare UTrf LTbf */
  for (i = 0; i < ws; i++)
    for (j = i+1; j < ws; j++)
      if (!bf[j][i] && rf[i][j]) {
        x++;
        if (VERBOSE > 2) 
          printf ("ERR: non-BF D[%d, %d]\n", i, j );
      }
                                           /* report OK */
  if ((VERBOSE >= 1 ) && (e == 0)) {
    printf ("OK : no dataline BFs.\n");
    if ((R(a_t) ^ tv) & dm)
      printf ("WRN: retention fault\n");
  }
                                           /* report noise */
  if ((VERBOSE >= 2) && (noise))
      printf ("WRN: detected %d%% noise.\n", noise);
                                           /* report loonies */
  if ((VERBOSE >= 2) && (x))
      printf ("WRN: %d inconsistent BF%s.\n", x, x == 1 ? "" : "s");
                                           /* report count VERBOSE = 1*/
  if (VERBOSE == 1)
    if (e)
      printf ("ERR: dataline BF: %d error%s.\n", e, e == 1 ? "" : "s");

                                                 /* return */
  return (e);
}


#define CRC16_POLY 0xA001

UINT32 updcrc (UINT32 crc16, UINT32 c) {
  int i;

  c = (c & 0xff) << 8;
  for (i = 0; i < 8; i++) {
    if((crc16 ^ c) & 0x8000) 
      crc16 = (crc16 << 1) ^ CRC16_POLY;
    else 
      crc16 <<= 1;
    c <<= 1;
  }
  return (crc16 & 0xffff);
}


                                                   /* calculate crc */
int cal_crc (int argc, char **argv) {
  /* 
    Usage: crc [-r]ref [options]
           -r<ref>   reference address 
           either
             -n<n>     length
           or
             -a<amask> address mask
             -c<cmask> count mask
           -w<n>     wordsize

           returns 16 bit crc number
  */
  UINT32 a_t,        /* target address */
         a_ref = 0,  /* reference address */
         a_tt,       /* temporary target address */
         crc16 = 0,  /* 16 bit CRC */
         d   = 0,    /* data read */
         dm  = ~0,   /* data mask */
         am  = 0,    /* address mask */
         cm  = 0,    /* count mask */
         count = 0;  /* count vector */
  int    ab  = 0,    /* address bit */
         e   = 0,    /* error */
         stride = 0, /* stride */
         n_loc = 1,  /* length */
         n   = 0,    /* locations counter */
         ws  = 32,   /* word-size */
         cnt = 0,    /* argument passing vars */
         j   = 0,
         i,
         iinc = 1;
  UINT32 value;
  char   option;

                                      /* argument passing */
  for (i = 1; i < argc; i += iinc) {
    if (argv[i][0] == '-') {  /* an option */
      option = argv[i][1];
      if (argv[i][2] == '\0') {
        value = myatoi (argv[i+1]);/* seperate value */
        iinc = 2;
      }else{
        value = myatoi (argv[i]+2);
        iinc = 1;
      }
      switch (option) {
      case 'r': a_ref= value; j = 1; break;
      case 'd': dm   = value; break;
      case 'n': n_loc= value; break; 
      case 's': stride=value; break; 
      case 'a': am   = value; break;
      case 'c': cm   = value; break;
      case 'w': ws   = value; break;
      default : e    = 1; iinc = 1;
      }
    }else{
      cnt++;
      switch (cnt + j) {
      case 1  : a_ref= myatoi (argv[i]); break;
      case 2  : n_loc= myatoi (argv[i]); break;
      default : e    = 1;
      }
    }
  }

  if ((n_loc > 1) && (am || cm)) {
    printf ("ERR: specify either -n OR -a and -c.\n");
    return 1;
  }
  if (!((ws == 32) || (ws == 16) || (ws == 8))) {
    printf ("Invalid wordsize.\n");
    return (1);
  }

  if (e || ((cnt + j) < 1) || (argc < 2)) {
    printf ("usage: calcrc16 [-r]ref [-n]len [options]\n");
    printf ("-r<ref>   reference address\n");
    printf ("-n<len>   length (bites -a and -c)\n");
    printf ("-s<n>     stride (bites -a and -c)\n");
    printf ("-a<amask> address mask (bites -n and -s)\n");
    printf ("-c<cmask> count mask (bites -n and -s)\n");
    printf ("-d<value> data mask (0x%08x)\n", dm);
    printf ("-w<n>     wordsize 8, 16 or 32 (%d)\n", ws);
    printf ("returns: 16 bit crc.\n");
    fflush (stderr);
    return 1;
  }

  if (!(n_loc || cm || am) || (dm == 0)) {
    if (VERBOSE > 0)
      printf ("Nothing to do.\n");
    return 0;
  }

  if (!n_loc) {/* cm and am specified */
    n_loc = 1;
    if (cm) {
      if ((cm & ((ws / 8)- 1)) && (VERBOSE >= 2))
        printf ("WRN: ignoring cmask bits that address within word.\n");
      cm &= ~((ws / 8)- 1);
    }
  }
  else {       /* n_loc specified */
    if (!stride)
      stride = ws /8;
    if ((stride % (ws/8)) && (VERBOSE >= 2))
      printf ("WRN: stride not in wordsize units causing misalignment.\n");
    if ((n_loc % stride) && (VERBOSE >= 2))
      printf ("WRN: using length %x instead of %x to fit stride.\n",
               n_loc - n_loc % stride, n_loc);
    n_loc /= stride;
  }
  
  do{                           /* generate target address */
    if (am) {                         /* address line toggle */
      while (!((am >> ab) & 1)) ab++; 
      a_tt = a_ref ^ (1 << ab);
      am &= ~(1 << ab);       
    }else{
      a_tt = a_ref;
    }
    count = 0;                                 /* counting */
    do {
      count--;
      count &= cm;
      a_t = a_tt ^ count;
      for (i = 0; i < n_loc; i++) {           /* locations */
        switch (ws) {
        case 8:
          d = RB(a_t) & dm;
          crc16 = updcrc (crc16, d); 
          break;
        case 16:
          d = RS(a_t) & dm;
          crc16 = updcrc (crc16, d     );
          crc16 = updcrc (crc16, d >> 8);
          break;
        default:
          d = R (a_t) & dm;
          crc16 = updcrc (crc16, d      );
          crc16 = updcrc (crc16, d >> 8 );
          crc16 = updcrc (crc16, d >> 16);
          crc16 = updcrc (crc16, d >>24);
        }
        a_t += stride;
        n++;
      }
    }while (count);
  }while (a_tt != a_ref);

  if (VERBOSE > 0)
    printf("Number of locations: %d\nCRC: %04x\n", n, crc16);

  return (crc16);
}


/* end of file tests.c */
#endif 
