#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <openssl/err.h>
#include <openssl/pem.h>
#include <openssl/x509.h>
#include <openssl/x509_vfy.h>

#define NAGIOS_OK	0
#define NAGIOS_WARN	1
#define NAGIOS_CRIT	2
#define NAGIOS_UNKNOWN	4

struct opts {
	const char *cacert;
	const char *crl;
	int warn;
	int crit;
};

struct appdata {
	struct opts *opts;
	time_t now;
	char *current_certfile;
	X509 *current_cert;
	char *msg_ok;
	char *msg_warn;
	char *msg_crit;
	int status;
};

static void
usage(char *progname)
{
	printf("usage: %s -cacert cert.pem [-crl crl.pem]"
	       "[-warn time][-crit time] certificate.pem ...\n",progname);
	exit(NAGIOS_UNKNOWN);
}

static int
get_options(int argc, char **argv, struct opts *opts)
{
	char *progname = argv[0];
	int i = 0;

	for (i = 1; i < argc; i++) {
		if (argv[i][0] != '-')
			break;

		/* all options require an argument */
		if (i + 1 > argc) {
			printf("%s requires an argument", argv[i]);
			usage(progname);
		}

		if (!strcmp(argv[i], "-cacert")) {
			opts->cacert = argv[++i];
			continue;
		}
		if (!strcmp(argv[i], "-crl")) {
			opts->crl = argv[++i];
			continue;
		}
		if (!strcmp(argv[i], "-warn")) {
			opts->warn = atoi(argv[++i]);
			continue;
		}
		if (!strcmp(argv[i], "-crit")) {
			opts->crit = atoi(argv[++i]);
			continue;
		}
		printf("unknown option %s\n", argv[i]);
		usage(progname);
	}

	if (opts->cacert == NULL)
		usage(progname);

	return i;
}

static void
swap_str(int size, char *new, char **old)
{
        if (size < 0) {
		printf("UNKNOWN: asprintf() failed\n");
		exit (NAGIOS_UNKNOWN);
        }

	if (*old)
		free(*old);

	*old = new;
}

static int 
verify_cb(int ok, X509_STORE_CTX *ctx) {
	int cert_error = X509_STORE_CTX_get_error(ctx);
	X509 *current_cert = X509_STORE_CTX_get_current_cert(ctx);
	struct appdata *appdata = X509_STORE_CTX_get_app_data(ctx);
	char buf[512];
	char *certname;
	char *msg;
	int ret;

	if (appdata == NULL) {
		printf("UNKNOWN: NULL appdata\n");
		exit (NAGIOS_UNKNOWN);
	}

	if (current_cert == appdata->current_cert) {
		certname = appdata->current_certfile;
	} else {
		X509_NAME *name = X509_get_subject_name(current_cert);
		if (name)
			certname = X509_NAME_oneline(name, buf, sizeof(buf));
		else
			certname = "(?)";
	}

	if (current_cert == NULL) {
		ret = asprintf(&msg, "%s%s %s unavailable in %s",
			       appdata->msg_crit ? appdata->msg_crit : "",
			       appdata->msg_crit ? "," : "", 
 			       appdata->current_certfile, __func__);
		swap_str(ret, msg, &appdata->msg_crit);
		appdata->status = NAGIOS_CRIT;	
	}

	if (ok) {
		struct tm tm;
		ASN1_TIME_to_tm(X509_get0_notAfter(current_cert), &tm);
		int days_left = (int)(mktime(&tm) - appdata->now) / 8640;

		if (days_left < appdata->opts->crit) {
			ret = asprintf(&msg, "%s%s %s expire in %d days",
			    appdata->msg_crit ? appdata->msg_crit : "",
			    appdata->msg_crit[0] ? "," : "", 
			    certname, days_left);
			swap_str(ret, msg, &appdata->msg_crit);
			appdata->status = NAGIOS_CRIT;
		} else if (days_left < appdata->opts->warn) {
			ret = asprintf(&msg, "%s%s %s expire in %d days",
			    appdata->msg_warn ? appdata->msg_warn : "",
			    appdata->msg_warn ? "," : "", 
			    certname, days_left);
			swap_str(ret, msg, &appdata->msg_warn);
			if (appdata->status != NAGIOS_CRIT)
				appdata->status = NAGIOS_WARN;
		} else {
			ret = asprintf(&msg, "%s%s %s expire in %d days",
			    appdata->msg_ok ? appdata->msg_ok : "",
			    appdata->msg_ok ? "," : "", 
			    certname, days_left);
			swap_str(ret, msg, &appdata->msg_ok);
		}

		return ok;
	}

	ret = asprintf(&msg, "%s%s %s %s",
		       appdata->msg_crit ? appdata->msg_crit : "",
		       appdata->msg_crit ? "," : "", 
		       certname,
		       X509_verify_cert_error_string(cert_error));
	swap_str(ret, msg, &appdata->msg_crit);
	appdata->status = NAGIOS_CRIT;

	return ok;
}


int
main(int argc, char **argv) {
	char *progname = argv[0];
	struct opts opts = { NULL, NULL, 7, 3 };
	struct appdata appdata =
	    { &opts, time(NULL), NULL, NULL, NULL, NULL, NULL, NAGIOS_OK };
	int opts_taken;
	FILE *fp;
	X509 *cacert;
	X509_CRL *crl;
	int i;
	X509 *cert = NULL;
	X509_STORE *store;
	X509_STORE_CTX *ctx;

	opts_taken = get_options(argc, argv, &opts);
	argc -= opts_taken;
	argv += opts_taken;

	if (argc < 1)
		usage(progname);

	if ((store = X509_STORE_new()) == NULL) {
                printf("CRITICAL X509_STORE_new() failed\n");
                exit(NAGIOS_CRIT);
	}

	if ((fp = fopen(opts.cacert, "r")) == NULL) {
                printf("CRITICAL cannot open %s\n",argv[argc - 1]);
                exit(NAGIOS_CRIT);
	}

	if ((cacert = PEM_read_X509(fp, NULL, NULL, NULL)) == NULL) {
        	printf("CRITICAL PEM_read_X509(%s) failed\n",opts.cacert);
               	exit(NAGIOS_CRIT);
	}

	(void)fclose(fp);

	if (!X509_STORE_add_cert(store, cacert)) {
                printf("CRITICAL X509_STORE_add_cert() failed\n");
                exit(NAGIOS_CRIT);
	}

	if (opts.crl) {
		if ((fp = fopen(opts.crl, "r")) == NULL) {
			printf("CRITICAL cannot open CRL %s\n",opts.crl);
			exit(NAGIOS_CRIT);
		}

		if ((crl = PEM_read_X509_CRL(fp, NULL, NULL, NULL)) == NULL) {
			printf("CRITICAL PEM_read_X509_CRM() failed\n");
			exit(NAGIOS_CRIT);
		}

		(void)fclose(fp);

		if (!X509_STORE_add_crl(store, crl)) {
			printf("CRITICAL X509_STORE_add_crl() failed\n");
			exit(NAGIOS_CRIT);
		}

		X509_STORE_set_flags(store, X509_V_FLAG_CRL_CHECK|
					    X509_V_FLAG_CRL_CHECK_ALL);
	}

	X509_STORE_set_verify_cb(store, verify_cb);

	if ((ctx = X509_STORE_CTX_new()) == NULL) {
                printf("CRITICAL X509_STORE_CTX_new() failed\n");
                exit(NAGIOS_CRIT);
	}
	
	for (i = 0; i < argc; i++) {
		FILE *fp;

		if ((fp = fopen(argv[i], "r")) == NULL) {
               		printf("CRITICAL fopen(%s) failed\n",argv[i]);
                	exit(NAGIOS_CRIT);
		}

		if ((cert = PEM_read_X509(fp, NULL, NULL, NULL)) == NULL) {
               		printf("CRITICAL PEM_read_X509(%s) failed\n",argv[i]);
                	exit(NAGIOS_CRIT);
		}

		X509_STORE_CTX_init(ctx, store, cert, NULL);
		if (ctx == NULL) {
               		printf("CRITICAL X509_STORE_CTX_init() failed\n");
                	exit(NAGIOS_CRIT);
		}

		appdata.current_certfile = argv[i];
		appdata.current_cert = cert;
		X509_STORE_CTX_set_app_data(ctx, &appdata);

		/* Ignore result, work is already done in callback */
		(void)X509_verify_cert(ctx);
        }

	switch(appdata.status) {
	case NAGIOS_OK:
		printf("OK%s\n",appdata.msg_ok);
		break;
	case NAGIOS_WARN:
		printf("WARNING%s\n",appdata.msg_warn);
		break;
	case NAGIOS_CRIT:
		printf("CRITICAL%s\n",appdata.msg_crit);
		break;
	default:
		printf("UNKNOWN: unexpected status %d",appdata.status);
		exit(NAGIOS_UNKNOWN);
		break;
	}
		
	return appdata.status;
}
