/*
 * Copyright (c) 2013,2018 Emmanuel Dreyfus
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *        This product includes software developed by Emmanuel Dreyfus
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <err.h>
#include <sysexits.h>
#include <ctype.h>
#include <ldap.h>
#include <time.h>

#ifndef STATE_OK
#define STATE_OK        0
#define STATE_WARNING   1
#define STATE_CRITICAL  2
#define STATE_UNKNOWN   3
#endif  

int
ld_errno(ld)
	LDAP *ld;
{
	return 0;
}

static char *
get_contextcsn(url, base)
	char *url;
	char *base;
{
	LDAP *ldp;
	int version;
	char *attrs[] = { "ContextCSN", NULL }; 
	LDAPMessage *res;
	BerElement *ber;
	struct berval bv;
	struct berval *bvals;
	struct berval **bvp = &bvals;
	int rc;
	

	if (ldap_initialize(&ldp, url) != 0)
		err(EX_OSERR, "LDAP initialization error for %s", url);

	version = 3;
	rc = ldap_set_option(ldp, LDAP_OPT_PROTOCOL_VERSION, &version);
	if (rc) { 
		printf("LDAP version error for %s: %s\n", url, 
		       ldap_err2string(rc));
		exit(STATE_WARNING);
	}	


	rc = ldap_simple_bind_s(ldp, NULL, NULL);
	if (rc) { 
		printf("CRTICAL: %s unreachable: %s\n", url,
		       ldap_err2string(rc));
		exit(STATE_CRITICAL);
	}

	rc = ldap_search_ext_s(ldp, base, LDAP_SCOPE_BASE, NULL, 
			       attrs, 0, NULL, NULL, NULL, 0, &res);
	if (rc) { 
		printf("CRITICAL: search failed in %s, base %s: %s\n",
		       url, base, ldap_err2string(rc));
		exit(STATE_CRITICAL);
	}

	if (ldap_count_entries(ldp, res) != 1) {
		printf("WARNING: Unexpected ContextCSN count in %s, base %s\n",
		       url, base);
		exit(STATE_WARNING);
	}

	rc = ldap_get_dn_ber(ldp, res, &ber, &bv);
	if (rc) { 
		printf("WARNING: Cannot get root DN in %s, base %s: %s\n",
		       url, base, ldap_err2string(rc));
		exit(STATE_WARNING);
	}

	rc = ldap_get_attribute_ber(ldp, res, ber, &bv, bvp);
	if (rc) { 
		printf("WARNING: Cannot get ContextCSN in %s, base %s: %s\n",
		       url, base, ldap_err2string(rc));
		exit(STATE_WARNING);
	}

	if (bv.bv_val == NULL) {
		printf("WARNING: no ContextCSN attr in %s, base %s\n",
		       url, base);
		exit(STATE_WARNING);
	}

	if (bvals[0].bv_val == NULL) {
		printf("WARNING: no ContextCSN attr in %s, base %s\n",
		       url, base);
		exit(STATE_WARNING);
	}

	if (bvals[1].bv_val != NULL) {
		printf("WARNING: multiple ContextCSN attr in %s, base %s\n",
		       url, base);
		exit(STATE_WARNING);
	}

	return bvals[0].bv_val;
}
		

static time_t
csn_to_ts(const char *csn)
{
	char *buf;
	char *cp;
	struct tm tm;

	if ((buf = strdup(csn)) == NULL) {
		printf("WARNING: strdup failed");
		exit(STATE_WARNING);
	}

	if ((cp = strchr(buf, (int)'.')) == NULL) {
		printf("WARNING: unexpected CSN format \"%s\"", csn);
		exit(STATE_WARNING);
	}

	*cp = '\0';

	memset(&tm, 0, sizeof(tm));
	if (strptime(buf, "%Y%m%d%H%M%S", &tm) == NULL) {
		printf("WARNING: bad date \"%s\"", buf);
		exit(STATE_WARNING);
	}

	return mktime(&tm);
}

int
main(argc, argv)
	int argc;
	char **argv;
{
	const char *progname = argv[0];
	extern char *optarg;
	extern int optind;
	int ch;
	char *base = NULL;
	char *master = NULL;
	char *replica = NULL;
	int delta_sec = 0;
	char *mcsn;
	char *rcsn;
	
	while ((ch = getopt(argc, argv, "b:m:r:s:h")) != -1) {
		switch (ch) {
		case 'b':
			base = optarg;
			break;
		case 'm':
			master = optarg;
			break;
		case 'r':
			replica = optarg;
			break;
		case 's':
			delta_sec = atoi(optarg);
			break;
		case 'h':
		default:
			break;
		}
	}

	argc -= optind;
	argv += optind;

	/*
	 * Master
	 */
	if (!master || !replica || !base) 
		errx(EX_USAGE, "%s -m master -r replica -b base "
			       " [-s delta_sec]", progname);

	mcsn = get_contextcsn(master, base);
	rcsn = get_contextcsn(replica, base);

	if (delta_sec == 0) {
		if (strcmp(mcsn, rcsn) != 0) {
			printf("CRITICAL: master ContextCSN = %s, "
			       "replica ContextCSN = %s\n", mcsn, rcsn);
			return STATE_CRITICAL;	
		}
	} else {
		if (abs(csn_to_ts(mcsn) - csn_to_ts(rcsn)) > delta_sec) {
			printf("CRITICAL: master ContextCSN = %s, "
			       "replica ContextCSN = %s\n", mcsn, rcsn);
			return STATE_CRITICAL;	
		}
	}

	printf("OK; contextCSN = %s\n", mcsn);
	return STATE_OK;
}
