#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <stdarg.h>
#include <errno.h>
#include <time.h>
#include <sys/uio.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <arpa/inet.h>

#include <libmilter/mfapi.h>

#ifndef SUN_LEN
#define SUN_LEN(su)	sizeof(struct(sockaddr_un))
#endif

#ifndef STATE_OK
#define STATE_OK        0
#define STATE_WARNING   1
#define STATE_CRITICAL  2
#define STATE_UNKNOWN   3
#endif

typedef uint32_t mi_int32;

static void
fatal(int code, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	vprintf(fmt, ap);
	exit(code);
}

static void
alarm_handler(sig)
	int sig;
{
	fatal(STATE_CRITICAL, "CRITICAL: socket timeout\n");
	return;
}

static void
setup_alarm(timeout)
	unsigned int timeout;
{
	if (signal(SIGALRM, alarm_handler) != 0)
		fatal(STATE_UNKNOWN, "UNKNOWN: signal() failed: %s\n",
		      strerror(errno));
	(void)alarm(timeout);

	return;
}

static int
setup_socket(path)
	char *path;
{
	int s;
	size_t len;
	struct sockaddr_un sun;

	if ((len = strlen(path)) >= sizeof(sun.sun_path))
		fatal(STATE_UNKNOWN, "UNKOWN: socket patch too long: \"%s\"\n",
		      path);

	sun.sun_len = SUN_LEN(&sun);
	sun.sun_family = AF_LOCAL;
	strcpy(sun.sun_path, path);

	if ((s = socket(PF_LOCAL, SOCK_STREAM, 0)) == -1)
		fatal(STATE_UNKNOWN, "UNKNOWN: socket() failed: %s\n",
		      strerror(errno));

	if (connect(s, (struct sockaddr *)&sun, sizeof(sun)) == -1)
		fatal(STATE_CRITICAL, "CRITICAL: cannot connect %s: %s\n",
		      path, strerror(errno));

	return s;
}

static void
send_cmd(s, path)
	int s;
	char *path;
{
	mi_int32 mta_prot_vers, mta_prot_flags, mta_prot_actions;
	char data[MILTER_OPTLEN];
	mi_int32 nl;
	struct iovec iov[2];
	char header[MILTER_LEN_BYTES + 1];
	char *dp;
	ssize_t written;

	mta_prot_vers = htonl(SMFI_PROT_VERSION);
	mta_prot_flags = htonl(SMFI_CURR_PROT);
	mta_prot_actions = htonl(SMFI_CURR_ACTS);

	/*
	 * Connexion payload
	 */
	dp = data;

	(void)memcpy(dp, &mta_prot_vers, MILTER_LEN_BYTES);
	dp += MILTER_LEN_BYTES;

	(void)memcpy(dp, &mta_prot_flags, MILTER_LEN_BYTES);
	dp += MILTER_LEN_BYTES;

	(void)memcpy(dp, &mta_prot_actions, MILTER_LEN_BYTES);
	dp += MILTER_LEN_BYTES;

	/*
	 * Connexion header
	 */
	nl = htonl(sizeof(data) + 1);
	(void)memcpy(header, &nl, MILTER_LEN_BYTES);
	header[MILTER_LEN_BYTES] = SMFIC_OPTNEG;

	iov[0].iov_base = header;
	iov[0].iov_len = sizeof(header);

	iov[1].iov_base = data;
	iov[1].iov_len = sizeof(data);
	
	written = writev(s, iov, sizeof(iov) / sizeof(*iov));
	if (written == -1)
		fatal(STATE_CRITICAL, "CRITICAL: write failed on %s: %s\n", 
		      path, strerror(errno));
	else if (written != iov[0].iov_len + iov[1].iov_len)
		fatal(STATE_CRITICAL, "CRITICAL: short write %d/%d on %s\n", 
		      written, iov[0].iov_len + iov[1].iov_len, path);

	return;
}

static void
recv_rep(s, path)
	int s;
	char *path;
{
	char header[MILTER_LEN_BYTES + 1];
	ssize_t readen;
	char cmd;

	readen = read(s, header, sizeof(header));

	/* disabla alarm */
	alarm(0);

	if (readen == -1)
		fatal(STATE_CRITICAL, "CRITICAL: read failed on %s: %s\n",
		      path, strerror(errno));
	else if (readen != sizeof(header))
		fatal(STATE_CRITICAL, "CRITICAL: short read %d/%d on %s\n",
		      readen, sizeof(header), path);
	
	cmd = header[MILTER_LEN_BYTES];

	if (cmd != SMFIC_OPTNEG)
		fatal(STATE_CRITICAL, "CRITICAL: protocol error, 0x%x/0x%x\n",
		     cmd, SMFIC_OPTNEG);

	return;
}


int
main(argc, argv)
	int argc;
	char **argv;
{
	int s;
	int ch;
	struct timeval err = { 1, 0 };
	struct timeval warn = { 2, 0 };
	struct timeval start;
	struct timeval end;
	struct timeval delay;
	char *path = NULL;
	
	while((ch = getopt(argc, argv, "w:c:p:?")) != -1) {
		switch (ch) {
		case 'w':
			warn.tv_sec = atoi(optarg);
			break;
		case 'c':
			err.tv_sec = atoi(optarg);
			break;
		case 'p':
			if (path != NULL)
				fatal(STATE_UNKNOWN,
				      "-p may be used only once\n");
			path = optarg;
			break;
		case '?':
		default:
			fatal(STATE_UNKNOWN, 
			      "usage: check_milter [-w warn] [-c crit] "
			      "-p socket\n");
			break;
		}
	
		argc -= optind;
		argv += optind;
	}

	if (path == NULL) 
		fatal(STATE_UNKNOWN, "No socket given, please use -p\n");

	setup_alarm(err.tv_sec);

	if (gettimeofday(&start, NULL) != 0)
		fatal(STATE_UNKNOWN, "UNKNOWN: gettimeofday() failed\n");

	s = setup_socket(path);
	send_cmd(s, path);
	recv_rep(s, path);
	(void)close(s);

	if (gettimeofday(&end, NULL) != 0)
		fatal(STATE_UNKNOWN, "UNKNOWN: gettimeofday() failed\n");

	timersub(&end, &start, &delay);

	if (timercmp(&delay, &err, >))
		fatal(STATE_CRITICAL, "CRITICAL: %ld.%06lds\n",
		      (unsigned long)delay.tv_sec,
		      (unsigned long)delay.tv_usec);
	else if (timercmp(&delay, &warn, >))
		fatal(STATE_WARNING, "WARNING: %ld.%06lds\n",
		      (unsigned long)delay.tv_sec,
		      (unsigned long)delay.tv_usec);

	printf("OK: %ld.%06lds\n",
	       (unsigned long)delay.tv_sec,
	       (unsigned long)delay.tv_usec);

	return STATE_OK;
}
	
