/*
 * Copyright (c) 2022 Emmanuel Dreyfus
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *        This product includes software developed by Emmanuel Dreyfus
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <dirent.h>
#include <err.h>
#include <fcntl.h>
#include <paths.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/param.h>

#define NAGIOS_OK	0
#define NAGIOS_WARN	1
#define NAGIOS_CRIT	2
#define NAGIOS_UNKNOWN	3

#define DEFAULT_WARN 3
#define DEFAULT_CRIT 10


int
main(int argc, char **argv)
{
	int ch;
	int warn = DEFAULT_WARN;
	int crit = DEFAULT_CRIT;
	char *bp = NULL;
	const char *db_array[] = { _PATH_PRINTCAP, NULL };
	char ok_str[512] = "";
	size_t ok_strlen = 0;
	char bad_str[512] = "";
	size_t bad_strlen = 0;
	int retval = NAGIOS_OK;

	while ((ch = getopt(argc, argv, "c:w:")) != -1) {
		switch (ch) {
		case 'c':
			crit = atoi(optarg);;
			break;
		case 'w':
			warn = atoi(optarg);;
			break;
		default:
			errx(NAGIOS_UNKNOWN,
			     "usage: check_lpd [-w warn] [-c crit]");
			break;
		}
	}

	while (cgetnext(&bp, db_array) > 0) {
		char *lpname = NULL;
		char *sd = NULL;
		size_t lpnamelen;
		char statuspath[MAXPATHLEN + 1];
		int statusfd;
		char status[512];
		DIR *queue;
		struct dirent *dp;
		int queuelen = 0; 

		lpnamelen = strcspn(bp, ":|");
		if ((lpname = strndup(bp, lpnamelen)) == NULL)
			err(NAGIOS_UNKNOWN, "strdup failed");

		if (cgetstr(bp, "sd", &sd) <= 0)
			err(NAGIOS_UNKNOWN, "cgetstr failed");

		(void)snprintf(statuspath, MAXPATHLEN, "%s/status", sd);
		if ((statusfd = open(statuspath, O_RDONLY)) == -1)
			err(NAGIOS_UNKNOWN, "open %s failed", statuspath);

		if (read(statusfd, status, sizeof(status)) == -1)
			err(NAGIOS_UNKNOWN, "read from %s failed", statuspath);

		if (strstr(status, "is ready and printing") == NULL) {
			bad_strlen -= snprintf(bad_str + bad_strlen, 
					       sizeof(bad_strlen) - bad_strlen,
					       "%s%s",
					       bad_strlen ? ", " : "",
					       status);
			retval = NAGIOS_CRIT;
		}

		if (statusfd != -1)
			close(statusfd);

		if ((queue = opendir(sd)) == NULL) 
			err(NAGIOS_UNKNOWN, "opendir %s failed", sd);

		while ((dp = readdir(queue)) != NULL) {
			if (dp->d_namlen < 1)
				continue;
			if (dp->d_name[0] == 'd')
				queuelen++;
		}

		if (queuelen >= crit && retval < NAGIOS_CRIT)
			retval = NAGIOS_CRIT;
		else if (queuelen >= warn && retval < NAGIOS_WARN)
			retval = NAGIOS_WARN;

		if (queuelen >= warn || queuelen >= crit) {
			bad_strlen += snprintf(bad_str + bad_strlen,
					       sizeof(bad_str) - bad_strlen,
					       "%s%s has %d jobs pending",
					       bad_strlen ? ", " : "",
					       lpname, queuelen);
		} else {
			ok_strlen += snprintf(ok_str + ok_strlen,
					       sizeof(ok_str) - ok_strlen,
					       "%s%s has %d jobs pending",
					       ok_strlen ? ", " : "",
					       lpname, queuelen);
		}

		if (queue != NULL)
			closedir(queue);
	
		if (sd)
			free(sd);
		if (lpname)
			free(lpname);

		free(bp);
	}

	(void)cgetclose();

	if (retval == NAGIOS_OK) {
		printf("OK: %s\n", ok_str); 
	} else {
		printf("%s: %s\n", 
		       retval == NAGIOS_CRIT ? "CRITICAL" : "WARNING",
		       bad_str);
	}

	return retval;
}
