/*
 * Copyright (c) 2014 Emmanuel Dreyfus
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *        This product includes software developed by Emmanuel Dreyfus
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fstab.h>
#include <limits.h>
#include <err.h>
#include <sys/types.h>
#include <sys/statvfs.h>
#include <sys/swap.h>

#ifndef STATE_OK
#define STATE_OK        0
#define STATE_WARNING   1
#define STATE_CRITICAL  2
#define STATE_UNKNOWN   3
#endif

#ifdef __NetBSD__
typedef struct statvfs statvfs_t;
typedef struct swapent swapent_t;
#endif

#ifdef linux
#define _VFS_MNAMELEN   1024
#define ST_NOWAIT 0

typedef struct {
	char 	f_mnttoname[_VFS_MNAMELEN];
	char 	f_mntfromname[_VFS_MNAMELEN];
} statvfs_t;

#define SWAP_NSWAP	3
#define SWAP_STATS	10

typedef struct {
	char 	se_path[PATH_MAX + 1];
} swapent_t;

int
getmntinfo(mntbufp, flags)
	statvfs_t **mntbufp;
	int flags;
{
	FILE *mounts;
	int lcount = 0;
	statvfs_t *st = NULL;
	int ret;

	if ((mounts = fopen("/proc/mounts", "r")) == NULL) 
		err(STATE_UNKNOWN, "cannot open /proc/mounts");

	while (!feof(mounts)) {
		if ((st = realloc(st, sizeof(*st) * (lcount + 1))) == NULL)
			err(STATE_UNKNOWN, "out of memory (count = %d)",	
			    lcount + 1);

		ret = fscanf(mounts, "%s%s%*s%*s%*s%*s", 
			     st[lcount].f_mntfromname, st[lcount].f_mnttoname);
		if (ret != 2)
			break;

		lcount++;
	}
	
	fclose(mounts);
	*mntbufp = st;

	return lcount;
}

int
swapctl(cmd, arg, misc)
	int cmd;
	void *arg;
	int misc;
{
	FILE *swaps;
	int lcount = 0;
	swapent_t *sep = (swapent_t *)arg;
	swapent_t se;
	char i;
	int ret;

	if (cmd != SWAP_NSWAP && cmd != SWAP_STATS)
		err(STATE_UNKNOWN, "unimplemented swapctl cmd = %d", cmd);

	if ((swaps = fopen("/proc/swaps", "r")) == NULL) 
		err(STATE_UNKNOWN, "cannot open /proc/swaps");

	while (!feof(swaps)) {
		if (cmd == SWAP_STATS) {
			if (lcount > misc)
				goto out;
			ret = fscanf(swaps, "%s%*s%*s%*s%*s",
				     sep[lcount].se_path);
			if (ret != 1)
				break;
			if (sep[lcount].se_path[0] == '/')
				lcount++;
		} else {
			ret = fscanf(swaps, "%s%*s%*s%*s%*s", se.se_path);
			if (ret != 1)
				break;
			if (se.se_path[0] == '/')
				lcount++;
		}
	}

out:
	fclose(swaps);

	return lcount;
}

#endif /* linux */

static int
has_noauto(mntopts)
	char *mntopts;
{
	char *cp;

	for (cp = strtok(mntopts, ","); cp; cp = strtok(NULL, ",")) {
		if (strcmp(cp, "noauto") == 0)
			return 1;
	}

	return 0;
}

static int
is_mounted(fs_spec, fs_file, mntbuf, mntcount)
	const char *fs_spec;
	const char *fs_file;
	statvfs_t *mntbuf;
	int mntcount;
{
	int i;

	for (i = 0; i < mntcount; i++) {
		if (strcmp(mntbuf[i].f_mntfromname, fs_spec) == 0)
			return 1;
#ifdef linux
		if (strcmp(mntbuf[i].f_mnttoname, fs_file) == 0)
			return 1;
#endif
#ifdef __NetBSD__
		if (strcmp(mntbuf[i].f_mntonname, fs_file) == 0)
			return 1;
#endif
	}

	return 0;
}

static int
is_swap(fs_spec, sebuf, secount)
	const char *fs_spec;
	swapent_t *sebuf;
	int secount;
{
	int i;

	for (i = 0; i < secount; i++) {
		if (strcmp(sebuf[i].se_path, fs_spec) == 0)
			return 1;
	}

	return 0;
}

static char *
append_str(str, append)
	char *str;
	char *append;
{
	size_t len;
	char *ostr;

	if (str == NULL) {
		if ((str = strdup(append)) == NULL)
			err(STATE_UNKNOWN, "strdup failed");
		return str;
	}

	len = strlen(str) + 1 + strlen(append) + 1;
	if ((str = realloc(str, len)) == NULL)
		err(STATE_UNKNOWN, "realloc failed");
	
	if ((ostr = strdup(str)) == NULL)
		err(STATE_UNKNOWN, "strdup failed");;

	(void)sprintf(str, "%s %s", ostr, append);

	free(ostr);

	return str;
}


int
main(argc, argv)
	int argc;
	char **argv;
{
	int mnt_used;
	int mnt_defined;
	statvfs_t *mntbuf;
	int swap_used;
	int swap_defined;
	swapent_t *sebuf;
	struct fstab *fs;
	statvfs_t *vfs;
	char *missing = NULL;
	char *correct = NULL;
	char *all = NULL;
	int i;

	/*
	 * Get mounts
	 */
	if ((mnt_used = getmntinfo(&mntbuf, ST_NOWAIT)) == 0)
		err(STATE_UNKNOWN, "cannot list mounts");

	/*
	 * Get swap devices
	 */
	swap_used = swapctl(SWAP_NSWAP, NULL, 0);
	if ((sebuf = malloc(sizeof(*sebuf) * swap_used)) == NULL)
		err(STATE_UNKNOWN, "malloc failed");

	if (swapctl(SWAP_STATS, sebuf, swap_used) != swap_used)
		errx(STATE_UNKNOWN, "swap_used miss %d", swap_used);

	/*
	 * read fstab
	 */
	swap_defined = 0;
	mnt_defined = 0;
	while ((fs = getfsent()) != NULL) {
		/*
		 * Swap entries
		 */
		if (strcmp(fs->fs_type, FSTAB_SW) == 0) {
			swap_defined++;
			if (!is_swap(fs->fs_spec, sebuf, swap_used)) 
				missing = append_str(missing, fs->fs_spec);
			else
				correct = append_str(correct, fs->fs_spec);
			all = append_str(all, fs->fs_spec);
		/*
		 * "Readable" entries
		 */
		} else if ((strcmp(fs->fs_type, FSTAB_RW) == 0) ||
			   (strcmp(fs->fs_type, FSTAB_RO) == 0) ||
			   (strcmp(fs->fs_type, FSTAB_RQ) == 0)) {
			/*
			 * Ignore mounts with -o noauto
			 */
			if (has_noauto(fs->fs_mntops))
				continue;

			mnt_defined++;

			if (!is_mounted(fs->fs_spec, fs->fs_file, 
					mntbuf, mnt_used)) 
				missing = append_str(missing, fs->fs_file);
			else
				correct = append_str(correct, fs->fs_file);
			all = append_str(all, fs->fs_spec);
		}
	}

	/*
	 * On Linux with UUID swaps, we cannot easily discover
	 * what devices are defined (we would need to read the
	 * superblock), but if the count is correct, we know
	 * we can return OK. Otherwise we are likely to report
	 * failure for all swaps.
	 */
	if (swap_used >= swap_defined && mnt_used >= mnt_defined) {
		printf("OK: %s\n", all); 
		return STATE_OK;
	}

	if (missing != NULL) {
		printf("CRITICAL: %s\n", missing);
		return STATE_CRITICAL;
	}

	printf("OK: %s\n", correct); 
	return STATE_OK;
}
