/*
 *  cambevao - writes jpeg images captured via bktr(4) or ov511+/ov7620
 *				based USB webcams via ugen(4) to a file,
 *				optionally serves static or streaming images via
 *				built-in mini webserver
 */

/* Copyright (c) 1999, 2000, 2001, 2002, 2003, 2004 Thomas Runge (coto@core.de)
 *
 * Partly based on libjpeg.doc by the Independent JPEG Group
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <sys/param.h>
#include <sys/time.h>
#include <fcntl.h>
#include <signal.h>
#include <errno.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <ctype.h>
#include <netdb.h>
#include <time.h>
#include <syslog.h>
#ifdef __NetBSD__
#include <util.h>
#else
#include "util.h"
#endif
#include "main.h"
#include "server.h"
#include "cams.h"
#include "log.h"
#include "font.h"
#include "prefs.h"

/* how much to dim */
#define DIMFAC .6

/* text offset */
#define OFFSET 1

/* max jpeg marker length */
#define JPEG_MARKER_LENGTH 65533
#define JPEG_MSG "written by "PROGRAM" by Thomas Runge (http://core.de/~coto/projects/"PROGRAM"/), version "VERSION

/* how many lines around the text */
#define BORDER 4

/* default sleep time, if none specified */
#define DEFSLEEP 30

static int keep_running;
static int is_daemon;
static int do_reset;

struct webcam_data *ccam;

void error(char *prefix, char *what)
{
	if(is_daemon)
		syslog(LOG_ERR, "%s: %s: %s\n", getprogname(), prefix, what);
	else
		dlog(__FILE__, "%s: %s: %s\n", getprogname(), prefix, what);
	exit(EXIT_FAILURE);
}

static void sigpipe(int sig)
{
	/* ignore */
}

static void sigint(int sig)
{
	dlog(__FILE__, "catched SIGINT, stopping, cleaning up.\n");
	keep_running = FALSE;
}

static void jpeg_init(struct webcam *cam)
{
	cam->cinfo.err = jpeg_std_error(&cam->jerr);
	jpeg_create_compress(&cam->cinfo);

	cam->cinfo.image_width  = cam->cols;
	cam->cinfo.image_height = cam->rows;
	cam->cinfo.input_components = cam->jpg_ncomp;
	cam->cinfo.in_color_space = cam->jpg_format;

	jpeg_set_defaults(&cam->cinfo);

	jpeg_set_quality(&cam->cinfo, cam->prefs->quality, TRUE);
}

static void jpeg_encode(struct webcam *cam)
{
	FILE *fp = stdout;
	JSAMPROW row_pointer[1];
	char marker[JPEG_MARKER_LENGTH];
	int i, dim, mline;
	time_t tp;
	struct tm *lt;
	char *cp;

	tp = time(NULL);
	lt = localtime(&tp);

	if(cam->prefs->filename)
	{
		/* make filename with timestamp */
		snprintf(cam->tfilename, FILENAME_MAX,
				"%s_%04d%02d%02d%02d%02d%02d", cam->prefs->filename,
				lt->tm_year+1900, lt->tm_mon+1, lt->tm_mday,
				lt->tm_hour, lt->tm_min, lt->tm_sec);

		if(cam->prefs->verbose)
			dlog(__FILE__, "opening tempfile \"%s\"\n", cam->tfilename);
		if((fp = fopen(cam->tfilename, "wb")) == NULL)
			error("fopen", strerror(errno));
	}

	dim = mline = 0;
	cp = asctime(lt);
	cp[strlen(cp)-1] = '\0';
	if(strlen(cp) > cam->max_msg)
		cp[cam->max_msg-1] = '\0';

	jpeg_stdio_dest(&cam->cinfo, fp);

	jpeg_start_compress(&cam->cinfo, TRUE);

	dim = cam->cinfo.image_height - (cam->fontheight + 2*BORDER);
	snprintf(marker, JPEG_MARKER_LENGTH, "%s (%s)", JPEG_MSG, cp);

	if(cam->prefs->verbose)
		dlog(__FILE__, "marker: %s\n", marker);

	jpeg_write_marker(&cam->cinfo, JPEG_COM, marker, strlen(marker));

	while(cam->cinfo.next_scanline < cam->cinfo.image_height)
	{
		ccam->capture_getline(cam, cam->cinfo.next_scanline);
		if(cam->msg != NULL)
		{
			if(cam->cinfo.next_scanline >= dim)
			{
				unsigned char *p = cam->copybuf;

				/* dim image */
				for(i = cam->cols; i; i--)
				{
					if(cam->jpg_format == JCS_YCbCr)
					{
						*p *= DIMFAC;
						p+=3;
					}
					else
					{
						*p++ *= DIMFAC;
						*p++ *= DIMFAC;
						*p++ *= DIMFAC;
					}
				}

				/* write message */
				if(	(cam->cinfo.next_scanline > (dim + BORDER)) &&
					(cam->cinfo.next_scanline <= (dim + BORDER + cam->fontheight)))
				{
					int j, k, len, ts;
					unsigned char b;

					len = strlen(cp);
					ts = cam->max_msg - len - OFFSET;
					p = cam->copybuf + ts * cam->fontwidth * 3;
					for(j = 0; j < len; j++)
					{
						b = cam->font[(cp[j] * cam->fontheight) + mline];
						for(k = 128; k > 0; k >>= 1)
						{
							if(b & k)
							{
								p[0] = 0xFF;
								p[1] = cam->jpg_format == JCS_RGB ? 0xFF : 0x80;
								p[2] = cam->jpg_format == JCS_RGB ? 0xFF : 0x80;
							}
							p += 3;
						}
					}
					p = cam->copybuf + OFFSET * cam->fontwidth * 3;
					len = MIN(strlen(cam->msg), ts-1-OFFSET);
					for(j = 0; j < len; j++)
					{
						b = cam->font[(cam->msg[j] * cam->fontheight) + mline];
						for(k = 128; k > 0; k >>= 1)
						{
							if(b & k)
							{
								p[0] = 0xFF;
								p[1] = cam->jpg_format == JCS_RGB ? 0xFF : 0x80;
								p[2] = cam->jpg_format == JCS_RGB ? 0xFF : 0x80;
							}
							p += 3;
						}
					}
					mline++;
				}
			}
		}
		row_pointer[0] = cam->copybuf;
		jpeg_write_scanlines(&cam->cinfo, row_pointer, 1);
	}

	jpeg_finish_compress(&cam->cinfo);

	if(cam->prefs->verbose)
		dlog(__FILE__, "wrote jpeg data\n");

	fflush(fp);
	if(cam->prefs->filename)
	{
		fclose(fp);
		if(cam->prefs->verbose)
			dlog(__FILE__, "save file closed\n");
		if(cam->prefs->keepold)
		{
			/* ln -s backup_file file */
			int ret = unlink(cam->prefs->filename);
			if((ret == -1) && (errno != ENOENT))
				error("unlink", strerror(errno));
			if(symlink(cam->tfilename, cam->prefs->filename) == -1)
				error("symlink", strerror(errno));
		}
		else
		{
			/* mv backup_file file */
			if(rename(cam->tfilename, cam->prefs->filename) == -1)
				error("rename", strerror(errno));
		}
		if(cam->prefs->verbose)
			dlog(__FILE__, "save file %s to %s\n",
					(cam->prefs->keepold ? "symlinked" : "renamed"),
					cam->prefs->filename);
	}
}

void camsleep(struct webcam *cam, long sleep)
{
	struct timeval timeout;
	int i;

	timeout.tv_sec=1;
	timeout.tv_usec=0;

	for(i = 0; i < sleep; i++)
	{
		if(cam->prefs->verbose)
		{
			dlog(__FILE__, "sleeping %ld secs.\r", sleep-i);
			fflush(stderr);
		}
		if(keep_running)
			select(0, NULL, NULL, NULL, &timeout);
	}
	if(cam->prefs->verbose)
		dlog(__FILE__, "\n");
}

void camreset(struct webcam *cam)
{
	do_reset = TRUE;
}

void dlog(const char *module, const char *format, ... )
{
	char buf[1024], tbuf[128];
	va_list ap;
	time_t tnow;

	va_start(ap, format);
	vsnprintf(buf, sizeof(buf), format, ap);
	va_end(ap);

	tnow = time(NULL);
	strftime(tbuf, sizeof(tbuf), RFC1123DATE, localtime(&tnow));

	fprintf(stderr, "%s [%s] %s", tbuf, module, buf);
}

char *trim(char *s)
{
	char *t;

	t = s + strlen(s) - 1;
	while(isspace(*t) && (t >= s))
		*(t--) = '\0';

	while(isspace(*s))
		s++;

	return(s);
}

static void usage()
{
	int i;
	struct webcam_data *camdata;

	fprintf(stdout, "Usage: %s [-h] [configfile.xml]\n", getprogname());
	fprintf(stdout, "\nlist of supported webcams:\n");

	i = 0;
	while(supported_cams[i] != NULL)
	{
		camdata = supported_cams[i++]();
		fprintf(stdout, "  %s\n", camdata->name);
	}

	fprintf(stdout, "this is version: %s\n", VERSION);
	exit(EXIT_SUCCESS);
}

int main(int argc, char **argv)
{
	struct prefs_t aprefs;
	struct servent *serv;
	struct webcam acam, *cam = &acam;
	char pm[PATH_MAX+1];
	int streamer_cnt;
	time_t tlast;

	if((argc > 2) || ((argc == 2) && !strcmp(argv[1], "-h")))
		usage();

	keep_running = TRUE;
	do_reset = FALSE;
	streamer_cnt = 0;
	tlast = 0;
	memset(cam, 0, sizeof(*cam));
	memset(&aprefs, 0, sizeof(aprefs));

	/* init cam specific data */
	cam->prefs = &aprefs;
	cam->prefs->sleep    = 0;
	cam->prefs->width    = 320;
	cam->prefs->height   = 240;
	cam->prefs->quality  = 75;
	cam->prefs->log_enabled = FALSE;
	cam->prefs->logfile  = NULL;
	cam->prefs->cam      = NULL;
	cam->prefs->filename = NULL;
	cam->prefs->fontfile = NULL;
	cam->prefs->msg      = NULL;
	cam->prefs->keepold  = FALSE;
	cam->prefs->server_enabled  = FALSE;
	cam->prefs->server_dostream = FALSE;
	cam->prefs->server_maxconn  = 3;
	cam->prefs->daemon    = FALSE;
	cam->prefs->verbose   = 0;
	cam->prefs->use_external    = 0;
	cam->prefs->rootdir   = NULL;
	cam->prefs->col_fore  = NULL;
	cam->prefs->col_back  = NULL;
	cam->prefs->col_link  = NULL;
	cam->prefs->col_hover = NULL;

	serv = getservbyname("http", "tcp");
	if(serv == NULL)
		cam->prefs->server_port = 80;
	else
		cam->prefs->server_port = ntohs(serv->s_port);

	if(argc == 2)
		snprintf(pm, PATH_MAX, "%s", argv[1]);
	else
		snprintf(pm, PATH_MAX, "%s/.%s.xml", getenv("HOME"), PROGRAM);
	pm[PATH_MAX] = '\0';

	ccam = prefs_read(pm, cam);

	if(ccam == NULL)
		error("No valid cam device found", "Exiting");

	is_daemon = cam->prefs->daemon;

	/* run als daemon, if required */
	if(is_daemon)
	{
		daemon(0, 0);
		pidfile(NULL);
	}

	if((cam->prefs->filename == NULL) && (cam->prefs->keepold == TRUE))
		error("If you want backups you must specify a filename.", "Exiting");

	if(cam->prefs->col_fore == NULL)
		cam->prefs->col_fore = strdup(COLOR_FORE);
	if(cam->prefs->col_back == NULL)
		cam->prefs->col_back = strdup(COLOR_BACK);
	if(cam->prefs->col_link == NULL)
		cam->prefs->col_link = strdup(COLOR_LINK);
	if(cam->prefs->col_hover == NULL)
		cam->prefs->col_hover = strdup(COLOR_HOVER);

	log_init(cam);

	cam->cols = cam->prefs->width;
	cam->rows = cam->prefs->height;

	cam->jpg_ncomp  = ccam->jpg_ncomp;
	cam->jpg_format = ccam->jpg_format;
	ccam->init(cam);

	cam->copybuf = (char*)malloc(cam->cols*ccam->jpg_ncomp);
	if(cam->copybuf == NULL)
		error("malloc", strerror(errno));

	if(cam->prefs->msg != NULL)
	{
		cam->msg = cam->prefs->msg;
		if(cam->prefs->fontfile != NULL)
		{
			cam->font = load_font(cam->prefs->fontfile,
									&cam->fontheight,
									&cam->fontwidth);
		}
		if(cam->font == NULL)
			cam->font = get_default_font(&cam->fontheight, &cam->fontwidth);
		cam->max_msg = cam->cols / cam->fontwidth;
		if(strlen(cam->msg) > cam->max_msg)
			cam->msg[cam->max_msg-1] = '\0';
	}

	if(cam->prefs->verbose)
		dlog(__FILE__, "filename: %s  msg: %s  cbufl: %d  width: %d  height: %d max_connections: %d  sleep: %d\n", cam->prefs->filename, cam->msg, cam->cols*ccam->jpg_ncomp, cam->cols, cam->rows, cam->prefs->server_maxconn, cam->prefs->sleep);

	signal(SIGPIPE, sigpipe);
	signal(SIGINT,  sigint);

	jpeg_init(cam);
	if(cam->prefs->server_enabled)
	{
		if(cam->prefs->sleep == 0)
			cam->prefs->sleep = DEFSLEEP;
		server_init(cam);
	}

	fprintf(stderr, "\n");
	while(keep_running)
	{
		time_t tnow = time(NULL);

		if((tnow >= tlast + cam->prefs->sleep) || (streamer_cnt > 0))
		{
			ccam->capture_pic(cam);
			jpeg_encode(cam);
			tlast = tnow;
		}

		if(cam->prefs->server_enabled)
			streamer_cnt = server_work(cam);
		else
			if(cam->prefs->sleep > 0)
				camsleep(cam, cam->prefs->sleep);
			else
				break;

		if(do_reset)
		{
			log_error(cam, "reset device");
			ccam->finish(cam);
			ccam->init(cam);
			do_reset = FALSE;
		}
	}

	if(cam->prefs->server_enabled)
		server_finish(cam);
	jpeg_destroy_compress(&cam->cinfo);
	ccam->finish(cam);

	if(cam->copybuf)
		free(cam->copybuf);

	if(cam->prefs->title)
		free(cam->prefs->title);

	if(cam->prefs->msg)
		free(cam->prefs->msg);

	if(cam->prefs->fontfile)
		free(cam->prefs->fontfile);

	if(cam->prefs->logfile)
		free(cam->prefs->logfile);

	if(cam->prefs->col_fore)
		free(cam->prefs->col_fore);

	if(cam->prefs->col_back)
		free(cam->prefs->col_back);

	if(cam->prefs->col_link)
		free(cam->prefs->col_link);

	if(cam->prefs->col_hover)
		free(cam->prefs->col_hover);

	if(cam->prefs->rootdir)
		free(cam->prefs->rootdir);

	if(cam->prefs->cam)
		free(cam->prefs->cam);

	log_finish(cam);

	exit(EXIT_SUCCESS);
}

