
/* Copyright (c) 1999, 2000, 2001, 2002, 2003, 2004 Thomas Runge (coto@core.de)
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include <errno.h>
#include "main.h"
#include "font.h"
#include "f8x8.h"

#define FONT8X8         2048    /* filesize for 8x8 font              */
#define HEIGHT8X8       8       /* 8 scan lines char cell height      */

#define FONT8X10        2560    /* filesize for 8x10 font             */
#define HEIGHT8X10      10      /* 10 scan lines char cell height     */

#define FONT8X14        3584    /* filesize for 8x14 font             */
#define HEIGHT8X14      14      /* 14 scan lines char cell height     */

#define FONT8X16        4096    /* filesize for 8x16 font             */
#define HEIGHT8X16      16      /* 16 scan lines char cell height     */



unsigned char *get_default_font(int *cheight, int *cwidth)
{
	*cheight = 8;
	*cwidth  = 8;
	return(f8x8_data);
}

unsigned char *load_font(const char *filename, int *cheight, int *cwidth)
{
	FILE *fp;
	struct stat sbuf;
	unsigned char *fontbuf;

	fp = fopen(filename, "r");
	if(fp == NULL)
	{
		dlog(__FILE__, "Failed to open font file %s\n", filename);
		return(NULL);
	}

	if((fstat(fileno(fp), &sbuf)) == -1)
	{
		dlog(__FILE__, "Failed to stat font file: %s\n", strerror(errno));
		return(NULL);
	}

	*cwidth = 8;
	switch(sbuf.st_size)
	{
		case FONT8X8:
			*cheight = HEIGHT8X8;
			break;
		case FONT8X10:
			*cheight = HEIGHT8X10;
			break;
		case FONT8X14:
			*cheight = HEIGHT8X14;
			break;
		case FONT8X16:
			*cheight = HEIGHT8X16;
			break;
		default:
			dlog(__FILE__, "file %s is no valid font file\n", filename);
			return(NULL);
	}

	fontbuf = (unsigned char*)malloc((size_t)sbuf.st_size);
	if(fontbuf == NULL)
	{
		dlog(__FILE__, "Failed to malloc\n");
		return(NULL);
	}

	if((fread(fontbuf, sizeof(*fontbuf), sbuf.st_size, fp)) != sbuf.st_size)
	{
		dlog(__FILE__, "Failed to fread\n");
		return(NULL);
	}

	fclose(fp);
	return(fontbuf);
}

#ifdef FONTTEST

void dumpchar(unsigned char *fontbuf, int cheight, int ch)
{
	int i, j;
	unsigned char b;

	printf("Dumping: '%c' (%d)\n", ch, ch);
	for(i = 0; i < cheight; i++)
	{
		b = fontbuf[(ch * cheight) + i];
		for(j = 128; j > 0; j >>= 1)
		{
			if(b & j)
				printf("O");
			else
				printf(" ");
		}
		printf("\n");
	}
	printf("--------\n");
}

void dumpfont(unsigned char *fontbuf, int cheight)
{
	int i;

	for(i = 0; i < 256; i++)
		dumpchar(fontbuf, cheight, i);
}

void printline(unsigned char *fontbuf, const char *str, int cheight, int line)
{
	int i, j, len;

	len = strlen(str);
	for(i = 0; i < len; i++)
	{
		unsigned char b;

		b = fontbuf[(str[i] * cheight) + line];
		for(j = 128; j > 0; j >>= 1)
		{
			if(b & j)
				printf("O");
			else
				printf(" ");
		}
	}
	printf("\n");
}

void printstring(unsigned char *fontbuf, const char *str, int cheight)
{
	int i;

	for(i = 0; i < cheight; i++)
		printline(fontbuf, str, cheight, i);
}

static void usage()
{
	printf("usage: %s [-f font] [-s string] [-h]\n", getprogname());
}

int main(int argc, char **argv)
{
	unsigned char *fontbuf;
	const char *str;
	int cheight, cwidth, ch;

	str = NULL;
	cheight = 8;
	fontbuf = f8x8_data;

	while((ch = getopt(argc, argv, "f:s:h")) != -1)
    { 
        switch(ch)
        {
            case 'f':
				fontbuf = load_font(optarg, &cheight, &cwidth);
				if(fontbuf == NULL)
					exit(EXIT_FAILURE);
                break;
            case 's':
				str = optarg;
                break;
            case 'h':
				usage();
				exit(EXIT_SUCCESS);
                break;
		}
	}

	if(str != NULL)
		printstring(fontbuf, str, cheight);
	else
		dumpfont(fontbuf, cheight);

	exit(EXIT_SUCCESS);
}

#endif /* FONTTEST */

