
/* Copyright (c) 1999, 2000, 2001, 2002, 2003, 2004 Thomas Runge (coto@core.de)
 *
 * Based on grab.c by Roger Hardiman and videocapture.c by Amancio Hasty
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <sys/param.h>
#include <sys/time.h>
#include <fcntl.h>
#include <signal.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <time.h>
#include "main.h"
#include "cam_bktr.h"

#ifdef __NetBSD__
#include <dev/ic/bt8xx.h>
#endif /* NetBSD */

#ifdef __FreeBSD__
#include <machine/ioctl_meteor.h>
#include <machine/ioctl_bt848.h>
#endif /* __FreeBSD__ */

/* which device */
#define VIDEO_DEV  "/dev/bktr"
#define TUNER_DEV  "/dev/tuner"

/* PAL is 768 x 576. NTSC is 640 x 480 */
#define PAL_MAXCOLS 768
#define PAL_MAXROWS 576
#define NTSC_MAXCOLS 640
#define NTSC_MAXROWS 480

/* device needs some time to settle down */
#define SETTLE 1

static int BRIGHT_MIN       = BT848_BRIGHTMIN;
static int BRIGHT_MAX       = (BT848_BRIGHTMIN + BT848_BRIGHTRANGE);
static int BRIGHT_CENTER    = BT848_BRIGHTCENTER;
static int BRIGHT_RANGE     = BT848_BRIGHTRANGE;
static int BRIGHT_DRV_MIN   = BT848_BRIGHTREGMIN;
static int BRIGHT_DRV_RANGE = (BT848_BRIGHTREGMAX - BT848_BRIGHTREGMIN + 1);

static int CONTRAST_MIN       = BT848_CONTRASTMIN;
static int CONTRAST_MAX       = (BT848_CONTRASTMIN + BT848_CONTRASTRANGE);
static int CONTRAST_CENTER    = BT848_CONTRASTCENTER;
static int CONTRAST_RANGE     = BT848_CONTRASTRANGE;
static int CONTRAST_DRV_MIN   = BT848_CONTRASTREGMIN;
static int CONTRAST_DRV_RANGE = (BT848_CONTRASTREGMAX - BT848_CONTRASTREGMIN + 1);

static void cam_bktr_setprefs(struct webcam *cam);
static void cam_bktr_addpref(struct prefs_t *prefs, char *key, char *value);
static void cam_bktr_startup(struct webcam *cam);
static void cam_bktr_finish(struct webcam *cam);
static void cam_bktr_capture_pic(struct webcam *cam);
static void cam_bktr_getline(struct webcam *cam, int line);

static int unit;
static int tvformat;
static int input;
static int channel;
static int brightness;
static int contrast;

static struct webcam_data cam_bktr =
	{
		"bktr", 3, JCS_RGB,
		cam_bktr_setprefs, cam_bktr_addpref, cam_bktr_startup,
		cam_bktr_finish, cam_bktr_capture_pic, cam_bktr_getline
	};

struct webcam_data *cam_bktr_init()
{
	return(&cam_bktr);
}

static int setValues(struct webcam *cam, int value, unsigned long request,
				int min, int max, int range,
				int drvmin, int drvrange)
{
	int arg;

	value = MAX(min, MIN(max, value));

	arg = (value - min) / (range + 0.01) * drvrange + drvmin;
	arg = MAX(drvmin, MIN(drvmin + drvrange - 1, arg));

	if(ioctl(cam->tuner, request, &arg ) < 0 )
		error("ioctl", strerror(errno));

	return(arg);
}

static void cam_bktr_setprefs(struct webcam *cam)
{
	cam->video = -1;
	cam->tuner = -1;

	unit       = 0;
	tvformat   = -1;
	input      = 0;
	channel    = -1;
	brightness = BRIGHT_CENTER;
	contrast   = CONTRAST_CENTER;
}

static void cam_bktr_addpref(struct prefs_t *prefs, char *key, char *value)
{
	char *s = trim(value);

	if(!strcasecmp(key, "unit"))
	{
		if(strlen(s) > 0)
			unit = atoi(s);
	}
	if(!strcasecmp(key, "tvformat"))
	{
		if(strlen(s) > 0)
			tvformat = atoi(s);
	}
	if(!strcasecmp(key, "input"))
	{
		if(strlen(s) > 0)
			input = atoi(s);
	}
	if(!strcasecmp(key, "channel"))
	{
		if(strlen(s) > 0)
			channel = atoi(s);
	}
	if(!strcasecmp(key, "brightness"))
	{
		if(strlen(s) > 0)
			brightness = atoi(s);
	}
	if(!strcasecmp(key, "contrast"))
	{
		if(strlen(s) > 0)
			contrast = atoi(s);
	}
}

static void cam_bktr_startup(struct webcam *cam)
{
	char dev_video[MAXPATHLEN];
	char dev_tuner[MAXPATHLEN];
	char errbuf[80];
	struct meteor_geomet geo;
	int ret;

	switch(tvformat)
	{
		case 1:  tvformat = BT848_IFORM_F_NTSCM; break;
		case 2:  tvformat = BT848_IFORM_F_NTSCJ; break;
		case 3:  tvformat = BT848_IFORM_F_PALN; break;
		case 4:  tvformat = BT848_IFORM_F_PALM; break;
		case 5:  tvformat = BT848_IFORM_F_SECAM; break;
		case 6:  tvformat = BT848_IFORM_F_RSVD; break;
		default:
		case 0:  tvformat = BT848_IFORM_F_PALBDGHI; break;
	}

	switch(input)
	{
		case 1:  input = METEOR_INPUT_DEV1; break;
		case 2:  input = METEOR_INPUT_DEV2; break;
		case 3:  input = METEOR_INPUT_DEV3; break;
		default:
		case 0:  input = METEOR_INPUT_DEV0; break;
	}

	if(	(contrast < CONTRAST_MIN) ||
		(contrast > CONTRAST_MAX))
	{
		snprintf(errbuf, sizeof(errbuf), "contrast out of range (%d <= x <= %d)",
					CONTRAST_MIN, CONTRAST_MAX);
		error(errbuf, "Exiting");
	}

	if(	(brightness < BRIGHT_MIN) ||
		(brightness > BRIGHT_MAX))
	{
		snprintf(errbuf, sizeof(errbuf), "brightness out of range (%d <= x <= %d)",
					BRIGHT_MIN, BRIGHT_MAX);
		error(errbuf, "Exiting");
	}

	if(	cam->cols > PAL_MAXCOLS || cam->cols <= 0 ||
		cam->rows > PAL_MAXROWS || cam->rows <= 0)
	{
		cam->cols = PAL_MAXCOLS;
		cam->rows = PAL_MAXROWS;
	}

	geo.columns = cam->cols;
	geo.rows    = cam->rows;
	geo.frames  = 1;
	geo.oformat = METEOR_GEO_RGB24;

	snprintf(dev_video, sizeof(dev_video), "%s%d", VIDEO_DEV, unit);
	snprintf(dev_tuner, sizeof(dev_tuner), "%s%d", TUNER_DEV, unit);

	if(cam->prefs->verbose)
	{
		dlog(__FILE__, "video device: %s\n", dev_video);
		dlog(__FILE__, "tuner device: %s\n", dev_tuner);
	}
	if(cam->prefs->verbose)
		dlog(__FILE__, "opening video device\n");

	if((cam->video = open(dev_video, O_RDONLY)) == -1)
		error("open video", strerror(errno));

	if(cam->prefs->verbose)
		dlog(__FILE__, "opening tuner device\n");

	if((cam->tuner = open(dev_tuner, O_RDONLY)) == -1)
		error("open tuner", strerror(errno));

	if(ioctl(cam->video, BT848SFMT, &tvformat) == -1)
		error("ioctl BT848SFMT", "BT848_IFORM");

	switch(tvformat)
	{
		case BT848_IFORM_F_NTSCM:
		case BT848_IFORM_F_NTSCJ:
		case BT848_IFORM_F_PALM:
				if(geo.rows <= NTSC_MAXROWS / 2)
					geo.oformat |= METEOR_GEO_ODD_ONLY; 
				break;
		case BT848_IFORM_F_PALBDGHI:
		case BT848_IFORM_F_PALN:
		case BT848_IFORM_F_SECAM:
		case BT848_IFORM_F_RSVD:
		default:
				if(geo.rows <= PAL_MAXROWS / 2)
					geo.oformat |= METEOR_GEO_ODD_ONLY; 
				break;
	}

	if(ioctl(cam->video, METEORSINPUT, &input) == -1)
		error("ioctl METEORSINPUT", "video SINPUT");
	if(cam->prefs->verbose)
		dlog(__FILE__, "set input to %d\n", input);

	if(ioctl(cam->video, METEORSETGEO, &geo) == -1)
		error("ioctl METEORSETGEO", "video SETGEO");

	ret = setValues(cam, brightness, BT848_SBRIG,
				BRIGHT_MIN, BRIGHT_MAX, BRIGHT_RANGE,
				BRIGHT_DRV_MIN, BRIGHT_DRV_RANGE);
	if(cam->prefs->verbose)
		dlog(__FILE__, "set brightness to: %d\n", ret);

	ret = setValues(cam, contrast, BT848_SCONT,
				CONTRAST_MIN, CONTRAST_MAX, CONTRAST_RANGE,
				CONTRAST_DRV_MIN, CONTRAST_DRV_RANGE);
	if(cam->prefs->verbose)
		dlog(__FILE__, "set contrast to: %d\n", ret);

	if(ioctl(cam->tuner, TVTUNER_SETCHNL, &channel) < 0 )
		error("set_channel", strerror(errno));
	if(cam->prefs->verbose)
		dlog(__FILE__, "set channel to: %d\n", channel);

	cam->buf = (char*) mmap(NULL, cam->rows * cam->cols * 4,
									PROT_READ, MAP_SHARED, cam->video, 0);

	if(cam->buf == MAP_FAILED)
		error("mmap", strerror(errno));

	/* wait for the device to settle down */
	if(cam->prefs->verbose)
		dlog(__FILE__, "waiting for device to settle down\n");
	camsleep(cam, SETTLE);
}

static void cam_bktr_finish(struct webcam *cam)
{
	munmap(cam->buf, cam->rows * cam->cols * 4);
	if(cam->prefs->verbose)
		dlog(__FILE__, "closing tuner device\n");
	close(cam->tuner);

	if(cam->prefs->verbose)
		dlog(__FILE__, "closing video device\n");
	close(cam->video);

}

static void cam_bktr_capture_pic(struct webcam *cam)
{
	int single  = METEOR_CAP_SINGLE;
	ioctl(cam->video, METEORCAPTUR, &single);
}

static void cam_bktr_getline(struct webcam *cam, int line)
{
	int i;
	unsigned char *p, *src;

	src = cam->buf + (cam->cols*line << 2);
	p = cam->copybuf;

	for(i = cam->cols; i; i--)
	{
		*p++ = src[2];
		*p++ = src[1];
		*p++ = src[0];
		src += 4;
	}
}

