/* blinkensim - a Blinkenlights simulator
 *
 * Copyright (c) 2001-2002  Sven Neumann <sven@gimp.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"

#include <stdlib.h>

#include <blib/blib.h>

#include "gfx.h"


static GObject *view = NULL;
static guchar   lut[256];


static gboolean
frame_callback (BReceiver *receiver,
                BPacket   *packet,
                gpointer   data)
{
  BTheme *theme = (BTheme *) data;

  if (packet->header.width    != theme->columns ||
      packet->header.height   != theme->rows    ||
      packet->header.channels != 1)
    {
      g_printerr ("Packet mismatch\n");
      return TRUE;
    }

  if (theme->maxval != 0xFF || packet->header.maxval != 0xFF)
    {
      gint    n = packet->header.width * packet->header.height;
      guchar *d = packet->data;

      switch (packet->header.maxval)
        {
        case 0xFF:
          while (n--)
            *d = lut[*d], d++;
          break;

        default:
          while (n--)
            *d = lut[((guint) *d * 0xFF) / packet->header.maxval], d++;
          break;
        }
    }

  if (view)
    gfx_view_update (view, packet->data);

  return TRUE;
}

static void
setup_lut (BTheme *theme)
{
  gint    i, d;
  gdouble v;

  for (i = 0, d = 0, v = 0.0; i < 256; i++)
    {
      lut[i] = v + 0.5;

      d += theme->maxval + 1;
      if (d > 255)
        {
          d -= 256;
          v += 255.0 / theme->maxval;
        }
    }
}

static void
print_theme_info (const gchar *name,
                  BTheme      *theme,
                  gpointer     user_data)
{
  if (!name || !theme)
    return;

  g_print ("  %-20s  %s (%dx%d) on %dx%d\n",
           name,
           b_object_get_name (B_OBJECT (theme)), theme->columns, theme->rows,
           theme->width, theme->height);
}

int
main (int    argc, 
      char **argv)
{
  BReceiver *receiver;
  BTheme    *theme;
  GMainLoop *loop;
  GError    *error = NULL;

  g_print ("blinkensim version " VERSION "\n");

  if (!gfx_init (&argc, &argv, &error))
    {
      g_printerr ("%s\n", error->message);
      return EXIT_FAILURE;
    }

  if (argc < 2)
    {
      g_printerr ("Usage: %s <theme>\n", argv[0]);
      g_printerr ("  Use '%s --list' to get a list of available themes.\n",
                  argv[0]);
      return EXIT_FAILURE;
    }

  g_type_init ();

  if (strcmp (argv[1], "--list") == 0)
    {
      g_print ("\nAvailable themes:\n");
      b_themes_foreach_theme (NULL, (GHFunc) print_theme_info, NULL);
      g_print ("\n");
      return EXIT_SUCCESS;
    }

  if (g_file_test (argv[1], G_FILE_TEST_IS_REGULAR))
    {
      theme = b_theme_new_from_file (argv[1], TRUE, &error);
      if (!theme)
        {
          g_printerr ("Error opening '%s': %s\n", argv[1], error->message);
          return EXIT_FAILURE;
        }
    }
  else
    {
      theme = b_themes_lookup_theme (argv[1], NULL, &error);
      if (!theme)
        {
          g_printerr ("\n%s\n", error->message);
          g_printerr ("Fix your spelling or try setting the "
                      "B_THEME_PATH environment variable.\n\n");
          g_printerr ("Use '%s --list' to get a list of available themes.\n",
                      argv[0]);
          return EXIT_FAILURE;
        }
      if (!b_theme_load (theme, &error))
        {
          g_printerr ("Error loading theme '%s': %s\n",
                      b_object_get_name (B_OBJECT (theme)), error->message);
          return EXIT_FAILURE;
        }
    }
  
  g_print ("Using theme '%s' (%dx%d) at size %dx%d\n",
           b_object_get_name (B_OBJECT (theme)), theme->columns, theme->rows,
           theme->width, theme->height);

  g_assert (theme->maxval > 0);

  setup_lut (theme);

  loop = g_main_loop_new (NULL, FALSE);

  view = gfx_view_new (theme, loop, &error);

  if (!view)
    {
      g_printerr ("%s\n", error->message);
      gfx_close ();
      return EXIT_FAILURE;
    }

  g_object_add_weak_pointer (view, (gpointer) &view);

  receiver = b_receiver_new (frame_callback, theme);

  b_receiver_listen (receiver, 2323);

  g_main_loop_run (loop);

  g_main_loop_unref (loop);
  g_object_unref (receiver);
  g_object_unref (theme);

  gfx_close ();

  return EXIT_SUCCESS;
}
