/*
 * Copyright (C) 2002-2003 Clemens Fuchslocher <clfuit00@fht-esslingen.de>
 *
 * xbel-dom.c - 17.07.2003 - v0.2 - galeon-dom.c => xbel-dom.c
 *              05.09.2002 - v0.1
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * 'The XML Bookmark Exchange Language Resource Page'
 * <http://pyxml.sourceforge.net/topics/xbel/>
 *
 */

#include <errno.h>
#include <stdlib.h>
#include <string.h>

#include <libxml/parser.h>

#include "galeon.h"
#include "plugin.h"
#include "g_stack.h"

#include "bk_edit.h"
#include "bk_edit_misc.h"
#include "bk_edit_icon.h"
#include "bk_edit_tree.h"

extern bk_edit_tree tree;

static void traverse_dom (xmlNodePtr node);

static void process_node_alias (xmlNodePtr node);
static void process_node_separator (xmlNodePtr node);
static void process_node_bookmark (xmlNodePtr node);
static void process_node_folder (xmlNodePtr node);
static void process_node_xbel (xmlNodePtr node);
static void process_node (xmlNodePtr node);

static xmlChar *node_get_metadata (xmlNodePtr node, bk_edit_tree_data *node_data);
static xmlChar *node_get_element_content (xmlNodePtr node, xmlChar *element, int element_length);
static xmlChar *node_dump (xmlNodePtr node);

static int convert_node_data_to_latin1 (bk_edit_tree_data *node_data);
static void clear_node_data (bk_edit_tree_data *node_data);


int galeon_dom_parser (char *filename)
{
	xmlDocPtr doc;
	xmlNodePtr root;

	doc = xmlParseFile (filename);
	if (doc == NULL)
	{
		return GALEON_PARSER_KO;
	}

	root = xmlDocGetRootElement (doc);
	if (root == NULL)
	{
		fprintf (stderr, "%s[%d]: xmlDocGetRootElement\n", __FILE__, __LINE__);
		xmlFreeDoc (doc);
		return GALEON_PARSER_KO;
	}

	traverse_dom (root);

	xmlFreeDoc (doc);

	return GALEON_PARSER_OK;
}


static xmlChar *node_get_metadata (xmlNodePtr node, bk_edit_tree_data *node_data)
{
	GString *metadata = g_string_sized_new (42);

	if (strncmp (node->name, "info", sizeof ("info")) == 0)
	{
		xmlNodePtr n = node->children;
		while (n)
		{
			if (n->type != XML_TEXT_NODE)
			{
				if (strncmp (n->name, "metadata", sizeof ("metadata")) == 0)
				{
					xmlChar *dump;

					xmlNodePtr nn = n->children;
					while (nn)
					{
						if (node_data->elements[PIXMAP] == NULL)
							node_data->elements[PIXMAP] = node_get_element_content (nn, "pixmap", sizeof ("pixmap"));

						if (node_data->elements[CREATE_TOOLBAR] == NULL)
							node_data->elements[CREATE_TOOLBAR] = node_get_element_content (nn, "create_toolbar", sizeof ("create_toolbar"));

						if (node_data->elements[CREATE_CONTEXT] == NULL)
							node_data->elements[CREATE_CONTEXT] = node_get_element_content (nn, "create_context", sizeof ("create_context"));

						if (node_data->elements[TOOLBAR_STYLE] == NULL)
							node_data->elements[TOOLBAR_STYLE] = node_get_element_content (nn, "toolbar_style", sizeof ("toolbar_style"));

						if (node_data->elements[LAST_VISIT] == NULL)
							node_data->elements[LAST_VISIT] = node_get_element_content (nn, "time_visited", sizeof ("time_visited"));

						if (node_data->elements[LAST_MODIFIED] == NULL)
							node_data->elements[LAST_MODIFIED] = node_get_element_content (nn, "time_modified", sizeof ("time_modified"));

						if (node_data->elements[ADD_DATE] == NULL)
							node_data->elements[ADD_DATE] = node_get_element_content (nn, "time_added", sizeof ("time_added"));

						if (node_data->elements[NICKNAME] == NULL)
							node_data->elements[NICKNAME] = node_get_element_content (nn, "nick", sizeof ("nick"));

						if (node_data->elements[SMARTURL] == NULL)
							node_data->elements[SMARTURL] = node_get_element_content (nn, "smarturl", sizeof ("smarturl"));

						if (node_data->elements[DEFAULT_FOLDER] == NULL)
							node_data->elements[DEFAULT_FOLDER] = node_get_element_content (nn, "default_folder", sizeof ("default_folder"));

						nn = nn->next;
					}
					node_data->elements[METADATA_OWNER] = xmlGetProp (n, "owner");

					dump = node_dump (n);
					g_string_append (metadata, dump);
					free (dump);
				}
			}
			n = n->next;
		}
	}

	if (metadata->len > 0)
	{
		xmlChar *str = metadata->str;
		g_string_free (metadata, FALSE);
		return str;
	}

	g_string_free (metadata, TRUE);
	return NULL;
}


static xmlChar *node_get_element_content (xmlNodePtr node, xmlChar *element, int element_length)
{
	xmlChar *content = NULL;

	if (strncmp (node->name, element, element_length) == 0)
	{
		content = xmlNodeGetContent (node->children);
	}

	return content;
}


static xmlChar *node_dump (xmlNodePtr node)
{
	xmlChar *metadata;

	xmlBufferPtr dump = xmlBufferCreate ();
	if (dump == NULL)
	{
		fprintf (stderr, "%s[%d]: xmlBufferCreate", __FILE__, __LINE__);
		return strdup ("");
	}

/* libxml > 2.5
	if (xmlNodeDump (dump, node->doc, node, 0, FALSE) == -1)
	{
		fprintf (stderr, "%s[%d]: xmlNodeDump", __FILE__, __LINE__);
		xmlBufferFree (dump);
		return strdup ("");
	}
*/
	xmlNodeDump (dump, node->doc, node, 0, FALSE);

	metadata = strdup (dump->content);
	xmlBufferFree (dump);

	return metadata;
}


static void process_node_alias (xmlNodePtr node)
{
	xmlChar *ref;

	bk_edit_tree_data node_data;
	memset (&node_data, 0, sizeof (node_data));

	ref = xmlGetProp (node, "ref");

	node_data.elements[REF] = ref;

	bk_edit_tree_add_bookmark_node (&node_data);

	if (ref) xmlFree (ref);
}


static void process_node_separator (xmlNodePtr node)
{
	bk_edit_tree_data node_data;
	memset (&node_data, 0, sizeof (node_data));

	node_data.elements[NAME] = "-----------------";

	bk_edit_tree_add_separator (&node_data);
}


static void process_node_bookmark (xmlNodePtr node)
{
	bk_edit_tree_data node_data;
	memset (&node_data, 0, sizeof (node_data));

	node_data.elements[ID] = xmlGetProp (node, "id");
	node_data.elements[URI] = xmlGetProp (node, "href");

	node = node->children;
	while (node)
	{
		if (node->type != XML_TEXT_NODE)
		{
			if (node_data.elements[NAME] == NULL)
				node_data.elements[NAME] = node_get_element_content (node, "title", sizeof ("title"));

			if (node_data.elements[METADATA] == NULL)
				node_data.elements[METADATA] = node_get_metadata (node, &node_data);

			if (node_data.elements[COMMENT] == NULL)
				node_data.elements[COMMENT] = node_get_element_content (node, "desc", sizeof ("desc"));
		}
		node = node->next;
	}

	convert_node_data_to_latin1 (&node_data);
	convert_node_data_to_latin1 (&node_data);
	bk_edit_tree_add_bookmark_node (&node_data);
	clear_node_data (&node_data);
}


static void process_node_folder (xmlNodePtr node)
{
	xmlChar *folded = NULL;

	bk_edit_tree_data node_data;
	memset (&node_data, 0, sizeof (node_data));

	node_data.elements[ID] = xmlGetProp (node, "id");
	folded = xmlGetProp (node, "folded");

	node = node->children;
	while (node)
	{
		if (node->type != XML_TEXT_NODE)
		{
			if (node_data.elements[NAME] == NULL)
				node_data.elements[NAME] = node_get_element_content (node, "title", sizeof ("title"));

			if (node_data.elements[METADATA] == NULL)
				node_data.elements[METADATA] = node_get_metadata (node, &node_data);

			if (node_data.elements[COMMENT] == NULL)
				node_data.elements[COMMENT] = node_get_element_content (node, "desc", sizeof ("desc"));
		}
		node = node->next;
	}

	convert_node_data_to_latin1 (&node_data);
	convert_node_data_to_latin1 (&node_data);
	bk_edit_tree_add_folder_node (&node_data);
	clear_node_data (&node_data);
}


static void process_node_xbel (xmlNodePtr node)
{
	bk_edit_tree_data node_data;
	memset (&node_data, 0, sizeof (node_data));

	node_data.elements[ID] = xmlGetProp (node, "id");
	node_data.elements[VERSION] = xmlGetProp (node, "version");

	node = node->children;
	while (node)
	{
		if (node->type != XML_TEXT_NODE)
		{
			if (node_data.elements[NAME] == NULL)
				node_data.elements[NAME] = node_get_element_content (node, "title", sizeof ("title"));

			if (node_data.elements[METADATA] == NULL)
				node_data.elements[METADATA] = node_get_metadata (node, &node_data);

			if (node_data.elements[COMMENT] == NULL)
				node_data.elements[COMMENT] = node_get_element_content (node, "desc", sizeof ("desc"));
		}
		node = node->next;
	}

	convert_node_data_to_latin1 (&node_data);
	convert_node_data_to_latin1 (&node_data);
	bk_edit_tree_add_root_node (&node_data);
	clear_node_data (&node_data);
}


static void process_node (xmlNodePtr node)
{
	if (strncmp (node->name, "bookmark", sizeof ("bookmark")) == 0)
	{
		process_node_bookmark (node);
	}
	else if (strncmp (node->name, "folder", sizeof ("folder")) == 0)
	{
		process_node_folder (node);
		traverse_dom (node->children);
		bk_edit_tree_leave_folder ();
	}
	else if (strncmp (node->name, "separator", sizeof ("separator")) == 0)
	{
		process_node_separator (node);
	}
	else if (strncmp (node->name, "alias", sizeof ("alias")) == 0)
	{
		process_node_alias (node);
	}
	else if (strncmp (node->name, "xbel", sizeof ("xbel")) == 0)
	{
		process_node_xbel (node);
		traverse_dom (node->children);
	}
}


static void traverse_dom (xmlNodePtr node)
{
	while (node)
	{
		process_node (node);

		node = node->next;
	}
}


static int convert_node_data_to_latin1 (bk_edit_tree_data *node_data)
{
	int i;

	for (i = NAME; i < ELEMENTS; i++)
	{
		if (node_data->elements[i] != NULL)
		{
			int len = strlen (node_data->elements[i]);

			int out_len = len;
			unsigned char *out = (unsigned char *) malloc (len + 1);
			if (out == NULL)
			{
				fprintf (stderr, "%s[%d]: malloc (): %s\n", __FILE__, __LINE__, strerror (errno));
				continue;
			}

			if (bk_edit_misc_mixed_utf8_to_latin1 (out, &out_len, node_data->elements[i], &len) != OK)
			{
				fprintf (stderr, "%s[%d]: convert_mixed_utf8_to_latin1 ()\n", __FILE__, __LINE__);
				continue;
			}
			out[out_len] = '\0';

			free (node_data->elements[i]);
			node_data->elements[i] = out;
		}
	}

	return GALEON_PARSER_OK;
}


static void clear_node_data (bk_edit_tree_data *node_data)
{
	int i;

	for (i = NAME; i < ELEMENTS; i++)
	{
		if (node_data->elements[i] != NULL)
		{
			free (node_data->elements[i]);
			node_data->elements[i] = NULL;
		}
	}
}

