/*
 * Copyright (C) 2003 Clemens Fuchslocher <clfuit00@fht-esslingen.de>
 *
 * convert.c - 10.07.2003 - v0.1
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 */

#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>

#include <sys/stat.h>

#include "bk_edit.h"
#include "bk_edit_misc.h"

#include "plugin.h"
#include "convert.h"

static void traverse (GNode *node);
static gboolean traverse_free (GNode *node, gpointer data);

convert conv;

extern plugins *plugs;


int convert_bookmarks (command_line_options *options)
{
	struct stat st_save;
	struct stat st_load;

	conv.mode = CONVERT;

	if (stat (options->load, &st_load) == -1)
	{
		fprintf (stderr, "%s[%d]: %s does not exist.\n", __FILE__, __LINE__, options->load);
		return KO;
	}

	if (stat (options->save, &st_save) == 0)
	{
		if (st_save.st_ino == st_load.st_ino)
		{
			fprintf (stderr, "%s[%d]: %s and %s are the same files.\n", __FILE__, __LINE__, options->load, options->save);
			return KO;
		}

		if (bk_edit_misc_create_backup_file (options->save) == KO)
		{
			return KO;
		}
	}

	if (!S_ISREG (st_load.st_mode))
	{
		fprintf (stderr, "%s[%d]: %s is not a regular file.\n", __FILE__, __LINE__, options->load);
		return KO;
	}

	conv.plugin_load = plugin_get_by_name (plugs, options->load_plugin);
	if (conv.plugin_load == NULL)
	{
		conv.plugin_load = plugin_get_by_short_name (plugs, options->load_plugin);
		if (conv.plugin_load == NULL)
		{
			fprintf (stderr, "%s[%d]: %s is an unknown plugin.", __FILE__, __LINE__, options->load_plugin);
			return KO;
		}
	}

	conv.plugin_save = plugin_get_by_name (plugs, options->save_plugin);
	if (conv.plugin_save == NULL)
	{
		conv.plugin_save = plugin_get_by_short_name (plugs, options->save_plugin);
		if (conv.plugin_save == NULL)
		{
			fprintf (stderr, "%s[%d]: %s is an unknown plugin.", __FILE__, __LINE__, options->save_plugin);
			return KO;
		}
	}

	conv.filename_save = options->save;
	conv.filename_load = options->load;

	conv.parent_stack = g_stack_new (conv.parent_stack);

	if (conv.plugin_load->load (options->load) != PLUGIN_OK)
	{
		fprintf (stderr, "%s[%d]: The selected file couldn't be loaded.\n", __FILE__, __LINE__);
	}

	conv.plugin_save->save_begin (conv.filename_save, conv.tree->data);
	traverse (conv.tree->children);
	conv.plugin_save->save_end ();

	g_stack_delete (conv.parent_stack);
	g_node_traverse (conv.tree, G_PRE_ORDER, G_TRAVERSE_ALL, -1, traverse_free, NULL);

	return OK;
}


static void traverse (GNode *node)
{
	while (node != NULL)
	{
		if (node->data != NULL)
		{
			bk_edit_tree_data *node_data = (bk_edit_tree_data *) node->data;
			if (node_data != NULL)
			{
				if (node_data->type == FOLDER)
				{
					conv.plugin_save->save_folder_in (node_data);
					traverse (node->children);
					conv.plugin_save->save_folder_out ();
				}
				else if (node_data->type == BOOKMARK)
				{
					conv.plugin_save->save_bookmark (node_data);
				}
				else if (node_data->type == SEPARATOR)
				{
					conv.plugin_save->save_separator (node_data);
				}
			}
		}

		node = node->next;
	}
}

static gboolean traverse_free (GNode *node, gpointer data)
{
	bk_edit_tree_data *node_data = (bk_edit_tree_data *) node->data;

	int i;
	for (i = 0; i < ELEMENTS; i++)
	{
		free (node_data->elements[i]);
	}
	free (node_data);

	return FALSE;
}

