/*
 * Copyright (C) 2003 Clemens Fuchslocher <clfuit00@fht-esslingen.de>
 *
 * bk_edit_urlhandler.c - 15.02.2003 - v0.1
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 */

#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <stddef.h>

#include <unistd.h>
#include <sys/wait.h>

#include <gtk/gtk.h>

#include "config.h"

#include "bk_edit_misc.h"
#include "bk_edit_tree.h"
#include "bk_edit_url_handler.h"

extern config *conf;
extern bk_edit_tree tree;

static int popt_parse_argv_string (const char *s, int *argcPtr, const char ***argvPtr);
static int popt_dup_argv (int argc, const char **argv, int *argcPtr, const char ***argvPtr);

static gboolean child_reaper (gpointer pid);

static wildcard wildcards[] = {
	{ "%u", URI },
	{ "%n", NAME },
	{ "%c", COMMENT },
	{ NULL, -1 }
};


void bk_edit_url_handler (gpointer callback_data, guint callback_action, GtkWidget *widget)
{
	int argc, pid, n;
	char **argv;

	char *command;

	GList *url_handlers = config_url_handlers_get (conf);
	GList *url_handler = url_handlers;

	bk_edit_tree_data *node_data;

	if (tree.selected_node == NULL)
	{
		return;
	}

	node_data = (bk_edit_tree_data *) gtk_ctree_node_get_row_data (GTK_CTREE (tree.tree), tree.selected_node);
	if (node_data->type != BOOKMARK)
	{
		return;
	}

	while (url_handler)
	{
		if (USE_DEFAULT_HANDLER == (int) callback_data)
		{
			if (((config_url_handler *) url_handler->data)->default_handler == 1)
			{
				break;
			}
		}
		else if (callback_action-- == 0)
		{
			break;
		}
		url_handler = url_handler->next;
	}

	if (url_handler == NULL)
	{
		fprintf (stderr, "%s[%d]: url_handler == NULL\n", __FILE__, __LINE__);
		config_url_handlers_free (conf, url_handlers);
		return;
	}

	if (*((config_url_handler *) url_handler->data)->command == '\0')
	{
		return;
	}

	command = strdup (((config_url_handler *) url_handler->data)->command);

	for (n = 0; wildcards[n].name != NULL; n++)
	{
		char *command_escaped, *c;

		command_escaped = bk_edit_misc_strescape (command, '%');

		if (command_escaped == NULL)
		{
			fprintf (stderr, "%s[%d]: bk_edit_misc_strescape\n", __FILE__, __LINE__);
			continue;
		}
		free (command);

		c = strstr (command_escaped, wildcards[n].name);
		if (c == NULL)
		{
			fprintf (stderr, "%s[%d]: strstr\n", __FILE__, __LINE__);
			command = g_strdup_printf (command_escaped);
			free (command_escaped);
			continue;
		}
		*(c + 1) = 's';

		while (*c != '\0')
		{
			*c = *(c + 1);

			c++;
		}

		command = g_strdup_printf (command_escaped, node_data->elements[wildcards[n].index] == NULL ? "" : node_data->elements[wildcards[n].index]);
		if (command == NULL)
		{
			fprintf (stderr, "%s[%d]: g_strdup_printf\n", __FILE__, __LINE__);
			free (command_escaped);
			continue;
		}

		free (command_escaped);
	}

	if (popt_parse_argv_string (command, &argc, (const char ***) &argv) == -1)
	{
		fprintf (stderr, "%s[%d]: popt_parse_argv_string\n", __FILE__, __LINE__);
		config_url_handlers_free (conf, url_handlers);
		free (command);
		return;
	}

	pid = fork ();
	if (pid == -1)
	{
		fprintf (stderr, "%s[%d]: fork\n", __FILE__, __LINE__);
		config_url_handlers_free (conf, url_handlers);
		free (command);
		return;
	}

	if (pid == 0)
	{
		execvp (argv[0], argv);
		fprintf (stderr, "%s[%d]: execvp\n", __FILE__, __LINE__);
		_exit (0);
	}

	/* We don't want zombies. */
	g_timeout_add (2500, (GSourceFunc) child_reaper, (gpointer) pid);

	config_url_handlers_free (conf, url_handlers);
	free (command);
	free (argv);
}


void bk_edit_url_handler_button_press_event (GtkWidget *widget, GdkEventButton *event, gpointer null)
{
	if (event->button == 1)
	{
		if (event->type == GDK_2BUTTON_PRESS)
		{
			bk_edit_url_handler ((gpointer) USE_DEFAULT_HANDLER, 0, NULL);
		}
	}
}


static gboolean child_reaper (gpointer pid)
{
	if (waitpid ((int) pid, NULL, WNOHANG) == (int) pid)
	{
		return FALSE;
	}

	return TRUE;
}


/* This is taken from popt-1.6.4/poptparse.c */
/* (C) 1998-2000 Red Hat, Inc. -- Licensing details are in the COPYING
   file accompanying popt source distributions, available from
   ftp://ftp.rpm.org/pub/rpm/dist. */

#define POPT_ERROR_NOARG -10
#define POPT_ERROR_MALLOC -21
#define POPT_ERROR_BADQUOTE -15

#define POPT_ARGV_ARRAY_GROW_DELTA 5

static int popt_dup_argv (int argc, const char **argv, int *argcPtr, const char ***argvPtr)
{
    size_t nb = (argc + 1) * sizeof(*argv);
    const char ** argv2;
    char * dst;
    int i;

    if (argc <= 0 || argv == NULL)	/* XXX can't happen */
	return POPT_ERROR_NOARG;
    for (i = 0; i < argc; i++) {
	if (argv[i] == NULL)
	    return POPT_ERROR_NOARG;
	nb += strlen(argv[i]) + 1;
    }

    dst = malloc(nb);
    if (dst == NULL)			/* XXX can't happen */
	return POPT_ERROR_MALLOC;
    argv2 = (void *) dst;
    dst += (argc + 1) * sizeof(*argv);

    /*@-branchstate@*/
    for (i = 0; i < argc; i++) {
	argv2[i] = dst;
	dst += strlen(strcpy(dst, argv[i])) + 1;
    }
    /*@=branchstate@*/
    argv2[argc] = NULL;

    if (argvPtr) {
	*argvPtr = argv2;
    } else {
	free(argv2);
	argv2 = NULL;
    }
    if (argcPtr)
	*argcPtr = argc;
    return 0;
}


static int popt_parse_argv_string (const char *s, int *argcPtr, const char ***argvPtr)
{
    const char * src;
    char quote = '\0';
    int argvAlloced = POPT_ARGV_ARRAY_GROW_DELTA;
    const char ** argv = malloc(sizeof(*argv) * argvAlloced);
    int argc = 0;
    int buflen = strlen(s) + 1;
    char * buf = memset(alloca(buflen), 0, buflen);
    int rc = POPT_ERROR_MALLOC;

    if (argv == NULL) return rc;
    argv[argc] = buf;

    for (src = s; *src != '\0'; src++) {
	if (quote == *src) {
	    quote = '\0';
	} else if (quote != '\0') {
	    if (*src == '\\') {
		src++;
		if (!*src) {
		    rc = POPT_ERROR_BADQUOTE;
		    goto exit;
		}
		if (*src != quote) *buf++ = '\\';
	    }
	    *buf++ = *src;
	} else if (isspace(*src)) {
	    if (*argv[argc] != '\0') {
		buf++, argc++;
		if (argc == argvAlloced) {
		    argvAlloced += POPT_ARGV_ARRAY_GROW_DELTA;
		    argv = realloc(argv, sizeof(*argv) * argvAlloced);
		    if (argv == NULL) goto exit;
		}
		argv[argc] = buf;
	    }
	} else switch (*src) {
	  case '"':
	  case '\'':
	    quote = *src;
	    /*@switchbreak@*/ break;
	  case '\\':
	    src++;
	    if (!*src) {
		rc = POPT_ERROR_BADQUOTE;
		goto exit;
	    }
	    /*@fallthrough@*/
	  default:
	    *buf++ = *src;
	    /*@switchbreak@*/ break;
	}
    }

    if (strlen(argv[argc])) {
	argc++, buf++;
    }

    rc = popt_dup_argv(argc, argv, argcPtr, argvPtr);

exit:
    if (argv) free(argv);
    return rc;
}

