/*
 * Copyright (C) 2001-2003 Clemens Fuchslocher <clfuit00@fht-esslingen.de>
 *
 * be_edit_undo.c - 23.05.2003 - v0.4 - modify indicator rewrite
 *                  17.04.2003 - v0.3 - sort order
 *                  26.10.2001 - v0.2 - copy, cut & paste functionality
 *                  05.09.2001 - v0.1
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 */

#include <stdlib.h>
#include <string.h>

#include "bk_edit_tree.h"
#include "bk_edit_icon.h"
#include "bk_edit_undo.h"
#include "bk_edit_sort.h"
#include "bk_edit_dialog_main.h"
#include "bk_edit_dialog_undo.h"

extern bk_edit_tree tree;
extern bk_edit_undo *undo;
extern bk_edit_dialog_main dialog_main;

static void save_folder_order (GtkCTreeNode *folder, GHashTable *order_table);
static void restore_folder_order (GtkCTreeNode *folder, GHashTable *order_table);
static gint folder_order_cmp (gconstpointer a, gconstpointer b);


bk_edit_undo *bk_edit_undo_new (bk_edit_undo *undo)
{
	undo = (bk_edit_undo *) malloc (sizeof (bk_edit_undo));
	if (undo == NULL)
	{
		return NULL;
	}

	undo->undo_stack = g_stack_new (undo->undo_stack);
	if (undo->undo_stack == NULL)
	{
		return NULL;
	}

	return undo;
}


void bk_edit_undo_delete (bk_edit_undo *undo)
{
	gint i, j;

	for (i = 0; i < undo->undo_stack->stack_pointer; i++)
	{
		switch (((bk_edit_undo_item *) (undo->undo_stack->stack[i]))->undo_type)
		{
			case UNDO_EDIT:
			{
				for (j = 0; j < ELEMENTS; j++)
				{
					free (((bk_edit_undo_item *) (undo->undo_stack->stack[i]))->undo_data.undo_edit->node_data.elements[j]);
				}
				free (((bk_edit_undo_item *) (undo->undo_stack->stack[i]))->undo_data.undo_edit);
				free (((bk_edit_undo_item *) (undo->undo_stack->stack[i])));

				break;
			}

			case UNDO_PASTE:
			case UNDO_NEW:
			{
				free (((bk_edit_undo_item *) (undo->undo_stack->stack[i]))->undo_data.undo_new);
				free (((bk_edit_undo_item *) (undo->undo_stack->stack[i])));

				break;
			}


			case UNDO_CUT:
			case UNDO_DELETE:
			{
				g_node_traverse (((bk_edit_undo_item *) (undo->undo_stack->stack[i]))->undo_data.undo_delete->nodes, G_IN_ORDER, G_TRAVERSE_ALL, -1, bk_edit_undo_traverse_gnode, NULL);
				g_node_destroy (((bk_edit_undo_item *) (undo->undo_stack->stack[i]))->undo_data.undo_delete->nodes);

				free (((bk_edit_undo_item *) (undo->undo_stack->stack[i]))->undo_data.undo_delete);
				free (((bk_edit_undo_item *) (undo->undo_stack->stack[i])));

				break;
			}

			case UNDO_MOVE:
			{
				free (((bk_edit_undo_item *) (undo->undo_stack->stack[i]))->undo_data.undo_move);
				free (((bk_edit_undo_item *) (undo->undo_stack->stack[i])));

				break;
			}
		}
	}

	g_stack_delete (undo->undo_stack);
	free (undo);
}


gboolean bk_edit_undo_traverse_gnode (GNode *node, gpointer data)
{
	gint i;

	for (i = 0; i < ELEMENTS; i++)
	{
		free (((bk_edit_tree_data *) (node->data))->elements[i]);
	}
	free ((bk_edit_tree_data *) (node->data));

	return FALSE;
}


gint bk_edit_undo_cmp (gconstpointer node_data, gconstpointer id)
{
	if (((bk_edit_tree_data *) node_data)->id == *((int *) id))
	{
		return FALSE;
	}

	return TRUE;
}


void bk_edit_undo_undo (void)
{
	gint i;
	GtkCTreeNode      *node;
	bk_edit_tree_data *node_data;
	bk_edit_undo_item *undo_item;

	undo_item = g_stack_pop (undo->undo_stack);
	if (undo_item == NULL)
	{
		fprintf (stderr, "%s[%d]: undo stack underrun.\n", __FILE__, __LINE__);
		return;
	}

	gtk_clist_freeze (GTK_CLIST (tree.tree));
	switch (undo_item->undo_type)
	{
		case UNDO_EDIT:
		{
			gint id = undo_item->undo_data.undo_edit->id;

			node = gtk_ctree_find_by_row_data_custom (GTK_CTREE (tree.tree), NULL, (gpointer *) &id, bk_edit_undo_cmp);
			if (node == NULL)
			{
				fprintf (stderr, "%s[%d]: bk_edit_undo_undo: can't find the related node.\n", __FILE__, __LINE__);
				return;
			}
			node_data = (bk_edit_tree_data *) gtk_ctree_node_get_row_data (GTK_CTREE (tree.tree), node);

			for (i = 0; i < ELEMENTS; i++)
			{
				free (node_data->elements[i]);
				node_data->elements[i] = strdup (undo_item->undo_data.undo_edit->node_data.elements[i]);

				free (undo_item->undo_data.undo_edit->node_data.elements[i]);
			}

			free (undo_item->undo_data.undo_edit);
			free (undo_item);

			if (node_data->type == FOLDER)
			{
				for (i = 0; i < ELEMENTS; i++)
				{
					gtk_ctree_node_set_text (GTK_CTREE (tree.tree), node, i, node_data->elements[i]);
		 		}

				gtk_ctree_node_set_pixtext (GTK_CTREE (tree.tree), node, NAME, node_data->elements[NAME], 4, GTK_CTREE_ROW (node)->expanded == 0 ? bookmark_folder_closed_pix : bookmark_folder_open_pix, GTK_CTREE_ROW (node)->expanded == 0 ? bookmark_folder_closed_mask : bookmark_folder_open_mask);

				bk_edit_tree_date_update (GTK_CTREE (tree.tree), node, node_data);
			}
			else if (node_data->type == BOOKMARK)
			{
				for (i = 0; i < ELEMENTS; i++)
				{
					gtk_ctree_node_set_text (GTK_CTREE (tree.tree), node, i, node_data->elements[i]);
		 		}

				gtk_ctree_node_set_pixtext (GTK_CTREE (tree.tree), node, NAME, node_data->elements[NAME], 4, bookmark_item_pix, bookmark_item_mask);

				bk_edit_tree_date_update (GTK_CTREE (tree.tree), node, node_data);
			}
			else if (node_data->type == SEPARATOR)
			{
			}
			else if (node_data->type == TOP)
			{
				gtk_ctree_node_set_text (GTK_CTREE (tree.tree), node, NAME, node_data->elements[NAME]);
				gtk_ctree_node_set_pixtext (GTK_CTREE (tree.tree), node, NAME, node_data->elements[NAME], 4, GTK_CTREE_ROW (node)->expanded == 0 ? bookmark_folder_closed_pix : bookmark_folder_open_pix, GTK_CTREE_ROW (node)->expanded == 0 ? bookmark_folder_closed_mask : bookmark_folder_open_mask);
			}

			bk_edit_sort_sort (bk_edit_sort_menu_get_menu_state (), gtk_ctree_sort_node, GTK_CTREE_ROW (node)->parent);

			break;
		}

		case UNDO_PASTE:
		case UNDO_NEW:
		{
			gint id = undo_item->undo_data.undo_new->id;

			node = gtk_ctree_find_by_row_data_custom (GTK_CTREE (tree.tree), NULL, (gpointer *) &id, bk_edit_undo_cmp);
			if (node == NULL)
			{
				fprintf (stderr, "%s[%d]: bk_edit_undo_undo: can't find the related node.\n", __FILE__, __LINE__);
				return;
			}

			free (undo_item->undo_data.undo_new);
			free (undo_item);

			restore_folder_order (GTK_CTREE_ROW (node)->parent, undo_item->undo_data.undo_new->order_table);
			g_hash_table_destroy (undo_item->undo_data.undo_new->order_table);

			gtk_ctree_remove_node (GTK_CTREE (tree.tree), node);

			break;
		}

		case UNDO_CUT:
		case UNDO_DELETE:
		{
			gint id_sibling = undo_item->undo_data.undo_delete->id_sibling;
			gint id_parent  = undo_item->undo_data.undo_delete->id_parent;

			GtkCTreeNode *node_sibling;
			GtkCTreeNode *node_parent;

			GNode *nodes = undo_item->undo_data.undo_delete->nodes;

			node_sibling = gtk_ctree_find_by_row_data_custom (GTK_CTREE (tree.tree), NULL, (gpointer *) &id_sibling, bk_edit_undo_cmp);
			if (node_sibling == NULL)
			{
				fprintf (stderr, "%s[%d]: bk_edit_undo_undo: can't find the related node.\n", __FILE__, __LINE__);
			}

			node_parent = gtk_ctree_find_by_row_data_custom (GTK_CTREE (tree.tree), NULL, (gpointer *) &id_parent, bk_edit_undo_cmp);
			if (node_parent == NULL)
			{
				fprintf (stderr, "%s[%d]: bk_edit_undo_undo: can't find the related node.\n", __FILE__, __LINE__);
			}

			node = gtk_ctree_insert_gnode (GTK_CTREE (tree.tree), node_parent, node_sibling, nodes, bk_edit_gnode_to_tree, NULL);
			bk_edit_tree_destroy_gnode_tree (nodes);

			restore_folder_order (node_parent, undo_item->undo_data.undo_delete->order_table);
			g_hash_table_destroy (undo_item->undo_data.undo_delete->order_table);

			free (undo_item->undo_data.undo_delete);
			free (undo_item);

			bk_edit_sort_sort (bk_edit_sort_menu_get_menu_state (), gtk_ctree_sort_node, GTK_CTREE_ROW (node)->parent);

			break;
		}

		case UNDO_MOVE:
		{
			gint id = undo_item->undo_data.undo_move->id;
			gint id_old_parent  = undo_item->undo_data.undo_move->id_old_parent;
			gint id_old_sibling = undo_item->undo_data.undo_move->id_old_sibling;

			GtkCTreeNode *old_parent;
			GtkCTreeNode *old_sibling;
			GtkCTreeNode *parent;

			node = gtk_ctree_find_by_row_data_custom (GTK_CTREE (tree.tree), NULL, (gpointer *) &id, bk_edit_undo_cmp);
			if (node == NULL)
			{
				fprintf (stderr, "%s[%d]: bk_edit_undo_undo: can't find the related node.\n", __FILE__, __LINE__);
				return;
			}

			parent = GTK_CTREE_ROW (node)->parent;
			old_parent = gtk_ctree_find_by_row_data_custom (GTK_CTREE (tree.tree), NULL, (gpointer *) &id_old_parent, bk_edit_undo_cmp);
			old_sibling = gtk_ctree_find_by_row_data_custom (GTK_CTREE (tree.tree), NULL, (gpointer *) &id_old_sibling, bk_edit_undo_cmp);

			gtk_signal_disconnect_by_func (GTK_OBJECT (tree.tree), GTK_SIGNAL_FUNC (bk_edit_tree_move), NULL);
			gtk_signal_disconnect_by_func (GTK_OBJECT (tree.tree), GTK_SIGNAL_FUNC (bk_edit_tree_move_after), NULL);
			gtk_ctree_move (GTK_CTREE (tree.tree), node, old_parent, old_sibling);
			gtk_signal_connect_after (GTK_OBJECT (tree.tree), "tree_move",
						GTK_SIGNAL_FUNC (bk_edit_tree_move_after), NULL);
			gtk_signal_connect (GTK_OBJECT (tree.tree), "tree_move",
						GTK_SIGNAL_FUNC (bk_edit_tree_move), NULL);

			restore_folder_order (old_parent, undo_item->undo_data.undo_move->target_order_table);
			restore_folder_order (parent, undo_item->undo_data.undo_move->source_order_table);

			g_hash_table_destroy (undo_item->undo_data.undo_move->source_order_table);
			g_hash_table_destroy (undo_item->undo_data.undo_move->target_order_table);

			free (undo_item->undo_data.undo_move);
			free (undo_item);

			bk_edit_sort_sort (bk_edit_sort_menu_get_menu_state (), gtk_ctree_sort_node, GTK_CTREE_ROW (node)->parent);

			break;
		}

		default:
		{
			fprintf (stderr, "%s[%d]: bk_edit_undo_undo: unkown action!\n", __FILE__, __LINE__);
		}
	}
	gtk_clist_thaw (GTK_CLIST (tree.tree));

	bk_edit_dialog_undo_update ();
}


int bk_edit_undo_add_edit_item (GtkCTreeNode *node, bk_edit_tree_data *node_data)
{
	gint i;

	bk_edit_undo_item *undo_item;
	bk_edit_undo_edit_item *undo_edit_item;

	undo_item = (bk_edit_undo_item *) malloc (sizeof (bk_edit_undo_item));
	if (undo_item == NULL)
	{
		return BK_EDIT_UNDO_OUT_OF_MEMORY;
	}

	undo_edit_item = (bk_edit_undo_edit_item *) malloc (sizeof (bk_edit_undo_edit_item));
	if (undo_edit_item == NULL)
	{
		free (undo_edit_item);
		return BK_EDIT_UNDO_OUT_OF_MEMORY;
	}

	for (i = 0; i < ELEMENTS; i++)
	{
		undo_edit_item->node_data.elements[i] = strdup (node_data->elements[i]);
	}
	undo_edit_item->id = node_data->id;

	undo_item->undo_type = UNDO_EDIT;
	undo_item->undo_data.undo_edit = undo_edit_item;

	if (g_stack_push (undo->undo_stack, (void *) undo_item) == G_STACK_FULL)
	{
		fprintf (stderr, "%s[%d]: bk_edit_undo_add_edit_item: undo stack full.\n", __FILE__, __LINE__);
		return BK_EDIT_UNDO_STACK_FULL;
	}

	bk_edit_dialog_undo_update ();

	return BK_EDIT_UNDO_OK;
}


int bk_edit_undo_add_move_item (GtkCTreeNode *node, GtkCTreeNode *new_parent, GtkCTreeNode *new_sibling)
{
	bk_edit_undo_item *undo_item;
	bk_edit_undo_move_item *undo_move_item;

	undo_item = (bk_edit_undo_item *) malloc (sizeof (bk_edit_undo_item));
	if (undo_item == NULL)
	{
		return BK_EDIT_UNDO_OUT_OF_MEMORY;
	}

	undo_move_item = (bk_edit_undo_move_item *) malloc (sizeof (bk_edit_undo_move_item));
	if (undo_move_item == NULL)
	{
		free (undo_item);
		return BK_EDIT_UNDO_OUT_OF_MEMORY;
	}

	undo_move_item->source_order_table = g_hash_table_new (g_direct_hash, folder_order_cmp);
	undo_move_item->target_order_table = g_hash_table_new (g_direct_hash, folder_order_cmp);

	save_folder_order (GTK_CTREE_ROW (node)->parent, undo_move_item->target_order_table);
	save_folder_order (new_parent, undo_move_item->source_order_table);

	undo_move_item->id = ((bk_edit_tree_data *) (GTK_CTREE_ROW (node)->row.data))->id;
	undo_move_item->id_old_parent = ((bk_edit_tree_data *) (GTK_CTREE_ROW (GTK_CTREE_ROW (node)->parent)->row.data))->id;

	if (GTK_CTREE_ROW (node)->sibling != NULL) /* node is at the end of the tree. there are no siblings left behind. */
	{
		undo_move_item->id_old_sibling = ((bk_edit_tree_data *) (GTK_CTREE_ROW (GTK_CTREE_ROW (node)->sibling)->row.data))->id;
	}
	else
	{
		undo_move_item->id_old_sibling = -1;
	}

	undo_item->undo_type = UNDO_MOVE;
	undo_item->undo_data.undo_move = undo_move_item;

	if (g_stack_push (undo->undo_stack, (void *) undo_item) == G_STACK_FULL)
	{
		fprintf (stderr, "%s[%d]: bk_edit_undo_add_move_item: undo stack full.\n", __FILE__, __LINE__);
		return BK_EDIT_UNDO_STACK_FULL;
	}

	bk_edit_dialog_undo_update ();

	return BK_EDIT_UNDO_OK;
}


int bk_edit_undo_add_delete_item (GtkCTreeNode *node)
{
	return bk_edit_undo_add_delete_or_cut_item (node, UNDO_DELETE);
}


int bk_edit_undo_add_cut_item (GtkCTreeNode *node)
{
	return bk_edit_undo_add_delete_or_cut_item (node, UNDO_CUT);
}


int bk_edit_undo_add_delete_or_cut_item (GtkCTreeNode *node, enum bk_edit_undo_type type)
{
	bk_edit_undo_item *undo_item;
	bk_edit_undo_delete_item *undo_delete_item;

	undo_item = (bk_edit_undo_item *) malloc (sizeof (bk_edit_undo_item));
	if (undo_item == NULL)
	{
		return BK_EDIT_UNDO_OUT_OF_MEMORY;
	}

	undo_delete_item = (bk_edit_undo_delete_item *) malloc (sizeof (bk_edit_undo_delete_item));
	if (undo_delete_item == NULL)
	{
		free (undo_item);
		return BK_EDIT_UNDO_OUT_OF_MEMORY;
	}

	undo_delete_item->order_table = g_hash_table_new (g_direct_hash, folder_order_cmp);
	save_folder_order (GTK_CTREE_ROW (node)->parent, undo_delete_item->order_table);

	if (GTK_CTREE_ROW (node)->sibling != NULL)
	{
		undo_delete_item->id_sibling = ((bk_edit_tree_data *) GTK_CTREE_ROW (GTK_CTREE_ROW (node)->sibling)->row.data)->id;
	}
	else
	{
		undo_delete_item->id_sibling = -1;
	}

	if (GTK_CTREE_ROW (node)->parent != NULL)
	{
		undo_delete_item->id_parent = ((bk_edit_tree_data *) GTK_CTREE_ROW (GTK_CTREE_ROW (node)->parent)->row.data)->id;
	}
	else
	{
		undo_delete_item->id_parent = -1;
	}

	undo_delete_item->nodes = gtk_ctree_export_to_gnode (GTK_CTREE (tree.tree), NULL, NULL, node, bk_edit_tree_to_gnode, NULL);

	undo_item->undo_type = type;
	undo_item->undo_data.undo_delete = undo_delete_item;

	if (g_stack_push (undo->undo_stack, (void *) undo_item) == G_STACK_FULL)
	{
		fprintf (stderr, "%s[%d]: bk_edit_undo_add_new_item: undo stack full.\n", __FILE__, __LINE__);
		return BK_EDIT_UNDO_STACK_FULL;
	}

	bk_edit_dialog_undo_update ();

	return BK_EDIT_UNDO_OK;
}


int bk_edit_undo_add_new_item (GtkCTreeNode *node, bk_edit_tree_data *node_data)
{
	return bk_edit_undo_add_new_or_paste_item (node, node_data, UNDO_NEW);
}


int bk_edit_undo_add_paste_item (GtkCTreeNode *node, bk_edit_tree_data *node_data)
{
	return bk_edit_undo_add_new_or_paste_item (node, node_data, UNDO_PASTE);
}


int bk_edit_undo_add_new_or_paste_item (GtkCTreeNode *node, bk_edit_tree_data *node_data, enum bk_edit_undo_type type)
{
	bk_edit_undo_item *undo_item;
	bk_edit_undo_new_item *undo_new_item;

	undo_item = (bk_edit_undo_item *) malloc (sizeof (bk_edit_undo_item));
	if (undo_item == NULL)
	{
		return BK_EDIT_UNDO_OUT_OF_MEMORY;
	}

	undo_new_item = (bk_edit_undo_new_item *) malloc (sizeof (bk_edit_undo_new_item));
	if (undo_new_item == NULL)
	{
		free (undo_item);
		return BK_EDIT_UNDO_OUT_OF_MEMORY;
	}
	undo_new_item->id = node_data->id;

	undo_new_item->order_table = g_hash_table_new (g_direct_hash, folder_order_cmp);
	save_folder_order (GTK_CTREE_ROW (node)->parent, undo_new_item->order_table);

	undo_item->undo_type = type;
	undo_item->undo_data.undo_new = undo_new_item;

	if (g_stack_push (undo->undo_stack, (void *) undo_item) == G_STACK_FULL)
	{
		fprintf (stderr, "%s[%d]: bk_edit_undo_add_new_item: undo stack full.\n", __FILE__, __LINE__);
		return BK_EDIT_UNDO_STACK_FULL;
	}

	bk_edit_dialog_undo_update ();

	return BK_EDIT_UNDO_OK;
}


static void save_folder_order (GtkCTreeNode *folder, GHashTable *order_table)
{
	GtkCTreeNode *child = (folder == NULL) ? GTK_CTREE_NODE (GTK_CLIST (tree.tree)->row_list) : GTK_CTREE_ROW (folder)->children;

	while (child != NULL)
	{
		bk_edit_tree_data *child_data = (bk_edit_tree_data *) gtk_ctree_node_get_row_data (GTK_CTREE (tree.tree), child);

		g_hash_table_insert (order_table, (gpointer) child_data->id, (gpointer) child_data->order);

		child = GTK_CTREE_ROW (child)->sibling;
	}
}


static void restore_folder_order (GtkCTreeNode *folder, GHashTable *order_table)
{
	GtkCTreeNode *child = (folder == NULL) ? GTK_CTREE_NODE (GTK_CLIST (tree.tree)->row_list) : GTK_CTREE_ROW (folder)->children;
	while (child != NULL)
	{
		bk_edit_tree_data *child_data = (bk_edit_tree_data *) gtk_ctree_node_get_row_data (GTK_CTREE (tree.tree), child);

		gpointer orig_key, value;

		if (g_hash_table_lookup_extended (order_table, (gpointer) child_data->id, &orig_key, &value))
		{
			child_data->order = (int) value;
		}
		else
		{
			fprintf (stderr, "%s[%d]: This shouldn't happen!\n", __FILE__, __LINE__);
		}

		child = GTK_CTREE_ROW (child)->sibling;
	}
}


static gint folder_order_cmp (gconstpointer a, gconstpointer b)
{
	if ((int) a == (int) b)
	{
		return TRUE;
	}

	return FALSE;
}
