/*
 * Copyright (C) 2003 Clemens Fuchslocher <clfuit00@fht-esslingen.de>
 *
 * bk_edit_sort.c - 17.04.2003 - v0.1
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "bk_edit.h"
#include "bk_edit_tree.h"
#include "bk_edit_sort.h"
#include "bk_edit_data.h"
#include "bk_edit_dialog_main.h"

extern bk_edit_tree tree;
extern bk_edit_dialog_main dialog_main;
extern bk_attribute_string bk_attribute_strings[];

static int order = 0;

static gint unsorted (GtkCList *clist, gconstpointer a, gconstpointer b);
static gint alphabetically (GtkCList *clist, gconstpointer a, gconstpointer b);
static gint alphabetically_folder_first (GtkCList *clist, gconstpointer a, gconstpointer b);
static gint alphabetically_reverse (GtkCList *clist, gconstpointer a, gconstpointer b);
static gint alphabetically_reverse_folder_first (GtkCList *clist, gconstpointer a, gconstpointer b);

order_string order_strings[] = {
	{ UNSORTED,                            "Unsorted",             unsorted                            },
	{ ALPHABETICALLY,                      "A > Z",                alphabetically                      },
	{ ALPHABETICALLY_FOLDER_FIRST,         "A > Z, Folders fisrt", alphabetically_folder_first         },
	{ ALPHABETICALLY_REVERSE,              "Z > A",                alphabetically_reverse              },
	{ ALPHABETICALLY_REVERSE_FOLDER_FIRST, "Z > A, Folders first", alphabetically_reverse_folder_first },
	{ -1,                                  NULL,                   NULL                                }
};


void bk_edit_sort (gpointer null, int order, GtkWidget *widget)
{
	bk_edit_sort_sort (order, gtk_ctree_sort_recursive, GTK_CTREE_NODE (GTK_CLIST (tree.tree)->row_list));
}


void bk_edit_sort_reorder_folder (GtkCTreeNode *folder, GtkCTreeNode *exclude_this_node_from_reorder)
{
	GtkCTreeNode *child;

	int n = 0;

	if (bk_edit_sort_menu_state_is_unsorted () == KO)
	{
		return;
	}

	if (folder == NULL)
	{
		/* Nothing to do. */
		return;
	}

	child = GTK_CTREE_ROW (folder)->children;
	while (child != NULL)
	{
		bk_edit_tree_data *data = (bk_edit_tree_data *) gtk_ctree_node_get_row_data (GTK_CTREE (tree.tree), child);

		if (child != exclude_this_node_from_reorder)
		{
			data->order = n++;
		}

		child = GTK_CTREE_ROW (child)->sibling;
	}
}


void bk_edit_sort_sort (int order, void (*sort) (GtkCTree *ctree, GtkCTreeNode *node), GtkCTreeNode *top)
{
	gtk_clist_set_compare_func (GTK_CLIST (tree.tree), order_strings[order].cmp);

	gtk_clist_freeze (GTK_CLIST (tree.tree));
	sort (GTK_CTREE (tree.tree), top);
	gtk_clist_thaw (GTK_CLIST (tree.tree));
}


void bk_edit_sort_menu_update (char *order)
{
	char *path = g_strdup_printf ("/Sort/%s", order);

	GtkWidget *item = gtk_item_factory_get_widget (dialog_main.menubar_factory, path);
	if (item != NULL)
	{
		gtk_menu_item_activate (GTK_MENU_ITEM (item));
	}

	free (path);
}


int bk_edit_sort_menu_get_menu_state (void)
{
	int n;

	for (n = 0; order_strings[n].string != NULL; n++)
	{
		char *path = g_strdup_printf ("/Sort/%s", order_strings[n].string);

		GtkWidget *item = gtk_item_factory_get_widget (dialog_main.menubar_factory, path);
		if (item != NULL)
		{
			if (GTK_CHECK_MENU_ITEM (item)->active)
			{
				return order_strings[n].order;
			}
		}

		free (path);
	}

	return -1;
}


void bk_edit_sort_menu_set_sort_order (gpointer null, int o, GtkWidget *widget)
{
	if (order != o)
	{
		order = o;

		bk_edit_sort_sort (order, gtk_ctree_sort_recursive, GTK_CTREE_NODE (GTK_CLIST (tree.tree)->row_list));
	}
}


void bk_edit_sort_reset_to_menu_value (void)
{
	GtkWidget *item = gtk_item_factory_get_widget (dialog_main.menubar_factory, "/Sort/Unsorted");
	if (item != NULL)
	{
		GSList *items = gtk_radio_menu_item_group (GTK_RADIO_MENU_ITEM (item));
		GSList *iter = items;

		while (iter != NULL)
		{
			GtkRadioMenuItem *item = GTK_RADIO_MENU_ITEM (iter->data);

			if (GTK_CHECK_MENU_ITEM (item)->active)
			{
				break;
			}

			iter = iter->next;
		}

		if (iter != NULL)
		{
			gtk_menu_item_activate (GTK_MENU_ITEM (GTK_RADIO_MENU_ITEM (iter->data)));
		}
	}
}


int bk_edit_sort_menu_state_is_unsorted (void)
{
	GtkWidget *item = gtk_item_factory_get_widget (dialog_main.menubar_factory, "/Sort/Unsorted");
	if (item == NULL)
	{
		return KO;
	}

	if (GTK_CHECK_MENU_ITEM (item)->active)
	{
		return OK;
	}

	return KO;
}


int bk_edit_sort_get_highest_order (GtkCTreeNode *folder, GtkCTreeNode *exclude_this_node_from_process)
{
	GtkCTreeNode *sibling = GTK_CTREE_ROW (folder)->children;

	int max_order = 0;

	while (sibling != NULL)
	{
		if (sibling != exclude_this_node_from_process)
		{
			bk_edit_tree_data *node_data = (bk_edit_tree_data *) gtk_ctree_node_get_row_data (GTK_CTREE (tree.tree), sibling);

			if (node_data->order > max_order)
			{
				max_order = node_data->order;
			}
		}
		sibling = GTK_CTREE_ROW (sibling)->sibling;
	}

	return ++max_order;
}


static gint unsorted (GtkCList *clist, gconstpointer a, gconstpointer b)
{
	bk_edit_tree_data *data_a = (bk_edit_tree_data *) ((GtkCListRow *) a)->data;
	bk_edit_tree_data *data_b = (bk_edit_tree_data *) ((GtkCListRow *) b)->data;

	if (data_a->order < data_b->order)
	{
		return -1;
	}

	return 0;
}


static gint alphabetically (GtkCList *clist, gconstpointer a, gconstpointer b)
{
	bk_edit_tree_data *data_a = (bk_edit_tree_data *) ((GtkCListRow *) a)->data;
	bk_edit_tree_data *data_b = (bk_edit_tree_data *) ((GtkCListRow *) b)->data;

	return strcasecmp (data_a->elements[order], data_b->elements[order]);
}


static gint alphabetically_folder_first (GtkCList *clist, gconstpointer a, gconstpointer b)
{
	bk_edit_tree_data *data_a = (bk_edit_tree_data *) ((GtkCListRow *) a)->data;
	bk_edit_tree_data *data_b = (bk_edit_tree_data *) ((GtkCListRow *) b)->data;

	if ((data_a->type == FOLDER) && (data_b->type != FOLDER))
	{
		return -1;
	}
	else if ((data_a->type != FOLDER) && (data_b->type == FOLDER))
	{
		return 1;
	}

	return strcasecmp (data_a->elements[order], data_b->elements[order]);
}


static gint alphabetically_reverse (GtkCList *clist, gconstpointer a, gconstpointer b)
{
	bk_edit_tree_data *data_a = (bk_edit_tree_data *) ((GtkCListRow *) a)->data;
	bk_edit_tree_data *data_b = (bk_edit_tree_data *) ((GtkCListRow *) b)->data;

	int cmp = strcasecmp (data_a->elements[order], data_b->elements[order]);
	if (cmp < 0)
	{
		return 1;
	}
	else if (cmp > 0)
	{
		return -1;
	}

	return 0;
}


static gint alphabetically_reverse_folder_first (GtkCList *clist, gconstpointer a, gconstpointer b)
{
	int cmp;
	bk_edit_tree_data *data_a = (bk_edit_tree_data *) ((GtkCListRow *) a)->data;
	bk_edit_tree_data *data_b = (bk_edit_tree_data *) ((GtkCListRow *) b)->data;

	if ((data_a->type == FOLDER) && (data_b->type != FOLDER))
	{
		return -1;
	}
	else if ((data_a->type != FOLDER) && (data_b->type == FOLDER))
	{
		return 1;
	}

	cmp = strcasecmp (data_a->elements[order], data_b->elements[order]);
	if (cmp < 0)
	{
		return 1;
	}
	else if (cmp > 0)
	{
		return -1;
	}

	return 0;
}

