/*
 * Copyright (C) 2001-2003 Clemens Fuchslocher <clfuit00@fht-esslingen.de>
 *
 * bk_edit_menu.c - 17.04.2003 - v0.13 - sort order
 *                  05.04.2003 - v0.12 - bugfix - underscore problem with "Recently Opened Documents"
 *                    (Reported by Andreas Busch <Andreas.Busch@politics.ox.ac.uk>)
 *                  01.03.2003 - v0.11 - "Recently Opened Documents"
 *                  15.02.2003 - v0.10 - preferences dialog, url handler
 *                  12.02.2003 - v0.9 - find functionality
 *                  11.02.2003 - v0.8 - modified menu shortcuts are saved now
 *                  18.12.2002 - v0.7 - external drag and drop support
 *                  08.03.2002 - v0.6 - new
 *                  26.02.2002 - v0.5 - plugin functionality
 *                  26.10.2001 - v0.4 - copy, cut & paste functionality
 *                  05.09.2001 - v0.3 - undo functionality
 *                  08.08.2001 - v0.2
 *                  24.07.2001 - v0.1
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 */

#include <string.h>
#include <stdlib.h>

#include <sys/stat.h>

#include <gdk/gdkkeysyms.h>

#include "config.h"

#include "bk_edit.h"
#include "bk_edit_ccp.h"
#include "bk_edit_file.h"
#include "bk_edit_undo.h"
#include "bk_edit_tree.h"
#include "bk_edit_edit.h"
#include "bk_edit_misc.h"
#include "bk_edit_sort.h"
#include "bk_edit_menu.h"
#include "bk_edit_data.h"
#include "bk_edit_url_handler.h"
#include "bk_edit_recent_document.h"

#include "bk_edit_dialog_dnd.h"
#include "bk_edit_dialog_undo.h"
#include "bk_edit_dialog_main.h"
#include "bk_edit_dialog_find.h"
#include "bk_edit_dialog_about.h"
#include "bk_edit_dialog_plugin.h"
#include "bk_edit_dialog_config.h"
#include "bk_edit_dialog_clipboard.h"

extern config *conf;
extern bk_edit_dialog_main dialog_main;
extern bk_attribute_string bk_attribute_strings[];

GtkItemFactoryEntry bk_edit_menubar_items[] = {
	{ "/_File",				NULL,	NULL,				0,	"<Branch>"	},
	{ "/_File/-",				NULL,	NULL,				0,	"<Tearoff>"	},
	{ "/_File/_New",			NULL,	bk_edit_new,			0,	"<Item>"	},
	{ "/_File/_Open...",			NULL,	bk_edit_file_open,		0,	"<Item>"	},
	{ "/_File/_Save...",			NULL,	bk_edit_file_save,		0,	"<Item>"	},
	{ "/_File/Save _as...",			NULL,	bk_edit_file_save_as,		0,	"<Item>"	},
	{ "/_File/Separator",			NULL,	NULL,				0,	"<Separator>"	},
	{ "/_File/_Preferences...",		NULL,	bk_edit_dialog_config_show,	0,	"<Item>"	},
	{ "/_File/Separator",			NULL,	NULL,				0,	"<Separator>"	},
	{ "/_File/RECENT-DOCUMENTS",		NULL,	NULL,				RECENT,	"<Item>"	},
	{ "/_File/Separator",			NULL,	NULL,				0,	"<Separator>"	},
	{ "/_File/E_xit",			NULL,	bk_edit_quit,			0,	"<Item>"	},
	{ "/_Edit",				NULL,	NULL,				0,	"<Branch>"	},
	{ "/_Edit/-",				NULL,	NULL,				0,	"<Tearoff>"	},
	{ "/_Edit/_Edit...",			NULL,	bk_edit_edit,			0,	"<Item>"	},
	{ "/_Edit/Separator",			NULL,	NULL,				0,	"<Separator>"	},
	{ "/_Edit/Cu_t ",			NULL,	bk_edit_ccp_cut,		0,	"<Item>"	},
	{ "/_Edit/_Copy ",			NULL,	bk_edit_ccp_copy,		0,	"<Item>"	},
	{ "/_Edit/_Paste ",			NULL,	bk_edit_ccp_paste,		0,	"<Item>"	},
	{ "/_Edit/_Delete",			NULL,	bk_edit_delete,			0,	"<Item>"	},
	{ "/_Edit/Separator ",			NULL,	NULL,				0,	"<Separator>"	},
	{ "/_Edit/_Undo",			NULL,	bk_edit_undo_undo,		0,	"<Item>"	},
	{ "/_Edit/Separator",			NULL,	NULL,				0,	"<Separator>"	},
	{ "/_Edit/New _Folder",			NULL,	bk_edit_new_folder,		0,	"<Item>"	},
	{ "/_Edit/New _Bookmark",		NULL,	bk_edit_new_bookmark,		0,	"<Item>"	},
	{ "/_Edit/New _Separator",		NULL,	bk_edit_new_separator,		0,	"<Item>"	},
	{ "/_Edit/Separator",			NULL,	NULL,				0,	"<Separator>"	},
	{ "/_Edit/_Find...",			NULL,	bk_edit_dialog_find_show,	0,	"<Item>"	},
	{ "/_Sort",				NULL,	NULL,				0,	"<Branch>"	},
	{ "/_Sort/-",				NULL,	NULL,				0,	"<Tearoff>"	},
	{ "/_Sort/Unsorted",			NULL,	bk_edit_sort,			UNSORTED,				"<RadioItem>"		},
	{ "/_Sort/A > Z",			NULL,	bk_edit_sort,			ALPHABETICALLY,				"/Sort/Unsorted"	},
	{ "/_Sort/A > Z, Folders fisrt",	NULL,	bk_edit_sort,			ALPHABETICALLY_FOLDER_FIRST,		"/Sort/Unsorted"	},
	{ "/_Sort/Z > A",			NULL,	bk_edit_sort,			ALPHABETICALLY_REVERSE,			"/Sort/Unsorted"	},
	{ "/_Sort/Z > A, Folders first",	NULL,	bk_edit_sort,			ALPHABETICALLY_REVERSE_FOLDER_FIRST,	"/Sort/Unsorted"	},
	{ "/_Sort/-",                           NULL,   NULL,                           0,      "<Separator>"   },
	{ "/_Sort/Sorted By",			NULL,	NULL,				SORT,	"<Branch>"	},
	{ "/_Misc",				NULL,	NULL,				0,	"<Branch>"	},
	{ "/_Misc/-",				NULL,	NULL,				0,	"<Tearoff>"	},
	{ "/_Misc/Show Plugins...",		NULL,	bk_edit_dialog_plugin_show,	0,	"<Item>"	},
	{ "/_Misc/Show Clipboard...",		NULL,	bk_edit_dialog_clipboard_show,	0,	"<Item>"	},
	{ "/_Misc/Show Undo Stack...",		NULL,	bk_edit_dialog_undo_show,	0,	"<Item>"	},
	{ "/_Misc/Show URL Dropzone...",	NULL,	bk_edit_dialog_dnd_show,	0,	"<Item>"	},
	{ "/_Misc/Separator",			NULL,	NULL,				0,	"<Separator>"	},
	{ "/_Misc/Expand Tree",			NULL,	bk_edit_tree_expand,		0,	"<Item>"	},
	{ "/_Misc/Collapse Tree",		NULL,	bk_edit_tree_collapse,		0,	"<Item>"	},
	{ "/_Help",				NULL,	NULL,				0,	"<LastBranch>"	},
	{ "/_Help/-",				NULL,	NULL,				0,	"<Tearoff>"	},
	{ "/_Help/About...",			NULL,	bk_edit_dialog_about,		0,	"<Item>"	}
};


GtkItemFactoryEntry bk_edit_popup_items[] = {
	{ "/Edit... ",		NULL,	bk_edit_edit,		0,	"<Item>"	},
	{ "/Separator",		NULL,	NULL,			0,	"<Separator>"	},
	{ "OPEN-WITH",		NULL,	NULL,			OPEN,	"<Item>"	},
	{ "/Separator",		NULL,	NULL,			0,	"<Separator>"	},
	{ "/Cut ",		NULL,	bk_edit_ccp_cut,	0,	"<Item>"	},
	{ "/Copy ",		NULL,	bk_edit_ccp_copy,	0,	"<Item>"	},
	{ "/Paste ",		NULL,	bk_edit_ccp_paste,	0,	"<Item>"	},
	{ "/Separator ",	NULL,	NULL,			0,	"<Separator>"	},
	{ "/Undo",		NULL,	bk_edit_undo_undo,	0,	"<Item>" 	},
	{ "/Separator ",	NULL,	NULL,			0,	"<Separator>"	},
	{ "/New Folder ",	NULL,	bk_edit_new_folder,	0,	"<Item>"	},
	{ "/New Bookmark ",	NULL,	bk_edit_new_bookmark,	0,	"<Item>"	},
	{ "/New Separator ",	NULL,	bk_edit_new_separator,	0,	"<Item>"	},
	{ "/Separator",		NULL,	NULL,			0,	"<Separator>"	},
	{ "/Delete ",		NULL,	bk_edit_delete,		0,	"<Item>"	}
};


gint bk_edit_menubar_size = sizeof (bk_edit_menubar_items) / sizeof (bk_edit_menubar_items[0]);
gint bk_edit_popup_size = sizeof (bk_edit_popup_items) / sizeof (bk_edit_popup_items[0]);


static gchar *default_shortcuts = "(menu-path \"<main>/File/New\" \"<Control>n\")\n"
	"(menu-path \"<main>/File/Open...\" \"<Control>o\")\n"
	"(menu-path \"<main>/File/Save...\" \"<Control>s\")\n"
	"(menu-path \"<main>/File/Save as...\" \"<Control>a\")\n"
	"(menu-path \"<main>/File/Preferences...\" \"<Control>p\")\n"
	"(menu-path \"<main>/File/Exit\" \"<Control>q\")\n"
	"(menu-path \"<main>/Edit/Edit...\" \"<Control>e\")\n"
	"(menu-path \"<main>/Edit/Cut \" \"<Control>x\")\n"
	"(menu-path \"<main>/Edit/Copy \" \"<Control>c\")\n"
	"(menu-path \"<main>/Edit/Paste \" \"<Control>v\")\n"
	"(menu-path \"<main>/Edit/Delete\" \"<Control>d\")\n"
	"(menu-path \"<main>/Edit/Undo\" \"<Control>z\")\n"
	"(menu-path \"<main>/Edit/New Folder\" \"<Control>l\")\n"
	"(menu-path \"<main>/Edit/New Bookmark\" \"<Control>b\")\n"
	"(menu-path \"<main>/Edit/New Separator\" \"<Control>r\")\n"
	"(menu-path \"<main>/Edit/Find...\" \"<Control>f\")\n"
	"(menu-path \"<main>/Misc/Expand Tree\" \"comma\")\n"
	"(menu-path \"<main>/Misc/Collapse Tree\" \"period\")\n"
	"(menu-path \"<main>/Help/About...\" \"F1\")\n";


void bk_edit_menu_init (void)
{
	struct stat st;

	char *filename = g_strdup_printf ("%s/" BK_EDIT_RC_MENU, bk_edit_misc_get_user_home_dir ());
	if (stat (filename, &st) == -1)
	{
		gtk_item_factory_parse_rc_string (default_shortcuts);

		free (filename);
		return;
	}
	gtk_item_factory_parse_rc (filename);

	free (filename);
}


void bk_edit_menu_serialize (void)
{
	char *filename = g_strdup_printf ("%s/" BK_EDIT_RC_MENU, bk_edit_misc_get_user_home_dir ());

	gtk_item_factory_dump_rc (filename, NULL, TRUE);

	free (filename);
}


int bk_edit_menubar_build (GtkItemFactoryEntry **menubar_items)
{
	GList *list = config_recent_documents_list_get (conf);

	int recent_documents_list_size = (g_list_length (list) > config_recent_documents_list_size_get (conf) ? config_recent_documents_list_size_get (conf) : g_list_length (list));
	int size = recent_documents_list_size + bk_edit_menubar_size - 1 + ELEMENTS;
	int n, nn;

	*menubar_items = (GtkItemFactoryEntry *) malloc (sizeof (GtkItemFactoryEntry) * size);

	for (n = 0, nn = 0; n < size; n++, nn++)
	{
		(*menubar_items)[n].path = strdup (bk_edit_menubar_items[nn].path);
		(*menubar_items)[n].accelerator = bk_edit_menubar_items[nn].accelerator;
		(*menubar_items)[n].callback = bk_edit_menubar_items[nn].callback;
		(*menubar_items)[n].callback_action = bk_edit_menubar_items[nn].callback_action;
		(*menubar_items)[n].item_type = strdup (bk_edit_menubar_items[nn].item_type);

		if (bk_edit_menubar_items[nn].callback_action == RECENT)
		{
			GList *l = list;

			while (l)
			{
				char *filename = bk_edit_misc_strescape (g_basename (((config_recent_document *) l->data)->filename), '_');

				(*menubar_items)[n].path = g_strdup_printf ("/_File/%s", filename);
				(*menubar_items)[n].accelerator = NULL;
				(*menubar_items)[n].callback = bk_edit_recent_document;
				(*menubar_items)[n].callback_action = n - nn;
				(*menubar_items)[n].item_type = strdup ("");

				l = l->next;

				n++;

				if (--recent_documents_list_size == 0)
				{
					break;
				}

				free (filename);
			}

			n--;
		}
		else if (bk_edit_menubar_items[nn].callback_action == SORT)
		{
			int i;

			for (i = 0; i < ELEMENTS; i++)
			{
				char *order = bk_edit_misc_strescape (bk_attribute_strings[i].string, '_');

				n++;

				(*menubar_items)[n].path = g_strdup_printf ("/_Sort/Sorted By/%s", order);
				(*menubar_items)[n].accelerator = NULL;
				(*menubar_items)[n].callback = bk_edit_sort_menu_set_sort_order;
				(*menubar_items)[n].callback_action = bk_attribute_strings[i].id;

				if (bk_attribute_strings[i].id == NAME)
				{
					(*menubar_items)[n].item_type = strdup ("<RadioItem>");
				}
				else
				{
					(*menubar_items)[n].item_type = g_strdup_printf ("/Sort/Sorted By/%s", bk_attribute_strings[0].string);
				}

				free (order);
			}
		}
	}

	config_recent_documents_list_free (conf, list);

	return size;
}


void bk_edit_menubar_free (GtkItemFactoryEntry menubar_items[], int menubar_size)
{
	int n;

	for (n = 0; n < menubar_size; n++)
	{
		free (menubar_items[n].path);
		free (menubar_items[n].item_type);
	}

	free (menubar_items);
}


void bk_edit_menubar_update_shortcuts (void)
{
	GList *l = config_recent_documents_list_get (conf);

	int n = 0, nn = config_recent_documents_list_size_get (conf);

	while (l)
	{
		GtkWidget *w;

		char *name = g_strdup_printf ("/File/%s", g_basename (((config_recent_document *) l->data)->filename));

		bk_edit_misc_remove_char_from_str (name, '_');

		w = gtk_item_factory_get_widget (dialog_main.menubar_factory, name);
		gtk_widget_remove_accelerators (w, "activate", TRUE);
		gtk_widget_add_accelerator (w, "activate", dialog_main.menubar_accelerators, GDK_1 + n, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE);

		free (name);

		if (++n >= nn)
		{
			break;
		}

		l = l->next;
	}

	config_recent_documents_list_free (conf, l);
}


int bk_edit_menu_popup_build (GtkItemFactoryEntry **popup_items)
{
	GList *url_handlers = config_url_handlers_get (conf);

	int size = g_list_length (url_handlers) + bk_edit_popup_size - 1;
	int n, nn;

	*popup_items = (GtkItemFactoryEntry *) malloc (sizeof (GtkItemFactoryEntry) * size);

	for (n = 0, nn = 0; n < size; n++, nn++)
	{
		(*popup_items)[n].path = strdup (bk_edit_popup_items[nn].path);
		(*popup_items)[n].accelerator = bk_edit_popup_items[nn].accelerator;
		(*popup_items)[n].callback = bk_edit_popup_items[nn].callback;
		(*popup_items)[n].callback_action = bk_edit_popup_items[nn].callback_action;
		(*popup_items)[n].item_type = strdup (bk_edit_popup_items[nn].item_type);

		if (bk_edit_popup_items[nn].callback_action == OPEN)
		{
			GList *url_handler = url_handlers;

			while (url_handler)
			{
				(*popup_items)[n].path = g_strdup_printf ("/Open with/%s", ((config_url_handler *) url_handler->data)->name);
				(*popup_items)[n].accelerator = NULL;
				(*popup_items)[n].callback = bk_edit_url_handler;
				(*popup_items)[n].callback_action = n - nn;
				(*popup_items)[n].item_type = strdup ("");

				url_handler = url_handler->next;

				n++;
			}

			n--;
		}
	}

	config_url_handlers_free (conf, url_handlers);

	return size;
}


void bk_edit_menu_popup_free (GtkItemFactoryEntry popup_items[], int popup_size)
{
	int n;

	for (n = 0; n < popup_size; n++)
	{
		free (popup_items[n].path);
		free (popup_items[n].item_type);
	}

	free (popup_items);
}

