/*
 * Copyright (C) 2002-2003 Clemens Fuchslocher <clfuit00@fht-esslingen.de>
 *
 * bk_edit_dnd.c - 17.04.2003 - v0.2 - sort order
 *                 18.12.2002 - v0.1
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 */

#include <time.h>
#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>

#include "bk_edit.h"
#include "bk_edit_dnd.h"
#include "bk_edit_edit.h"
#include "bk_edit_tree.h"
#include "bk_edit_data.h"
#include "bk_edit_undo.h"
#include "bk_edit_icon.h"
#include "bk_edit_sort.h"


extern bk_edit_tree tree;


GtkTargetEntry target_entry[] = {
	{ "application/x-xbel", 0, XBEL_URL },
	{ "GALEON_BOOKMARK", 0, GALEON_URL },
	{ "application/opera-hotlist", 0, OPERA_URL },
	{ "_NETSCAPE_URL", 0, NETSCAPE_URL },
	{ "text/uri-list", 0, TEXT_URI_LIST },
	{ "STRING", 0, STRING },
	{ "text/plain", 0, TEXT_PLAIN },
	{ "text/x-moz-url", 0, MOZILLA_URL },
	{ "gtk-clist-drag-reorder", 0, CLIST_DRAG }
};


static void uri_add (GtkWidget *dropzone, bk_edit_tree_data *node_data, int x, int y);
static void uri_list_add (GList *uris, GtkCTreeNode *parent, GtkCTreeNode *sibling);
static GList *uri_list_extract_uris (const gchar* uri_list);


void bk_edit_dnd_add_drop_support (GtkWidget *widget)
{
	if(!GTK_WIDGET_NO_WINDOW (widget))
	{
		gtk_drag_dest_set (widget, GTK_DEST_DEFAULT_ALL | GTK_DEST_DEFAULT_MOTION | GTK_DEST_DEFAULT_HIGHLIGHT | GTK_DEST_DEFAULT_DROP,
				target_entry, sizeof (target_entry) / sizeof (GtkTargetEntry), GDK_ACTION_COPY | GDK_ACTION_MOVE | GDK_ACTION_DEFAULT | GDK_ACTION_LINK | GDK_ACTION_PRIVATE | GDK_ACTION_ASK);

		gtk_signal_connect_after (GTK_OBJECT (widget), "drag_data_received", GTK_SIGNAL_FUNC (bk_edit_dnd_drag_data_received), NULL);
	}
	else
	{
		fprintf (stderr, "%s[%d]: !GTK_WIDGET_NO_WINDOW (widget)\n", __FILE__, __LINE__);
	}
}


void bk_edit_dnd_drag_data_received (GtkWidget *widget, GdkDragContext *dc, gint x, gint y, GtkSelectionData *selection_data, guint info, guint t, gpointer data)
{
	if (selection_data == NULL)
	{
		return;
	}

	if (selection_data->length < 0)
	{
		return;
	}

	if (selection_data->data[selection_data->length] != '\0')
	{
		selection_data->data[selection_data->length] = '\0';
	}

	switch (info)
	{
		case XBEL_URL:
		case GALEON_URL:
		{
			char *url;
			char *url_eor;
			char *name;
			char *name_eor;
			char timestr[20];

			gint i;

			bk_edit_tree_data *node_data = (bk_edit_tree_data *) malloc (sizeof (bk_edit_tree_data));

			if ((name = strstr (selection_data->data, "<title>")))
			{
				name += sizeof ("<title>") - 1;
				name_eor = strchr (name, '<');
			}

			if ((url = strstr (selection_data->data, "href=\"")))
			{
				url += sizeof ("href=\"") - 1;
				url_eor = strchr (url, '"');
			}

			if (name == NULL || name_eor == NULL)
			{
				name = "";
			}
			else
			{
				*name_eor = '\0';
			}

			if (url == NULL || url_eor == NULL)
			{
				url = "";
			}
			else
			{
				*url_eor = '\0';
			}

			/* Empty GtkCTree? */
			if (GTK_CLIST (tree.tree)->rows == 0)
			{
				/* Yes. Fill it with the root folder. */
				bk_edit_new_ok (NULL, NULL);
			}

			/* */
			node_data->type = BOOKMARK;
			node_data->id = tree.id++;
			for (i = 0; i < ELEMENTS; i++)
			{
				node_data->elements[i] = strdup ("");
			}

			free (node_data->elements[NAME]);
			node_data->elements[NAME] = strdup (name);

			free (node_data->elements[URI]);
			node_data->elements[URI] = strdup (url);

			snprintf (timestr, sizeof (timestr), "%ld", time (NULL));
			free (node_data->elements[ADD_DATE]);
			node_data->elements[ADD_DATE] = strdup (timestr);

			uri_add (widget, node_data, x, y);

			break;
		}

		case OPERA_URL:
		{
			char *url;
			char *url_eor;
			char *name;
			char *name_eor;
			char timestr[20];

			gint i;

			bk_edit_tree_data *node_data = (bk_edit_tree_data *) malloc (sizeof (bk_edit_tree_data));

			if ((name = strstr (selection_data->data, "NAME=")))
			{
				name += sizeof ("NAME=") - 1;
				name_eor = strchr (name, '\n');
			}

			if ((url = strstr (selection_data->data, "URL=")))
			{
				url += sizeof ("URL=") - 1;
				url_eor = strchr (url, '\n');
			}

			if (name == NULL || name_eor == NULL)
			{
				name = "";
			}
			else
			{
				*name_eor = '\0';
			}

			if (url == NULL || url_eor == NULL)
			{
				url = "";
			}
			else
			{
				*url_eor = '\0';
			}

			/* Empty GtkCTree? */
			if (GTK_CLIST (tree.tree)->rows == 0)
			{
				/* Yes. Fill it with the root folder. */
				bk_edit_new_ok (NULL, NULL);
			}

			/* */
			node_data->type = BOOKMARK;
			node_data->id = tree.id++;
			for (i = 0; i < ELEMENTS; i++)
			{
				node_data->elements[i] = strdup ("");
			}

			free (node_data->elements[NAME]);
			node_data->elements[NAME] = strdup (name);

			free (node_data->elements[URI]);
			node_data->elements[URI] = strdup (url);

			snprintf (timestr, sizeof (timestr), "%ld", time (NULL));
			free (node_data->elements[ADD_DATE]);
			node_data->elements[ADD_DATE] = strdup (timestr);

			uri_add (widget, node_data, x, y);

			break;
		}

		case MOZILLA_URL:
		{
			break;
		}

		case NETSCAPE_URL:
		{
			char *url = "";
			char *name = "";
			char timestr[20];

			gint i;

			bk_edit_tree_data *node_data = (bk_edit_tree_data *) malloc (sizeof (bk_edit_tree_data));

			/* Empty GtkCTree? */
			if (GTK_CLIST (tree.tree)->rows == 0)
			{
				/* Yes. Fill it with the root folder. */
				bk_edit_new_ok (NULL, NULL);
			}

			/* */
			node_data->type = BOOKMARK;
			node_data->id = tree.id++;
			for (i = 0; i < ELEMENTS; i++)
			{
				node_data->elements[i] = strdup ("");
			}

			url = selection_data->data;

			if ((name = strchr (selection_data->data, '\n')) != NULL)
			{
				*name++ = '\0';
			}
			else
			{
				name = "";
			}

			free (node_data->elements[NAME]);
			node_data->elements[NAME] = strdup (name);

			free (node_data->elements[URI]);
			node_data->elements[URI] = strdup (url);

			snprintf (timestr, sizeof (timestr), "%ld", time (NULL));
			free (node_data->elements[ADD_DATE]);
			node_data->elements[ADD_DATE] = strdup (timestr);

			uri_add (widget, node_data, x, y);

			break;
		}

		case TEXT_PLAIN:
		{
			char timestr[20];

			gint i;

			bk_edit_tree_data *node_data = (bk_edit_tree_data *) malloc (sizeof (bk_edit_tree_data));

			/* Empty GtkCTree? */
			if (GTK_CLIST (tree.tree)->rows == 0)
			{
				/* Yes. Fill it with the root folder. */
				bk_edit_new_ok (NULL, NULL);
			}

			/* */
			node_data->type = BOOKMARK;
			node_data->id = tree.id++;
			for (i = 0; i < ELEMENTS; i++)
			{
				node_data->elements[i] = strdup ("");
			}

			free (node_data->elements[NAME]);
			node_data->elements[NAME] = strdup (selection_data->data);

			free (node_data->elements[COMMENT]);
			node_data->elements[COMMENT] = strdup (selection_data->data);

			snprintf (timestr, sizeof (timestr), "%ld", time (NULL));
			free (node_data->elements[ADD_DATE]);
			node_data->elements[ADD_DATE] = strdup (timestr);

			uri_add (widget, node_data, x, y);

			break;
		}

		/*
		 * '5. The text/uri-list Internet Media Type'
		 * <http://ftp.isi.edu/in-notes/rfc2483.txt>
		 */
		case TEXT_URI_LIST:
		{
			gint row, column;

			GList *uris = uri_list_extract_uris (selection_data->data);

			if (uris == NULL)
			{
				break;
			}

			/* Empty GtkCTree? */
			if (GTK_CLIST (tree.tree)->rows == 0)
			{
				/* Yes. Fill it with the root folder. */
				bk_edit_new_ok (NULL, NULL);
			}

			/* Was the url dropped over the url dropzone? */
			if (GTK_WIDGET_TYPE (widget) != GTK_TYPE_CTREE)
			{
				/* Yes. Add the new node to the root folder. */
				GtkCTreeNode *parent = gtk_ctree_node_nth (GTK_CTREE (tree.tree), 0);
				uri_list_add (uris, parent, NULL);
			}
			/* Was the url dropped outside the GtkCTree? */
			else if ((gtk_clist_get_selection_info (GTK_CLIST (tree.tree), x, y, &row, &column)) == 0)
			{
				/* Yes. Add the new node to the root folder. */
				GtkCTreeNode *parent = gtk_ctree_node_nth (GTK_CTREE (tree.tree), 0);
				uri_list_add (uris, parent, NULL);
			}
			else
			{
				/* No. */
				GtkCTreeNode *sibling = gtk_ctree_node_nth (GTK_CTREE (tree.tree), row - 1);

				/* Is there a valid sibling? */
				if (sibling == NULL)
				{
					/* No. Add the new node to the root folder. */
					GtkCTreeNode *parent = gtk_ctree_node_nth (GTK_CTREE (tree.tree), 0);
					uri_list_add (uris, parent, NULL);
				}
				else
				{
					/* Is the sibling a folder or a leaf? */
					if (GTK_CTREE_ROW (sibling)->is_leaf == 1)
					{
						/* It's a leaf! Add the new node next to the sibling. */
						uri_list_add (uris, GTK_CTREE_ROW (sibling)->parent, sibling);
					}
					else
					{
						/* It's a folder! Add the new node to this folder. */
						uri_list_add (uris, sibling, NULL);
					}
				}
			}

			break;
		}
	}
}


static void uri_add (GtkWidget *dropzone, bk_edit_tree_data *node_data, int x, int y)
{
	GtkCTreeNode *node, *folder = NULL;

	gint row, column;

	/* Was the url dropped over the url dropzone? */
	if (GTK_WIDGET_TYPE (dropzone) != GTK_TYPE_CTREE)
	{
		/* Yes. Add the new node to the root folder. */
		GtkCTreeNode *parent = gtk_ctree_node_nth (GTK_CTREE (tree.tree), 0);
		node_data->order = bk_edit_sort_get_highest_order (parent, NULL);
		node = gtk_ctree_insert_node (GTK_CTREE (tree.tree), parent, NULL, node_data->elements, 4, bookmark_item_pix, bookmark_item_mask, bookmark_item_pix, bookmark_item_mask, TRUE, FALSE);
	}
	/* Was the url dropped outside the GtkCTree? */
	else if ((gtk_clist_get_selection_info (GTK_CLIST (tree.tree), x, y, &row, &column)) == 0)
	{
		/* Yes. Add the new node to the root folder. */
		GtkCTreeNode *parent = gtk_ctree_node_nth (GTK_CTREE (tree.tree), 0);
		node_data->order = bk_edit_sort_get_highest_order (parent, NULL);
		node = gtk_ctree_insert_node (GTK_CTREE (tree.tree), parent, NULL, node_data->elements, 4, bookmark_item_pix, bookmark_item_mask, bookmark_item_pix, bookmark_item_mask, TRUE, FALSE);
	}
	else
	{
		/* No. */
		GtkCTreeNode *sibling = gtk_ctree_node_nth (GTK_CTREE (tree.tree), row - 1);

		/* Is there a valid sibling? */
		if (sibling == NULL)
		{
			/* No. Add the new node to the root folder. */
			GtkCTreeNode *parent = gtk_ctree_node_nth (GTK_CTREE (tree.tree), 0);
			node_data->order = bk_edit_sort_get_highest_order (parent, NULL);
			node = gtk_ctree_insert_node (GTK_CTREE (tree.tree), parent, NULL, node_data->elements, 4, bookmark_item_pix, bookmark_item_mask, bookmark_item_pix, bookmark_item_mask, TRUE, FALSE);
		}
		else
		{
			/* Is the sibling a folder or a leaf? */
			if (GTK_CTREE_ROW (sibling)->is_leaf == 1)
			{
				/* tree is sorted -> bk_edit_sort_get_highest_order */
				if (bk_edit_sort_menu_state_is_unsorted () != OK)
				{
					node_data->order = bk_edit_sort_get_highest_order (GTK_CTREE_ROW (sibling)->parent, NULL);
				}
				else
				{
					folder = GTK_CTREE_ROW (sibling)->parent;
				}

				/* It's a leaf! Add the new node next to the sibling. */
				node = gtk_ctree_insert_node (GTK_CTREE (tree.tree), GTK_CTREE_ROW (sibling)->parent, sibling, node_data->elements, 4, bookmark_item_pix, bookmark_item_mask, bookmark_item_pix, bookmark_item_mask, TRUE, FALSE);
			}
			else
			{
				/* tree is sorted -> bk_edit_sort_get_highest_order */
				if (bk_edit_sort_menu_state_is_unsorted () != OK)
				{
					node_data->order = bk_edit_sort_get_highest_order (sibling, NULL);
				}
				else
				{
					folder = sibling;
				}

				/* It's a folder! Add the new node to this folder. */
				node = gtk_ctree_insert_node (GTK_CTREE (tree.tree), sibling, NULL, node_data->elements, 4, bookmark_item_pix, bookmark_item_mask, bookmark_item_pix, bookmark_item_mask, TRUE, FALSE);
			}
		}
	}

	gtk_ctree_node_set_row_data_full (GTK_CTREE (tree.tree), node, node_data, bk_edit_tree_destroy);
	bk_edit_tree_date_update (GTK_CTREE (tree.tree), node, node_data);
	gtk_clist_select_row (GTK_CLIST (tree.tree), gtk_clist_find_row_from_data (GTK_CLIST (tree.tree), node_data), 0);

	/* undo */
	bk_edit_undo_add_new_item (node, node_data);

	if (folder != NULL)
	{
		bk_edit_sort_reorder_folder (folder, NULL);
	}
	else
	{
		bk_edit_sort_sort (bk_edit_sort_menu_get_menu_state (), gtk_ctree_sort_node, GTK_CTREE_ROW (node)->parent);
	}
}


static void uri_list_add (GList *uris, GtkCTreeNode *parent, GtkCTreeNode *sibling)
{
	GtkCTreeNode *node;
	bk_edit_tree_data *node_data;

	gtk_clist_freeze (GTK_CLIST (tree.tree));

	uris = g_list_reverse (uris);
	for (; uris; uris = uris->next)
	{
		int i;
		char timestr[20];

		node_data = (bk_edit_tree_data *) malloc (sizeof (bk_edit_tree_data));
		node_data->type = BOOKMARK;
		node_data->id = tree.id++;
		for (i = 0; i < ELEMENTS; i++)
		{
			node_data->elements[i] = strdup ("");
		}

		free (node_data->elements[NAME]);
		node_data->elements[NAME] = strdup (uris->data);

		free (node_data->elements[URI]);
		node_data->elements[URI] = strdup (uris->data);

		snprintf (timestr, sizeof (timestr), "%ld", time (NULL));
		free (node_data->elements[ADD_DATE]);
		node_data->elements[ADD_DATE] = strdup (timestr);

		node = gtk_ctree_insert_node (GTK_CTREE (tree.tree), parent, sibling, node_data->elements, 4, bookmark_item_pix, bookmark_item_mask, bookmark_item_pix, bookmark_item_mask, TRUE, FALSE);

	 	bk_edit_tree_date_update (GTK_CTREE (tree.tree), node, node_data);
		gtk_ctree_node_set_row_data_full (GTK_CTREE (tree.tree), node, node_data, bk_edit_tree_destroy);

		/* undo */
		bk_edit_undo_add_new_item (node, node_data);

		if (bk_edit_sort_menu_state_is_unsorted () != OK)
		{
			node_data->order = bk_edit_sort_get_highest_order (parent, node);
		}
		else
		{
			bk_edit_sort_reorder_folder (parent, NULL);
		}
	}

	if (bk_edit_sort_menu_state_is_unsorted () != OK)
	{

		bk_edit_sort_sort (bk_edit_sort_menu_get_menu_state (), gtk_ctree_sort_node, GTK_CTREE_ROW (node)->parent);
	}

	gtk_clist_thaw (GTK_CLIST (tree.tree));

	gtk_clist_select_row (GTK_CLIST (tree.tree), gtk_clist_find_row_from_data (GTK_CLIST (tree.tree), node_data), 0);
}


/* This is taken from gnome-libs-1.4.2/libgnome/gnome-mime.c */
static GList *uri_list_extract_uris (const gchar* uri_list)
{
	const gchar *p, *q;
	gchar *retval;
	GList *result = NULL;

	g_return_val_if_fail (uri_list != NULL, NULL);

	p = uri_list;

	/* We don't actually try to validate the URI according to RFC
	 * 2396, or even check for allowed characters - we just ignore
	 * comments and trim whitespace off the ends.  We also
	 * allow LF delimination as well as the specified CRLF.
	 */
	while (p) {
		if (*p != '#') {
			while (isspace(*p))
				p++;

			q = p;
			while (*q && (*q != '\n') && (*q != '\r'))
				q++;

			if (q > p) {
			        q--;
				while (q > p && isspace(*q))
					q--;

				retval = g_malloc (q - p + 2);
				strncpy (retval, p, q - p + 1);
				retval[q - p + 1] = '\0';

				result = g_list_prepend (result, retval);
			}
		}
		p = strchr (p, '\n');
		if (p)
			p++;
	}

	return g_list_reverse (result);
}

