/*
 * Copyright (C) 2003 Clemens Fuchslocher <clfuit00@fht-esslingen.de>
 *
 * bk_edit_dialog_config.c - 17.04.2003 - v0.5 - sort order
 *                           05.04.2003 - v0.4 - edit with middle mouse click
 *                           05.04.2003 - v0.4 - default browser type
 *                           07.03.2003 - v0.3 - dialog options
 *                           01.03.2003 - v0.2 - "Recently Opened Documents"
 *                           15.02.2003 - v0.1
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 */

#include <string.h>

#include <gdk/gdk.h>
#include <gdk/gdkkeysyms.h>

#include "config.h"
#include "plugin.h"

#include "bk_edit_sort.h"
#include "bk_edit_misc.h"
#include "bk_edit_menu.h"
#include "bk_edit_url_handler.h"

#include "bk_edit_dialog_dnd.h"
#include "bk_edit_dialog_main.h"
#include "bk_edit_dialog_config.h"

extern config *conf;
extern plugins *plugs;
extern char *selected_plugin;
extern order_string order_strings[];

extern bk_edit_dialog_dnd dialog_dnd;
extern bk_edit_dialog_main dialog_main;
extern bk_edit_dialog_config dialog_config;

/* dialog options */
static GtkWidget *dialog_options_ui (void);
static void dialog_options_update (void);
static void dialog_options_get_position (void);
static void dialog_options_get_size (void);
static void size_toggled (GtkToggleButton *button, gpointer null);
static void position_toggled (GtkToggleButton *button, gpointer null);

/* general */
static GtkWidget *general_settings_ui (void);
static void general_settings_update (void);

/* url handler */
static GtkWidget *url_handler_ui (void);
static void url_handler_update (void);
static void url_handler_set_default_handler (void);
static void url_handler_new (void);
static void url_handler_delete (void);
static void url_handler_fill_in_entries (int row);
static void url_handler_fill_in_row (int row);
static void url_handler_entry_changed (GtkWidget *widget, GdkEventKey *event, gpointer *null);

static GtkItemFactory *url_handler_popup;
static GtkItemFactoryEntry url_handler_popup_items[] = {
	{ "/New",                    NULL, url_handler_new,                 0, NULL },
	{ "/Delete",                 NULL, url_handler_delete,              0, NULL },
	{ "/Separator",              NULL, NULL,                            0, "<Separator>" },
	{ "/Set as Default Handler", NULL, url_handler_set_default_handler, 0, NULL },
};


static void ok (void);
static void key_pressed (GtkWidget *widget, GdkEventKey *event, gpointer *null);
static void button_press_event (GtkWidget *widget, GdkEventButton *event, gpointer popup);


void bk_edit_dialog_config_show (void)
{
	if (dialog_config.dialog == NULL)
	{
		GtkWidget *button_box, *label;

		/* dialog_config.dialog */
		dialog_config.dialog = gtk_window_new (GTK_WINDOW_DIALOG);
		gtk_window_position (GTK_WINDOW (dialog_config.dialog), GTK_WIN_POS_MOUSE);
		gtk_window_set_title (GTK_WINDOW (dialog_config.dialog), "bk edit - preferences");
		gtk_window_set_policy (GTK_WINDOW (dialog_config.dialog), TRUE, TRUE, FALSE);
		gtk_widget_set_usize (GTK_WIDGET (dialog_config.dialog), 400, 470);
		gtk_container_set_border_width (GTK_CONTAINER (dialog_config.dialog), 4);
		gtk_signal_connect_object (GTK_OBJECT (dialog_config.dialog), "delete_event",
				GTK_SIGNAL_FUNC (gtk_widget_hide), GTK_OBJECT (dialog_config.dialog));
		gtk_widget_realize (dialog_config.dialog);

		/* enter & escape */
		gtk_signal_connect (GTK_OBJECT (dialog_config.dialog),
				"key_press_event", GTK_SIGNAL_FUNC (key_pressed), NULL);

		/* content */
		dialog_config.content = gtk_vbox_new (FALSE, 1);
		gtk_container_set_border_width (GTK_CONTAINER (dialog_config.content), 4);

		/* dialog_config.notebook */
                dialog_config.notebook = gtk_notebook_new ();

		/* dialog options */
		label = gtk_label_new ("Dialog Options");
		gtk_notebook_append_page (GTK_NOTEBOOK (dialog_config.notebook), dialog_options_ui (), label);

		/* general */
		label = gtk_label_new ("General Settings");
		gtk_notebook_append_page (GTK_NOTEBOOK (dialog_config.notebook), general_settings_ui (), label);

		/* url handler */
		label = gtk_label_new ("URL Handlers");
		gtk_notebook_append_page (GTK_NOTEBOOK (dialog_config.notebook), url_handler_ui (), label);

		gtk_box_pack_start (GTK_BOX (dialog_config.content), dialog_config.notebook, TRUE, TRUE, 0);

		/* ok */
		dialog_config.ok = gtk_button_new_with_label ("Ok");
		GTK_WIDGET_SET_FLAGS (dialog_config.ok, GTK_CAN_DEFAULT);
		gtk_signal_connect_object (GTK_OBJECT (dialog_config.ok), "clicked",
				GTK_SIGNAL_FUNC (ok), NULL);
		gtk_signal_connect_object (GTK_OBJECT (dialog_config.ok), "clicked",
				GTK_SIGNAL_FUNC (gtk_widget_hide), GTK_OBJECT (dialog_config.dialog));

		/* cancle */
		dialog_config.cancel = gtk_button_new_with_label ("Cancel");
		GTK_WIDGET_SET_FLAGS (dialog_config.cancel, GTK_CAN_DEFAULT);
		gtk_signal_connect_object (GTK_OBJECT (dialog_config.cancel), "clicked",
						GTK_SIGNAL_FUNC (gtk_widget_hide),
						GTK_OBJECT (dialog_config.dialog));

		/* box */
		button_box = gtk_hbutton_box_new ();
		gtk_button_box_set_layout (GTK_BUTTON_BOX (button_box), GTK_BUTTONBOX_DEFAULT_STYLE);
		gtk_button_box_set_spacing (GTK_BUTTON_BOX (button_box), 5);

		/* pack */
		gtk_container_add (GTK_CONTAINER (button_box), dialog_config.ok);
		gtk_container_add (GTK_CONTAINER (button_box), dialog_config.cancel);

		gtk_box_pack_start (GTK_BOX (dialog_config.content), button_box, FALSE, FALSE, 5);
		gtk_container_add (GTK_CONTAINER (dialog_config.dialog), dialog_config.content);

		/* */
		gtk_widget_grab_default (dialog_config.ok);

		/* */
		gtk_widget_show_all (dialog_config.dialog);
	}

	gtk_widget_grab_focus (dialog_config.ok);

	/* dialog options */
	dialog_options_update ();

	/* general settings */
	general_settings_update ();

	/* url handler */
	url_handler_update ();
	url_handler_fill_in_entries (0);

	/* raise */
	gtk_widget_show (dialog_config.dialog);
	gdk_window_show (dialog_config.dialog->window);
}


static GtkWidget *dialog_options_ui (void)
{
	GtkWidget *content = gtk_vbox_new (FALSE, 1);
	GtkWidget *frame, *table, *hbox, *align, *button;
	GtkObject *adjustment;

	int width;

	table = gtk_table_new (2, 5, FALSE);

	frame = gtk_frame_new ("URL Dropzone");
	gtk_container_set_border_width (GTK_CONTAINER (frame), 4);
	gtk_box_pack_start (GTK_BOX (content), frame, FALSE, FALSE, 4);

	dialog_config.sticky = gtk_check_button_new_with_label (" Sticky");
	gtk_table_attach (GTK_TABLE (table), dialog_config.sticky, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 0, 4);

	dialog_config.always_on_top = gtk_check_button_new_with_label (" Always On Top");
	gtk_table_attach (GTK_TABLE (table), dialog_config.always_on_top, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 0, 4);

	/* size */
	hbox = gtk_hbox_new (FALSE, 2);

	dialog_config.size = gtk_check_button_new_with_label (" Size");
	gtk_signal_connect (GTK_OBJECT (dialog_config.size), "toggled", size_toggled, NULL);
	gtk_table_attach (GTK_TABLE (table), dialog_config.size, 0, 1, 2, 3, GTK_FILL, GTK_FILL, 0, 4);

	adjustment = gtk_adjustment_new (0, 0, 10000, 1, 1, 0);
	dialog_config.width = gtk_spin_button_new (GTK_ADJUSTMENT (adjustment), 0.5, 0);
	width = gdk_text_width (dialog_config.width->style->font, "0", 1);
	gtk_widget_set_usize (GTK_WIDGET (dialog_config.width), width * 8, -1);
	gtk_box_pack_start (GTK_BOX (hbox), dialog_config.width, FALSE, FALSE, 4);

	adjustment = gtk_adjustment_new (0, 0, 10000, 1, 1, 0);
	dialog_config.height = gtk_spin_button_new (GTK_ADJUSTMENT (adjustment), 0.5, 0);
	width = gdk_text_width (dialog_config.height->style->font, "0", 1);
	gtk_widget_set_usize (GTK_WIDGET (dialog_config.height), width * 8, -1);
	gtk_box_pack_start (GTK_BOX (hbox), dialog_config.height, FALSE, FALSE, 4);
	gtk_table_attach (GTK_TABLE (table), hbox, 1, 2, 2, 3, GTK_FILL, GTK_FILL, 0, 4);

	button = gtk_button_new_with_label (" Get Size ");
	gtk_signal_connect_object (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC (dialog_options_get_size), NULL);
	gtk_table_attach (GTK_TABLE (table), button, 2, 3, 2, 3, GTK_FILL, GTK_FILL, 0, 4);

	/* position */
	hbox = gtk_hbox_new (FALSE, 2);

	dialog_config.position = gtk_check_button_new_with_label (" Position");
	gtk_signal_connect (GTK_OBJECT (dialog_config.position), "toggled", position_toggled, NULL);
	gtk_table_attach (GTK_TABLE (table), dialog_config.position, 0, 1, 3, 4, GTK_FILL, GTK_FILL, 0, 4);

	adjustment = gtk_adjustment_new (0, -10000, 10000, 1, 1, 0);
	dialog_config.x = gtk_spin_button_new (GTK_ADJUSTMENT (adjustment), 0.5, 0);
	width = gdk_text_width (dialog_config.x->style->font, "0", 1);
	gtk_widget_set_usize (GTK_WIDGET (dialog_config.x), width * 8, -1);
	gtk_box_pack_start (GTK_BOX (hbox), dialog_config.x, FALSE, FALSE, 4);

	adjustment = gtk_adjustment_new (0, -10000, 10000, 1, 1, 0);
	dialog_config.y = gtk_spin_button_new (GTK_ADJUSTMENT (adjustment), 0.5, 0);
	width = gdk_text_width (dialog_config.y->style->font, "0", 1);
	gtk_widget_set_usize (GTK_WIDGET (dialog_config.y), width * 8, -1);
	gtk_box_pack_start (GTK_BOX (hbox), dialog_config.y, FALSE, FALSE, 4);
	gtk_table_attach (GTK_TABLE (table), hbox, 1, 2, 3, 4, GTK_FILL, GTK_FILL, 0, 4);

	button = gtk_button_new_with_label (" Get Position ");
	gtk_signal_connect_object (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC (dialog_options_get_position), NULL);
	gtk_table_attach (GTK_TABLE (table), button, 2, 3, 3, 4, GTK_FILL, GTK_FILL, 0, 4);

	align = gtk_alignment_new (0.5, 0.5, 0, 0);
	gtk_container_add (GTK_CONTAINER (align), table);
	gtk_container_add (GTK_CONTAINER (frame), align);

	return content;
}


static void dialog_options_update (void)
{
	int x, y, width, height;

	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog_config.sticky), config_dialog_options_url_dropzone_sticky_get (conf));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog_config.always_on_top), config_dialog_options_url_dropzone_always_on_top_get (conf));

	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog_config.size), config_dialog_options_url_dropzone_size_state_get (conf));
	config_dialog_options_url_dropzone_size_get (conf, &width, &height);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (dialog_config.width), (float) width);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (dialog_config.height), (float) height);

	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog_config.position), config_dialog_options_url_dropzone_position_state_get (conf));
	config_dialog_options_url_dropzone_position_get (conf, &x, &y);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (dialog_config.x), (float) x);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (dialog_config.y), (float) y);

	size_toggled (NULL, NULL);
	position_toggled (NULL, NULL);
}


static void dialog_options_get_position (void)
{
	int x, y;

	if (dialog_dnd.dialog == NULL)
	{
		return;
	}

	gdk_window_get_position (dialog_dnd.dialog->window, &x, &y);

	gtk_spin_button_set_value (GTK_SPIN_BUTTON (dialog_config.x), x);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (dialog_config.y), y);
}


static void dialog_options_get_size (void)
{
	int width, height;

	if (dialog_dnd.dialog == NULL)
	{
		return;
	}

	gdk_window_get_geometry (dialog_dnd.dialog->window, NULL, NULL, &width, &height, NULL);

	gtk_spin_button_set_value (GTK_SPIN_BUTTON (dialog_config.width), width);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (dialog_config.height), height);
}


static void size_toggled (GtkToggleButton *button, gpointer null)
{
	int state = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog_config.size));
	gtk_widget_set_sensitive (GTK_WIDGET (dialog_config.width), state);
	gtk_widget_set_sensitive (GTK_WIDGET (dialog_config.height), state);
}


static void position_toggled (GtkToggleButton *button, gpointer null)
{
	int state = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog_config.position));
	gtk_widget_set_sensitive (GTK_WIDGET (dialog_config.x), state);
	gtk_widget_set_sensitive (GTK_WIDGET (dialog_config.y), state);
}


static GtkWidget *general_settings_ui (void)
{
	GtkWidget *content = gtk_vbox_new (FALSE, 1);
	GtkWidget *frame, *align, *box, *label, *menu;
	GtkObject *adjustment;
	GList *iter;

	int width, n;

	/* Recently Opened Documents */
	frame = gtk_frame_new ("Recently Opened Documents");
	gtk_container_set_border_width (GTK_CONTAINER (frame), 4);
	gtk_box_pack_start (GTK_BOX (content), frame, FALSE, FALSE, 4);

	box = gtk_hbox_new (FALSE, 2);
	gtk_container_set_border_width (GTK_CONTAINER (box), 4);

	label = gtk_label_new ("Recent Documents List Size: ");
	gtk_box_pack_start (GTK_BOX (box), label, FALSE, FALSE, 4);

	adjustment = gtk_adjustment_new (0, 0, 1000, 1, 1, 0);
	dialog_config.recent_documents_list_size = gtk_spin_button_new (GTK_ADJUSTMENT (adjustment), 0.5, 0);
	width = gdk_text_width (dialog_config.recent_documents_list_size->style->font, "0", 1);
	gtk_widget_set_usize (GTK_WIDGET (dialog_config.recent_documents_list_size), width * 10, -1);

	gtk_box_pack_start (GTK_BOX (box), dialog_config.recent_documents_list_size, FALSE, FALSE, 4);

	align = gtk_alignment_new (0.5, 0.5, 0, 0);
	gtk_container_add (GTK_CONTAINER (align), box);
	gtk_container_add (GTK_CONTAINER (frame), align);

	/* Default Browser Type */
	frame = gtk_frame_new ("Default Browser Type");
	gtk_container_set_border_width (GTK_CONTAINER (frame), 4);
	gtk_box_pack_start (GTK_BOX (content), frame, FALSE, FALSE, 4);

	box = gtk_hbox_new (FALSE, 2);
	gtk_container_set_border_width (GTK_CONTAINER (box), 4);

	menu = gtk_menu_new ();
	dialog_config.browser_type = gtk_option_menu_new ();

	for (iter = plugs->list; iter != NULL; iter = g_list_next (iter))
	{
		if (((plugin *) iter->data)->type & PLUGIN_LOAD)
		{
			GtkWidget *menu_item = gtk_menu_item_new_with_label (((plugin *) iter->data)->name);
			gtk_menu_append (GTK_MENU (menu), menu_item);
			gtk_widget_show (menu_item);
		}
	}
	gtk_option_menu_set_menu (GTK_OPTION_MENU (dialog_config.browser_type), menu);
	bk_edit_misc_set_menu_option (GTK_OPTION_MENU (dialog_config.browser_type), selected_plugin);

	label = gtk_label_new ("Browser Type: ");
	gtk_box_pack_start (GTK_BOX (box), label, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (box), dialog_config.browser_type, TRUE, TRUE, 0);
	gtk_container_add (GTK_CONTAINER (frame), box);

	/* Sort Order */
	frame = gtk_frame_new ("Default Sort Order");
	gtk_container_set_border_width (GTK_CONTAINER (frame), 4);
	gtk_box_pack_start (GTK_BOX (content), frame, FALSE, FALSE, 4);

	box = gtk_hbox_new (FALSE, 2);
	gtk_container_set_border_width (GTK_CONTAINER (box), 4);

	menu = gtk_menu_new ();
	dialog_config.sort_order = gtk_option_menu_new ();

	for (n = 0; order_strings[n].string != NULL; n++)
	{
		GtkWidget *menu_item = gtk_menu_item_new_with_label (order_strings[n].string);
		gtk_menu_append (GTK_MENU (menu), menu_item);
		gtk_widget_show (menu_item);
	}

	gtk_option_menu_set_menu (GTK_OPTION_MENU (dialog_config.sort_order), menu);
	bk_edit_misc_set_menu_option (GTK_OPTION_MENU (dialog_config.sort_order), selected_plugin);

	label = gtk_label_new ("Sort Order: ");
	gtk_box_pack_start (GTK_BOX (box), label, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (box), dialog_config.sort_order, TRUE, TRUE, 0);
	gtk_container_add (GTK_CONTAINER (frame), box);

	/* Misc */
	frame = gtk_frame_new ("Misc");
	gtk_container_set_border_width (GTK_CONTAINER (frame), 4);
	gtk_box_pack_start (GTK_BOX (content), frame, FALSE, FALSE, 4);

	box = gtk_hbox_new (FALSE, 2);
	gtk_container_set_border_width (GTK_CONTAINER (box), 4);

	dialog_config.middle_mouse = gtk_check_button_new_with_label (" Edit bookmark with middle-mouse click");
	align = gtk_alignment_new (0.5, 0.5, 0, 0);
	gtk_container_add (GTK_CONTAINER (align), box);
	gtk_box_pack_start (GTK_BOX (box), dialog_config.middle_mouse, FALSE, FALSE, 4);
	gtk_container_add (GTK_CONTAINER (frame), align);

	return content;
}


static void general_settings_update (void)
{
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (dialog_config.recent_documents_list_size), (float) config_recent_documents_list_size_get (conf));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog_config.middle_mouse), config_middle_mouse_get (conf));
	bk_edit_misc_set_menu_option (GTK_OPTION_MENU (dialog_config.browser_type), config_browser_type_get (conf));
	bk_edit_misc_set_menu_option (GTK_OPTION_MENU (dialog_config.sort_order), config_sort_order_get (conf));
}


static GtkWidget *url_handler_ui (void)
{
	GtkWidget *content = gtk_vbox_new (FALSE, 1);
	GtkWidget *scroll, *label, *table, *new, *delete, *frame, *box;

	char *titles[] = { "Name", "Command" };

	/* list */
	dialog_config.url_handler_list = gtk_clist_new_with_titles (2, titles);
	gtk_clist_set_row_height (GTK_CLIST (dialog_config.url_handler_list), 19);
	gtk_clist_set_shadow_type (GTK_CLIST (dialog_config.url_handler_list), GTK_SHADOW_IN);
	gtk_clist_set_selection_mode (GTK_CLIST (dialog_config.url_handler_list), GTK_SELECTION_BROWSE);
	gtk_clist_set_column_auto_resize (GTK_CLIST (dialog_config.url_handler_list), 0, TRUE);
	gtk_clist_set_reorderable (GTK_CLIST (dialog_config.url_handler_list), TRUE);
	gtk_clist_column_titles_active (GTK_CLIST (dialog_config.url_handler_list));

	scroll = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll), GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);

	gtk_container_add (GTK_CONTAINER (scroll), dialog_config.url_handler_list);
	gtk_box_pack_start (GTK_BOX (content), scroll, TRUE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (content), 4);

	/* name & command */
	table = gtk_table_new (2, 3, FALSE);

	label = gtk_label_new ("Name: ");
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	dialog_config.url_handler_name_entry = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 5, 1);
	gtk_table_attach (GTK_TABLE (table), dialog_config.url_handler_name_entry, 1, 2, 0, 1, GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 1, 1);

	gtk_signal_connect_after (GTK_OBJECT (dialog_config.url_handler_name_entry),
			"key_press_event", GTK_SIGNAL_FUNC (url_handler_entry_changed), NULL);

	label = gtk_label_new ("Command: ");
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	dialog_config.url_handler_command_entry = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 5, 1);
	gtk_table_attach (GTK_TABLE (table), dialog_config.url_handler_command_entry, 1, 2, 1, 2, GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 1, 1);

	gtk_signal_connect_after (GTK_OBJECT (dialog_config.url_handler_command_entry),
			"key_press_event", GTK_SIGNAL_FUNC (url_handler_entry_changed), NULL);

	/* new & delete */
	new = gtk_button_new_with_label ("  New  ");
	gtk_signal_connect_object (GTK_OBJECT (new), "clicked",
			GTK_SIGNAL_FUNC (url_handler_new), NULL);

	delete = gtk_button_new_with_label ("  Delete  ");
	gtk_signal_connect_object (GTK_OBJECT (delete), "clicked",
			GTK_SIGNAL_FUNC (url_handler_delete), NULL);

	gtk_table_attach (GTK_TABLE (table), new, 2, 3, 0, 1, GTK_FILL, GTK_FILL, 5, 1);
	gtk_table_attach (GTK_TABLE (table), delete, 2, 3, 1, 2, GTK_FILL, GTK_FILL, 5, 1);

	box = gtk_vbox_new (FALSE, 1);
	gtk_box_pack_start (GTK_BOX (box), table, TRUE, FALSE, 4);

	frame = gtk_frame_new ("URL Handler");
	gtk_container_add (GTK_CONTAINER (frame), box);
	gtk_box_pack_start (GTK_BOX (content), frame, FALSE, FALSE, 4);

	/* legend */
	table = gtk_table_new (3, 1, FALSE);
	gtk_table_attach (GTK_TABLE (table), gtk_label_new ("%n: Name"),    0, 1, 0, 1, GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 1, 5);
	gtk_table_attach (GTK_TABLE (table), gtk_label_new ("%u: URI"),     1, 2, 0, 1, GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 1, 5);
	gtk_table_attach (GTK_TABLE (table), gtk_label_new ("%c: Comment"), 2, 3, 0, 1, GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 1, 5);

	frame = gtk_frame_new ("Legend");
	gtk_container_add (GTK_CONTAINER (frame), table);
	gtk_box_pack_start (GTK_BOX (content), frame, FALSE, FALSE, 4);

	/* popup */
	url_handler_popup = gtk_item_factory_new (GTK_TYPE_MENU, "<popup>", NULL);
	gtk_item_factory_create_items (url_handler_popup,
			sizeof (url_handler_popup_items) / sizeof (url_handler_popup_items[0]),
			url_handler_popup_items, NULL);

	/* */
	gtk_signal_connect_after (GTK_OBJECT (dialog_config.url_handler_list), "button_press_event",
			GTK_SIGNAL_FUNC (button_press_event), GTK_OBJECT (url_handler_popup));

	return content;
}


static void url_handler_update (void)
{
	GList *url_handlers = config_url_handlers_get (conf);
	GList *url_handler = url_handlers;

	gtk_clist_freeze (GTK_CLIST (dialog_config.url_handler_list));
	gtk_clist_clear (GTK_CLIST (dialog_config.url_handler_list));
	while (url_handler != NULL)
	{
		char *entry[] = { ((config_url_handler *) url_handler->data)->name,
			((config_url_handler *) url_handler->data)->command };

		int row = gtk_clist_append (GTK_CLIST (dialog_config.url_handler_list), entry);
		gtk_clist_set_row_data (GTK_CLIST (dialog_config.url_handler_list), row, (gpointer) ((config_url_handler *) url_handler->data)->default_handler);

		if (((config_url_handler *) url_handler->data)->default_handler == CONFIG_URL_HANDLER_DEFAULT)
		{
			GdkColor gray = { 0, 0xCCCC, 0xCCCC, 0xCCCC };
			gtk_clist_set_background (GTK_CLIST (dialog_config.url_handler_list), row, &gray);
		}

		url_handler = url_handler->next;
	}
	gtk_clist_thaw (GTK_CLIST (dialog_config.url_handler_list));

	config_url_handlers_free (conf, url_handlers);
}


static void url_handler_set_default_handler (void)
{
	GdkColor white = { 0, 0xFFFF, 0xFFFF, 0xFFFF };
	GdkColor gray = { 0, 0xCCCC, 0xCCCC, 0xCCCC };

	int old = gtk_clist_find_row_from_data (GTK_CLIST (dialog_config.url_handler_list), (gpointer) CONFIG_URL_HANDLER_DEFAULT);
	int row = bk_edit_misc_get_selection (GTK_CLIST (dialog_config.url_handler_list));
	if (row < 0)
	{
		return;
	}

	gtk_clist_set_background (GTK_CLIST (dialog_config.url_handler_list), old, &white);
	gtk_clist_set_background (GTK_CLIST (dialog_config.url_handler_list), row, &gray);

	gtk_clist_set_row_data (GTK_CLIST (dialog_config.url_handler_list), old, (gpointer) CONFIG_URL_HANDLER_NORMAL);
	gtk_clist_set_row_data (GTK_CLIST (dialog_config.url_handler_list), row, (gpointer) CONFIG_URL_HANDLER_DEFAULT);
}


static void url_handler_new (void)
{
	char *entry[] = { "New", "" };

	url_handler_fill_in_entries (gtk_clist_append (GTK_CLIST (dialog_config.url_handler_list), entry));
}


static void url_handler_delete (void)
{
	int row = bk_edit_misc_get_selection (GTK_CLIST (dialog_config.url_handler_list));
	if (row < 0)
	{
		return;
	}

	gtk_clist_remove (GTK_CLIST (dialog_config.url_handler_list), row);
	url_handler_fill_in_entries (row);

	if (bk_edit_misc_get_selection (GTK_CLIST (dialog_config.url_handler_list)) < 0)
	{
		gtk_entry_set_text (GTK_ENTRY (dialog_config.url_handler_name_entry), "");
		gtk_entry_set_text (GTK_ENTRY (dialog_config.url_handler_command_entry), "");
	}
}


static void url_handler_fill_in_entries (int row)
{
	char *name;
	char *command;

	if (gtk_clist_get_text (GTK_CLIST (dialog_config.url_handler_list), row, 0, &name) == 0)
	{
		return;
	}

	if (gtk_clist_get_text (GTK_CLIST (dialog_config.url_handler_list), row, 1, &command) == 0)
	{
		return;
	}

	gtk_entry_set_text (GTK_ENTRY (dialog_config.url_handler_name_entry), name);
	gtk_entry_set_text (GTK_ENTRY (dialog_config.url_handler_command_entry), command);

	gtk_clist_select_row (GTK_CLIST (dialog_config.url_handler_list), row, 0);
	gtk_widget_grab_focus (GTK_WIDGET (dialog_config.url_handler_name_entry));
}


static void url_handler_fill_in_row (int row)
{
	char *name = gtk_entry_get_text (GTK_ENTRY (dialog_config.url_handler_name_entry));
	char *command = gtk_entry_get_text (GTK_ENTRY (dialog_config.url_handler_command_entry));

	gtk_clist_set_text (GTK_CLIST (dialog_config.url_handler_list), row, 0, name);
	gtk_clist_set_text (GTK_CLIST (dialog_config.url_handler_list), row, 1, command);
}


static void url_handler_entry_changed (GtkWidget *widget, GdkEventKey *event, gpointer *null)
{
	int row = bk_edit_misc_get_selection (GTK_CLIST (dialog_config.url_handler_list));
	if (row < 0)
	{
		return;
	}

	url_handler_fill_in_row (row);
}


static void ok (void)
{
	int popup_size, n;
	GtkItemFactoryEntry *popup_items;

	GList *url_handlers = NULL;

	gtk_widget_grab_focus (dialog_config.ok);

	/* dialog options */
	config_dialog_options_url_dropzone_sticky_set (conf, gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog_config.sticky)));
	config_dialog_options_url_dropzone_always_on_top_set (conf, gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog_config.always_on_top)));

	config_dialog_options_url_dropzone_size_state_set (conf, gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog_config.size)));
	config_dialog_options_url_dropzone_size_set (conf, gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (dialog_config.width)),
		gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (dialog_config.height)));

	config_dialog_options_url_dropzone_position_state_set (conf, gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog_config.position)));
	config_dialog_options_url_dropzone_position_set (conf, gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (dialog_config.x)),
		gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (dialog_config.y)));

	bk_edit_dialog_dnd_update ();

	/* gerneral settings */
	config_recent_documents_list_size_set (conf, gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (dialog_config.recent_documents_list_size)));
	config_middle_mouse_set (conf, gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog_config.middle_mouse)));
	config_browser_type_set (conf, bk_edit_misc_get_menu_option (GTK_OPTION_MENU (dialog_config.browser_type)));
	config_sort_order_set (conf, bk_edit_misc_get_menu_option (GTK_OPTION_MENU (dialog_config.sort_order)));

	/* url handlers */
	popup_size = bk_edit_menu_popup_build (&popup_items);
	for (n = 0; n < popup_size; n++)
	{
		gtk_item_factory_delete_entry (dialog_main.popup, &popup_items[n]);
	}
	gtk_item_factory_delete_item (dialog_main.popup, (const char *) "/Open with");
	bk_edit_menu_popup_free (popup_items, popup_size);

	for (n = 0; n < GTK_CLIST (dialog_config.url_handler_list)->rows; n++)
	{
		char *name;
		char *command;

		config_url_handler *url_handler = (config_url_handler *) malloc (sizeof (url_handler));
		if (url_handler == NULL)
		{
			continue;
		}

		gtk_clist_get_text (GTK_CLIST (dialog_config.url_handler_list), n, 0, &name);
		gtk_clist_get_text (GTK_CLIST (dialog_config.url_handler_list), n, 1, &command);

		url_handler->name = strdup (name);
		url_handler->command = strdup (command);
		url_handler->default_handler = (int) gtk_clist_get_row_data (GTK_CLIST (dialog_config.url_handler_list), n);


		url_handlers = g_list_append (url_handlers, url_handler);
	}

	config_url_handlers_set (conf, url_handlers);
	config_url_handlers_free (conf, url_handlers);

	popup_size = bk_edit_menu_popup_build (&popup_items);
	gtk_item_factory_create_items (dialog_main.popup, popup_size, popup_items, NULL);
	bk_edit_menu_popup_free (popup_items, popup_size);
}


static void key_pressed (GtkWidget *widget, GdkEventKey *event, gpointer *null)
{
	if (event == NULL)
	{
		return;
	}

	switch (event->keyval)
	{
		case GDK_KP_Enter:
		case GDK_Return:
		{
			if (GTK_IS_TEXT ((GTK_WINDOW (widget)->focus_widget)))
			{
				return;
			}

			if (GTK_WIDGET_HAS_DEFAULT (dialog_config.ok))
			{
				ok ();
				gtk_widget_hide (dialog_config.dialog);
			}
			break;
		}

		case GDK_Escape:
		{
			gtk_widget_hide (dialog_config.dialog);
			break;
		}
	}
}


static void button_press_event (GtkWidget *widget, GdkEventButton *event, gpointer popup)
{
	gint row = -1, col = -1;

	if (event->type == GDK_KEY_PRESS)
	{
	}
	else if (event->type == GDK_BUTTON_PRESS)
	{
		gtk_clist_get_selection_info (GTK_CLIST (dialog_config.url_handler_list), event->x, event->y, &row, &col);

		if ((row == -1) && (col == -1))
		{
			return;
		}

		if (event->button == 1)
		{
			url_handler_fill_in_entries (row);
		}
		else if (event->button == 3) /* popup */
		{
			gtk_clist_select_row (GTK_CLIST (dialog_config.url_handler_list), row, col);
			gtk_item_factory_popup (GTK_ITEM_FACTORY (popup), event->x_root, event->y_root, event->button, event->time);
		}
	}
}

