/**  --------------------------------------------------------------------
 *  @file session.cc
 *  @brief Implementation of the Session class.
 *  ------------------------------------------------------------------ **/
#include <unistd.h>
#include <syslog.h>

#include "argparser.h"
#include "convert.h"
#include "globals.h"
#include "session.h"
#include "tools.h"
#include <string>
#include <map>

using namespace ::std;
using namespace Binc;

extern char **environ;

//----------------------------------------------------------------------
Session::Session(void)
{
  readbytes = 0;
  writebytes = 0;
  statements = 0;
  bodies = 0;
  mailboxchanges = true;
  logfacility = LOG_DAEMON;
}

//----------------------------------------------------------------------
Session &Session::getInstance(void)
{
  static Session session;
  return session;
}

//----------------------------------------------------------------------
const int Session::getState(void) const
{
  return state;
}

//----------------------------------------------------------------------
void Session::setState(int n)
{
  state = n;
}

//----------------------------------------------------------------------
const string &Session::getUserID(void) const
{
  return userid;
}

//----------------------------------------------------------------------
void Session::setUserID(const string &s)
{
  userid = s;
}

//----------------------------------------------------------------------
const string &Session::getIP(void) const
{
  return ip;
}

//----------------------------------------------------------------------
void Session::setIP(const string &s)
{
  ip = s;
}

//----------------------------------------------------------------------
void Session::setLogFacility(int facility)
{
  logfacility = facility;
}

//----------------------------------------------------------------------
int Session::getLogFacility(void) const
{
  return logfacility;
}

//----------------------------------------------------------------------
void Session::addBody(void)
{
  ++bodies;
}

//----------------------------------------------------------------------
void Session::addStatement(void)
{
  ++statements;
}

//----------------------------------------------------------------------
void Session::addReadBytes(int i)
{
  readbytes += i;
}

//----------------------------------------------------------------------
void Session::addWriteBytes(int i)
{
  writebytes += i;
}

//----------------------------------------------------------------------
int Session::getBodies(void) const
{
  return bodies;
}

//----------------------------------------------------------------------
int Session::getStatements(void) const
{
  return statements;
}

//----------------------------------------------------------------------
int Session::getWriteBytes(void) const
{
  return writebytes;
}

//----------------------------------------------------------------------
int Session::getReadBytes(void) const
{
  return readbytes;
}

//----------------------------------------------------------------------
bool Session::parseCommandLine(int argc, char * argv[])
{
  args.addOptional("h|help", "Display this help screen", true);
  args.addOptional("version", "Display the version of Binc IMAP", true);
  args.addOptional("a|allow-plain", "Allow authentication when not TLS protected", true);
  args.addOptional("v|show-version", "Enable verbose IMAP greeting", false);
  args.addOptional("l|log-type", "Sets the method used for logging", false);
  args.addOptional("d|depot", "Sets the depot type", false);
  args.addOptional("D|delimiter", "Sets the mailbox delimiter", false);

  if (!args.parse(argc, argv)) {
    setLastError("Command line error, " + args.errorString());
    return false;
  }

  command.help = args["help"] == "yes" ? true : false;
  command.version = args["version"] == "yes" ? true : false;

  unparsedArgs = argv + args.argc();

  return true;
}

//----------------------------------------------------------------------
void Session::assignCommandLineArgs(void)
{
  if (args.hasArg("allow-plain"))
    setEnv("ALLOW_NONSSL_PLAINTEXT_LOGINS", "yes");

  if (args.hasArg("show-version"))
    setEnv("SHOW_VERSION_IN_GREETING", "yes");

  if (args.hasArg("log-type"))
    setEnv("LOG_TYPE", args["log-type"]);

  if (args.hasArg("depot"))
    setEnv("DEPOT", args["depot"]);

  if (args.hasArg("delimiter"))
    setEnv("DELIMITER", args["delimiter"]);
}

//----------------------------------------------------------------------
const string &Session::getLastError(void) const
{
  return lastError;
}

//----------------------------------------------------------------------
void Session::setLastError(const string &error) const
{
  lastError = error;
}

//----------------------------------------------------------------------
const string &Session::getResponseCode(void) const
{
  return responseCode;
}

//----------------------------------------------------------------------
void Session::setResponseCode(const string &code) const
{
  responseCode = "[" + code + "] ";
}

//----------------------------------------------------------------------
void Session::clearResponseCode(void) const
{
  responseCode = "";
}

//----------------------------------------------------------------------
pid_t Session::getPid(void)
{
  if (pid == 0)
    pid = getpid();

  return pid;
}

//----------------------------------------------------------------------
int Session::timeout() const
{
  return state == NONAUTHENTICATED ? AUTH_TIMEOUT : IDLE_TIMEOUT;
}

//----------------------------------------------------------------------
bool Session::hasEnv(const string &key) const
{
  return getenv(key.c_str()) != 0;
}

//----------------------------------------------------------------------
string Session::getEnv(const string &key)
{
  char *c = getenv(key.c_str());
  return c ? c : "";
}

//----------------------------------------------------------------------
void Session::setEnv(const string &key, const string &value)
{
  string env = key + "=" + value;
  putenv(strdup(env.c_str()));
}
