/**  --------------------------------------------------------------------
 *  @file  session-initialize-bincimap-up.cc
 *  @brief bincimap-up requires sslserver 
 *  @author Andreas Aardal Hanssen, Erwin Hoffmann
 *  @date 2002-2005, 2023 
 *  -----------------------------------------------------------------  **/
#include <syslog.h>
#include <ctype.h>

#include "broker.h"
#include "convert.h"
#include "globals.h"
#include "iodevice.h"
#include "iofactory.h"
#include "multilogdevice.h"
#include "syslogdevice.h"
#include "stdiodevice.h"
#include "session.h"
#include "tools.h"

#include <fcntl.h>
#include <string>
#include <map>

using namespace ::std;
using namespace Binc;

extern char **environ;

//----------------------------------------------------------------------
bool Session::initialize(int argc, char *argv[])
{
  IOFactory &ioFactory = IOFactory::getInstance();

  IODevice *stdioDevice = new StdIODevice(IODevice::IsEnabled);
  stdioDevice->setFlags(IODevice::HasOutputLimit);
  stdioDevice->setMaxOutputBufferSize(TRANSFER_BUFFER_SIZE);
  ioFactory.addDevice(stdioDevice);

  Session &session = Session::getInstance();

  IOFactory::getLogger().clearFlags(IODevice::FlushesOnEndl);

  // Read command line arguments
  if (!session.parseCommandLine(argc, argv))
    return false;

  // Show help if asked for it
  if (session.command.help) {
    printf("%s\n", session.args.usageString().c_str());
    return false;
  }

  // Show version if asked for it
  if (session.command.version) {
    printf("Binc IMAP v" BINC_VERSION"\n");
    return false;
  }

  // Let the command line args override the global settings.
  session.assignCommandLineArgs();

  // for log input
  string ip = getenv("TCP6REMOTEIP") ? getenv("TCP6REMOTEIP") :
              getenv("TCPREMOTEIP") ? getenv("TCPREMOTEIP") : "?";
  session.setIP(ip);

  string logtype = session.getEnv("LOG_TYPE");
  lowercase(logtype);
  trim(logtype);
  if (logtype == "multilog" || logtype == "stderr") {
    MultilogDevice *device = new MultilogDevice(IODevice::IsEnabled
                                              | IODevice::FlushesOnEndl);
    ioFactory.addDevice(device);
  } else if (logtype == "" || logtype == "syslog") {
    const string f = session.getEnv("SYSLOG_FACILITY");
    int facility;

    if (isdigit(f[0])) {
      facility = atoi(f);
    } else {
      if (f == "LOG_USER") facility = LOG_USER;
      else if (f == "LOG_LOCAL0") facility = LOG_LOCAL0;
      else if (f == "LOG_LOCAL1") facility = LOG_LOCAL1;
      else if (f == "LOG_LOCAL2") facility = LOG_LOCAL2;
      else if (f == "LOG_LOCAL3") facility = LOG_LOCAL3;
      else if (f == "LOG_LOCAL4") facility = LOG_LOCAL4;
      else if (f == "LOG_LOCAL5") facility = LOG_LOCAL5;
      else if (f == "LOG_LOCAL6") facility = LOG_LOCAL6;
      else if (f == "LOG_LOCAL7") facility = LOG_LOCAL7;
      else facility = LOG_DAEMON;
    }

    SyslogDevice *device = new SyslogDevice(IODevice::IsEnabled
                                          | IODevice::FlushesOnEndl,
                                            "bincimap-up",
                                            LOG_NDELAY | LOG_PID,
                                            facility);
    ioFactory.addDevice(device);
  }

  // Now that we know the log type, we can flush.
  IOFactory::getLogger().setFlags(IODevice::FlushesOnEndl);
  IOFactory::getLogger().setOutputLevelLimit(IODevice::InfoLevel);


  // imaps (port 993) -- requires sslserver with option -e

  int stls = 0;
  if (getenv("SSL_PROTOCOL")) {
    session.command.ssl = true;
    stls = -1;
    // else we will do starttls - requires new FDs
  } else if (getenv("UCSPITLS")) {
      string ucspitls = session.getEnv("UCSPITLS");
      if (ucspitls == "+") stls = 1;
      if (ucspitls == "-") stls = 0;
      if (ucspitls == "!") stls = 2;
  }

  BrokerFactory &brokerfactory = BrokerFactory::getInstance();

  brokerfactory.assign("AUTHENTICATE", new AuthenticateOperator());
  brokerfactory.assign("CAPABILITY", new CapabilityOperator());
  brokerfactory.assign("LOGIN", new LoginOperator());
  brokerfactory.assign("LOGOUT", new LogoutOperator());
  brokerfactory.assign("NOOP", new NoopOperator());
  brokerfactory.assign("ID", new IdOperator());
  if (stls > 0) brokerfactory.assign("STARTTLS", new StarttlsOperator());

  bincClient.setTimeout(AUTH_TIMEOUT);

  session.setState(Session::NONAUTHENTICATED);

  // If the depot was not initialized properly, return false.
  return true;
}
