/**  --------------------------------------------------------------------
 *  @file  operator-status.cc
 *  @brief  Implementation of the STATUS command.
 *  @author Andreas Aardal Hanssen
 *  @date 2002-2005
 *  -----------------------------------------------------------------  **/
#include <string>
#include <iostream>

#include <dirent.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "convert.h"
#include "depot.h"
#include "iodevice.h"
#include "iofactory.h"
#include "mailbox.h"
#include "operators.h"
#include "recursivedescent.h"
#include "session.h"
#include "status.h"

using namespace ::std;
using namespace Binc;

//----------------------------------------------------------------------
StatusOperator::StatusOperator(void)
{
}

//----------------------------------------------------------------------
StatusOperator::~StatusOperator(void)
{
}

//----------------------------------------------------------------------
const string StatusOperator::getName(void) const
{
  return "STATUS";
}

//----------------------------------------------------------------------
int StatusOperator::getState(void) const
{
  return Session::AUTHENTICATED | Session::SELECTED;
}

//------------------------------------------------------------------------
Operator::ProcessResult StatusOperator::process(Depot &depot,
						Request &command)
{
  Session &session = Session::getInstance();

  Status status;
  if (!depot.getStatus(command.getMailbox(), status)) {
    session.setLastError(depot.getLastError());
    return NO;
  }

  bincClient << "* STATUS " << toImapString(command.getMailbox()) << " (";
    
  string prefix;
  for (vector<string>::const_iterator i = command.statuses.begin();
       i != command.statuses.end(); ++i) {
    string tmp = *i;
    uppercase(tmp);
    if (tmp == "UIDNEXT") {
      bincClient << prefix << "UIDNEXT " << status.getUidNext();
      prefix = " ";
    } else if (tmp == "MESSAGES") {
      bincClient << prefix << "MESSAGES " << status.getMessages();
      prefix = " ";
    } else if (tmp == "RECENT") {
      bincClient << prefix << "RECENT " << status.getRecent(); 
      prefix = " ";
    } else if (tmp == "UIDVALIDITY") {
      bincClient << prefix << "UIDVALIDITY " << status.getUidValidity();
      prefix = " ";
    } else if (tmp == "UNSEEN") {
      bincClient << prefix << "UNSEEN " << status.getUnseen(); 
      prefix = " ";
    }
  }
  bincClient << ")" << endl;

  return OK;
}

//----------------------------------------------------------------------
Operator::ParseResult StatusOperator::parse(Request &c_in) const
{
  Session &session = Session::getInstance();

  if (c_in.getUidMode())
    return REJECT;

  Operator::ParseResult res;
  if ((res = expectSPACE()) != ACCEPT) {
    session.setLastError("Expected SPACE");
    return res;
  }

  string mailbox;
  if ((res = expectMailbox(mailbox)) != ACCEPT) {
    session.setLastError("Expected mailbox");
    return res;
  }

  c_in.setMailbox(mailbox);

  if ((res = expectSPACE()) != ACCEPT) {
    session.setLastError("Expected SPACE");
    return res;
  }

  if ((res = expectThisString("(")) != ACCEPT) {
    session.setLastError("Expected (");
    return res;
  }

  while (1) {
    if ((res = expectThisString("MESSAGES")) == ACCEPT)
      c_in.getStatuses().push_back("MESSAGES");
    else if ((res = expectThisString("RECENT")) == ACCEPT)
      c_in.getStatuses().push_back("RECENT");
    else if ((res = expectThisString("UIDNEXT")) == ACCEPT)
      c_in.getStatuses().push_back("UIDNEXT");
    else if ((res = expectThisString("UIDVALIDITY")) == ACCEPT)
      c_in.getStatuses().push_back("UIDVALIDITY");
    else if ((res = expectThisString("UNSEEN")) == ACCEPT)
      c_in.getStatuses().push_back("UNSEEN");
    else {
      session.setLastError("Expected status_att");
      return res;
    }

    if (expectSPACE() != ACCEPT) break;
  }

  if ((res = expectThisString(")")) != ACCEPT) {
    session.setLastError("Expected )");
    return res;
  }

  if ((res = expectCRLF()) != ACCEPT) {
    session.setLastError("Expected CRLF");
    return ERROR;
  }

  return ACCEPT;
}
