/**  --------------------------------------------------------------------
 *  @file operator-rename.cc
 *  @brief Implementation of the RENAME command.
 *  -----------------------------------------------------------------  **/
#include <string>
#include <iostream>

#include <dirent.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "convert.h"
#include "depot.h"
#include "mailbox.h"
#include "operators.h"
#include "recursivedescent.h"
#include "session.h"

using namespace ::std;
using namespace Binc;

//----------------------------------------------------------------------
RenameOperator::RenameOperator(void)
{
}

//----------------------------------------------------------------------
RenameOperator::~RenameOperator(void)
{
}

//----------------------------------------------------------------------
const string RenameOperator::getName(void) const
{
  return "RENAME";
}

//----------------------------------------------------------------------
int RenameOperator::getState(void) const
{
  return Session::AUTHENTICATED | Session::SELECTED;
}

//------------------------------------------------------------------------
Operator::ProcessResult RenameOperator::process(Depot &depot,
						Request &command)
{
  Session &session = Session::getInstance();

  const string &srcmailbox = command.getMailbox();
  const string &canonmailbox = toCanonMailbox(srcmailbox);
  const string &canondestmailbox = toCanonMailbox(command.getNewMailbox());

  // renaming INBOX should actually create the destination mailbox,
  // move over all the messages and then leave INBOX empty.
  if (canonmailbox == "INBOX") {
    session.setLastError("Sorry, renaming INBOX is not yet supported"
                         " by this IMAP server. Try copying the messages"
                         " instead");
    return NO;
  }

  if (canondestmailbox == "INBOX") {
    session.setLastError("It is not allowed to rename a mailbox to INBOX");
    return NO;
  }

  if (depot.renameMailbox(canonmailbox, canondestmailbox))
    return OK;
  else
    return NO;
}

//----------------------------------------------------------------------
Operator::ParseResult RenameOperator::parse(Request &c_in) const
{
  Session &session = Session::getInstance();
 
  if (c_in.getUidMode())
    return REJECT;

  Operator::ParseResult res;
  if ((res = expectSPACE()) != ACCEPT) {
    session.setLastError("Expected SPACE after RENAME");
    return res;
  }

  string mailbox;
  if ((res = expectMailbox(mailbox)) != ACCEPT) {
    session.setLastError("Expected mailbox after RENAME SPACE");
    return res;
  }

  if ((res = expectSPACE()) != ACCEPT) {
    session.setLastError("Expected SPACE after RENAME SPACE mailbox");
    return res;
  }

  string newmailbox;
  if ((res = expectMailbox(newmailbox)) != ACCEPT) {
    session.setLastError("Expected mailbox after RENAME SPACE"
			 " mailbox SPACE");
    return res;
  }

  if ((res = expectCRLF()) != ACCEPT) {
    session.setLastError("Expected CRLF after RENAME SPACE"
			 " mailbox SPACE mailbox");
    return res;
  }

  session.mailboxchanges = true;

  c_in.setName("RENAME");
  c_in.setMailbox(mailbox);
  c_in.setNewMailbox(newmailbox);

  return ACCEPT;
}
