/* ------------------------------------------------------------------------
@NAME       : bibparse.c
@DESCRIPTION: Parses a BibTeX file using the default options
				  for the btparse library. Prints the parsed entries as
				  (almost) bibtexml.
@GLOBALS    : 
@CREATED    : May 1996, Greg Ward
@MODIFIED   : Modified by Johannes Henkel (jhenkel@jhenkel.de)
@VERSION    : $Id: bib2xml.c,v 1.3 2002/10/06 23:16:07 henkel Exp $
@COPYRIGHT  : Copyright (c) 1996-97 by Gregory P. Ward.  All rights reserved.

              This file is part of the btparse distribution (but not part
              of the library itself).  This is free software; you can
              redistribute it and/or modify it under the terms of the GNU
              General Public License as published by the Free Software
              Foundation; either version 2 of the License, or (at your
              option) any later version.
-------------------------------------------------------------------------- */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <limits.h>
#include <stdarg.h>
#include <assert.h>
#include <btparse.h>

#define DEBUG 0

void escxml_fputs(char *,FILE*);

char *  Usage = "usage: bib2xml file\n";
char *  Help = 
"\n";

/**
	this function escapes all characters that are not allowed to appear
	in XML character data.
*/
void escxml_fputs(char * str, FILE * stream){
	int pos;
	for(pos = 0; str[pos]!=0; pos++){
		char c=str[pos];
		switch(c){
			case '&': fputs("&#38;#38;", stream); break;
			case '<': fputs("&#38;#60;", stream); break;
			case '>': fputs("&#62;",stream); break;
			case '\'': fputs("&#39;",stream); break;
			case '"': fputs("&#34;",stream); break;
			default: fputc(c,stream); break;
		}
	}
}


/* ------------------------------------------------------------------------
@NAME       : print_assigned_entry()
@INPUT      : stream
              top
@OUTPUT     : 
@RETURNS    : 
@DESCRIPTION: Prints out a "field-assignment"-type entry (ie. a macro 
              definition or regular entry).
@GLOBALS    : 
@CALLS      : btparse traversal functions
@CALLERS    : print_entry()
@CREATED    : 1997/08/12, GPW
@MODIFIED   : Johannes Henkel (jhenkel@jhenkel.de)
-------------------------------------------------------------------------- */
static void
print_assigned_entry (FILE *stream, AST *top)
{
   char  *type, *key;
   char  *field_name;
   AST   *field;

   type = bt_entry_type (top);
   key = bt_entry_key (top);

	fputs("  <bibtex:entry id=\"",stream);
	escxml_fputs(key?key:"undefined", stream);
	fputs("\">\n    <bibtex:",stream);
	escxml_fputs(type,stream);
	fputs(">\n",stream);

   field = NULL;
   while ((field = bt_next_field (top, field, &field_name)))
   {
      AST *   value;
      bt_nodetype nodetype;
      char *  text;
      boolean first;

      fputs("      <bibtex:",stream);
		escxml_fputs(field_name, stream);
		fputs(">",stream);

      value = NULL;
      first = TRUE;
      
      while ((value = bt_next_value (field, value, &nodetype, &text)))
      {
         if (!first) fputc (' ', stream);
         if (text) 
         {
			escxml_fputs (text, stream);
         }
         first = FALSE;
      }
		
	  fputs("</bibtex:",stream);
	  escxml_fputs(field_name, stream);
	  fputs(">\n",stream);
	  
   }

	fputs("    </bibtex:",stream);
	escxml_fputs(type,stream);
	fputs(">\n  </bibtex:entry>\n",stream);
	
} /* print_assigned_entry() */


/* ------------------------------------------------------------------------
@NAME       : print_value_entry()
@INPUT      : stream
              top
@OUTPUT     : 
@RETURNS    : 
@DESCRIPTION: Prints out a "value-only" entry (ie. comment or preamble).
@GLOBALS    : 
@CALLS      : btparse traversal functions
@CALLERS    : print_entry()
@CREATED    : 1997/08/13, GPW
@MODIFIED   : Johannes Henkel jhenkel@jhenkel.de
-------------------------------------------------------------------------- */
static void
print_value_entry (FILE *stream, AST *top)
{
   char *  type;
   AST *   value;
   char *  text;

   type = bt_entry_type (top);
	
	fputs("<!-- @",stream);
	escxml_fputs(type, stream);
	fputs("\n",stream);

   value = NULL;
      
   while ((value = bt_next_value (top, value, NULL, &text)))
   {
      if (text) {
			escxml_fputs(text,stream);
			fputs("\n",stream);
		}
   }

   fputs("-->\n",stream);
   
} /* print_value_entry() */


/* ------------------------------------------------------------------------
@NAME       : print_entry()
@INPUT      : 
@OUTPUT     : 
@RETURNS    : 
@DESCRIPTION: Prints a BibTeX entry in a very simplistic form by calling
              either print_assigned_entry() or print_value_entry().  These
              in turn work by calling the AST traversal routines in the
              btparse library, providing canonical examples of how to use
              these routines.
@GLOBALS    : 
@CALLS      : 
@CREATED    : 1997/01/22, GPW
@MODIFIED   : 1997/08/13, GPW: changed to differentiate between the two,
                               ahem, meta-meta-types of entries
-------------------------------------------------------------------------- */
static void
print_entry (FILE *stream, AST *top)
{

   switch (bt_entry_metatype (top))
   {
      case BTE_MACRODEF:
		  break; // don't print macrodef's, they'll be expanded anyways ...
      case BTE_REGULAR:
         print_assigned_entry (stream, top);
         break;

      case BTE_COMMENT:
      case BTE_PREAMBLE:
         print_value_entry (stream, top);
         break;

      default:
         fprintf (stderr, "warning: unknown entry type \"%s\"\n",
                  bt_entry_type (top));
         break;
   }
} /* print_entry() [2nd version] */


/* ------------------------------------------------------------------------
@NAME       : process_file
@INPUT      : filename
@OUTPUT     : 
@RETURNS    : true if there were no errors or only trivial errors
              false if there were serious errors
@DESCRIPTION: Parses an entire BibTeX file one entry at a time.  Each 
              entry is separately read, parsed, and printed back out
              to minimize memory use.
@GLOBALS    : 
@CALLS      : 
@CREATED    : Jan 1997, GPW
@MODIFIED   : 
@COMMENTS   : this *might* eventually wind up in the library, with
              a function pointer argument to specify what to do 
              to each entry
-------------------------------------------------------------------------- */
static int process_file (char *filename, FILE * infile, FILE * stream)
{
   AST    *cur_entry;
   boolean status, overall_status;

   bt_set_stringopts (BTE_MACRODEF, BTO_FULL);
   bt_set_stringopts (BTE_REGULAR, BTO_FULL);
   bt_set_stringopts (BTE_COMMENT, BTO_FULL);
   bt_set_stringopts (BTE_PREAMBLE, BTO_FULL);
	      
   overall_status = 1;                  /* assume success */
   
   fputs("<bibtex:file xmlns:bibtex=\"http://bibtexml.sf.net/\">\n",stream);

	bt_parse_entry_s("@string{jan = \"January\"}","builtin-nofile",
			0,0,&status);
	bt_parse_entry_s("@string{feb = \"February\"}","builtin-nofile",
			0,0,&status);
	bt_parse_entry_s("@string{mar = \"March\"}","builtin-nofile",
			0,0,&status);
	bt_parse_entry_s("@string{apr = \"April\"}","builtin-nofile",
			0,0,&status);
	bt_parse_entry_s("@string{may = \"May\"}","builtin-nofile",
			0,0,&status);
	bt_parse_entry_s("@string{jun = \"June\"}","builtin-nofile",
			0,0,&status);
	bt_parse_entry_s("@string{jul = \"July\"}","builtin-nofile",
			0,0,&status);
	bt_parse_entry_s("@string{aug = \"August\"}","builtin-nofile",
			0,0,&status);
	bt_parse_entry_s("@string{sep = \"September\"}","builtin-nofile",
			0,0,&status);
	bt_parse_entry_s("@string{oct = \"October\"}","builtin-nofile",
			0,0,&status);
	bt_parse_entry_s("@string{nov = \"November\"}","builtin-nofile",
			0,0,&status);
	bt_parse_entry_s("@string{dec = \"December\"}","builtin-nofile",
			0,0,&status);
		   
   while (1)
   {
      cur_entry = bt_parse_entry (infile, filename,0,&status);
      overall_status &= status;
      if (!cur_entry) break;
	  	print_entry (stream, cur_entry);
      bt_free_ast (cur_entry);
   }
   fputs("</bibtex:file>\n",stream);
   
   fclose (infile);
   return overall_status;

} /* process_file() */

int main (int argc, char *argv[])
{

   bt_initialize ();

   if(argc!=2){
      fprintf (stderr, Usage);
      fprintf (stderr, Help);
      fprintf (stderr, "Not enough arguments\n");
      exit (1);
   } else {
		 /*
    	 * If a string was given, and it's *not* "-", then open that filename.
   	 * Otherwise just use stdin.
   	 */

		char * filename = argv[1];
		FILE * infile;

   	if (filename != NULL && strcmp (filename, "-") != 0) {
      	infile = fopen (filename, "r");
      	if (infile == NULL) {
         	perror (filename);
         	return 0;
      	}
   	} else {
      	filename = "(stdin)";
      	infile = stdin;
   	}

		process_file(filename,infile,stdout);
   }

   bt_cleanup ();
   exit (bt_error_status (NULL));
}
