/*
 * Copyright (c) 1997 Massachusetts Institute of Technology
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to use, copy, modify, and distribute the Software without
 * restriction, provided the Software, including any modified copies made
 * under this license, is not distributed for a fee, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE MASSACHUSETTS INSTITUTE OF TECHNOLOGY BE LIABLE
 * FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of the Massachusetts
 * Institute of Technology shall not be used in advertising or otherwise
 * to promote the sale, use or other dealings in this Software without
 * prior written authorization from the Massachusetts Institute of
 * Technology.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "bench_utils.h"

short askuser(const char *s);
void printhelp(const char *progname);
void benchmark_submission(double maxmem);

static short submit_info_only = 0;
double min_bench_time;

int main(int argc, char **argv)
{
     short do_submission = 0;
     short do_1d = 0, do_3d = 0, do_r1d = 0, do_r2d = 0;
     short compute_accuracy = 0;
     double maxmem = 50.0; /* 50 MB max, by default */
     factor_type ftype = ALL_FACTORS;
     int which_N = 0;
     int i;
     char name[100] = "bench",fname[200];
     int dos_names = 0;
     char dos_name[20] = "";

     min_bench_time = 1.0;
     if (argc <= 1) {
	  if (argc == 1)
	       printhelp(argv[0]);

	  printf("\n");
	  /* Read information manually: */
	  if (askuser("Are you submitting results to "
		      "the benchFFT repository?")) {
	       do_submission = 1;
               printf("Enter the maximum memory to use (in megabytes): ");
               scanf("%lf",&maxmem);
               if (maxmem < 0.0) {
                    printf("  -- Error: must be positive!");
                    return 411;
               }	       
	  }
	  else {
	       printf("\nEnter benchmark parameters:\n\n");
	       do_1d = askuser("Benchmark 1D transforms?");
	       do_3d = askuser("Benchmark 3D transforms?");
	       do_r1d = askuser("Benchmark 1D real-complex transforms?");
	       do_r2d = askuser("Benchmark 2D real-complex transforms?");
	       compute_accuracy = !askuser("Measure speed (not accuracy)?");
	       printf("Enter size of transform to benchmark (0 for all): ");
	       scanf("%d",&which_N);
	       if (which_N < 0) {
		    printf("  -- Error: must be non-negative!");
		    return 411;
	       }
	       if (which_N == 0 && which_fft[0] == 0) {
		    if (askuser("Only use transforms of sizes"
				" that are powers of two?"))
			 ftype = POWERS_OF_TWO_ONLY;
		    else if (askuser("Only use sizes that are not "
				     "powers of two?"))
			 ftype = NON_POWERS_OF_TWO_ONLY;
		    else {
			 printf("Using all sizes.\n");
			 ftype = ALL_FACTORS;
		    }

		    printf("Maximum memory to use (in megabytes): ");
		    scanf("%lf",&maxmem);
		    if (maxmem < 0.0) {
			 printf("  -- Error: must be positive!");
			 return 411;
		    }   
		    
		    dos_names = askuser("Use cryptic DOS 8.3 filenames?");
	       }
	  }
     }
     else {
	  for (i = 1; i < argc; ++i) {
	       if (!strcmp(argv[i],"-h")) {
		    printhelp(argv[0]);
		    return 0;
	       }
	       else if (!strcmp(argv[i],"-submit"))
		    do_submission = 1;
	       else if (!strcmp(argv[i],"-submit-info-only"))
		    submit_info_only = 1;
	       else if (!strcmp(argv[i],"-dos"))
		    dos_names = 1;
	       else if (!strcmp(argv[i],"-1d"))
		    do_1d = 1;
	       else if (!strcmp(argv[i],"-r1d"))
		    do_r1d = 1;
	       else if (!strcmp(argv[i],"-3d"))
		    do_3d = 1;
	       else if (!strcmp(argv[i],"-r2d"))
		    do_r2d = 1;
	       else if (!strcmp(argv[i],"-s"))
		    compute_accuracy = 0;
	       else if (!strcmp(argv[i],"-a"))
		    compute_accuracy = 1;
	       else if (!strcmp(argv[i],"-p2"))
		    ftype = POWERS_OF_TWO_ONLY;
	       else if (!strcmp(argv[i],"-np2"))
		    ftype = NON_POWERS_OF_TWO_ONLY;
	       else if (!strcmp(argv[i],"-pa"))
		    ftype = ALL_FACTORS;
	       else if (!strcmp(argv[i],"-size")) {
		    ++i;
		    if (i >= argc) {
			 printf("Option \"-size\" must have an argument.\n");
			 printhelp(argv[0]);
			 return 411;
		    }
		    if (1 != sscanf(argv[i],"%d",&which_N) || which_N < 0) {
			 printf("Invalid \"-size\" argument \"%s\".\n",
				argv[i]);
			 printhelp(argv[0]);
			 return 411;
		    }
	       }
	       else if (!strcmp(argv[i],"-min-time")) {
		    ++i;
		    if (i >= argc) {
			 printf("Option \"-min-time\" must have an argument.\n");
			 printhelp(argv[0]);
			 return 411;
		    }
		    if (1 != sscanf(argv[i],"%lf",&min_bench_time) ||
			min_bench_time < 0) {
			 printf("Invalid \"-min-time\" argument \"%s\".\n",
				argv[i]);
			 printhelp(argv[0]);
			 return 411;
		    }
	       }
	       else if (!strcmp(argv[i],"-name")) {
		    ++i;
		    if (i >= argc) {
			 printf("Option \"-name\" must have an argument.\n");
			 printhelp(argv[0]);
			 return 411;
		    }
		    strcpy(name,argv[i]);
	       }
	       else if (!strcmp(argv[i],"-fft")) {
		    ++i;
		    if (i >= argc) {
			 printf("Option \"-name\" must have an argument.\n");
			 printhelp(argv[0]);
			 return 411;
		    }
		    strcpy(which_fft,argv[i]);
	       }
	       else if (!strcmp(argv[i],"-maxmem")) {
		    ++i;
		    if (i >= argc) {
			 printf("Option \"-maxmem\" must have an argument.\n");
			 printhelp(argv[0]);
			 return 411;
		    }
		    if (1 != sscanf(argv[i],"%lf",&maxmem) || maxmem <= 0) {
			 printf("Invalid \"-maxmem\" argument \"%s\".\n",
				argv[i]);
			 printhelp(argv[0]);
			 return 411;
		    }
	       }
	       else {
		    printf("\nUnknown option \"%s\"\n",argv[i]);
		    printhelp(argv[0]);
		    return 411;
	       }
	      

	  }
     }

     if (do_submission) {
	  benchmark_submission(maxmem);
	  return 0;
     }

     if (which_N != 0) {
	  maxmem = 1.0e20; /*
			    * big enough for whatever which_N is */

	  if (IS_POWER_OF_TWO(which_N))
	       ftype = POWERS_OF_TWO_ONLY;
	  else
	       ftype = NON_POWERS_OF_TWO_ONLY;
     }

     if (sizeof(FFTW_REAL) == sizeof(double))
	  ;
     else if (sizeof(FFTW_REAL) == sizeof(float)) {
	  strcat(name,".single");
	  strcat(dos_name,"f");
     }
     else {
	  char fptype[30];	  
	  sprintf(fptype,".real%d",(int)sizeof(FFTW_REAL));
	  strcat(name,fptype);		       
     }

     if (ftype == ALL_FACTORS) {
	  strcat(name,".pa");
	  strcat(dos_name,"a");
     }
     else if (ftype == POWERS_OF_TWO_ONLY) {
	  strcat(name,".p2");
	  strcat(dos_name,"2");
     }
     else if (ftype == NON_POWERS_OF_TWO_ONLY) {
	  strcat(name,".np2");
	  strcat(dos_name,"n");
     }

     if (compute_accuracy) {
	  strcat(name,".a");
	  strcat(dos_name,"a");
     }
     else
	  strcat(dos_name,"s");

     if (which_N != 0 || which_fft[0] != 0) {
	  bench_log_file = stdout;
	  bench_dat_file = 0;
	  bench_echo_dat = 0;
     }
     else {
	  if (dos_names) {
	       strcpy(fname,dos_name);
	       if (do_1d)
		    strcat(fname,"1");
	       if (do_r1d)
		    strcat(fname,"r");
	       if (do_3d)
		    strcat(fname,"3");
	       if (do_r2d)
		    strcat(fname,"2");
	  }
	  else {
	       strcpy(fname,name);
	       if (do_1d)
		    strcat(fname,".1d");
	       if (do_r1d)
		    strcat(fname,".r1d");
	       if (do_3d)
		    strcat(fname,".3d");
	       if (do_r2d)
		    strcat(fname,".r2d");
	  }
	  strcat(fname,".log");
	  bench_log_file = fopen(fname,"w");
	  if (!bench_log_file) {
	       perror("Error creating log file");
	       return 1;
	  }
     }

     log_printf(
	  "FFT Benchmark Program by M. Frigo and S. G. Johnson.\n"
	  "           email: fftw@theory.lcs.mit.edu\n"
	  "             www: http://theory.lcs.mit.edu/~fftw\n");
     log_printf("\nUsing %s\n",fftw_version);
     
     log_printf("\nMaximum memory to use: %g MB\n"
		"Factors to allow: %s\n",
		maxmem,
		ftype == ALL_FACTORS ? "any" :
		(ftype == POWERS_OF_TWO_ONLY ? "2" :
		 "anything but 2"));
     if (which_N != 0)
	  log_printf(
	       "Only benchmarking size %d transforms.\n",which_N);
     if (which_fft[0] != 0)
	  log_printf(
	       "Only benchmarking transform \"%s\"\n",which_fft);
     
     if (sizeof(FFTW_REAL) == sizeof(double))
	  log_printf("Using double precision.\n");
     else if (sizeof(FFTW_REAL) == sizeof(float))
	  log_printf("Using single precision.\n");
     else
	  log_printf("Unknown fp type! "
		     "(single=%d, double=%d, FFTW_REAL=%d)\n",
		     sizeof(float),sizeof(double),sizeof(FFTW_REAL));
     
     if (do_1d) {
	  if (bench_echo_dat)
	       printf("\n");

	  if (which_N == 0 && which_fft[0] == 0) {
	       if (dos_names) {
		    strcpy(fname,dos_name);
		    strcat(fname,"1");
	       }
	       else {
		    strcpy(fname,name);
		    strcat(fname,".1d");
	       }
	       strcat(fname,".dat");
	       bench_dat_file = fopen(fname,"w");
	       if (!bench_dat_file) {
		    perror("Error creating 1D data file");
		    return 1;
	       }
	  }

	  log_printf("\nMeasuring %s of 1D transforms:\n\n",
		     compute_accuracy ? "accuracy" : "speed");

	  bench_1d(compute_accuracy,ftype,which_N,maxmem);  

	  if (which_N == 0 && which_fft[0] == 0)
	       fclose(bench_dat_file);
     }

     if (do_r1d) {
	  if (bench_echo_dat)
	       printf("\n");

	  if (which_N == 0 && which_fft[0] == 0) {
	       if (dos_names) {
		    strcpy(fname,dos_name);
		    strcat(fname,"r");
	       }
	       else {
		    strcpy(fname,name);
		    strcat(fname,".r1d");
	       }
	       strcat(fname,".dat");
	       bench_dat_file = fopen(fname,"w");
	       if (!bench_dat_file) {
		    perror("Error creating 1D data file");
		    return 1;
	       }
	  }

	  log_printf("\nMeasuring %s of real-complex 1D transforms:\n\n",
		     compute_accuracy ? "accuracy" : "speed");

	  rbench_1d(compute_accuracy,ftype,which_N,maxmem);  

	  if (which_N == 0 && which_fft[0] == 0)
	       fclose(bench_dat_file);
     }

     if (do_3d) {
	  if (bench_echo_dat)
	       printf("\n");

	  if (which_N == 0 && which_fft[0] == 0) {
	       if (dos_names) {
		    strcpy(fname,dos_name);
		    strcat(fname,"3");
	       }
	       else {
		    strcpy(fname,name);
		    strcat(fname,".3d");
	       }
	       strcat(fname,".dat");
	       bench_dat_file = fopen(fname,"w");
	       if (!bench_dat_file) {
		    perror("Error creating 3D data file");
		    return 1;
	       }
	  }

	  log_printf("\nMeasuring %s of 3D transforms:\n\n",
		       compute_accuracy ? "accuracy" : "speed");
	  
	  bench_3d(compute_accuracy,ftype,which_N,maxmem);  

	  if (which_N == 0 && which_fft[0] == 0)
	       fclose(bench_dat_file);
     }

     if (do_r2d) {
	  if (bench_echo_dat)
	       printf("\n");

	  if (which_N == 0 && which_fft[0] == 0) {
	       if (dos_names) {
		    strcpy(fname,dos_name);
		    strcat(fname,"2");
	       }
	       else {
		    strcpy(fname,name);
		    strcat(fname,".r2d");
	       }
	       strcat(fname,".dat");
	       bench_dat_file = fopen(fname,"w");
	       if (!bench_dat_file) {
		    perror("Error creating r2D data file");
		    return 1;
	       }
	  }

	  log_printf("\nMeasuring %s of real-complex 2D transforms:\n\n",
		       compute_accuracy ? "accuracy" : "speed");
	  
	  rbench_2d(compute_accuracy,ftype,which_N,maxmem);  

	  if (which_N == 0 && which_fft[0] == 0)
	       fclose(bench_dat_file);
     }

     if (which_N == 0 && which_fft[0] == 0)
	  fclose(bench_log_file);

     return 0;
}

void printhelp(const char *progname)
{
     printf("\nUsage: %s [ options ]\nOptions:\n"
	    "     -h = help (print this message)\n"
	    "-submit = submit benchmark results to the benchfft web page\n"
            "-name s = make s the root name for output files "
	    "(default s is \"bench\")\n"
	    "   -dos = use DOS 8.3 filename restrictions (off by default)\n"
	    "    -1d = benchmark 1D transforms\n"
	    "    -3d = benchmark 3D transforms\n"
	    "   -r1d = benchmark 1D real-complex transforms\n"
	    "     -s = measure speed of transforms (the default)\n"
	    "     -a = measure accuracy of transforms (instead of speed)\n"
	    "    -p2 = use only sizes that are powers of 2\n"
	    "   -np2 = use only sizes that are not powers of 2\n"
	    "    -pa = use all sizes (the default)\n"
	    "-size N = only benchmark size N transform\n"
	    "          (for 3d transforms, NxNxN)\n"
	    "          (N=0, the default, to use a fixed range of sizes)\n"
	    " -maxmem MB = MB is approx. max memory to use (in megabytes)\n"
	    "              (default is 50 megabytes)\n"
	    " -fft s = only benchmark the transform named s (string)\n",
	    progname);
}

short askuser(const char *s)
{
     char line[200] = "", c;
     int i, count = 0;
     
     do {
	  if (count++ > 0)
	       printf("Invalid response.  Please enter \"y\" or \"n\".\n");
	  printf("%s (y/n) ",s);
	  while (line[0] == 0 || line[0] == '\n') /* skip blank lines */
	       fgets(line,200,stdin);
	  for (i = 0; line[i] && (line[i] == ' ' || line[i] == '\t'); ++i)
	       ;
	  c = line[i];
     } while (c != 'n' && c != 'N' && c != 'y' && c != 'Y');

     return(c == 'y' || c == 'Y');
}

#ifndef CFLAGS
#define CFLAGS "none"
#endif
#ifndef FFLAGS
#define FFLAGS "none"
#endif

#define MY_GETS(s) { \
     fgets(line,2048,stdin); \
     if (line[strlen(line)-1] == '\n') \
	  line[strlen(line)-1] = 0; \
}

void get_benchmark_submission_info(FILE *f)
{
     char line[2048] = "";
     int read_flags;

     printf("\n      *** THIS PROGRAM DOES NOT SEND EMAIL ***\n"
	    "  You must mail the output file \"submit.txt\" yourself.\n");

     printf("\n     *** THE BENCHMARK MAY TAKE SEVERAL HOURS ***\n"
	    "                     Be patient!\n");

     printf("\nYou will now be prompted for various information describing\n"
	    "you, your machine, and your compiler.  This information will\n"
	    "be displayed on the benchfft web page when data from your\n"
	    "machine is requested.\n");
     printf("\nIf you do not know the answer to a question,\n"
	    "leave it blank.  All entries are completed by\n"
	    "hitting the return/enter key.\n");
     
     printf("\nPlease tell us your name.  If you are unsure, you may\n"
	    "instead enter your credit-card number.\n"
	    "Enter your name: ");
     while (line[0] == 0) MY_GETS(line);
     fprintf(f,"@ submitter = %s\n",line);

     printf("\nOptionally, we may display your email address on our web page\n"
	    "along with the data from your machine.\n");
     if (askuser("Should we display your email address on our web page?")) {
	  printf("Enter your preferred email address: ");
	  MY_GETS(line);
	  fprintf(f,"@ submitter email = %s\n",line);
     }
     else
	  fprintf(f,"@ submitter email = PRIVATE\n");

     printf("\n");
     if (askuser("Should we also display the name of your company, \n"
		 "school, or organization?")) {
          printf("Enter the name of your company/school/organization: ");
          MY_GETS(line);
          fprintf(f,"@ submitter organization = %s\n",line);
     }
     else
          fprintf(f,"@ submitter organization = NONE\n");

     printf("\nNow, we want to know about your computer.\n\n"
	    "First, tell us which company makes your computer\n"
	    "(e.g. Compaq, Apple, Ford, DEC, SGI, Sun, Radio Shack).\n"
	    "This is *not* necessarily the manufacturer of your processor.\n"
	    "If your computer is not built by any company (e.g. you\n"
	    "made it yourself), leave this blank.\n"
	    "Enter the name of your computer manufacturer: ");
     MY_GETS(line);
     fprintf(f,"@ computer manufacturer = %s\n",line);

     printf("\nWhat is the model name of your computer?  For example,\n"
	    "PowerMac 9500/120, AlphaServer 4100, RS/6000 Model 3BT,\n"
	    "TRS 80, etc.  This is *not* the name of the processor.\n"
	    "(Leave this blank if your computer does not have a model\n"
	    "name that you know of.)\n"
	    "Enter the model name of your computer: ");
     MY_GETS(line);
     fprintf(f,"@ computer model = %s\n",line);
     
     printf("\nWhat company makes the processor (CPU) in your computer?\n"
	    "(e.g. Motorola, Intel, Sun, Moon, SGI, NEC, IBM)\n"
	    "Enter the company that makes your CPU: ");
     MY_GETS(line);
     fprintf(f,"@ CPU manufacturer = %s\n",line);

     printf("\nWhat is the model name of the processor in your computer?\n"
	    "(e.g. PowerPC 604e, PowerPC 686, R10000, 386, Pentium,\n"
	    "Pentium Pro, Pentium II, 68040, Alpha EV56, etc.)  Do not\n"
	    "include the processor speed or manufacturer in the name.\n"
	    "Enter the model name of your CPU: ");
     MY_GETS(line);
     fprintf(f,"@ CPU model = %s\n",line);

     printf("\nEnter the clock speed of your CPU in MHz (e.g. 100, \n"
	    "200, etcetera).  Do not include \"MHz\" in your answer.\n"
	    "You may use scientific notation for clock speeds\n"
	    "greater than 10^6 MHz.\n"
	    "Enter the clock speed of your CPU in MHz: ");
     MY_GETS(line);
     fprintf(f,"@ CPU speed = %s MHz\n",line);

     printf("\nHow much RAM does your computer have, in megabytes (MB)?\n"
	    "(e.g. 8, 64, 52.7, 1024, etcetera)\n"
	    "Enter the amount of RAM in your computer: ");
     MY_GETS(line);
     fprintf(f,"@ RAM = %s MB\n",line);

     printf("\nWhat is the size of your Level 2 (L2) cache?\n"
	    "Specify the units (\"kB\" or \"MB\").  (e.g. 128 kB,\n"
	    "1 MB, 2 MB).  If you have no L2 cache, enter 0.  If you\n"
	    "do not know, leave this blank.\n"
	    "Enter the size of your L2 cache: ");
     MY_GETS(line);
     fprintf(f,"@ L2 cache size = %s\n",line);

     printf("\nNext, enter the name and version number of the operating\n"
	    "system you are using.  (e.g. Red Hat Linux 4.1 (Kernel 2.0.27),\n"
	    "MacOS 8.1, Windows NT 4.0, Windows 3.1, AIX 4.1, BeOS DR9,\n"
	    "SunOS 5.5.1)  (Do not abbreviate \"Windows.\")\n"
	    "Enter the name and version of your OS: ");
     MY_GETS(line);
     fprintf(f,"@ operating system = %s\n",line);

     printf("\nFinally, tell us about your compilers.\n");

     printf("\nPlease tell us the full name and version number of your\n"
	    "C/C++ compiler.  (e.g. gcc 2.7.2, Metrowerks Codewarrior Pro 2,\n"
	    "Microsoft Visual C++ 4.0, Sun WorkShop cc 4.2)\n"
	    "Enter your compiler and version: ");
     MY_GETS(line);
     fprintf(f,"@ C compiler = %s\n",line);

     printf("\n");
     if (strcmp(CFLAGS,"none")) {
	  printf("Your C compiler flags seem to be:\n"
		 "          %s\n", CFLAGS);
	  read_flags = !askuser("Is this correct?");
     }
     else
	  read_flags = 1;
     if (read_flags) {
	  printf("What C compiler optimization options did you use?\n"
		 "On a command-line system, please list the compiler\n"
		 "flags.  In GUI development environments, if you simply\n"
		 "turned on all the optimization switches, just enter\n"
		 "\"all\".  Otherwise, list the options that you used.\n"
		 "Enter your C compiler optimization flags: ");
	  MY_GETS(line);
     }
     else
	  strcpy(line,CFLAGS);
     fprintf(f,"@ C compiler flags = %s\n",line);
     
     #ifdef HAVE_F77

     printf("\nPlease tell us the full name and version number of your\n"
	    "Fortran compiler.  (e.g. g77 0.5.21).\n"
	    "Enter your compiler and version: ");
     MY_GETS(line);
     fprintf(f,"@ Fortran compiler = %s\n",line);
     
     printf("\n");
     if (strcmp(FFLAGS,"none")) {
	  printf("Your Fortran compiler flags seem to be:\n"
		 "          %s\n", FFLAGS);
	  read_flags = !askuser("Is this correct?");
     }
     else
	  read_flags = 1;
     if (read_flags) {
	  printf("What Fortran compiler optimization options did you use?\n"
		 "On a command-line system, please list the compiler\n"
		 "flags.  In GUI development environments, if you simply\n"
		 "turned on all the optimization switches, just enter\n"
		 "\"all\".  Otherwise, list the options that you used.\n"
		 "Enter your Fortran compiler optimization flags: ");
	  MY_GETS(line);
     }
     else
	  strcpy(line,FFLAGS);
     fprintf(f,"@ Fortran compiler flags = %s\n",line);

     #else
     
     fprintf(f,"@ Fortran compiler = NONE\n");
     fprintf(f,"@ Fortran compiler flags = NONE\n");

     #endif /* HAVE_F77 */

     printf("\nIf there is any other information about your computer\n"
	    "that you believe to be pertinent, please enter it here.\n"
	    "End your comments by hitting the return/enter key.  (If you\n"
	    "have no remarks, leave this blank.)\n"
	    "Your comments: ");
     MY_GETS(line);
     fprintf(f,"@ remarks = %s\n",line);

     /* FFTW 1.2 has deceptive fftw_version string: */
     if (!strcmp(fftw_version,"FFTW V1.1 ($Id: executor.c,v 1.34 "
		 "1997/04/30 13:15:56 fftw Exp $)"))
	  fprintf(f,"@ FFTW version = %s\n","FFTW V1.2");
     else
	  fprintf(f,"@ FFTW version = %s\n",fftw_version);

     if (sizeof(FFTW_REAL) == sizeof(double))
	  fprintf(f,"@ floating-point precision = double\n");
     else if (sizeof(FFTW_REAL) == sizeof(float))
	  fprintf(f,"@ floating-point precision = single\n");
     else
	  fprintf(f,"@ floating-point precision = UNKNOWN\n");
     fprintf(f,"@ floating-point size = %d bytes\n",(int)sizeof(FFTW_REAL));
}

void do_append_file(FILE *f, char *append_fname)
{
     FILE *append_f;
     int c;
     
     append_f = fopen(append_fname,"r");
     if (!append_f) {
	  perror("Error opening file for append");
	  exit(1);
     }

     while ((c = getc(append_f)) != EOF)
	  putc(c,f);

     fclose(append_f);
}

#define SEPARATOR "\n------------------------------------------------------\n"

void benchmark_submission(double maxmem)
{
     FILE *submit_f;

     submit_f = fopen("submit.txt","w");
     if (!submit_f) {
	  perror("Error creating submission file");
	  exit(1);
     }

     fprintf(submit_f,
	     "To: benchfft@theory.lcs.mit.edu\n"
	     "Subject: SUBMIT\n"
	     "-------------------\n"
	     "@@SUBMIT@@\n");
     get_benchmark_submission_info(submit_f);

     if (submit_info_only) {
	  fclose(submit_f);
	  return;
     }

#ifdef SUBMIT_LOGS
     bench_log_file = submit_f;
#else
     bench_log_file = NULL;
#endif

     bench_dat_file = fopen("submit.dat","w");
     if (!bench_dat_file) {
	  perror("Error creating submission data file");
	  exit(1);
     }

     bench_echo_dat = 1;

     printf(SEPARATOR
	    "\nYou are done entering information now, and the benchmark is\n"
	    "being run.  This will take some time.  When it is done, you\n"
	    "should mail the file \"submit.txt\" to\n"
	    "               benchfft@theory.lcs.mit.edu\n"
	    "with \"SUBMIT\" in the email subject.  This is a plain ASCII\n"
	    "file, so please do not encode it in any way.\n\n");

     fflush(stdout);
     fflush(submit_f);

     printf("Running 1D speed benchmark (powers of two)...\n"); fflush(stdout);

     log_printf(SEPARATOR "@@@@ bench.1d.p2.log\n");
     dat_printf(SEPARATOR "@@@@ bench.1d.p2.dat\n");

     bench_1d(0,POWERS_OF_TWO_ONLY,0,maxmem);


     printf("Running 1D speed benchmark (non powers of two)...\n"); 
     fflush(stdout);

     log_printf(SEPARATOR "@@@@ bench.1d.np2.log\n");
     dat_printf(SEPARATOR "@@@@ bench.1d.np2.dat\n");

     bench_1d(0,NON_POWERS_OF_TWO_ONLY,0,maxmem);

     printf("Running 3D speed benchmark (powers of two)...\n"); fflush(stdout);

     log_printf(SEPARATOR "@@@@ bench.3d.p2.log\n");
     dat_printf(SEPARATOR "@@@@ bench.3d.p2.dat\n");

     bench_3d(0,POWERS_OF_TWO_ONLY,0,maxmem);


     printf("Running 3D speed benchmark (non powers of two)...\n");
     fflush(stdout);

     log_printf(SEPARATOR "@@@@ bench.3d.np2.log\n");
     dat_printf(SEPARATOR "@@@@ bench.3d.np2.dat\n");

     bench_3d(0,NON_POWERS_OF_TWO_ONLY,0,maxmem);

     printf("Running 1D real-complex speed benchmark (powers of two)...\n");
     fflush(stdout);

     log_printf(SEPARATOR "@@@@ bench.r1d.p2.log\n");
     dat_printf(SEPARATOR "@@@@ bench.r1d.p2.dat\n");

     rbench_1d(0,POWERS_OF_TWO_ONLY,0,maxmem);

     printf("Running 1D real-complex speed benchmark (non powers of two)...\n"); 
     fflush(stdout);

     log_printf(SEPARATOR "@@@@ bench.r1d.np2.log\n");
     dat_printf(SEPARATOR "@@@@ bench.r1d.np2.dat\n");

     rbench_1d(0,NON_POWERS_OF_TWO_ONLY,0,maxmem);

     printf("Running 2D real-complex speed benchmark (powers of two)...\n");
     fflush(stdout);

     log_printf(SEPARATOR "@@@@ bench.r2d.p2.log\n");
     dat_printf(SEPARATOR "@@@@ bench.r2d.p2.dat\n");

     rbench_2d(0,POWERS_OF_TWO_ONLY,0,maxmem);


     printf("Running 2D real-complex speed benchmark (non powers of two)...\n");
     fflush(stdout);

     log_printf(SEPARATOR "@@@@ bench.r2d.np2.log\n");
     dat_printf(SEPARATOR "@@@@ bench.r2d.np2.dat\n");

     rbench_2d(0,NON_POWERS_OF_TWO_ONLY,0,maxmem);

     printf("Done with benchmarks.  Cleaning up.\n"); fflush(stdout);

     fclose(bench_dat_file);

     do_append_file(submit_f,"submit.dat");
     remove("submit.dat");

     fprintf(submit_f,"@@@@ end\n");

     fclose(submit_f);
}
