/*  $Id: BatrachiansEngine.h,v 1.39 2017/12/09 18:26:16 sarrazip Exp $
    BatrachiansEngine.h - Main engine

    batrachians - A fly-eating frog game.
    Copyright (C) 2004-2011 Pierre Sarrazin <http://sarrazip.com/>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
    02110-1301, USA.
*/

#ifndef _H_BatrachiansEngine
#define _H_BatrachiansEngine

#include <flatzebra/GameEngine.h>
#include <flatzebra/RSprite.h>
#include <flatzebra/SoundMixer.h>

#include "util.h"
#include "Controller.h"
#include "Frog.h"

#include <math.h>
#include <string>
#include <list>
#include <stdexcept>

using namespace std;
using namespace flatzebra;


class BatrachiansEngine : public GameEngine
{
public:

    BatrachiansEngine(const string &windowManagerCaption,
                        size_t maxNumFlies,
                        bool useSound,
                        bool fullScreen,
                        bool processActiveEvent);
    /*
        See base class.

        'windowManagerCaption' must contain the title to give to the window.

        'processActiveEvent' determines if the game automatically pauses
        when the game window loses focus.

        Throws a string or an integer code if an error occurs.
    */

    virtual ~BatrachiansEngine();
    /*
        Nothing interesting.
    */

    virtual void processKey(SDLKey keysym, bool pressed);
    /*
        Inherited.
    */

    virtual void processActivation(bool appActive);
    /*  Called when the app is activated or deactivated (iconified).
        'appActive' is true if the app has just become active,
        or false if it has just become inactive.

        After this method has been called, the tick() method will not be
        called until the app has been reactivated. However, the SDL
        screen is flipped after this call, so the effect of drawing
        commands made by this method will appear.
        Never called if processActiveEvent argument of constructor is false.

        Inherited from GameEngine.
    */

    virtual bool tick();
    /*
        Inherited.
    */

private:

    class LilyPad
    {
    public:
        static int y;
        static int height;

        int x;
        int width;

        LilyPad(int xx, int ww) : x(xx), width(ww) {}
        int xMiddle() const { return x + width / 2; }
        int xRight() const { return x + width; }
        bool isPointOverPad(RCouple p) const
        {
            return (x <= p.x && p.x < x + width);
        }
    };

    enum  // frog pixmap indexes
    {
        FROG_L_STAND,
        FROG_L_SWIM,
        FROG_R_STAND,
        FROG_R_SWIM
    };

    typedef RSpriteList::iterator Iter;
    typedef RSpriteList::const_iterator CIter;

    enum { NUM_SHADES = 7 };  // must be odd (actually, must be 7...)

    enum { WATER_HEIGHT_ABOVE_LILYPADS = 50 };

    enum
    {
        GAME_LENGTH_IN_MS = 180000,
        DARK_MS           =  80000,  // time in ms before end of game when darkening starts
        BEGIN_STARS_MS    =  50000,  // ... when stars start to appear
        FULL_STARS_MS     =  30000,  // ... when stars are at their brightest
        FLICKERING_FLIES_MS = 20000  // ... when flies start to flicker
        //GAME_LENGTH_IN_MS = 14000, // debugging values
        //DARK_MS = 12000,
        //BEGIN_STARS_MS = 6000,
        //FULL_STARS_MS = 2000
    };

    static const double PI;

    bool firstTime;
    int tickCount;
    bool gameOver;
    Uint32 gameStartTime;   // time in ms when game started
    Uint32 totalPauseTime;  // total ms spent in pause mode
    Uint32 pauseStartTime;  // time in ms when most recent pause started

    Uint32 blackColor;

    SDL_Surface *lilyPadSurface;
    Couple lilyPadSurfacePos;
    LilyPad leftLilyPad;
    LilyPad rightLilyPad;

    SDL_Surface *skySurface;
    SDL_Surface *waterSurface;
    SDL_Surface *cloudSurfaces[2];

    PixmapArray starPA;
    RSpriteList stars;

    PixmapArray frogPA;

    Frog *userFrog;
    Frog *compFrog;

    PixmapArray splashPA;

    PixmapArray tonguePA;

    PixmapArray crosshairsPA;
    RSprite *crosshairs;

    PixmapArray fly0PA;
    PixmapArray fly1PA;
    PixmapArray fly2PA;
    PixmapArray fly3PA;
    RSpriteList flies;
    size_t maxNumFlies;
    int ticksBeforeNextFly;

    PixmapArray userDigitPA;
    PixmapArray computerDigitPA;  //TODO: use other way to write numbers, one color per frog
    RSpriteList scoreSprites;

    Controller controller;

    SoundMixer *theSoundMixer;
    bool useSound;

    class Sound
    {
    public:
        SoundMixer::Chunk gameStarts;
        SoundMixer::Chunk frogJumps;
        SoundMixer::Chunk flyEaten;
        SoundMixer::Chunk tongueOut;
        SoundMixer::Chunk splash;
        SoundMixer::Chunk gameEnds;

        Sound()
        :   gameStarts(), frogJumps(), flyEaten(),
            tongueOut(), splash(), gameEnds() {}
    } sounds;

    const Couple fontDim;

private:

    void setTicksBeforeNextFly()
    {
        ticksBeforeNextFly = 1 + 0 * FPS + rand() % (2 * FPS);
    }

    bool isFrogSwimming(RSprite &s) const
    {
        return (s.currentPixmapIndex % 4 == FROG_L_SWIM
                || s.currentPixmapIndex % 4 == FROG_R_SWIM);
    }

    void putSprite(const RSprite *s)
    {
        copySpritePixmap(*s, s->currentPixmapIndex, s->getPos());
    }

    static RCouple getCoupleFromAngle(double radius, double angle)
    {
        return RCouple(radius * cos(angle), - radius * sin(angle));
    }

    static RCouple chooseRandomSpeed(double radius)
    {
        double angle = rand() * PI / RAND_MAX;
        return RCouple(radius * cos(angle), - radius * sin(angle));
    }

    static double getAngleFromCouple(RCouple c)
    {
        return atan2(-c.y, c.x);
    }

    void putSpriteList(RSpriteList &sl)
    {
        for (Iter it = sl.begin(); it != sl.end(); it++)
            putSprite(*it);
    }

    void changeStarColors()
    {
        for (Iter it = stars.begin(); it != stars.end(); it++)
            (*it)->currentPixmapIndex = rand() % 2;
    }

    int getYWater() const
    {
        return lilyPadSurfacePos.y - WATER_HEIGHT_ABOVE_LILYPADS;
    }


    void initFrogPositions();
    void initGame();
    void moveFrog(RSprite &aFrog, RSprite &itsTongue,
                                        RSprite *&itsSplash, bool isUser);
    void animateSplash(RSprite *&splash);
    void moveFlies();
    void setTonguePosition(const RSprite &frog,
                                size_t piOffset, RSprite &tongue);
    bool draw();
    void detectCollisions();
    size_t detectTongueFlyCollisions(Frog &frog);
    void controlCrosshairs();
    void controlUserFrogJump();
    void controlUserFrogTongue();
    RSprite *findNearestFly(RCouple pos);
    void controlComputerFrogJump();
    void controlComputerFrogTongue();
    RCouple computeJumpSpeed(RCouple start, RCouple target, double g);
    void playSoundEffect(SoundMixer::Chunk &chunk);
    void createScoreSprites(long n, RCouple center, Frog &frog);
    void animateTemporarySprites(RSpriteList &sl) const;
    int getFlyType(const RSprite &fly) const;

    /*        Forbidden operations:
    */
    BatrachiansEngine(const BatrachiansEngine &);
    BatrachiansEngine &operator = (const BatrachiansEngine &);
};


#endif  /* _H_BatrachiansEngine */
