#include "fft.h"

#include "command.h"
#include "ft.h"
#include "utility.h"

typedef void (*Fft_process)(float *data, int n);

static int ncodes = 0;
static int npoints_in[MAX_NCODES];
static int npoints_out[MAX_NCODES];
static Fft_process fft_process[MAX_NCODES];

void do_fft_process(int code, float *data)
{
    int m, n;
    float *d;

    n = npoints_in[code];
    m = npoints_out[code] - n;
    d = data + n;

    ZERO_VECTOR(d, m);
    (*fft_process[code])(data, npoints_out[code]);
}

static Status init_fft_process(String fft_msg, Fft_process process,
				int type_in, int type_out, String error_msg)
{
    int type, npts;

    if (setup_command(&type, &npts, ncodes, fft_msg, do_fft_process,
							error_msg) == ERROR)
	return  ERROR;

    if (type != type_in)
    {
	if (type_in == REAL_DATA)
	    sprintf(error_msg, "must have real data to do '%s'", fft_msg);
	else /* (type_in == COMPLEX_DATA) */
	    sprintf(error_msg, "must have complex data to do '%s'", fft_msg);

	return  ERROR;
    }

    npoints_in[ncodes] = npts;
    npts = ceil_power_of_2(npts);
    npoints_out[ncodes] = npts;
    fft_process[ncodes] = process;

    CHECK_STATUS(end_command(type_out, npts, fft_msg, error_msg));

    ncodes++;

    return  OK;
}

Status init_fft(Generic_ptr *param, String error_msg)
{
    return  init_fft_process("fft", fft, COMPLEX_DATA, COMPLEX_DATA,
								error_msg);
}

Status init_ifft(Generic_ptr *param, String error_msg)
{
    return  init_fft_process("ifft", ifft, COMPLEX_DATA, COMPLEX_DATA,
								error_msg);
}

Status init_rft(Generic_ptr *param, String error_msg)
{
    return  init_fft_process("rft", rft, REAL_DATA, COMPLEX_DATA,
								error_msg);
}

Status init_irft(Generic_ptr *param, String error_msg)
{
    return  init_fft_process("irft", irft, COMPLEX_DATA, REAL_DATA,
								error_msg);
}

Status init_sft(Generic_ptr *param, String error_msg)
{
    return  init_fft_process("sft", sft, REAL_DATA, REAL_DATA,
								error_msg);
}

Status init_isft(Generic_ptr *param, String error_msg)
{
    return  init_fft_process("isft", isft, REAL_DATA, REAL_DATA,
								error_msg);
}

Status init_cft(Generic_ptr *param, String error_msg)
{
    return  init_fft_process("cft", cft, REAL_DATA, REAL_DATA,
								error_msg);
}

Status init_icft(Generic_ptr *param, String error_msg)
{
    return  init_fft_process("icft", icft, REAL_DATA, REAL_DATA,
								error_msg);
}

Status init_hft(Generic_ptr *param, String error_msg)
{
    int type, npts;

    if (setup_command(&type, &npts, ncodes, "hft", do_fft_process,
							error_msg) == ERROR)
	return  ERROR;

    if (type != REAL_DATA)
	RETURN_ERROR_MSG("must have real data to do 'hft'");

    npoints_in[ncodes] = npts;
    npts = ceil_power_of_2(npts);
    npoints_out[ncodes] = npts;
	/* not really, but easiest to define it this way */
    fft_process[ncodes] = hft;

    type = COMPLEX_DATA;
    CHECK_STATUS(end_command(type, 2*npts, "hft", error_msg));

    ncodes++;

    return  OK;
}

Status init_fftn(Generic_ptr *param, String error_msg)
{
    return  init_fft_process("fftn", fftn, COMPLEX_DATA, COMPLEX_DATA,
								error_msg);
}

Status init_ifftn(Generic_ptr *param, String error_msg)
{
    return  init_fft_process("ifftn", ifftn, COMPLEX_DATA, COMPLEX_DATA,
								error_msg);
}

Status init_rftn(Generic_ptr *param, String error_msg)
{
    return  init_fft_process("rftn", rftn, REAL_DATA, COMPLEX_DATA,
								error_msg);
}

Status init_irftn(Generic_ptr *param, String error_msg)
{
    return  init_fft_process("irftn", irftn, COMPLEX_DATA, REAL_DATA,
								error_msg);
}

Status init_sftn(Generic_ptr *param, String error_msg)
{
    return  init_fft_process("sftn", sftn, REAL_DATA, REAL_DATA,
								error_msg);
}

Status init_isftn(Generic_ptr *param, String error_msg)
{
    return  init_fft_process("isftn", isftn, REAL_DATA, REAL_DATA,
								error_msg);
}

Status init_cftn(Generic_ptr *param, String error_msg)
{
    return  init_fft_process("cftn", cftn, REAL_DATA, REAL_DATA,
								error_msg);
}

Status init_icftn(Generic_ptr *param, String error_msg)
{
    return  init_fft_process("icftn", icftn, REAL_DATA, REAL_DATA,
								error_msg);
}

Status init_fft_max(int *code, int npts, int data_type, String error_msg)
{
    if ((ncodes+2) > MAX_NCODES)
	RETURN_ERROR_MSG("too many '*ft*' commands to be able to do 'maxent'");

    npoints_in[ncodes] = npts;
    npoints_out[ncodes] = npts;
    npoints_in[ncodes+1] = npts;
    npoints_out[ncodes+1] = npts;

    if (data_type == COMPLEX_DATA)
    {
	fft_process[ncodes] = fft;
	fft_process[ncodes+1] = ifft; /* also the transpose */
    }
    else
    {
	fft_process[ncodes] = rft;
	fft_process[ncodes+1] = rftt;
    }

    *code = ncodes;
    ncodes += 2;

    return  OK;
}
