#include "data.h"

#include "object.h"
#include "parser.h"
#include "ref.h"

#define  SMALL_NUMBER	5.0e-3
#define  LARGE_NUMBER	1.0e20

#define  NDATA_ALLOC	10

static int ndata_sets = 0;
static int ndata_alloc = 0;
static Data_info *data_info;

static void free_data_memory()
{
    if (ndata_alloc > 0)
    {
	FREE(data_info, Data_info);
	ndata_sets = ndata_alloc = 0;
    }
}

static Status alloc_data_memory(String error_msg)
{
    int n;

    if (ndata_alloc == 0)
    {
	sprintf(error_msg, "allocating data memory");

	n = NDATA_ALLOC;
	MALLOC(data_info, Data_info, n);
	ndata_alloc = n;
    }
    else
    {
	sprintf(error_msg, "reallocating data memory");

	n = ndata_alloc + NDATA_ALLOC;
	REALLOC(data_info, Data_info, n);
	ndata_alloc = n;
    }

    return  OK;
}

Status register_data_set(int name_arg, int info_arg, int x_arg, int y_arg,
			Parser_store **store, Ref_info *ref, String error_msg)
{
    Parser_store *s;

    sprintf(error_msg, "allocating memory for data set");
    MALLOC(s, Parser_store, 1);

    if (ndata_sets == ndata_alloc)
	CHECK_STATUS(alloc_data_memory(error_msg));

    s->ndata = s->nalloc = 0;
    s->data_type = PARSER_NONE;
    s->data = (Generic_ptr) (data_info + ndata_sets);

    store[info_arg] = s;

    data_info[ndata_sets].name = store[name_arg];
    data_info[ndata_sets].ref = ref;

    if (x_arg >= 0)
	data_info[ndata_sets].x = store[x_arg];
    else
	data_info[ndata_sets].x = (Parser_store *) NULL;

    data_info[ndata_sets].y = store[y_arg];

    ndata_sets++;

    return  OK;
}

int get_ndata_sets()
{
    return  ndata_sets;
}

String get_data_name(int data_set)
{
    if (data_set < ndata_sets)
	return  (String) (data_info[data_set].name->data);
    else
	return  (String) NULL;
}

Data_info *get_data_info()
{
    return  data_info;
}

void set_data_visibility(int data_set, int visibility)
{
    if ((data_set >= 0) && (data_set < ndata_sets))
	data_info[data_set].property.visibility = visibility;
}

int get_data_visibility(int data_set)
{
    if ((data_set >= 0) && (data_set < ndata_sets))
	return  data_info[data_set].property.visibility;
    else
	return  VISIBILITY_ON;
}

static void print_xy_point_stats(int data_set, float x, int step,
						Print_funcs *print_funcs)
{
}

static void print_y_point_stats(int ref_type, int data_set, float x, int step,
						Print_funcs *print_funcs)
{
    int i, n, pt0, pt1;
    float *d;
    Line message;

    if (ref_type != REF_POINTS)
    {
	convert_to_point(ref_type, data_info[data_set].y->ndata,
						data_info[data_set].ref, &x);

	sprintf(message, "Crosshair is at point %5.2f\n", x);

	(*(print_funcs->print_message))(message);
    }

    pt0 = FLOOR(x) - 1;  /* points start at 1 */
    pt1 = pt0 + 2;
    pt0 = MAX(0, pt0);
    n = data_info[data_set].y->ndata;
    pt1 = MIN(n, pt1);

    d = (float *) (data_info[data_set].y->data);

    for (i = pt0; i < pt1; i++)
    {
	sprintf(message, "\tPoint %d: \t%5.2f\n", i+1, d[i*step]);
	(*(print_funcs->print_message))(message);
    }
}

void print_point_stats(int ref_type, float *point, Print_funcs *print_funcs)
{
    int i, step;
    Line error_msg, message;

    if (ndata_sets == 0)  return;

    if ((*(print_funcs->start_print))(error_msg) == ERROR)
	ERROR_AND_RETURN(error_msg);

    sprintf(message, "Crosshair is at (%5.2f %s, %5.2f)\n", point[0],
	ref_type == REF_POINTS  ?  "\b"  :  ref_names[ref_type], point[1]);

    (*(print_funcs->print_message))(message);

    for (i = 0; i < ndata_sets; i++)
    {
	sprintf(message, "\nStatistics for data set '%s'\n",
					(char *) (data_info[i].name->data));

	(*(print_funcs->print_message))(message);

	if (data_info[i].y->data_type & PARSER_REAL)
	    step = 1;
	else
	    step = 2;

	if (data_info[i].x)
	    print_xy_point_stats(i, point[0], step, print_funcs);
	else
	    print_y_point_stats(ref_type, i, point[0], step, print_funcs);
    }

    (*(print_funcs->end_print))();
}

static void print_stats(int n, float data_min, float data_max, float data_sum,
				float data_sqr, Print_funcs *print_funcs)
{
    Line message;

    sprintf(message, "\tNumber points used:\t%d\n", n);
    (*(print_funcs->print_message))(message);

    sprintf(message, "\tMinimum data value:\t%5.2f\n", data_min);
    (*(print_funcs->print_message))(message);

    sprintf(message, "\tMaximum data value:\t%5.2f\n", data_max);
    (*(print_funcs->print_message))(message);

    sprintf(message, "\tSum of data values:\t%5.2f\n", data_sum);
    (*(print_funcs->print_message))(message);

    data_sum /= (float) n;
    sprintf(message, "\tAverage of data values:\t%5.2f\n", data_sum);
    (*(print_funcs->print_message))(message);

    data_sqr /= (float) n;
    data_sqr -= data_sum*data_sum;
    data_sqr = sqrt((double) data_sqr);

    if (n > 1)
	data_sqr *= ((float) n) / ((float) (n-1));

    sprintf(message, "\tStandard deviation of data values:\t%5.2f\n",
                                                                data_sqr);
    (*(print_funcs->print_message))(message);
}

static void print_xy_region_stats(int data_set, float *x, int step,
						Print_funcs *print_funcs)
{
}

static void print_y_region_stats(int ref_type, int data_set, float *xx,
					int step, Print_funcs *print_funcs)
{
    int i, n, pt0, pt1;
    float *d, data, data_min, data_max, data_sum, data_sqr, x[LIMITS_DIM];
    Line message;

    COPY_VECTOR(x, xx, LIMITS_DIM);

    if (ref_type != REF_POINTS)
    {
	convert_range_to_points(ref_type, 1, &(data_info[data_set].y->ndata),
				data_info[data_set].ref, &x[0], &x[1]);

	sprintf(message, "Chosen x range is (%5.2f %5.2f)\n", x[0], x[1]);

	(*(print_funcs->print_message))(message);
    }

    pt0 = CEILING(x[0]-SMALL_NUMBER) - 1;  /* points start at 1 */
    pt1 = FLOOR(x[1]+SMALL_NUMBER);
    pt0 = MAX(0, pt0);
    n = data_info[data_set].y->ndata;
    pt1 = MIN(n, pt1);

    d = (float *) (data_info[data_set].y->data);

    data_min = LARGE_NUMBER;
    data_max = - LARGE_NUMBER;
    data_sum = 0;
    data_sqr = 0;

    for (i = pt0; i < pt1; i++)
    {
	data = d[i*step];
	data_min = MIN(data, data_min);
	data_max = MAX(data, data_max);
	data_sum += data;
	data_sqr += data * data;
    }

    n = pt1 - pt0;

    print_stats(n, data_min, data_max, data_sum, data_sqr, print_funcs);
}

void print_region_stats(int ref_type, float *x, Print_funcs *print_funcs)
{
    int i, step;
    Line error_msg, message;

    if (ndata_sets == 0)  return;

    if ((*(print_funcs->start_print))(error_msg) == ERROR)
	ERROR_AND_RETURN(error_msg);

    sprintf(message, "Chosen x range is (%5.2f %5.2f %s)\n", x[0], x[1],
		ref_type == REF_POINTS  ?  "\b"  :  ref_names[ref_type]);

    (*(print_funcs->print_message))(message);

    for (i = 0; i < ndata_sets; i++)
    {
	sprintf(message, "\nStatistics for data set '%s'\n",
					(char *) (data_info[i].name->data));

	(*(print_funcs->print_message))(message);

	if (data_info[i].y->data_type & PARSER_REAL)
	    step = 1;
	else
	    step = 2;

	if (data_info[i].x)
	    print_xy_region_stats(i, x, step, print_funcs);
	else
	    print_y_region_stats(ref_type, i, x, step, print_funcs);
    }

    (*(print_funcs->end_print))();
}
