#include "assign.h"

#include "parser.h"
#include "script.h"

#define  SMALL_NUMBER	1.0e-4

#define  STEP_FIRST	1
#define  STEP_LAST	2
#define  STEP_STEP	3

static Status do_number(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int n;
    float *data_in, *data_out;

    n = store[INPUT_X]->ndata;
    store_int_to_float(store[INPUT_X]);
    data_in = (float *) (store[INPUT_X]->data);

    store_type_float(store[OUTPUT_X]);
    CHECK_STATUS(check_parser_alloc(store[OUTPUT_X], n, error_msg));
    data_out = (float *) (store[OUTPUT_X]->data);

    COPY_VECTOR(data_out, data_in, n);

    return  OK;
}

static Status do_array(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int i, m, n, type;
    float *data_in, *data_out;

    nstore--;

    n = 0;
    for (i = 0; i < nstore; i++)
    {
	store_int_to_float(store[i+INPUT_X]);
	m = store[i+INPUT_X]->ndata;

	n += m;
    }

    store_type_float(store[OUTPUT_X]);
    CHECK_STATUS(check_parser_alloc(store[OUTPUT_X], n, error_msg));
    data_out = (float *) (store[OUTPUT_X]->data);

    type = store[OUTPUT_X]->data_type;

    for (i = 0; i < nstore; i++)
    {
	m = store[i+INPUT_X]->ndata;

	if (type & PARSER_COMPLEX)
	    m *= 2;

	data_in = (float *) (store[i+INPUT_X]->data);
	COPY_VECTOR(data_out, data_in, m);
	data_out += m;
    }

    return  OK;
}

static Status do_string(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int n;
    char *data;

    n = store[INPUT_X]->ndata;
    CHECK_STATUS(check_parser_alloc(store[OUTPUT_X], n, error_msg));
    data = (char *) (store[OUTPUT_X]->data);

    strcpy(data, (char *) (store[INPUT_X]->data));

    return  OK;
}

static Status do_floor(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int i, n;
    float *data_in;
    int *data_out;

    n = store[INPUT_X]->ndata;
    data_in = (float *) (store[INPUT_X]->data);
    store_type_int(store[OUTPUT_X]);
    CHECK_STATUS(check_parser_alloc(store[OUTPUT_X], n, error_msg));
    data_out = (int *) (store[OUTPUT_X]->data);

    if (store[OUTPUT_X]->data_type & PARSER_COMPLEX)
	n *= 2;

    for (i = 0; i < n; i++)
	data_out[i] = FLOOR(data_in[i]);

    return  OK;
}

static Status do_ceiling(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int i, n;
    float *data_in;
    int *data_out;

    n = store[INPUT_X]->ndata;
    data_in = (float *) (store[INPUT_X]->data);
    store_type_int(store[OUTPUT_X]);
    CHECK_STATUS(check_parser_alloc(store[OUTPUT_X], n, error_msg));
    data_out = (int *) (store[OUTPUT_X]->data);

    if (store[OUTPUT_X]->data_type & PARSER_COMPLEX)
	n *= 2;

    for (i = 0; i < n; i++)
	data_out[i] = CEILING(data_in[i]);

    return  OK;
}

static Status do_step(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int i, n;
    float first, last, step;
    float *data_out;

    store_int_to_float(store[STEP_FIRST]);
    store_int_to_float(store[STEP_LAST]);
    store_int_to_float(store[STEP_STEP]);

    first = *((float *) (store[STEP_FIRST]->data));
    last = *((float *) (store[STEP_LAST]->data));
    step = *((float *) (store[STEP_STEP]->data));

    if (ABS(step) < SMALL_NUMBER)
    {
	sprintf(error_msg, "step = %e, must be > %e in absolute value",
							step, SMALL_NUMBER);
	return  ERROR;
    }

    if ((step > 0) && (first > last))
    {
	sprintf(error_msg,
		"(first, last, step) = (%f, %f, %f): must have first <= last",
							first, last, step);
	return  ERROR;
    }

    if ((step < 0) && (first < last))
    {
	sprintf(error_msg,
		"(first, last, step) = (%f, %f, %f): must have first >= last",
							first, last, step);
	return  ERROR;
    }

    if (step > 0)
    {
	last += SMALL_NUMBER/2;
	n = (last - first + step - 1) / step;
    }
    else /* (step < 0) */
    {
	last -= SMALL_NUMBER/2;
	n = (first - last - step - 1) / (-step);
    }

    store_type_float(store[OUTPUT_X]);
    CHECK_STATUS(check_parser_alloc(store[OUTPUT_X], n, error_msg));
    data_out = (float *) (store[OUTPUT_X]->data);

    for (i = 0; i < n; i++)
	data_out[i] = first + i*step;

    return  OK;
}

static Status do_sizeof(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int n;
    int *data_out;

    n = store[INPUT_X]->ndata;
    store_type_int(store[OUTPUT_X]);
    CHECK_STATUS(check_parser_alloc(store[OUTPUT_X], 1, error_msg));
    data_out = (int *) (store[OUTPUT_X]->data);
    *data_out = n;

    return  OK;
}

static Status check_input_types(int nstore, Parser_store **store,
							String error_msg)
{
    int i;

    nstore--;

    if (store[INPUT_X]->data_type & PARSER_REAL)
    {
	for (i = 1; i < nstore; i++)
	{
	    if (store[i+INPUT_X]->data_type & PARSER_COMPLEX)
	    {
		sprintf(error_msg, "argument #1 real, argument #%d complex", i);
		return  ERROR;
	    }
	}
    }
    else /* (store[INPUT_X]->data_type & PARSER_COMPLEX) */
    {
	for (i = 1; i < nstore; i++)
	{
	    if (store[i+INPUT_X]->data_type & PARSER_REAL)
	    {
		sprintf(error_msg, "argument #1 complex, argument #%d real", i);
		return  ERROR;
	    }
	}
    }

    return  OK;
}

Status init_number(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "number", do_number, error_msg) == ERROR)
        return  ERROR;

    store[OUTPUT_X]->data_type = PARSER_RS;

    return  OK;
}

Status init_array(int nstore, Parser_store **store, String error_msg)
{
    if (nstore < 2)
	RETURN_ERROR_MSG("no arguments");

    if (setup_command(nstore, store, "array", do_array, error_msg) == ERROR)
        return  ERROR;

    CHECK_STATUS(check_input_types(nstore, store, error_msg));

    store[OUTPUT_X]->data_type = PARSER_FA;

    if (store[INPUT_X]->data_type & PARSER_REAL)
	store[OUTPUT_X]->data_type &= ~PARSER_COMPLEX;
    else
	store[OUTPUT_X]->data_type &= ~PARSER_REAL;


    return  OK;
}

Status init_string(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "string", do_string, error_msg) == ERROR)
        return  ERROR;

    store[OUTPUT_X]->data_type = store[INPUT_X]->data_type;

    return  OK;
}

Status init_floor(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "floor", do_floor, error_msg) == ERROR)
        return  ERROR;

    store[OUTPUT_X]->data_type = store[INPUT_X]->data_type & ~PARSER_FLOAT;
    store[OUTPUT_X]->data_type |= PARSER_INT;

    return  OK;
}

Status init_ceiling(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "ceiling", do_ceiling, error_msg) == ERROR)
        return  ERROR;

    store[OUTPUT_X]->data_type = store[INPUT_X]->data_type & ~PARSER_FLOAT;
    store[OUTPUT_X]->data_type |= PARSER_INT;

    return  OK;
}

Status init_step(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "step", do_step, error_msg) == ERROR)
        return  ERROR;

    store[OUTPUT_X]->data_type = PARSER_RA;

    return  OK;
}

Status init_sizeof(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "sizeof", do_sizeof, error_msg) == ERROR)
        return  ERROR;

    store[OUTPUT_X]->data_type = PARSER_IRS;

    return  OK;
}
