# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['aws_error_utils']

package_data = \
{'': ['*']}

install_requires = \
['botocore']

setup_kwargs = {
    'name': 'aws-error-utils',
    'version': '2.7.0',
    'description': 'Error-handling functions for boto3/botocore',
    'long_description': "# aws-error-utils\n**Making botocore.exceptions.ClientError easier to deal with**\n\nAll AWS service exceptions are raised by `boto3` as a `botocore.exceptions.ClientError`, with the contents of the exception indicating what kind of exception happened.\nThis is not very pythonic, and the contents themselves are rather opaque, most being held in dicts rather than as properties.\nThe functions in this package help dealing with that, to make your code less verbose and require less memorization of `ClientError` contents.\n\n## Installation\n\n[The package is on PyPI](https://pypi.org/project/aws-error-utils/) for pip-installing, but you can also just copy the [`aws_error_utils.py` file](https://raw.githubusercontent.com/benkehoe/aws-error-utils/stable/aws_error_utils/aws_error_utils.py) into your project (note that the standalone file requires Python 3.7 or greater).\n\n## Usage\nIf you've got code like this:\n\n```python\nimport boto3, botocore.exceptions\n\ns3 = boto3.client('s3')\ntry:\n    s3.get_object(Bucket='my-bucket', Key='example')\nexcept botocore.exceptions.ClientError as error:\n    if error.response['Error']['Code'] == 'NoSuchBucket':\n        print(error.response['Error']['Message'])\n        # error handling\n    else:\n        raise\n```\n\nyou can replace it with:\n\n```python\nimport boto3\nfrom aws_error_utils import errors\n\ns3 = boto3.client('s3')\ntry:\n    s3.get_object(Bucket='my-bucket', Key='example')\nexcept errors.NoSuchBucket as error:\n# or\n# except catch_aws_error('NoSuchBucket') as error:\n    print(error.message)\n    # error handling\n```\n\nIf you have trouble remembering where all the contents in `ClientError` are, like these:\n\n```python\nclient_error.response['Error']['Code']\nclient_error.response['Error']['Message']\nclient_error.response['ResponseMetadata']['HTTPStatusCode']\nclient_error.operation_name\n```\n\nyou can replace it with:\n\n```python\nimport boto3\nfrom aws_error_utils import get_aws_error_info\n\nerr_info = get_aws_error_info(client_error)\n\nerr_info.code\nerr_info.message\nerr_info.http_status_code\nerr_info.operation_name\n```\n\nIf you're using `errors` or `catch_aws_error()`, you can skip the `get_aws_error_info()` step, because the fields are set directly on the `ClientError` object:\n\n```python\nimport boto3\nfrom aws_error_utils import errors\n\ns3 = boto3.client('s3')\ntry:\n    s3.get_object(Bucket='my-bucket', Key='example')\nexcept errors.NoSuchBucket as error:\n    error.code\n    error.message\n    error.http_status_code\n    error.operation_name\n```\n\nAdditionally, when you *do* need to use `ClientError`, it's imported by `aws_error_utils` so you can just use `aws_error_utils.ClientError` rather than `botocore.exceptions.ClientError` (the same is true for `BotoCoreError`, the base class of all non-`ClientError` exceptions).\n\n## `errors`\nIt's easiest to use the `errors` class if you don't have complex conditions to match.\nUsing the error code as a field name in an `except` block will match that error code.\nAdditionally, when you use this style, it sets the fields from `AWSErrorInfo` (see below) directly on the `ClientError` object.\nFor example:\n\n```python\nimport boto3\nfrom aws_error_utils import errors\n\ns3 = boto3.client('s3')\ntry:\n    s3.get_object(Bucket='my-bucket', Key='example')\nexcept errors.NoSuchBucket as error:\n    print(error.message)\n\n    # error handling\n```\n\nYou can include multiple error codes in an `except` statement, though this is slower than combining them with a single `catch_aws_error()` call.\n\n```python\nimport boto3\nfrom aws_error_utils import errors\n\ns3 = boto3.client('s3')\ntry:\n    s3.get_object(Bucket='my-bucket', Key='example')\nexcept (errors.NoSuchBucket, errors.NoSuchKey) as error:\n    print(error.message)\n\n    # error handling\n```\n\nYou can catch all `ClientError`s with `errors.ALL`.\n\nYou can only use this style for error codes that work as Python property names.\nFor error codes like EC2's `InvalidInstanceID.NotFound`, you have to use `catch_aws_error()` (see below).\n\nUnfortunately, you cannot get tab completion for error codes on the `errors` class, as a comprehensive list of error codes is not available as a Python package (`botocore` has a small number, but they are few and incomplete).\n\nNote that the value of `errors.NoSuchBucket` is not an exception class representing the `NoSuchBucket` error, it is an alias for `catch_aws_error('NoSuchBucket')`.\nIt can only be used in an `except` statement; it will raise `RuntimeError` otherwise.\nYou also cannot instantiate the `errors` class.\n\n## `catch_aws_error()`\nThe function takes as input error code(s), and optionally operation name(s), to match against the current raised exception. If the exception matches, the `except` block is executed.\nIf your error handling still needs the error object, you can still use an `as` expression, otherwise it can be omitted (just `except catch_aws_error(...):`).\nAdditionally, `catch_aws_error()` sets the fields from `AWSErrorInfo` (see below) directly on the `ClientError` object.\n\n```python\nimport boto3\nfrom aws_error_utils import catch_aws_error\n\ns3 = boto3.client('s3')\ntry:\n    s3.get_object(Bucket='my-bucket', Key='example')\nexcept catch_aws_error('NoSuchBucket') as error:\n    print(error.message)\n\n    # error handling\n```\n\nYou can provide error codes either as positional args, or as the `code` keyword argument with either as a single string or a list of strings.\n\n```python\ncatch_aws_error('NoSuchBucket')\ncatch_aws_error(code='NoSuchBucket')\n\ncatch_aws_error('NoSuchBucket', 'NoSuchKey')\ncatch_aws_error(code=['NoSuchBucket', 'NoSuchKey'])\n```\n\nIf there are multiple API calls in the `try` block, and you want to match against specific ones, the `operation_name` keyword argument can help.\nSimilar to the `code` keyword argument, the operation name(s) can be provided as either as a single string or a list of strings.\n\n```python\nimport boto3\nfrom aws_error_utils import catch_aws_error\n\ntry:\n    s3 = boto3.client('s3')\n    s3.list_objects_v2(Bucket='bucket-1')\n    s3.get_object(Bucket='bucket-2', Key='example')\nexcept catch_aws_error('NoSuchBucket', operation_name='GetObject') as error:\n    # This will be executed if the GetObject operation raises NoSuchBucket\n    # but not if the ListObjects operation raises it\n```\n\nYou must provide an error code.\nTo match exclusively against operation name, use the `aws_error_utils.ALL_CODES` token.\nFor completeness, there is also an `ALL_OPERATIONS` token.\n\n```python\nimport boto3\nfrom aws_error_utils import catch_aws_error\n\ntry:\n    s3 = boto3.client('s3')\n    s3.list_objects_v2(Bucket='bucket-1')\n    s3.get_object(Bucket='bucket-1', Key='example')\nexcept catch_aws_error(ALL_CODES, operation_name='ListObjectsV2') as e:\n    # This will execute for all ClientError exceptions raised by the ListObjectsV2 call\n```\n\nFor more complex conditions, instead of providing error codes and operation names, you can provide a callable to evaluate the exception.\nNote that unlike error codes, you can only provide a single callable.\n\n```python\nimport re\nimport boto3\nfrom aws_error_utils import catch_aws_error, get_aws_error_info\n\ndef matcher(e):\n    info = get_aws_error_info(e)\n    return re.search('does not exist', info.message)\n\ntry:\n    s3 = boto3.client('s3')\n    s3.list_objects_v2(Bucket='bucket-1')\nexcept catch_aws_error(matcher) as e:\n    # This will be executed if e is a ClientError and matcher(e) returns True\n    # Note the callable can assume the exception is a ClientError\n```\n\n## `get_aws_error_info()`\nThis function takes a returns an `AWSErrorInfo` object, which is a `collections.NamedTuple` with the following fields:\n\n* `code`\n* `message`\n* `http_status_code`\n* `operation_name`\n* `response` (the raw response dictionary)\n\nIf you're not modifying your `except` statements to use `catch_aws_error()`, this function can be useful instead of remembering exactly how this information is stored in the `ClientError` object.\n\nIf you're using `catch_aws_error()`, this function isn't necessary, because it sets these fields directly on the `ClientError` object.\n\n## `aws_error_matches()`\nThis is the matching logic behind `catch_aws_error()`.\nIt takes a `ClientError`, with the rest of the arguments being error codes and operation names identical to `catch_aws_error()`, except that it does not support providing a callable.\n\n## `make_aws_error()`\nFor testing, you can create `ClientError`s with the `make_aws_error()` function, which takes the following arguments:\n* `code` (required)\n* `message` (required)\n* `operation_name` (required)\n* `http_status_code`\n* `response`\n\nIf a `response` dict is provided, it will be copied rather than modified.\n",
    'author': 'Ben Kehoe',
    'author_email': 'None',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/benkehoe/aws-error-utils',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4',
}


setup(**setup_kwargs)
