/*
 * Copyright (c) 1997, 1998
 *      Jens Arvid Nilsson, jnilsson@ludd.luth.se. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY 
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*$Id: arpd.c,v 1.17 1998/05/24 14:14:13 jens Exp $*/  
#include <sys/types.h>
#include <sys/fcntl.h>
#include <sys/errno.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/uio.h>
#include <net/if.h>
#include <net/bpf.h>
#include <net/if_dl.h>
#include <net/if_arp.h>
#include <net/route.h>
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/if_ether.h>
#include <netinet/ip.h>
#include <netinet/udp.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <syslog.h>
#include <varargs.h>
#include <pcap.h>
#include "error.h"
#include "arpd.h"
#include "wraplib.h"
#include "util.h"

/* Only accepts ARP requests, see bpf(4) line 625 */
static struct bpf_insn insns[] = {
	BPF_STMT(BPF_LD | BPF_H | BPF_ABS, 12),
	BPF_JUMP(BPF_JMP | BPF_JEQ | BPF_K, ETHERTYPE_ARP, 0, 3),
	BPF_STMT(BPF_LD | BPF_H | BPF_ABS, 20),
	BPF_JUMP(BPF_JMP | BPF_JEQ | BPF_K, ARPOP_REQUEST, 0, 1),
	BPF_STMT(BPF_RET | BPF_K,
	    sizeof(struct arphdr) +
	    2 * ETHER_ADDR_LEN + 2 * sizeof(struct in_addr) +
	    sizeof(struct ether_header)),
	BPF_STMT(BPF_RET | BPF_K, 0),
};

static struct bpf_program filter = {
	sizeof insns / sizeof(insns[0]),
	insns
};


int
Open_bpf(char *szD)
{
	struct ifreq ifr;
	char    buf[20];
	int     n = 0, bs, i;
	u_int   dlt;


	do
		snprintf(buf, 19, "/dev/bpf%d", n++);
	while ((bs = open(buf, O_RDWR)) < 0 && errno == EBUSY);

	if (bs < 0)
		err_quit("Open_bpf: couldn't open bpf");

	strncpy(ifr.ifr_name, szD, sizeof ifr.ifr_name);

	if (ioctl(bs, BIOCSETIF, (caddr_t) & ifr) < 0)
		err_quit("Open_bpf: couldn't BIOCSETIF");

	if (ioctl(bs, BIOCGDLT, (caddr_t) & dlt) < 0)
		err_quit("Open_bpf: couldn't BIOCGDLT");

	if (dlt != DLT_EN10MB)
		err_quit("Open_bpf: %s is not an ethernet", szD);

	/* if (ioctl(bs, BIOCPROMISC, NULL) < 0) err_quit("Open_bpf: couldn't
	 * BIOCPROMISC"); */

	i = 1;
	if (ioctl(bs, BIOCIMMEDIATE, &i) < 0)
		err_quit("Open_bpf: couldn't BIOCIMMEDIATE");

	i = 0;
	if (ioctl(bs, FIONBIO, &i) < 0)
		err_quit("Opend_bpf: couldn't FIONBIO %s", strerror(errno));

	if (ioctl(bs, BIOCSETF, (caddr_t) & filter) < 0)
		err_quit("Open_bpf: couldn't BIOCSETF %s", strerror(errno));

	return bs;
}

#define ARPLEN 42
void
arp_reply(int fd, struct ether_addr *my_ether, struct in_addr *for_ip, struct in_addr *to_ether, struct in_addr *to_ip)
{
	char   *p;
	struct ether_header *ep;
	struct arphdr *ap;

	p = Malloc(ARPLEN + 8);

	ep = (struct ether_header *) BPF_WORDALIGN((long) p);
	ap = (struct arphdr *) (ep + 1);

	memcpy(ep->ether_dhost, to_ether, 6);
	ep->ether_type = htons(ETHERTYPE_ARP);
	ap->ar_hrd = htons(ARPHRD_ETHER);
	ap->ar_pro = htons(ETHERTYPE_IP);
	ap->ar_hln = 6;
	ap->ar_pln = 4;
	ap->ar_op = htons(ARPOP_REPLY);
	memcpy(ar_sha(ap), my_ether, 6);
	memcpy(ar_spa(ap), for_ip, 4);
	memcpy(ar_tha(ap), to_ether, 6);
	memcpy(ar_tpa(ap), to_ip, 4);

#if 0
	printf("ether_dhost = %s\n", ether_ntoa(&ep->ether_dhost));
	printf("ether_shost = %s\n", ether_ntoa(&ep->ether_shost));
	printf("arp_sha = %s\n", ether_ntoa(ar_sha(ap)));
	printf("arp_spa = %s\n", inet_ntoa(ar_spa(ap)));
	printf("arp_tha = %s\n", ether_ntoa(ar_tha(ap)));
	printf("arp_tpa = %s\n", inet_ntoa(ar_tpa(ap)));
#endif

	write(fd, p, ARPLEN);
	free(p);
}


int
arp_check(u_char * p, int len)
{
	struct ether_header *ep = (struct ether_header *) p;
	struct arphdr *ap = (struct arphdr *) (p + sizeof(*ep));

	if (len < sizeof(*ep) + sizeof(*ap)) {
		err_msg("arp_check: truncated request %d", len);
		return 0;
	}
	/* now that we know the fixed part of the ARP hdr is there: */
	if (len < sizeof(*ap) + 2 * ap->ar_hln + 2 * ap->ar_pln) {
		err_msg("arp_check: truncated request %d", len);
		return 0;
	}
	if (ntohs(ep->ether_type) != ETHERTYPE_ARP ||
	    ntohs(ap->ar_hrd) != ARPHRD_ETHER ||
	    ntohs(ap->ar_op) != ARPOP_REQUEST ||
	    ntohs(ap->ar_pro) != ETHERTYPE_IP ||
	    ap->ar_hln != 6 || ap->ar_pln != 4) {
			err_msg("arp_check: request fails sanity check");
			return 0;
	}
	if (memcmp((char *) &ep->ether_shost, ar_sha(ap), 6) != 0) {
		err_msg("arp_check: ether/arp sender address mismatch");
		return 0;
	}
#if 0
	if (memcmp((char *) &ep->ether_dhost, ar_tha(ap), 6) != 0) {
		err_msg("arp_check: ether/arp target address mismatch");
		return 0;
	}
#endif
	return 1;
}

int
main(int argc, char **argv)
{
	void   *buf;
	int     len, max_fd;
	FILE	*pF;
	fd_set  fds, listeners;
	struct bpf_hdr *bp;
	struct ether_header *ep;
	struct arphdr *ap;
	struct if_elm *pIf_elm;

	progname = "arpd"; /* for err_xxx */
	if (argc > 1)
		pF = fopen(argv[1], "r");
	else
		pF = fopen("/usr/local/etc/arpd.conf", "r");
	if (pF == NULL)
		err_sys("fopen");

	Read_conf(pF);
	fclose(pF);
	
	max_fd = 0;
	FD_ZERO(&fds);
	for (pIf_elm = sIf_list.lh_first; pIf_elm != NULL; 
				pIf_elm = pIf_elm->entries.le_next) {
		pIf_elm->fdDevice = Open_bpf(pIf_elm->szDevice);
		FD_SET(pIf_elm->fdDevice, &fds);
		max_fd = max(max_fd, pIf_elm->fdDevice);
	}

	daemon_init(progname, LOG_DAEMON);

	buf = (void *) BPF_WORDALIGN((long) malloc(8000));
	for (;;) {
		listeners = fds;
		if (select(max_fd + 1, &listeners, NULL, NULL, NULL) < 0)
			err_msg("select failed, %s", strerror(errno));

		for (pIf_elm = sIf_list.lh_first; pIf_elm != NULL; 
					pIf_elm = pIf_elm->entries.le_next) {
			if (!FD_ISSET(pIf_elm->fdDevice, &listeners))
				continue;

			len = read(pIf_elm->fdDevice, buf, 4096);

			if (len == -1)
				continue;
			
			bp = buf;

			if (!arp_check(buf + bp->bh_hdrlen, bp->bh_caplen))
				continue;

			ep = (struct ether_header *) (buf + bp->bh_hdrlen);
			ap = (struct arphdr *) (ep + 1);
			if (!ip_pass_rules(	(struct in_addr *) ar_spa(ap), 
								(struct in_addr *) ar_tpa(ap), 
								pIf_elm))
				continue;

			arp_reply(	pIf_elm->fdDevice, 
						(struct ether_addr *) &pIf_elm->ifhwa, 
						(struct in_addr *) ar_tpa(ap), 
						(struct in_addr *) ar_sha(ap), 
						(struct in_addr *) ar_spa(ap));
		}
	}
	exit(0);
}
