// realconsole.cpp
// Revision 14-may-2005

#include "realconsole.h"

#ifndef NO_CURSES

#include <ncurses.h>
#include <term.h>

#endif

#include <termios.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/poll.h>

#include <stdlib.h>
#include <sys/ioctl.h>

// This is from ncurses.
#ifdef TIOCGSIZE
# define IOCTL_WINSIZE TIOCGSIZE
# define STRUCT_WINSIZE struct ttysize
# define WINSIZE_ROWS(n) (int)n.ts_lines
# define WINSIZE_COLS(n) (int)n.ts_cols
#else
# ifdef TIOCGWINSZ
#  define IOCTL_WINSIZE TIOCGWINSZ
#  define STRUCT_WINSIZE struct winsize
#  define WINSIZE_ROWS(n) (int)n.ws_row
#  define WINSIZE_COLS(n) (int)n.ws_col
# endif
#endif

#undef lines
#undef columns

#include <string.h>

#include <string>
#include <map>

#include <assert.h>
#define ASSERT assert

class RealConsole::In
{
public:
	In ();
	~In ();
	void rawin ();
	void rawout ();
	void activate ();
	void deactivate ();
	bool charavailable ();
	unsigned char charin ();
	void charout (unsigned char c);
	void left ();
	void right ();
	unsigned lines ();
	unsigned columns ();
private:
	static bool static_init;
	static void initstatics ();

	bool isactive;
	bool rawmodein;
	bool rawmodeout;
	struct termios termsave;
	bool ischarpending;
	char charpending;
	enum State { StNormal, StEsc, StWaitingYX, StWaitingX };
	State state;
	unsigned char cursory;
};

bool RealConsole::In::static_init= false;

RealConsole::In::In () :
	isactive (false),
	rawmodein (false),
	rawmodeout (false),
	ischarpending (false),
	state (StNormal)
{
	if (! static_init)
	{
		static_init= true;
		initstatics ();
	}
}

RealConsole::In::~In ()
{
	if (isactive)
		deactivate ();
}

void RealConsole::In::rawin ()
{
	rawmodein= true;
}

void RealConsole::In::rawout ()
{
	rawmodeout= true;
}

namespace {

template <class C, size_t N>
size_t dim_array (C (&) [N])
{ return N; }

template <class C, size_t N>
size_t dim_array (const C (&) [N])
{ return N; }


unsigned char charset [256]= {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 
        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 
        0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 
        ' ',  '!',  '"',  '#',  '$',  '%',  '&',  '\'', 
        '(',  ')',  '*',  '+',  ',',  '-',  '.',  '/',  
        '0',  '1',  '2',  '3',  '4',  '5',  '6',  '7',  
        '8',  '9',  ':',  ';',  '<',  '=',  '>',  '?',  
        '@',  'A',  'B',  'C',  'D',  'E',  'F',  'G',  
        'H',  'I',  'J',  'K',  'L',  'M',  'N',  'O',  
        'P',  'Q',  'R',  'S',  'T',  'U',  'V',  'W',  
        'X',  'Y',  'Z',  '[',  '\\', ']',  '^',  '_',  
        '`',  'a',  'b',  'c',  'd',  'e',  'f',  'g',  
        'h',  'i',  'j',  'k',  'l',  'm',  'n',  'o',  
        'p',  'q',  'r',  's',  't',  'u',  'v',  'w',  
        'x',  'y',  'z',  '{',  '|',  '}',  '~',  0x7F, 
        '#',  '+',  '+',  '+',  '+',  '|',  '+',  '+',
        '+',  '+',  '-',  '+',  '+',  '+',  '+',  '+',
        '',  '|',  '-',  '+',  '|',  '|',  '+',  '+',
        '-',  '+',  '-',  '+',  '+',  '+',  '+',  '+', 
        0xA0, 0xA1, 0xA2, 0xA3, 0xA4, 0xA5, 0xA6, 0xA7, 
        0xA8, 0xA9, 0xAA, 0xAB, 0xAC, 0xAD, 0xAE, 0xAF, 
        0xB0, 0xB1, 0xB2, 0xB3, 0xB4, 0xB5, 0xB6, 0xB7, 
        0xB8, 0xB9, 0xBA, 0xBB, 0xBC, 0xBD, 0xBE, 0xBF, 
        0xC0, 0xC1, 0xC2, 0xC3, 0xC4, 0xC5, 0xC6, 0xC7, 
        0xC8, 0xC9, 0xCA, 0xCB, 0xCC, 0xCD, 0xCE, 0xCF, 
        0xD0, 0xD1, 0xD2, 0xD3, 0xD4, 0xD5, 0xD6, 0xD7, 
        0xD8, 0xD9, 0xDA, 0xDB, 0xDC, 0xDD, 0xDE, 0xDF, 
        0xE0, 0xE1, 0xE2, 0xE3, 0xE4, 0xE5, 0xE6, 0xE7, 
        0xE8, 0xE9, 0xEA, 0xEB, 0xEC, 0xED, 0xEE, 0xEF, 
        0xF0, 0xF1, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xF7, 
        0xF8, 0xF9, 0xFA, 0xFB, 0xFC, 0xFD, 0xFE, 0xFF,
};

inline void docharoutraw (unsigned char c)
{
	write (STDOUT_FILENO, & c, 1);
}

inline void docharout (unsigned char c)
{
	docharoutraw (charset [c] );
}

#ifndef NO_CURSES

// Term related variables and functions.

const char
	* strKeypadXmit= NULL,    * strKeypadLocal= NULL,
	* strCls= NULL,           * strCup= NULL,
	* strCursorNormal= NULL,  * strCursorInvisible= NULL,
	* strForeground= NULL,    * strBackground= NULL,
	* strEnterBold= NULL,     * strExitBold= NULL,
	* strMoveForward= NULL,   * strMoveBack= NULL,
	* strMoveForwardN= NULL,  * strMoveBackN= NULL,
	* strMoveUp= NULL,        * strMoveDown= NULL,
	* strMoveUpN= NULL,       * strMoveDownN= NULL,
	* strSaveCursorPos= NULL, * strRestoreCursorPos= NULL,
	* strClrEos= NULL,        * strClrEol= NULL,
	* strInsertLine= NULL,
	* strDeleteLine= NULL,
	* strSmam= NULL,          * strRmam= NULL,
	* strBell= NULL;

const char * newstr (const char * str)
{
	if (str == NULL)
		return NULL;
	size_t l= strlen (str);
	char *n= new char [l + 1];
	strcpy (n, str);
	return n;
}

inline const char * calltigetstr (const char * id)
{
	const char * str= tigetstr ( (char *) id);
	if (str == (char *) -1)
		return NULL;
	return str;
}

inline const char * mytigetstr (const char * id)
{
        return newstr (calltigetstr (id) );
}

int putfunc (int ic)
{
	docharoutraw (static_cast <unsigned char> (ic) );
	return 0;
}

inline void calltputs (const char * str)
{
	if (str != NULL)
		tputs (str, 1, putfunc);
}

inline void calltparm (const char * str, int n)
{
	if (str != NULL)
	{
		calltputs (tparm ( (char *) str, n) );
	}
}

struct str_terminfo {
	const char * & str;
	const char * tinfoname;
        str_terminfo (const char * & str, const char * tinfoname) :
                str (str), tinfoname (tinfoname)
        { }
};

const str_terminfo strinfo []= {
	str_terminfo (strKeypadXmit, "smkx"),
	str_terminfo (strKeypadLocal, "rmkx"),

	str_terminfo (strCls, "clear" ),
	str_terminfo (strCup, "cup" ),

	str_terminfo (strCursorNormal, "cnorm" ),
	str_terminfo (strCursorInvisible, "civis" ),

	str_terminfo (strForeground, "setaf" ),
	str_terminfo (strBackground, "setab" ),

	str_terminfo (strEnterBold, "bold" ),
	str_terminfo (strExitBold, "sgr0" ),

	str_terminfo (strMoveForward, "cuf1" ),
	str_terminfo (strMoveBack, "cub1" ),
	str_terminfo (strMoveForwardN, "cuf" ),
	str_terminfo (strMoveBackN, "cub" ),
	str_terminfo (strMoveUp, "cuu1" ),
	str_terminfo (strMoveDown, "cud1" ),
	str_terminfo (strMoveUpN, "cuu" ),
	str_terminfo (strMoveDownN, "cud" ),

	str_terminfo (strSaveCursorPos, "sc" ),
	str_terminfo (strRestoreCursorPos, "rc" ),

	str_terminfo (strClrEos, "ed"),
	str_terminfo (strClrEol, "el"),

	str_terminfo (strInsertLine, "il1"),
	str_terminfo (strDeleteLine, "dl1"),

	str_terminfo (strSmam, "smam"),
	str_terminfo (strRmam, "rmam"),

	str_terminfo (strBell, "bel"),
};

void cls ()
{
	calltputs (strCls);
}

void gotoxy (int x, int y)
{
	calltputs (tgoto (strCup, x, y) );
}

void home ()
{
	gotoxy (0, 0);
}

void cursorup ()
{
	calltputs (strMoveUp);
}

void cursordown ()
{
	calltputs (strMoveDown);
}

void cursorleft ()
{
	calltputs (strMoveBack);
}

void cursorright ()
{
	calltputs (strMoveForward);
}

void savecursorpos ()
{
	calltputs (strSaveCursorPos);
}

void restorecursorpos ()
{
	calltputs (strRestoreCursorPos);
}

void showcursor ()
{
	calltputs (strCursorNormal);
}

void hidecursor ()
{
	calltputs (strCursorInvisible);
}

void clreol ()
{
	calltputs (strClrEol);
}

void clreos ()
{
	calltputs (strClrEos);
}

void insertline ()
{
	calltputs (strInsertLine);
}

void deleteline ()
{
	calltputs (strDeleteLine);
}

class MapSpecial {
public:
	enum Result { NoMapped, Found, MoreNeeded };
	void addkey (const std::string & str, std::string::size_type pos,
		const std::string & keyvalue)
	{
		//TraceFunc tr ("MapSpecial::addkey");

		//ASSERT (pos < str.size () );
		char c= str [pos];
		if (pos == str.size () - 1)
			kname [c]= keyvalue;
		else
		{
			//if (kmap.find (c) == kmap.end () )
			//	kmap [c]= MapSpecial ();
			kmap [c].addkey (str, pos + 1, keyvalue);
		}
	}
	Result findkey (const std::string & str, std::string::size_type pos,
		std::string & keyvalue, std::string::size_type & consumed)
	{
		if (pos >= str.size () )
			return MoreNeeded;
		char c= str [pos];
		//cout << "Buscando: " << c << endl;
		{
			kname_t::iterator it=
				kname.find (c);
			if (it != kname.end () )
			{
				keyvalue= it->second;
				consumed= pos;
				return Found;
			}
		}
		std::map <char, MapSpecial>::iterator it= kmap.find (c);
		if (it != kmap.end () )
			return it->second.findkey
				(str, pos + 1, keyvalue, consumed);
		else
			return NoMapped;
	}
private:
	typedef std::map <char, std::string> kname_t;
	kname_t kname;
	std::map <char, MapSpecial> kmap;
};

MapSpecial ms;

struct KeyDescription {
	const char * tiId;
	const char * keyvalue;
	KeyDescription (const char * tiId, const char * keyvalue) :
		tiId (tiId),
		keyvalue (keyvalue)
	{ }
};

const KeyDescription keyname [] = {
	KeyDescription ("kpp",   "\x20"),  // previous-page key
	KeyDescription ("knp",   "\x20"),  // next-page key

	KeyDescription ("kend",  "\x20"),  // end key
	KeyDescription ("kslt",  "\x20"),  // select key
	KeyDescription ("kc1",   "\x20"),  // lower left of keypad
	KeyDescription ("khome", "\x20"),  // home key
	KeyDescription ("kfnd",  "\x20"),  // find key
	KeyDescription ("ka1",   "\x20"),  // upper left of keypad

	KeyDescription ("kcub1", "\x01"),  // left-arrow key
	KeyDescription ("kcuu1", "\x1F"),  // up-arrow key
	KeyDescription ("kcuf1", "\x06"),  // right-arrow key
	KeyDescription ("kcud1", "\x1E"),  // down-arrow key

	KeyDescription ("kich1", "\x15"),  // insert-character key
	KeyDescription ("kdch1", "\x07"),  // delete-character key
	KeyDescription ("kent",  "\x20"),  // enter/send key

	KeyDescription ("kf1",   "\x1A"),  // F1 function key
	KeyDescription ("kf2",   "\x1A"),  // F2 function key
	KeyDescription ("kf3",   "\x11"),  // F3 function key
	KeyDescription ("kf4",   "\x11"),  // F4 function key
	KeyDescription ("kf5",   "\x13"),  // F5 function key
	KeyDescription ("kf6",   "\x13"),  // F6 function key
	KeyDescription ("kf7",   "\x10"),  // F7 function key
	KeyDescription ("kf8",   "\x10"),  // F8 function key
	KeyDescription ("kf9",   "\x20"),  // F9 function key
	KeyDescription ("kf10",  "\x20"),  // F10 function key
	KeyDescription ("kf11",  "\x16"),  // F11 function key
	KeyDescription ("kf12",  "\x1C"),  // F12 function key
	KeyDescription ("kf54",  "\x20"),  // F54 function key, / in xterm
	KeyDescription ("kf55",  "\x20"),  // F55 function key, * in xterm
	KeyDescription ("kf56",  "\x20"),  // F56 function key, - in xterm
	KeyDescription ("kf57",  "\x20"),  // f57 function key, + in xterm
};

const KeyDescription keyws []= {
	KeyDescription ("kpp",   "\x12"),  // previous-page key
	KeyDescription ("knp",   "\x03"),  // next-page key

	KeyDescription ("kend",  "\x11" "D"),  // end key
	KeyDescription ("khome", "\x11" "S"),  // home key

	KeyDescription ("kbs",   "\x08"),   // Backspece key

	KeyDescription ("kcub1", "\x13"),  // left-arrow key
	KeyDescription ("kcuu1", "\x05"),  // up-arrow key
	KeyDescription ("kcuf1", "\x04"),  // right-arrow key
	KeyDescription ("kcud1", "\x18"),  // down-arrow key

};

void setkeytable (const KeyDescription * keyarray, size_t nkeys)
{
	for (size_t i= 0; i < nkeys; ++i)
	{
		const KeyDescription & keydesc= keyarray [i];
		const char * const strkey= keydesc.tiId;
		const char * str= calltigetstr (strkey);
		if (str != NULL)
		{
			//tr.message (std::string ("Adding ") + keydesc.blName);
			ms.addkey (str, 0, keydesc.keyvalue);
		}
	}
}

void initkeytable ()
{
	#if 0
	const size_t nkeys= sizeof (keyname) / sizeof (* keyname);

	for (size_t i= 0; i < nkeys; ++i)
	{
		const KeyDescription & keydesc= keyname [i];
		const char * const strkey= keydesc.tiId;
		const char * str= calltigetstr (strkey);
		if (str != NULL)
		{
			//tr.message (std::string ("Adding ") + keydesc.blName);
			ms.addkey (str, 0, keydesc.keyvalue);
		}
	}
	#else
	setkeytable (keyname, dim_array (keyname) );
	#endif
}

void setwskeys ()
{
	setkeytable (keyws, dim_array (keyws) );
}

#endif
// NO_CURSES

class PollInput {
public:
	PollInput ()
	{
		pfd.fd= STDIN_FILENO;
		pfd.events= POLLIN | POLLERR | POLLNVAL;
	}
	int poll ()
	{
		return ::poll (& pfd, 1, 100);
	}
private:
	struct pollfd pfd;
};

void do_poll ()
{
	PollInput ().poll ();
}

enum ReadType { ReadWait, ReadNoWait };

#ifndef NO_CURSES
bool readkey (ReadType type, char & result, bool rawmode)
#else
bool readkey (ReadType type, char & result, bool)
#endif
{
	static std::string outputpending;
	static std::string charpending;
        std::string str;
	bool reset_blocking_mode= false;
	int l;
	const int lbuf= 32;
	char buffer [lbuf + 1];

	if (! outputpending.empty () )
		goto returnchar;

	if (! charpending.empty () )
		goto check_it;

	fcntl (STDIN_FILENO, F_SETFL, O_NONBLOCK);
	reset_blocking_mode= true;
	l= read (STDIN_FILENO, buffer, lbuf);
	if (l < 1 && type == ReadWait)
	{
		do
		{
			do_poll ();
			l= read (STDIN_FILENO, buffer, lbuf);
		} while (l < 1);
	}
	if (l >= 1)
	{
		buffer [l]= '\0';
		str+= buffer;
	}

	#ifndef NO_CURSES
	read_another:
	#endif

	charpending+= str;
	str.erase ();

	check_it:

	#ifndef NO_CURSES
	std::string::size_type pos;
	#endif

	if (charpending.empty () )
	{
		if (reset_blocking_mode)
			fcntl (STDIN_FILENO, F_SETFL, 0);
		return false;
	}
	else
	{
		#ifndef NO_CURSES

		MapSpecial::Result r= MapSpecial::NoMapped;

		std::string found;
		if (! rawmode)
			r= ms.findkey (charpending, 0, found, pos);

		switch (r)
		{
		case MapSpecial::NoMapped:
			//str= charpending [0];
			//result= charpending [0];
			outputpending+= charpending [0];
			charpending.erase (0, 1);
			break;
		case MapSpecial::Found:
			//str= keyname;
			outputpending+= found;
			charpending.erase (0, pos + 1);
			break;
		case MapSpecial::MoreNeeded:
			fcntl (STDIN_FILENO, F_SETFL, O_NONBLOCK);
			reset_blocking_mode= true;
			do_poll ();
			l= read (STDIN_FILENO, buffer, lbuf);
			if (l >= 1)
			{
				buffer [l]= '\0';
				str= buffer;
				goto read_another;
			}
			//str= charpending [0];
			//result= charpending [0];
			outputpending+= charpending [0];
			charpending.erase (0, 1);
			break;
		}
		#else

		//result= charpending [0];
		//charpending.erase (0, 1);
		outputpending+= charpending;
		charpending.erase ();

		#endif
	}

        //if (type == ReadWait)
        //        cursorinvisible ();

returnchar:

	if (reset_blocking_mode)
		fcntl (STDIN_FILENO, F_SETFL, 0);

        //return str;
	//return true;

	ASSERT (! outputpending.empty () );
	result= outputpending [0];
	outputpending.erase (0, 1);
	return true;
}

} // namespace

void RealConsole::In::initstatics ()
{
	#ifndef NO_CURSES

	int erret;
	// Initilize terminal using the value of TERM,
	// using the standard output.
	if (setupterm (0, STDOUT_FILENO, & erret) != OK)
		return;

	for (size_t i= 0; i < sizeof (strinfo) / sizeof (* strinfo); ++i)
		strinfo [i].str= mytigetstr (strinfo [i].tinfoname);
	initkeytable ();

	#endif
}

void RealConsole::wskeys ()
{
	#ifndef NO_CURSES
	setwskeys ();
	#endif
}

void RealConsole::In::activate ()
{
	if (isactive)
		return;

	isactive= true;

	// Save state to restore it on deactivate.
	tcgetattr (STDIN_FILENO, & termsave);

	// Load state, modify and stablihs.
	struct termios term;
	tcgetattr (STDIN_FILENO, & term);

	// No traducir CR y LF en la entrada.
	term.c_iflag&= ~ (INLCR | ICRNL);
	// No traducir CR y LF en la salida.
	term.c_oflag&= ~ (ONLCR | OCRNL | ONOCR | ONLRET);
	// Sin echo, modo no cannico, sin interrupciones.
	term.c_lflag&= ~ (ECHO | ICANON | ISIG);
	// Esperar slo por un caracter.
	term.c_cc [VMIN]= 1;

	tcsetattr (STDIN_FILENO, TCSANOW, & term);

	#ifndef NO_CURSES

	// TERM related activation.

	if (isatty (STDOUT_FILENO) )
		calltputs (strKeypadXmit);

	#endif
}

void RealConsole::In::deactivate ()
{
	if (! isactive)
		return;

	isactive= false;

	// Restore saved state.

	tcsetattr (STDIN_FILENO, TCSANOW, & termsave);

	// TERM related deactivation.

	#ifndef NO_CURSES

	if (isatty (STDOUT_FILENO) )
		calltputs (strKeypadLocal);

	#endif
}

bool RealConsole::In::charavailable ()
{
	// Read a character without blocking.

	char c;
	if (readkey (ReadNoWait, c, rawmodein) )
	{
		ischarpending= true;
		charpending= c;
		return true;
	}
	return false;
}

unsigned char RealConsole::In::charin ()
{
	if (ischarpending)
	{
		ischarpending= false;
		return charpending;
	}
	char c;
	readkey (ReadWait, c, rawmodein);
	return c;
}

void RealConsole::In::charout (unsigned char c)
{
	#ifdef NO_CURSES

	docharout (c);

	#else

	if (rawmodeout)
	{
		docharoutraw (c);
		return;
	}
	switch (state)
	{
	case StNormal:
		switch (c)
		{
		case 0x07:
			calltputs (strBell);
			break;
		case 0x1B:
			state= StEsc;
			break;
		default:
			docharout (c);
		}
		break;
	case StEsc:
		switch (c)
		{
		case 'A': // Cursor up
			cursorup ();
			state= StNormal;
			break;
		case 'B': // Cursor down
			cursordown ();
			state= StNormal;
			break;
		case 'C': // Cursor right
			cursorright ();
			state= StNormal;
			break;
		case 'D': // Cursor left
			cursorleft ();
			state= StNormal;
			break;
		case 'E': // Clear page.
			cls ();
			state= StNormal;
			break;
		case 'H': // Cursor home.
			home ();
			state= StNormal;
			break;
		case 'J': // Clear to end of screen
			clreos ();
			state= StNormal;
			break;
		case 'K': // Clear to end of line
			clreol ();
			state= StNormal;
			break;
		case 'L': // Insert line
			insertline ();
			state= StNormal;
			break;
		case 'M': // Delete Line
			deleteline ();
			state= StNormal;
			break;
		case 'Y': // Cursor to line, col
			state= StWaitingYX;
			break;
		case 'e': // Show cursor
			showcursor ();
			state= StNormal;
			break;
		case 'f': // Hide cursor
			hidecursor ();
			state= StNormal;
			break;
		case 'j': // Save cursor position
			savecursorpos ();
			state= StNormal;
			break;
		case 'k': // Restore cursor position
			restorecursorpos ();
			state= StNormal;
			break;
		case 'p': // Set inverse mode.
			// Ignored
			state= StNormal;
			break;
		case 'q': // Unset inverse mode.
			// Ignored
			state= StNormal;
			break;
		case 'v': // Activate auto wrap lines.
			calltputs (strSmam);
			state= StNormal;
			break;
		case 'w': // Deactivate auto wrap lines.
			calltputs (strRmam);
			state= StNormal;
			break;
		case 'x': // Set 24x80 mode.
			cls ();
			state= StNormal;
			break;
		case 'y': // Unset 24x80 mode.
			cls ();
			state= StNormal;
			break;
		default:
			if (c < 0x20)
				docharout ('#');
			else
			{
				docharout ('[');
				docharout ('E');
				docharout ('S');
				docharout ('C');
				docharout (']');
				docharout (c);
			}
			state= StNormal;
		}
		break;
	case StWaitingYX:
		cursory= c - 0x20;
		state= StWaitingX;
		break;
	case StWaitingX:
		gotoxy (c - 0x20, cursory);
		state= StNormal;
		break;
	default:
		throw "Unexpected error";
	}

	#endif
}

void RealConsole::In::left ()
{
	#ifndef NO_CURSES
	cursorleft ();
	#endif
}

void RealConsole::In::right ()
{
	#ifndef NO_CURSES
	cursorright ();
	#endif
}

unsigned RealConsole::In::lines ()
{
	STRUCT_WINSIZE win;
	if (ioctl (0, IOCTL_WINSIZE, & win) == 0)
		return WINSIZE_ROWS (win);
	const char * aux= getenv ("LINES");
	if (aux)
		return strtoul (aux, 0, 10);
	return 25;
}

unsigned RealConsole::In::columns ()
{
	STRUCT_WINSIZE win;
	if (ioctl (0, IOCTL_WINSIZE, & win) == 0)
		return WINSIZE_COLS (win);
	const char * aux= getenv ("COLUMNS");
	if (aux)
		return strtoul (aux, 0, 10);
	return 25;
}

//***********************************************
//		RealConsole
//***********************************************

RealConsole::RealConsole () :
	p (new In)
{
}

RealConsole::~RealConsole ()
{
	delete p;
}

void RealConsole::rawin ()
{
	p->rawin ();
}

void RealConsole::rawout ()
{
	p->rawout ();
}

void RealConsole::activate ()
{
	p->activate ();
}

void RealConsole::deactivate ()
{
	p->deactivate ();
}

bool RealConsole::charavailable ()
{
	return p->charavailable ();
}

unsigned char RealConsole::charin ()
{
	return p->charin ();
}

void RealConsole::charout (unsigned char c)
{
	p->charout (c);
}

void RealConsole::left ()
{
	p->left ();
}

void RealConsole::right ()
{
	p->right ();
}

unsigned RealConsole::lines ()
{
	return p->lines ();
}

unsigned RealConsole::columns ()
{
	return p->columns ();
}

// End of realconsole.cpp
